/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GMessageBox.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/GGraphics.h"
#include "glib/gui/GDialogFrame.h"
#include "glib/gui/event/GDialogMessage.h"
#include "glib/primitives/GCharacter.h"
#include "glib/util/GMath.h"
#include "glib/util/GLog.h"
#include "glib/GProgram.h"
#include "glib/sys/GSystemInfo.h"
#include "glib/sys/GSystem.h"

GMessageBox::GMessageBox ( GMessageBox::Type type,
                           const GString& msg,
                           const GString& title,
                           const GString& flags,
                           bool monoFont,
                           const GString& userText1,
                           const GString& userText2,
                           const GString& userText3 )
            :iNrOfButtons(0),
             defaulBt(-1),
             cancelBt(-1),
             btTexts(6),
             btIDs(6),
             buttons(6),
             timeOutSeconds(0),
             doBeep(false),
             dlgFrame(null),
             hasDetails(false)
{
   GString strMessage = msg;
   GString titleStr = title;

   GString flagsStr = flags;
   if (flagsStr == "")
      flagsStr = "O";

   GString defaultTitleStr = "";
   switch (type)
   {
      case TYPE_QUESTION:
           flagsStr = "2" + flagsStr;
           defaultTitleStr = GProgram::LoadText("%MSG_TITLE_QUESTION");
           break;

      case TYPE_ERROR:
           flagsStr = "3" + flagsStr;
           defaultTitleStr = GProgram::LoadText("%MSG_TITLE_ERROR");
           break;

      case TYPE_WARNING:
           flagsStr = "4" + flagsStr;
           defaultTitleStr = GProgram::LoadText("%MSG_TITLE_WARNING");
           break;

      case TYPE_CRITICAL_ERROR:
           flagsStr = "6" + flagsStr;
           defaultTitleStr = GProgram::LoadText("%MSG_TITLE_CRITERROR");
           break;

      case TYPE_INFO:
      case TYPE_LOGO:
      case TYPE_NONE:
      default:
           flagsStr = "5" + flagsStr;
           defaultTitleStr = GProgram::LoadText("%MSG_TITLE_INFO");
           break;
   }

   if (titleStr == "")
      titleStr = defaultTitleStr;

   strMessage = GProgram::LoadText(strMessage);
   titleStr = GProgram::LoadText(titleStr);

   this->titleStr = titleStr;

   for (int i=0; i<flagsStr.length(); i++)
   {
      char nextFlag = flagsStr[i];
      switch (nextFlag)
      {
         case '1': //
              iconID = "ICON_INFORMATION";
              break;

         case '2': //
              iconID = "ICON_QUESTION";
              break;

         case '3': //
              iconID = "ICON_ERROR";
              break;

         case '4': //
              iconID = "ICON_WARNING";
              break;

         case '5': //
              iconID = "ICON_INFORMATION";
              break;

         case 'A': // Abort button
              defaulBt = iNrOfButtons;
         case 'a': // Abort button
              btIDs.add(new GInteger(Answer(IDABORT)));
              btTexts.add(new GString(GProgram::LoadText("%DLG_ABORT")));
              iNrOfButtons++;
              break;

         case 'R': // Retry button
              defaulBt = iNrOfButtons;
         case 'r': // Retry button
              btIDs.add(new GInteger(Answer(IDRETRY)));
              btTexts.add(new GString(GProgram::LoadText("%DLG_RETRY")));
              iNrOfButtons++;
              break;

         case 'F': // Retry button
              defaulBt = iNrOfButtons;
         case 'f': // Retry button
              btIDs.add(new GInteger(Answer(IDFAIL)));
              btTexts.add(new GString(GProgram::LoadText("%DLG_FAIL")));
              iNrOfButtons++;
              break;

         case 'I': // Ignore button
              defaulBt = iNrOfButtons;
         case 'i': // Ignore button
              btIDs.add(new GInteger(Answer(IDIGNORE)));
              btTexts.add(new GString(GProgram::LoadText("%DLG_IGNORE")));
              iNrOfButtons++;
              break;

         case 'H': // Help button
              defaulBt = iNrOfButtons;
         case 'h': // Help button
              btIDs.add(new GInteger(Answer(IDHELP)));
              btTexts.add(new GString(GProgram::LoadText("%DLG_HELP")));
              iNrOfButtons++;
              break;

         case 'O': // OK Button
              defaulBt = iNrOfButtons;
         case 'o': // OK Button
              btIDs.add(new GInteger(Answer(IDOK)));
              btTexts.add(new GString(GProgram::LoadText("%DLG_OK")));
              iNrOfButtons++;
              break;

         case 'C': // Cancel button
              defaulBt = iNrOfButtons;
         case 'c': // Cancel button
              btIDs.add(new GInteger(Answer(IDCANCEL)));
              btTexts.add(new GString(GProgram::LoadText("%DLG_CANCEL")));
              cancelBt = iNrOfButtons;
              iNrOfButtons++;
              break;

         case 'Y': // Yes button
              defaulBt = iNrOfButtons;
         case 'y': // Yes button
              if (i < flagsStr.length()-1 && flagsStr[i+1] == '!')
              {
                 i++;
                 btIDs.add(new GInteger(Answer(IDYESTOALL)));
                 btTexts.add(new GString(GProgram::LoadText("%DLG_YESTOALL")));
              }
              else
              {
                 btIDs.add(new GInteger(Answer(IDYES)));
                 btTexts.add(new GString(GProgram::LoadText("%DLG_YES")));
              }
              iNrOfButtons++;
              break;

         case 'N': // No button
              defaulBt = iNrOfButtons;
         case 'n': // No button
              btIDs.add(new GInteger(Answer(IDNO)));
              btTexts.add(new GString(GProgram::LoadText("%DLG_NO")));
              iNrOfButtons++;
              break;

         case 'D': // Details button
              defaulBt = iNrOfButtons;
         case 'd': // Details button
              btIDs.add(new GInteger(Answer(IDDETAILS)));
              btTexts.add(new GString(GProgram::LoadText("%DLG_DETAILS")));
              iNrOfButtons++;
              hasDetails = true;
              break;

         case 'S': // Skip button
              defaulBt = iNrOfButtons;
         case 's': // Skip button
              if (i < flagsStr.length()-1 && flagsStr[i+1] == '!')
              {
                 i++;
                 btIDs.add(new GInteger(Answer(IDSKIPALL)));
                 btTexts.add(new GString(GProgram::LoadText("%DLG_SKIPALL")));
              }
              else
              {
                 btIDs.add(new GInteger(Answer(IDSKIP)));
                 btTexts.add(new GString(GProgram::LoadText("%DLG_SKIP")));
              }
              iNrOfButtons++;
              break;

         case '(': // One of the pharantesized multi-character tags.
         {
            // Find the ending pharantesis, and track all characters between.
            GString tagChars(3);
            for (i++; i<flagsStr.length()-1 && flagsStr[i] != ')'; i++)
               tagChars += flagsStr[i];
            if (tagChars.length() <= 0)
               break;
            if (tagChars == "u1" || tagChars == "U1")
            {
               if (tagChars == "U1")
                  defaulBt = iNrOfButtons;
               btIDs.add(new GInteger(Answer(IDUSER1)));
               btTexts.add(new GString(GProgram::LoadText(userText1)));
               iNrOfButtons++;
            }
            else
            if (tagChars == "u2" || tagChars == "U2")
            {
               if (tagChars == "U2")
                  defaulBt = iNrOfButtons;
               btIDs.add(new GInteger(Answer(IDUSER2)));
               btTexts.add(new GString(GProgram::LoadText(userText2)));
               iNrOfButtons++;
            }
            else
            if (tagChars == "u3" || tagChars == "U3")
            {
               if (tagChars == "U3")
                  defaulBt = iNrOfButtons;
               btIDs.add(new GInteger(Answer(IDUSER3)));
               btTexts.add(new GString(GProgram::LoadText(userText3)));
               iNrOfButtons++;
            }
            else
            {
               gthrow_(GIllegalArgumentException(GString("Illegal flags: \"%s\"", GVArgs(flags))));
            }
            break;
         }

         case 'b':
         case 'B': // Beep
              doBeep = true;
              break;

         case 't':
         case 'T': { // Timeout
              int startIndex = i;
              GString str(5);
              if (i < flagsStr.length() - 1)
              {
                 char digit = flagsStr[++i];
                 for (;;)
                 {
                    if (!GCharacter::IsDigit(digit))
                    {
                       i -= 1;
                       break;
                    }

                    str += digit;
                    if (++i >= flagsStr.length())
                       break;
                    else
                       digit = flagsStr[i];
                 }
              }
              try {
                 timeOutSeconds = GInteger::ParseInt(str);
              } catch (GNumberFormatException& e) {
                 GLog::Log(this, "Illegal timeout string: \"%s\". Defaults to 60 seconds. Message is: %s", GVArgs(flagsStr.substring(startIndex)).add(e.toString()));
                 // Default timeout after 60 seconds. Just to prevent
                 // the box from never close even if program expects
                 // it to close automatically.
                 timeOutSeconds = 60;
              }
              break; }

         default:
              gthrow_(GIllegalArgumentException(GString("Illegal flags: \"%s\"", GVArgs(flags))));
      }
   }

   // Make the first button default, by default
   if (iNrOfButtons > 0 && defaulBt <= -1)
      defaulBt = iNrOfButtons - 1;

   for (int i=0; i<iNrOfButtons; i++)
      buttons.add(new GString(btTexts[i]));

   if (hasDetails)
      detailsStr = userText1;

   // Find the width of the widest button, minimum 12 characters.
   int buttonWidth = 12;
   for (int i2=0; i2<iNrOfButtons; i2++)
   {
      int sl = btTexts[i2].length();
      buttonWidth = GMath::Max(buttonWidth, sl);
   }

   int xmarg = 3;
   int widthOfIconArea = 8;
   int widthOfTextArea = GMath::Min(50, strMessage.length());
   int minDlgWidth = xmarg + widthOfIconArea + widthOfTextArea + 6;
   double dlgWidth = (xmarg*2) + (iNrOfButtons*(buttonWidth+2));
   if (dlgWidth < minDlgWidth)
      dlgWidth = minDlgWidth;
   else
      widthOfTextArea = int(dlgWidth - widthOfIconArea - (2*xmarg) - 2);

   // Create the dialog as a dynamic dialog box.
   GProgram& prg = GProgram::GetProgram();
   aptr<GDialogFrame> dlgAPtr = prg.makeDialog(GString::Empty, this);
   dlgFrame = dlgAPtr.release();
   GDialogPanel& dlg = dlgFrame->getDialogPanel();
   dlg.setTitleText(titleStr);
   if (iNrOfButtons == 1)
   {
      // If we have one and just one single button, selecting the 
      // "close window" command will always emulate that single button.
      dlgFrame->setEnableCloseButton(true);
      cancelBt = 0; // Use the first (and only) button as the cancel/escape button.
      defaulBt = 0; // Use it as the default button as well.
   }
   else
   if (cancelBt <= -1)
   {
      // Disable the close-button, since we have no cancel-button.
      dlgFrame->setEnableCloseButton(false);
   }

   GString str = GTextResource::MakeTextLiteral(strMessage);
   GString textCompScript("statictext(id=Txt text=\"%s\" nohotkey top left wordwrap)", GVArgs(str));
   GWindow& textWin = dlg.addComponent(textCompScript);
   if (monoFont)
      textWin.setFontNameSize(GDialogFrame::DefaultMonoFont);

   // Calculate number of text message lines. Try to be smart enough to
   // give the message box a good looking size.
   int numLines;
   GGraphics g(textWin);
   for (;;)
   {
      GArray<GString> strings(16);
      int maxWidth = ((widthOfTextArea-1) * dlg.getFontWidth()) - 1;
      g.getWordWrappedText(strMessage, maxWidth, strings);
      numLines = strings.getCount();
      if ((numLines > 6 && widthOfTextArea < 60) || 
          (numLines > 10 && widthOfTextArea < 80))
      {
         widthOfTextArea += 10;
         dlgWidth += 10;
      }
      else
         break;
   }

   // Calculate the height of the message box.
   GComponentSize dlgSize = dlg.convertToCharDimension(0, g.getFontHeight() * numLines);
   double dlgHeight = 2.9 + dlgSize.height;
   if (iconID == "")
   {
      if (dlgHeight < 3.5)
         dlgHeight = 3.5;
   }
   else
   {
      if (dlgHeight < 4)
         dlgHeight = 4;
   }

   // Set the calculated size of the message box.
   dlg.setDialogSize(dlgWidth, dlgHeight);
   dlg.setComponentSize("Txt", widthOfTextArea + 6, dlgHeight - 2.5);
   dlg.setComponentPos("Txt", xmarg + widthOfIconArea, 1.7);

   // ---
   if (iconID != "")
   {
      GVArgs args;
      args.add(xmarg).add(numLines <= 3 ? 1.3f : 1.7f).add(iconID);
      GString script("staticicon(xpos=%d ypos=%f width=7 height=2.2 icon=%s)", args);
      dlg.addComponent(script);
      dlg.setIcon(iconID);
   }

   // Center the buttons in the bottom part of the message box.
   double xpos = (dlgWidth/2) - (((iNrOfButtons*(buttonWidth+2))-2)/2) + 1;
   for (int i=0; i<iNrOfButtons; i++)
   {
      const GString& id = buttons[i];
      GVArgs args;
      args.add(id).add(xpos).add(dlgHeight-0.6).add(buttonWidth).add(i == defaulBt ? "default" : "").add(i == cancelBt ? "escape" : "");
      GString script("pushbutton(id=\"%s\" xpos=%f ypos=%f height=1.3 width=%d %s %s)", args);
      GWindow& butt = dlg.addComponent(script);
      xpos += buttonWidth + 2;
      butt.setText(btTexts[i]);
      butt.setUserData(new GInteger(i), true);
   }
}

GMessageBox::~GMessageBox ()
{
   delete dlgFrame;
}

GMessageBox::Answer GMessageBox::show ( GWindow* ownerWin )
{
   GString dismissArg = dlgFrame->executeModal(ownerWin);
   if (dismissArg == ":Timer!")
      return IDTIMEOUT;
   GDialogPanel& dlg = dlgFrame->getDialogPanel();
   GWindow& butt = dlg.getComponentByID(dismissArg);
   GInteger* data = (GInteger*) butt.getUserData();
   int index = data->intValue();
   return Answer(btIDs[index].intValue());
}

bool GMessageBox::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
      {
         if (defaulBt >= 0)
            dlg.setComponentFocus(buttons[defaulBt]);
         if (doBeep)
            GSystem::Beep();
         startTimeStamp = GSystem::CurrentTimeMillis();
         if (timeOutSeconds > 0)
         {
            dlg.startTimer("TIMER1HZ", 1000);
            // Explicitly send the first timer event so that the seconds 
            // remaining is displayed immediately.
            dlg.sendDialogMessage(GM_TIMER, "TIMER1HZ");
         }
         return true;
      }

      case GM_DISMISSDIALOG:
      {
         if (timeOutSeconds > 0)
            dlg.stopTimer("TIMER1HZ");
         return true;
      }

      case GM_COMMAND:
      {
         const GString& cmd = msg.getParam1String();         
         GWindow& butt = dlg.getComponentByID(cmd);
         GInteger* userData = dynamic_cast<GInteger*>(butt.getUserData());
         if (userData != null && btIDs[userData->intValue()] == IDDETAILS && detailsStr != "")
         {
            // The user want to se the details text.
            // we must enlarge the dialog and show the text area of where 
            // to display the details text.
            butt.setEnabled(false);
            GDialogFrame& oframe = dlg.getOwnerFrame();
            GComponentSize fsize = oframe.getDialogSize();
            fsize.width = GMath::Max(fsize.width, 80.0);
            GString teDef("multilineedit(id=Details xpos=3.0 ypos=%f width=%f height=6.6 readonly hidden)", GVArgs(fsize.height+1).add(fsize.width-3));
            GWindow& mle = dlg.addComponent(teDef);
            mle.setText(detailsStr);
            mle.setVisible(true);
            oframe.setDialogSize(fsize.width, fsize.height + 7);
            mle.grabFocusAsynchronously();
         }
         else
         {
            dlg.dismiss(cmd);
         }
         return true;
      }

      case GM_TIMER:
      {
         GString id = msg.getParam1String();
         if (id == "TIMER1HZ")
         {
            ulonglong tsNow = GSystem::CurrentTimeMillis();
            int secondsDiff = int((tsNow - startTimeStamp) / 1000);
            int remaining = timeOutSeconds - secondsDiff;
            if (remaining <= 0)
            {
               dlg.dismiss(":Timer!");
            }
            else
            if (defaulBt >= 0) // Will always be true if message box has at leat one button.
            {
               GString defID = buttons[defaulBt];
               GWindow& defButt = dlg.getComponentByID(defID);
               GString txt("%s (%d)", GVArgs(btTexts[defaulBt]).add(remaining));
               defButt.setText(txt);
            }
         }
         return true;
      }

      default:
         return false;
   }
}
