/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GKeyBar.h"
#include "glib/gui/GGraphics.h"
#include "glib/gui/GTooltip.h"
#include "glib/gui/border/GLineBorder.h"
#include "glib/util/GLog.h"
#include "glib/sys/GSystem.h"
#include "glib/GProgram.h"

GKeyBar::Button::Button ( GKeyBar& keyBar, const GString& keyName )
                :GAbstractCommand(GString::Empty),
                 keyName(keyName),
                 keyBar(keyBar)
{
}

GKeyBar::Button::~Button ()
{
}

GKeyBar::Key::Key ( GKeyBar& keyBar,
                    const GString& iconName,
                    const GString& keyName )
             :GToolbarButton(GString::Empty,
                             GString::Empty,
                             keyBar,
                             GString::Empty,
                             GString::Empty,
                             iconName,
                             GString::Empty),
              keyBar(keyBar),
              key(8)
{
   setInsets(new GInsets(2, 2, 2, 2), true);
   setBorder(new GLineBorder(GLineBorder::AlignInner), true);
   key.add(new Button(keyBar, keyName)); // KB_DEF
   key.add(new Button(keyBar, keyName + "_S")); // KB_SHIFT
   key.add(new Button(keyBar, keyName + "_A")); // KB_ALT
   key.add(new Button(keyBar, keyName + "_A_S")); // KB_ALT_SHIFT
   key.add(new Button(keyBar, keyName + "_C")); // KB_CTRL
   key.add(new Button(keyBar, keyName + "_C_S")); // KB_CTRL_SHIFT
   key.add(new Button(keyBar, keyName + "_C_A")); // KB_CTRL_ALT
   key.add(new Button(keyBar, keyName + "_C_A_S")); // KB_CTRL_ALT_SHIFT
}

GKeyBar::Key::~Key ()
{
}

const GString& GKeyBar::Key::getIDString () const
{
   const Button& b = key[keyBar.shiftBitMap];
   return b.getIDString();
}

void GKeyBar::Key::performAction ()
{
   const GString& cmdID = getIDString();
   if (cmdID != "") // Don't execute command if it has no ID!
      postCommand(&key[keyBar.shiftBitMap]);
}

void GKeyBar::Key::paintButtonBorderImpl ( GGraphics& g, 
                                           const GRectangle& rect, 
                                           bool force )
{
   const GString& id = getIDString();
   if (id != "")
      GToolbarButton::paintButtonBorderImpl(g, rect, force);
}

void GKeyBar::Key::paintButtonFaceImpl ( GGraphics& g, const GRectangle& rect )
{
   int iconWidth = 0;
   const GIcon* icon = getIcon();
   if (icon != null)
   {
      iconWidth = icon->getWidth();
      int iconHeight = icon->getHeight();
      int xpos = rect.x + 2;
      int ypos = rect.y + rect.height - iconHeight - 3;
      g.drawIcon(xpos, ypos, *icon);
   }

   GRectangle r = rect;
   r.x += iconWidth + 6;
   r.width -= iconWidth + 8;

   // Load text text which length best matches the available horizontal 
   // button space. The rectangle "r" defines the available space.
   GString text;
   int textWidth = 0;
   const GString& buttid = getIDString();
   if (buttid != "")
   {
      GResourceTable& restbl = keyBar.getResourceTable();
      int maxlen = r.width / g.getFontWidth();
      GString textid = "%" + buttid;
      while (maxlen > 0)
      {
         text = restbl.loadText(textid, GResourceTable::LT_PREFER_ALTERNATIVE, maxlen);
         textWidth = g.getWidthOfString(text);
         if (text == "" || textWidth <= r.width)
            break;
         maxlen--;
      }
   }

   // Clip the text if needed.
   while (text != "" && textWidth > r.width)
   {
      text.removeLastChar();
      while (text.endsWith(".")) // Remove all trailing dots
         text.removeLastChar();
      textWidth = g.getWidthOfString(text);
   }

   // Draw the text.
   GColor frg = getForegroundColor();
   g.drawText(text, r, frg);
}

GKeyBar::GKeyBar ( const GString& name,
                   const GString& constraints,
                   GWindow& parentWin )
        :GAbstractToolbarWindow(name,
                                constraints,
                                parentWin,
                                false,
                                WS_VISIBLE,
                                WS2_DEFAULTPAINT),
         preferredHeight(0),
         tmstWhenAltWasSet(0),
         keys(10),
         shiftBitMap(KB_DEF),
         previousFocusWin(null)
{
   // Make tooltip occur above (not below) buttons, since the keybar
   // is typically located at the bottom of the "screen".
   setTooltipPosition(GTooltip::PosAbove);

   // Create and add the keybar buttons.
   addKey("IDP_F1", "VK_F1");
   addKey("IDP_F2", "VK_F2");
   addKey("IDP_F3", "VK_F3");
   addKey("IDP_F4", "VK_F4");
   addKey("IDP_F5", "VK_F5");
   addKey("IDP_F6", "VK_F6");
   addKey("IDP_F7", "VK_F7");
   addKey("IDP_F8", "VK_F8");
   addKey("IDP_F9", "VK_F9");
   addKey("IDP_F10", "VK_F10");
}

void GKeyBar::addKey ( const char* iconName, const char* keyName )
{
   Key* key = new Key(*this, iconName, keyName);
   keys.add(key, false); // Button will be destroyed by the destructor of our base class.
   addElement(key);
}

void GKeyBar::setAltFlag ( bool on )
{
   if (((shiftBitMap & KB_ALT) != 0) != (on != 0))
   {
      if (on)
         shiftBitMap |= KB_ALT;
      else
         shiftBitMap &= ~KB_ALT;
      GTooltip::ReDisplay();
      invalidateAll(true);
   }
}

void GKeyBar::setCtrlFlag ( bool on )
{
   if (((shiftBitMap & KB_CTRL) != 0) != (on != 0))
   {
      if (on)
         shiftBitMap |= KB_CTRL;
      else
         shiftBitMap &= ~KB_CTRL;
      GTooltip::ReDisplay();
      invalidateAll(true);
   }
}

void GKeyBar::setShiftFlag ( bool on )
{
   if (((shiftBitMap & KB_SHIFT) != 0) != (on != 0))
   {
      if (on)
         shiftBitMap |= KB_SHIFT;
      else
         shiftBitMap &= ~KB_SHIFT;
      GTooltip::ReDisplay();
      invalidateAll(true);
   }
}

void GKeyBar::layout ()
{
   invalidateAll(true);

   GRectangle r = getWindowRect();
   int width = r.width;
   int height = r.height;

   const int numKeys = keys.getCount();
   if (numKeys <= 0)
      return;

   int keyWidth = width / numKeys;
   int rest = width % numKeys;
   int xpos = 0;

   for (int i=0; i<numKeys; i++, rest--)
   {
      int buttonWidth = keyWidth + (rest > 0 ? 1 : 0);
      Key& k = keys[i];
      k.setWindowBounds(xpos, 0, buttonWidth, height);
      xpos += buttonWidth;
   }
}

int GKeyBar::getPreferredHeight () const
{
   if (isVisible())
      return getHeightOfString("X") + 10;
   else
      return 0;
}

void GKeyBar::updateKeyCommands ( GWindow* focusWin )
{
   if (focusWin == null)
      focusWin = getParentWindow();

   if (focusWin == previousFocusWin)
      return;

   previousFocusWin = focusWin;

   const int numKeys = keys.getCount();
   for (int i=0; i<numKeys; i++)
   {
      Key& k = keys[i]; // Make this a fast one
      const int num = k.key.getCount();
      for (int ii=0; ii<num; ii++)
      {
         Button& kk = k.key[ii]; // Make this a fast one
         const GString& keyName = kk.getKeyName();
         GAccelItem* ai = focusWin->getAccelItemByKeyName(keyName);
         if (ai == null)
         {
            kk.setIDString(GString::Empty);
         }
         else
         {
            const GString& id = ai->getIDString();
            kk.setIDString(id);
         }
      }
   }

   invalidateAll(true);
}
