/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GFrameWindow.h"
#include "glib/gui/GDialogFrame.h"
#include "glib/gui/GKeyBar.h"
#include "glib/gui/GMenu.h"
#include "glib/gui/GStatusbar.h"
#include "glib/gui/GToolbar.h"
#include "glib/gui/GTooltip.h"
#include "glib/gui/border/GLineBorder.h"
#include "glib/gui/event/GKeyMessage.h"
#include "glib/gui/event/GWindowMessage.h"
#include "glib/util/GArray.h"
#include "glib/util/GTokenizer.h"
#include "glib/util/GSectionBag.h"
#include "glib/GProgram.h"
#include "glib/sys/GSystem.h"

// Implementation of the GFrameWindow::FrameWin class. Needed in order
// to support the "keepCaptionActive" for non-dialog frame windows on 
// OS/2. For OS/2 dialog frames this property is supported by handling 
// the WM_ACTIVATE for the GFrameWindow it self.

GFrameWindow::FrameWin::FrameWin ( GFrameWindow& owner, HWND hWndFrame )
                       :GWindow("Frame", 
                                hWndFrame, 
                                WS2_DEFAULTPAINT | 
                                    WS2_OS2Y |
                                    WS2_NOT_STATIC |
                                    WS2_IGNORE_COLORS_AND_FONT_PROFILE | 
                                    WS2_DONT_LAYOUT_PARENT_ON_FONTNAMESIZE_CHANGED),
                        owner(owner)
{
}

GFrameWindow::FrameWin::~FrameWin ()
{
}

GWindowMessage::Answer GFrameWindow::FrameWin::handleWindowMessage ( GWindowMessage& msg )
{
   switch (msg.getID())
   {
      case WM_ACTIVATE:
      {
         // From the OS/2 Toolkit Documentation:
         // "The frame control window procedure responds to this message 
         // by first sending a TBM_SETHILITE message to the FID_TITLEBAR 
         // control, if it exists, to highlight or unhighlight the title 
         // bar. If the style is FCF_DLGBORDER, the border is redrawn in 
         // either highlighted or unhighlighted state, as necessary. 
         // It then sends the WM_ACTIVATE message to the FID_CLIENT window. 
         HWND hwndOpposite = HWND(msg.getParam2Int());
         bool active = msg.getParam1Bool();
         bool drawActive = active || owner.keepCaptionActive;
         owner.titleBar->sendMessage(TBM_SETHILITE, GWindowMessage::Param1(USHORT(drawActive)));
         owner.sendMessage(WM_ACTIVATE, GWindowMessage::Param1(USHORT(active)), GWindowMessage::Param2(hwndOpposite));
         if (!active && owner.keepCaptionActive)
         {
            // Return without calling the super/default message handler
            // because that would paint the caption in an "inactive color"
            // which we are requested to not do by now.
            return 0;
         }
         return GWindow::handleWindowMessage(msg);
      }

      default:
         return GWindow::handleWindowMessage(msg);
   }
}

GFrameWindow::GFrameWindow ( const GString& name,
                             const GString& titleStr,
                             GWindow* ownerWin,
                             long winStyle,
                             long winStyle2,
                             bool isdlg )
             :GWindow(GWindowClass::GENERIC, GString::Empty, true),
              hWndFrame(null),
              frameWin(null),
              titleBar(null),
              hWndTitlebar(null),
              statbarTextStack(3),
              clientWin(null),
              menuBar(null),
              toolbar(null),
              keybar(null),
              statbar(null),
              autoManageWindowPos(true),
              keepCaptionActive(false),
              isdlg(isdlg)
{
   GColor bck = GSystem::GetSystemColor(GSystem::SCID_DIALOGBCK);
   GColor frg = GSystem::GetSystemColor(GSystem::SCID_DIALOGTXT);
   
   winStyle |= WS_CLIPCHILDREN;
   winStyle2 |= WS2_OS2Y | WS2_NOT_STATIC | WS2_DONT_LAYOUT_PARENT_ON_FONTNAMESIZE_CHANGED;

   if (isdlg)
   {
      // Initiate the window object, but don't create the HWND yet.
      winStyle2 |= WS2_DEFAULTPAINT | WS2_DONT_CREATE_HWND;
      GWindow::init(name, null, parentWin, 0, winStyle2, bck, frg, GDialogFrame::DefaultFont);

      // ---
      static DLGTEMPLATE* dlgt = null;
      if (dlgt == null) // No need to reload this over and over.
      {
         // Create a dialog template in memory.
         // Avoids needing a resource file to build up the dialog window.
         ULONG dtCtlData = FCF_SYSMENU | FCF_TITLEBAR;
         ULONG dtStyle = FS_DLGBORDER | WS_CLIPSIBLINGS | WS_SAVEBITS;
         SHORT dtX = 5; // location, In dialog-box units
         SHORT dtY = 5;
         SHORT dtCX = 171; // size, in dialog-box units
         SHORT dtCY = 63;
         PSZ dtCaption = ""; // caption text for dialog

         // Calculate number of bytes required by following fields:
         // Block must be large enough to contain the following:
         // # bytes: fixed part DLGTEMPLATE and one DLGTITEM
         // # bytes: dialog box caption.
         // # bytes: presentation parameters.

         ULONG wCaptionLen = (1 + strlen(dtCaption));
         ULONG wBlockLen = sizeof(DLGTEMPLATE) + sizeof(dtCtlData) + wCaptionLen;

         // Allocate global block of memory for dialog template.
         dlgt = (DLGTEMPLATE*) (new char[wBlockLen]);

         // Set the members of the DLGTEMPLATE structure.
         dlgt->cbTemplate = wBlockLen;
         dlgt->type = 0;                // not used
         dlgt->codepage = ::GpiQueryCp(GGraphics(this).getHandle());
         ULONG wBaseMem = (ULONG) dlgt;
         ULONG wItemMem = (ULONG) dlgt->adlgti;
         dlgt->offadlgti = (SHORT) wItemMem - wBaseMem;
         dlgt->fsTemplateStatus = 1;    // not used (docs say 0 but debug says 1)
         dlgt->iItemFocus = (USHORT) -1;         // no focus specified
         dlgt->coffPresParams = 0;      // never used!

         // Set the members of the frame DLGTITEM structure.
         // the first item is always the frame window class (WC_FRAME)
         dlgt->adlgti[0].fsItemStatus = 0;    // not used
         // no controls yet, Incremented with calls to AddDlgControl.
         dlgt->adlgti[0].cChildren = 0;
         dlgt->adlgti[0].cchClassName = 0;      	// use PM classes
         dlgt->adlgti[0].offClassName = SHORT1FROMMP((MPARAM) WC_FRAME);    // PM class
         dlgt->adlgti[0].cchText = wCaptionLen-1;     // strip null from string
         dlgt->adlgti[0].flStyle = dtStyle;
         dlgt->adlgti[0].x = dtX;
         dlgt->adlgti[0].y = dtY;
         dlgt->adlgti[0].cx = dtCX;
         dlgt->adlgti[0].cy = dtCY;
         dlgt->adlgti[0].id = 1; // ID of the dialog frame.
         dlgt->adlgti[0].offPresParams = (USHORT) -1; // No presentation params!

         // szDlgTemplate points to start of variable part of DLGTEMPLATE.
         PSZ szDlgTemplate = (PSZ) (dlgt + 1);

         // store variable length caption
         // we compute the offset from here to start of DLGTEMPLATE and save
         wItemMem = (ULONG) szDlgTemplate;

         dlgt->adlgti[0].offText =  (SHORT) (wItemMem - wBaseMem);
         memcpy(szDlgTemplate, dtCaption, wCaptionLen);
         szDlgTemplate += wCaptionLen;

         // store ULONG control data here
         wItemMem = (ULONG) szDlgTemplate;
         dlgt->adlgti[0].offCtlData =  (SHORT) wItemMem - wBaseMem;
         memcpy(szDlgTemplate, &dtCtlData, sizeof(dtCtlData));
         szDlgTemplate += sizeof(dtCtlData);
      }
      HWND hWndOwner = (parentWin == null ? HWND_DESKTOP : parentWin->getHWND());
      HWND hDlg = ::WinCreateDlg(HWND_DESKTOP, hWndOwner, GDialogFrame::DefaultDlgProc, dlgt, this);
      ::WinSetWindowBits(hDlg, QWL_STYLE, FS_NOBYTEALIGN, FS_NOBYTEALIGN);
      hWndFrame = hDlg;
      ::WinSetWindowPtr(hDlg, QWL_USER, (void*) getWindowEntry());
      hWndTitlebar = ::WinWindowFromID(hWndFrame, FID_TITLEBAR);
      // Make "this" window operate on the client area rather than the frame
      // it self. The application code is usually more interesting in the client
      // area. The frame is just an abstract property that is not "seen" by the
      // application code, except in a few situations.
      GWindow::setHWND(hDlg);
   }
   else
   {
      // Initiate the window object, but don't create the HWND yet.
      winStyle2 |= WS2_DONT_CREATE_HWND;
      GWindow::init(name, parentWin, null, winStyle, winStyle2, bck, frg, GWindow::SysDefFontNameSize);
      setText(titleStr);
      GString clsName = getClassName();

      HWND hWndParent = (parentWin == null ? HWND_DESKTOP : parentWin->getHWND());

      HWND hWin = null;
      ULONG flags = FCF_TITLEBAR |
                    FCF_SYSMENU |
                    FCF_ICON |
                    FCF_SIZEBORDER |
                    FCF_MINMAX |
                    FCF_SHELLPOSITION |
                    FCF_TASKLIST |
                    FCF_NOBYTEALIGN;
      const int ID_ICON1 = 1; // Resource-id of the default icon.
      hWndFrame = ::WinCreateStdWindow(hWndParent,
                                       winStyle | WS_ANIMATE,
                                       &flags,
                                       clsName,
                                       titleStr,
                                       0, 0,
                                       ID_ICON1,
                                       &hWin);
      ::WinSetWindowPtr(hWin, QWL_USER, (void*) getWindowEntry());
      hWndTitlebar = ::WinWindowFromID(hWndFrame, FID_TITLEBAR);
      // Make "this" window operate on the client area rather than the frame
      // it self. The application code is usually more interesting in the client
      // area. The frame is just an abstract property that is not "seen" by the
      // application code, except in a few situations.
      GWindow::setHWND(hWin);
   }

   // Make sure that the colors are actually activated 
   // on the HWND, because we postponed the creation of the HWND.
   setBackgroundColor(defaultBackgroundColor = bck);
   setForegroundColor(defaultForegroundColor = frg);
   // OS/2 uses two separate HWND's, one for the frame and one for the
   // client area. However, for dialog windows OS/2 works as for standard
   // frame windows in Windows.
   if (!isdlg) // OS/2 Dialog Boxes doesn have any separate frame window.
      frameWin = new FrameWin(*this, hWndFrame);
   titleBar = new GWindow("TitleBar", hWndTitlebar, WS2_DEFAULTPAINT | WS2_NOT_STATIC | WS2_OS2Y);
}

GFrameWindow::~GFrameWindow ()
{
   delete menuBar;
   delete toolbar;
   delete keybar;
   delete statbar;
   delete titleBar;
   delete frameWin;
   ::WinDestroyWindow(hWndFrame);
}

void GFrameWindow::layout ()
{
   // ---
   bool os2y = isOS2Y();
   GDimension wsize = getWindowSize();
   int left = 0;
   int right = wsize.width - 1;
   int bottom = os2y ? 0 : wsize.height - 1;
   int top = os2y ? wsize.height - 1 : 0;

   // Respect the insets, if any.
   GInsets ins = getInsets();
   left += ins.left;
   right -= ins.right;
   top += os2y ? -ins.top : ins.top;
   bottom += os2y ? ins.bottom : -ins.bottom;

   // Toolbar.
   if (toolbar != null)
   {
      int h = getPreferredToolbarHeight();
      int y = os2y ? top - h + 1 : top;
      toolbar->setWindowBounds(left, y, right - left + 1, h);
      top += os2y ? -h : h;
   }

   // Statusbar.
   if (statbar != null)
   {
      int h = getPreferredStatusbarHeight();
      int y = os2y ? bottom : bottom - h + 1;
      statbar->setWindowBounds(left, y, right - left + 1, h);
      bottom += os2y ? h : -h;
   }

   // Keybar.
   if (keybar != null)
   {
      int h = getPreferredKeybarHeight();
      int y = os2y ? bottom : bottom - h + 1;
      keybar->setWindowBounds(left, y, right - left + 1, h);
      bottom += os2y ? h : -h;
   }

   // Let the client area window take up the rest of the space.
   int y = os2y ? bottom : top;
   int h = os2y ? top - bottom + 1 : bottom - top + 1;
   clientArea.setBounds(left, y, right - left + 1, h);
   if (clientWin != null)
   {
      GRectangle& ca = clientArea;
      clientWin->setWindowBounds(ca.x, ca.y, ca.width, ca.height);
   }

   invalidateAll(true);
}

void GFrameWindow::setAutoLayoutClientWin ( GWindow* clientWin )
{
   this->clientWin = clientWin;
   if (isVisible())
      layout();
}

void GFrameWindow::setAutoManageWindowPos ( bool yes )
{
   autoManageWindowPos = yes;
}

bool GFrameWindow::isAutoManageWindowPos () const
{
   return autoManageWindowPos;
}

GFrameWindow::State GFrameWindow::getWindowState () const
{
   if (isMaximized())
      return GFrameWindow::Maximized;
   if (isMinimized())
      return GFrameWindow::Minimized;
   return GFrameWindow::Normalized;
}

void GFrameWindow::storeWindowPosImpl ( GPoint& pos, 
                                        GDimension& size, 
                                        GFrameWindow::State& state )
{
   SWP swp;
   HWND frameHWnd = (frameWin == null ? getHWND() : frameWin->getHWND());
   ::WinQueryWindowPos(frameHWnd, &swp);
   state = getWindowState();
   if (state == GFrameWindow::Normalized)
   {
      pos.x = swp.x;
      pos.y = swp.y;
      size.width = swp.cx;
      size.height = swp.cy;
   }
   else
   {
      pos.x = ::WinQueryWindowUShort(frameHWnd, QWS_XRESTORE);
      pos.y = ::WinQueryWindowUShort(frameHWnd, QWS_YRESTORE);
      size.width = ::WinQueryWindowUShort(frameHWnd, QWS_CXRESTORE);
      size.height = ::WinQueryWindowUShort(frameHWnd, QWS_CYRESTORE);
   }
}

void GFrameWindow::restoreWindowPosImpl ( const GPoint& pos, 
                                          const GDimension& size, 
                                          GFrameWindow::State state )
{
   HWND hframe = (frameWin == null ? getHWND() : frameWin->getHWND());
   int flags = SWP_SIZE | SWP_MOVE | SWP_SHOW;
   switch (state)
   { 
      case GFrameWindow::Minimized: flags |= SWP_MINIMIZE; break;
      case GFrameWindow::Maximized: flags |= SWP_MAXIMIZE; break;
      default: flags |= SWP_RESTORE; break;
   }
   ::WinSetWindowPos(hframe, null, pos.x, pos.y, size.width, size.height, flags);
   ::WinUpdateWindow(hframe);
}

void GFrameWindow::restoreWindowPos ( const GString& sectionName, 
                                      const GString& entryName,
                                      bool preventMinimized )
{
   GProgram& prg = GProgram::GetProgram();
   GSectionBag& ini = prg.getIniProperties();
   GString str = ini.getString(sectionName, entryName, "");
   GArray<GInteger> array(5);
   GTokenizer tokenizer(str, " \t\r\n", "", false);
   for (int i=0; i<5; i++)
   {
      const GToken* token = tokenizer.getNextToken();
      if (token->isEmpty())
      {
         // No more tokens. ==> Unknown format of string!
         setVisible();
         return;
      }

      GString item = token->toString();
      if (!GInteger::IsInteger(item))
      {
         setVisible();
         return; // Unknown format of string item!
      }

      array.add(new GInteger(item));
   }

   // ---
   GPoint pos(array[0], array[1]);
   GDimension size(array[2], array[3]);
   GFrameWindow::State state = GFrameWindow::State(array[4].intValue());
   if (preventMinimized && state == GFrameWindow::Minimized)
      state = GFrameWindow::Normalized;
   restoreWindowPosImpl(pos, size, state);
}

void GFrameWindow::storeWindowPos ( const char* sectionName, const char* entryName )
{
   GPoint pos;
   GDimension size;
   GFrameWindow::State state;
   storeWindowPosImpl(pos, size, state);

   GString str("%d %d %d %d %d", GVArgs(pos.x).add(pos.y).add(size.width).add(size.height).add(state));
   GProgram& prg = GProgram::GetProgram();
   GSectionBag& ini = prg.getIniProperties();
   ini.putString(sectionName, entryName, str);
}

void GFrameWindow::removeMenubar ()
{
   HWND hframe = (frameWin == null ? getHWND() : frameWin->getHWND());
   HWND hmenu = WinWindowFromID(hframe, FID_MENU);
   if (hmenu != null)
   {
      WinSetParent(hmenu, HWND_OBJECT, true);
      WinSetOwner(hmenu, HWND_OBJECT);
      if (menuBar != null)
      {
         delete menuBar;
         menuBar = null;
      }
   }
}

void GFrameWindow::setIcon ( const GString& iconName )
{
   // TODO: How to make this work on OS2?
   // const GIcon* icon = GIcon::getIcon(iconName);
   // int hicon = icon->getSystemHandle(); // HPOINTER
   // WinSendMsg(hWndFrame, WM_SETICON, MPARAM(hicon));
}

void GFrameWindow::setMenubar ( const GString& menuID,
                                bool useFancy,
                                bool topMnemonicsOn )
{
   // Use default resource table if the one specified is null.
   const GResourceTable& res = getResourceTable();
   GMenuResource* men = res.getMenuResource(menuID);
   if (men == null)
      return;

   // Remove the previous menubar, if any.
   removeMenubar();

   // Activate the new menubar.
   menuBar = new GMenu("MenuBar", GString::Empty, *this, men, useFancy, true, topMnemonicsOn);
   GMenuPopup::TheWindow& mbwin = menuBar->getTheWindow();
   HWND hmenu = mbwin.getHWND();
   HWND hframe = (frameWin == null ? getHWND() : frameWin->getHWND());
   ::WinSetParent(hmenu, hframe, true);
   ::WinSetOwner(hmenu, hframe);
   GWindow& frame = getFrame();
   frame.sendMessage(WM_UPDATEFRAME, MPARAM(FCF_MENU));
}

void GFrameWindow::setToolbar ( const GString& resID, const GString& name )
{
   toolbar = new GToolbar(name, GString::Empty, *this, resID);
}

void GFrameWindow::setKeybarVisible ( bool show )
{
   if (keybar == null)
      keybar = new GKeyBar("KeyBar", GString::Empty, *this);

   keybar->setVisible(show);

   // Layout all parts of the frame, since the visibility of
   // the keybar has changed.
   layout();
   ensureFocusOK();
}

void GFrameWindow::setStatusbarVisible ( bool show )
{
   if (statbar == null)
      statbar = new GStatusbar("StatusBar", GString::Empty, *this);

   statbar->setVisible(show);

   // Layout all parts of our self, since the visibility of the
   // statusbar has changed.
   layout();
   ensureFocusOK();
}

void GFrameWindow::setToolbarVisible ( bool show )
{
   if (toolbar == null)
      return;

   toolbar->setVisible(show);

   // Layout all parts of ours self, since the visibility of the
   // toolbar has changed.
   layout();
   ensureFocusOK();
}

void GFrameWindow::startMenubarMode ( const GString& cmdID )
{
   if (menuBar != null)
   {
      int id = menuBar->getCommandMapID(cmdID);
      if (id > 0)
      {
         HWND hframe = (frameWin == null ? getHWND() : frameWin->getHWND());
         HWND hmenu = ::WinWindowFromID(hframe, FID_MENU);
         ::WinSendMsg(hmenu, MM_SELECTITEM, MPFROM2SHORT(id, true), 0);
      }
   }
}

void GFrameWindow::setCommandEnableState ( const char* cmdID, bool state )
{
   // The popup menu
   GWindow::setCommandEnableState(cmdID, state);

   // The menubar
   if (menuBar != null)
      menuBar->setCommandEnableState(cmdID, state);

   // The toolbar
   // TODO: ???
}

void GFrameWindow::setCommandToggleState ( const char* cmdID, bool state )
{
   // The popup menu
   GWindow::setCommandToggleState(cmdID, state);

   // The menubar
   if (menuBar != null)
      menuBar->setCommandToggleState(cmdID, state);

   // The toolbar
   // TODO: ???
}

bool GFrameWindow::onDeactivate ()
{
   // Window has been deactivated, so reset all Shift-Flags of the Key Bar
   if (keybar != null)
   {
      keybar->setShiftFlag(false);
      keybar->setAltFlag(false);
      keybar->setCtrlFlag(false);
   }

   // Also make sure to close the tooltip if it is visible.
   GTooltip::Hide();

   return true;
}

bool GFrameWindow::onCommand ( int cmd )
{
   // Check if the command comes from the main menu
   if (menuBar != null)
   {
      GMenuPopupItem *mi = menuBar->getCommand(cmd);
      if (executeAbstractCommand(mi))
         return true;
   }

   // Check if the command comes from the popup menu
   return GWindow::onCommand(cmd);
}

bool GFrameWindow::onInitMenu ()
{
   updateCommandStates();
   return true;
}

bool GFrameWindow::onKey ( const GKeyMessage& key )
{
   if (keybar == null)
      return GWindow::onKey(key);

   bool down = key.isKeyDown();
   switch (key.getVirtualKey())
   {
      case VK_SHIFT:
      {
         keybar->setShiftFlag(down);
         return true;
      }

      case VK_ALT:
      {
         keybar->setAltFlag(down);
         // Since the keybar is currently visible we don't want the system 
         // to activate the menubar when the user press and release the 
         // ALT-key alone, except if he does this very quickly.
         if (down)
         {
            if (keybar->tmstWhenAltWasSet == 0)
               keybar->tmstWhenAltWasSet = GSystem::CurrentTimeMillis();
         }
         else
         {
            if (keybar->tmstWhenAltWasSet != 0)
            {
               ulonglong now = GSystem::CurrentTimeMillis();
               ulonglong diff = now - keybar->tmstWhenAltWasSet;
               keybar->tmstWhenAltWasSet = 0;
               if (diff > 200) 
               {
                  // More than 200 milliseconds since the ALT-key was pressed.
                  // So stop the system from activating the menu bar.
                  return true;
               }
            }
         }
         return GWindow::onKey(key);
      }

      case VK_CTRL:
      {
         keybar->setCtrlFlag(down);
         return true;
      }

      default:
         return GWindow::onKey(key);
   }
}

bool GFrameWindow::onMenuSelect ( int id )
{
   // Since menu has been activated (or are active) it is best to make sure
   // that all the Shift-Flags of the Key Bar are reset.
   if (keybar != null)
   {
      keybar->setShiftFlag(false);
      keybar->setAltFlag(false);
      keybar->setCtrlFlag(false);
   }

   // Check if the command comes from the menubar
   if (menuBar != null)
   {
      GMenuPopupItem *mi = menuBar->getCommand(id);
      if (mi != null)
      {
         const GString& idString = mi->getIDString();
         updateHintString(idString);
      }
      else
         updateHintString("");
   }

   // Check if the command comes from the popup menu
   return GWindow::onMenuSelect(id);
}

void GFrameWindow::updateHintString ( const GString& cmdID )
{
   GWindow* focusWin = getFocusedWindow();
   if (focusWin == null)
      focusWin = this;
   GString hint;
   if (cmdID != "")
   {
      GAccelItem* ai = focusWin->getAccelItemByCommandID(cmdID);
      if (ai != null)
         hint += ai->getKeyNameForHumans() + ": ";
      GString textID = "%" + cmdID;
      GString text = GProgram::LoadText(textID, GResourceTable::LT_PREFER_HINT);
      if (text != textID || hint != "")
         hint += text;
   }
   setStatusbarHint(hint);
}

const GRectangle& GFrameWindow::getClientAreaRect () const
{
   return clientArea;
}

void GFrameWindow::updateKeybarCommands ( GWindow* focusWin )
{
   GKeyBar* kb = getKeybar();
   if (kb != null)
      kb->updateKeyCommands(focusWin);
   GWindow::updateKeybarCommands(focusWin);
}

const GWindow& GFrameWindow::getFrame () const
{
   if (frameWin == null)
      return *this;
   else
      return *frameWin;
}

GWindow& GFrameWindow::getFrame ()
{
   if (frameWin == null)
      return *this;
   else
      return *frameWin;
}

GKeyBar* GFrameWindow::getKeybar ()
{
   return keybar;
}

GMenu* GFrameWindow::getMenubar ()
{
   return menuBar;
}

int GFrameWindow::getPreferredKeybarHeight () const
{
   if (keybar == null)
      return 0;
   else
      return keybar->getPreferredHeight();
}

int GFrameWindow::getPreferredStatusbarHeight () const
{
   if (statbar == null)
      return 0;
   else
      return statbar->getPreferredHeight();
}

int GFrameWindow::getPreferredToolbarHeight () const
{
   if (toolbar == null)
      return 0;
   else
      return toolbar->getPreferredHeight();
}

GWindow* GFrameWindow::getTitlebar ()
{
   return titleBar;
}

GStatusbar* GFrameWindow::getStatusbar ()
{
   return statbar;
}

GToolbar* GFrameWindow::getToolbar ()
{
   return toolbar;
}

GPoint GFrameWindow::getWindowPos () const
{
   if (frameWin == null)
      return GWindow::getWindowPos();
   else
      return frameWin->getWindowPos();
}

void GFrameWindow::setActive ( bool force )
{
   if (frameWin == null)
      GWindow::setActive(force);
   else
      frameWin->setActive(force);
}

bool GFrameWindow::isActive () const
{
   if (frameWin == null)
      return GWindow::isActive();
   else
      return frameWin->isActive();
}

bool GFrameWindow::isKeybarVisible () const
{
   if (keybar != null)
      return keybar->isVisible();
   else
      return false;
}

bool GFrameWindow::isMaximized () const
{
   if (frameWin == null)
      return GWindow::isMaximized();
   else
      return frameWin->isMaximized();
}

bool GFrameWindow::isMinimized () const
{
   if (frameWin == null)
      return GWindow::isMinimized();
   else
      return frameWin->isMinimized();
}

bool GFrameWindow::isStatusbarVisible () const
{
   if (statbar != null)
      return statbar->isVisible();
   else
      return false;
}

bool GFrameWindow::isToolbarVisible () const
{
   if (toolbar != null)
      return toolbar->isVisible();
   else
      return false;
}

bool GFrameWindow::isVisible () const
{
   if (frameWin == null)
      return GWindow::isVisible();
   else
      return frameWin->isVisible();
}

bool GFrameWindow::isResizable () const
{
   ULONG style = WinQueryWindowULong(getHWND(), QWL_STYLE);
   if (style & FCF_SIZEBORDER)
      return true;
   else
      return false;
}

void GFrameWindow::setMaximized ( bool flag )
{
   if (frameWin == null)
      GWindow::setMaximized(flag);
   else
      return frameWin->setMaximized(flag);
}

void GFrameWindow::setMinimized ( bool flag )
{
   if (frameWin == null)
      GWindow::setMinimized(flag);
   else
      return frameWin->setMinimized(flag);
}

void GFrameWindow::setVisible ( bool flag )
{
   if (frameWin == null)
      GWindow::setVisible(flag);
   else
      return frameWin->setVisible(flag);
}

void GFrameWindow::setWindowBounds ( int xpos, int ypos, int width, int height )
{
   if (frameWin == null)
      GWindow::setWindowBounds(xpos, ypos, width, height);
   else
      return frameWin->setWindowBounds(xpos, ypos, width, height);
}

void GFrameWindow::setWindowSize ( int width, int height )
{
   if (frameWin == null)
      GWindow::setWindowSize(width, height);
   else
      return frameWin->setWindowSize(width, height);
}

void GFrameWindow::setWindowPos ( int xpos, int ypos )
{
   if (frameWin == null)
      GWindow::setWindowPos(xpos, ypos);
   else
      return frameWin->setWindowPos(xpos, ypos);
}

void GFrameWindow::setStatusbarHint ( const GString& hint )
{
   if (statbar == null)
      return;
   statbar->setHint(hint);
}

void GFrameWindow::setStatusbarText ( const GString& text )
{
   if (statbar == null)
      return;
   statbar->setText(text);
}

void GFrameWindow::pushStatusbarText ( const GString& txt )
{
   if (statbar == null)
      return;
   GString prevText = statbar->getText();
   statbarTextStack.push(new GString(prevText));
   setStatusbarText(txt);
}

void GFrameWindow::popStatusbarText ()
{
   if (statbar == null)
      return;
   GString txt;
   if (!statbarTextStack.isEmpty())
   {
      GString* s = statbarTextStack.pop(false);
      txt = *s;
      delete s;
   }
   setStatusbarText(txt);
}

void GFrameWindow::setText ( const GString& text )
{
   if (titleBar != null)
      titleBar->setText(text);
}

void GFrameWindow::setResizable ( bool flag )
{
   if (isResizable() == flag)
      return;
   ULONG style = WinQueryWindowULong(getHWND(), QWL_STYLE);
   if (flag)
      WinSetWindowULong(getHWND(), QWL_STYLE, style | FCF_SIZEBORDER);
   else
      WinSetWindowULong(getHWND(), QWL_STYLE, style | ~FCF_SIZEBORDER);
}

void GFrameWindow::setEnabled ( bool flag, bool repaint )
{
   if (frameWin == null)
      GWindow::setEnabled(flag, repaint);
   else
      frameWin->setEnabled(flag, repaint);
}

void GFrameWindow::writeProfile ( const GString& sectName, bool force )
{
   GWindow::writeProfile(sectName, force);
   if (isAutoManageWindowPos())
      storeWindowPos(sectName, "Pos");
   if (frameWin != null)
      frameWin->writeProfile(sectName  + "." + frameWin->getName(), force);
   if (titleBar != null)
      titleBar->writeProfile(sectName + "." + titleBar->getName(), force);
}

void GFrameWindow::queryProfile ( const GString& sectName )
{
   if (frameWin != null)
   {
      GString subSect(sectName + "." + frameWin->getName());
      frameWin->queryProfile(subSect);
   }
   if (titleBar != null)
   {
      GString subSect(sectName + "." + titleBar->getName());
      titleBar->queryProfile(subSect);
   }
   GWindow::queryProfile(sectName);
   if (toolbar != null)
      setToolbarVisible(toolbar->isVisible());
   if (isAutoManageWindowPos())
      restoreWindowPos(sectName, "Pos");
}

void GFrameWindow::setKeepCaptionActive ( bool b )
{
   keepCaptionActive = b;
}

GWindowMessage::Answer GFrameWindow::handleWindowMessage ( GWindowMessage& msg )
{
   switch (msg.getID())
   {
      case WM_ACTIVATE:
      {
         // For dialog windows OS/2 paints the caption as part of the 
         // default handling of WM_ACTIVATE. Prevent this from happening
         // if we are requested to keep the caption in an "activ look" 
         // even when it has been deactivated.
         if (isdlg && !msg.getParam1Int() && keepCaptionActive)
            return 0;
         return GWindow::handleWindowMessage(msg);
      }

      default:
         return GWindow::handleWindowMessage(msg);
   }
}
