/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_DROPLIST
#define __GLIB_DROPLIST

#include "glib/gui/GTextEntry.h"
#include "glib/gui/GPushButton.h"
#include "glib/gui/GListBox.h"
#include "glib/gui/event/GCommandMap.h"
#include "glib/gui/event/GMouseListener.h"

/**
 * This is the low level window class that implements a drop list control.
 *
 * @author  Leif Erik Larsen
 * @since   2001.01.07
 */
class GDropList : public GWindow,
                  public GAbstractListBox,
                  protected GListSelectionListener
{
   friend class GListBox;

   private:

      /**
       * We need a peer window to contain the actual list box, in order
       * to work around some window owner/parent relationship problems,
       * at least on OS/2.
       */
      class DropContainer : public GWindow
      {
         public:

            /**
             * We need to sub-class GListBox in order to be able to override
             * some event methods.
             */
            class ListBox : public GListBox, public GMouseListener
            {
               public:

                  GDropList& dlist;

               public:

                  ListBox ( GDropList& dlist, GWindow& parentWin );
                  virtual ~ListBox ();

               private:

                  /** Disable the copy constructor. */
                  ListBox ( const ListBox& src ) 
                     : GListBox(GString::Empty, GString::Empty, *src.getParentWindow()), dlist(dlist) {}

                  /** Disable the assignment operator. */
                  ListBox& operator= ( const ListBox& ) { return *this; }

               private:

                  /** Part of {@link GMouseListener}. */
                  virtual bool mouseReleased ( const GMouseEvent& ev );

                  virtual bool onKeyDown ( const GKeyMessage& key );
            };

            ListBox list;

         public:

            DropContainer ( GDropList& dlist, GWindow& ownerWin );
            virtual ~DropContainer ();

         private:

            /** Disable the copy constructor. */
            DropContainer ( const DropContainer& src ) : list(src.list.dlist, *this) {}

            /** Disable the assignment operator. */
            DropContainer& operator= ( const DropContainer& ) { return *this; }
      };

   private:

      /**
       * We need to sub-class GTextEntry in order to be able to override
       * some event methods.
       */
      class TextEntry : public GTextEntry
      {
         private:

            GDropList& dlist;

         public:

            TextEntry ( GDropList& dlist );
            virtual ~TextEntry ();

         private:

            /** Disable the copy constructor. */
            TextEntry ( const TextEntry& src ) 
               : GTextEntry(GString::Empty, GString::Empty, *this), dlist(src.dlist) {}

            /** Disable the assignment operator. */
            TextEntry& operator= ( const TextEntry& ) { return *this; }

         private:

            virtual bool onKeyDown ( const GKeyMessage& key );
      };

      /**
       * Sub-class GPushButton in case we need to override something.
       */
      class Button : public GPushButton
      {
         private:

            GDropList& dlist;

         public:

            Button ( GDropList& dlist );
            virtual ~Button ();

         private:

            /** Disable the copy constructor. */
            Button ( const Button& src ) 
               : GPushButton(GString::Empty, GString::Empty, *this), dlist(src.dlist) {}

            /** Disable the assignment operator. */
            Button& operator= ( const Button& ) { return *this; }
      };

   private:

      friend class TextEntry;
      friend class Button;

      GVector<GListSelectionListener*> selectionListeners;
      bool selectListenerWasCalled;
      DropContainer drop;
      TextEntry entry;
      Button button;

   public:

      GDropList ( const GString& name,
                  const GString& constraints,
                  GWindow& parentWin,
                  long winStyle = WS_VISIBLE,
                  long winStyle2 = 0 );

      virtual ~GDropList ();

   private:

      /** Disable the copy constructor. */
      GDropList ( const GDropList& );

      /** Disable the assignment operator. */
      void operator= ( const GDropList& );

   private:

      virtual void layout ();

   public:

      virtual bool isEmpty () const;
      virtual void changeValue ( const GString& newValue, bool notify = true );
      virtual GString queryValue () const;

      virtual int getPreferredHeight () const;
      virtual int getPreferredWidth () const;

      /**
       * Grab the input focus and give it to contained text entry window.
       */
      virtual void grabFocus ( bool force );

      /**
       * Implements {@link GAbstractListBox#addItem}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @see     GAbstractListBox#addItem
       */
      virtual void addItem ( const GString& text, 
                             const GString& iconName = GString::Empty, 
                             GObject* userData = null, 
                             bool autoDelUD = false );

      /**
       * Add the specified listener to the set of object to be called
       * whenever the current selection of the drop-list is changed.
       *
       * @author  Leif Erik Larsen
       * @since   2001.02.23
       * @param   l    A reference to the listener object to add.
       */
      void addListBoxSelectionListener ( GListSelectionListener* l );

      /**
       * Implements {@link GAbstractListBox#insertItem}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @see     GAbstractListBox#insertItem
       */
      virtual void insertItem ( int index, 
                                const GString& text, 
                                const GString& iconName = GString::Empty, 
                                GObject* userData = null, 
                                bool autoDelUD = false );

      /**
       * Implements {@link GAbstractListBox#getSelectedIndex}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @see     GAbstractListBox#getSelectedIndex
       */
      virtual int getSelectedIndex () const;

      /**
       * Implements {@link GAbstractListBox#getItem}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @see     GAbstractListBox#getItem
       * @throws  GArrayIndexOutOfBoundsException if the specified index is
       *                                          out of bounds.
       */
      virtual const GListBoxItem& getItem ( int index ) const;

      /**
       * Implements {@link GAbstractListBox#setItemIcon}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.06.07
       * @see     GAbstractListBox#setItemIcon
       */
      virtual void setItemIcon ( int index, const GString& iconName );

      /**
       * Implements {@link GAbstractListBox#setItemTextAndIcon}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.06.07
       * @see     GAbstractListBox#setItemTextAndIcon
       */
      virtual void setItemTextAndIcon ( int index, const GString& text, const GString& iconName );

      /**
       * Implements {@link GAbstractListBox#setItemText}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @see     GAbstractListBox#setItemText
       */
      virtual void setItemText ( int index, const GString& text );

      /**
       * Implements {@link GAbstractListBox#getItemText}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @see     GAbstractListBox#getItemText
       */
      virtual GString getItemText ( int index ) const;

      /**
       * Implements {@link GAbstractListBox#setItemUserData}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @see     GAbstractListBox#setItemUserData
       */
      virtual void setItemUserData ( int index, GObject* userData, bool autoDelete );

      /**
       * Implements {@link GAbstractListBox#getItemUserData}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @see     GAbstractListBox#getItemUserData
       */
      virtual GObject* getItemUserData ( int index ) const;

      /**
       * Implements {@link GAbstractListBox#removeItem}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @see     GAbstractListBox#removeItem
       */
      virtual bool removeItem ( int index );

      /**
       * Implements {@link GAbstractListBox#removeAllItems}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @see     GAbstractListBox#removeAllItems
       */
      virtual void removeAllItems ();

      /**
       * Implements {@link GAbstractListBox#getItemCount}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @see     GAbstractListBox#getItemCount
       */
      virtual int getItemCount () const;

      /**
       * Implements {@link GAbstractListBox#setSelectedIndex}.
       *
       * @author  Leif Erik Larsen
       * @since   2000.05.09
       * @see     GAbstractListBox#setSelectedIndex
       */
      virtual void setSelectedIndex ( int index );

      /**
       * This methid will enable or diable all of our contained
       * child components as well, before the super implementation is
       * called.
       *
       * @author  Leif Erik Larsen
       * @since   2001.02.24
       */
      virtual void setEnabled ( bool flag = true, bool repaint = true );

      /**
       * We override this method in order to set the oily state of 
       * our contained focusable peer as well as our self.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.07
       */
      virtual void setOily ( bool flag );

      /**
       * Show or hide the drop list box.
       *
       * @author  Leif Erik Larsen
       * @since   2001.01.27
       * @param   show            True to show, or else false to hide, it.
       * @param   grabEntryFocus  True if we shall set the keyboard focus on
       *                          the entry field, but only if <i>show</i>
       *                          is false.
       */
      void setDropListVisible ( bool show = true, bool grabEntryFocus = true );

      /**
       * Test whether or not the drop list box is currently visible
       * on screen.
       */
      bool isDropListVisible ();

      /** This method is called when the user click on the drop-button. */
      void cmdDrop ( GAbstractCommand *cmd = null );

   protected:

      /** Part of {@link GListSelectionListener}. */
      virtual void onListBoxSelectionChanged ( GAbstractListBox& listb );

      /** Part of {@link GListSelectionListener}. */
      virtual void onListBoxSelectionChangedByCharKey ( GAbstractListBox& listb );

      /** Part of {@link GListSelectionListener}. */
      virtual void onListBoxSelectionDblClick ( GAbstractListBox& listb );

      /**
       * @author  Leif Erik Larsen
       * @since   2004.10.27
       */
      virtual bool onUserMessage ( class GUserMessage& msg );

   DECLARE_COMMAND_TABLE(GDropList);
};

#endif
