/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_DIALOGPANEL
#define __GLIB_DIALOGPANEL

#include "glib/gui/GDialogFrame.h"

/**
 * Main class to provide Dialog Panels within <i>GDialogFrame</i>.
 *
 * A Dialog Panel forms the container of a number of child components. 
 * The child components them self, however, are actually contained
 * in a layer panel which is called the Client Area Panel. The Client Area
 * Panel will automatically fill the space within the insents of the Owner
 * Dialog Panel. The insets are such elements as the Border, Command Palette,
 * Status Bar and Title Bar, if any. The top level Dialog Panel will always
 * be contained in an instance of {@link GDialogFrame} which is the root
 * panel of a "physical" dialog box on screen.
 *
 * Child components are typically defined in the body of a dialog resource 
 * in the resource script of the application, but they can also be added
 * programatically to a GDialogPanel via method {@link #addComponent}. 
 * A number of component types are supported (e.g. statictext, textentry,
 * pushbutton, radiobutton, listbox, droplist, tabbedpanel, progressbar
 * and others). Regardless of the component type there are a number of 
 * supported component properties. Here is a list of all properties 
 * that are commonly supported by all types of components:
 *
 * <TABLE BORDER CELLSPACING=0 CELLPADDING=5 WIDTH="100%" BGCOLOR="#CCFFFF">
 * <TR>
 * <TD><B><I>ID=X</I></B></TD>
 * <TD>The ID String of the Component. The ID String is case significant.
 *     Every Component within the same Dialog Panel must have unique ID's
 *     that are different from each other. If no ID is specified then the
 *     ICA System will automatically create an unique ID for the Component
 *     automatically. However, in that case the component will not be easily
 *     accessible by the program code. Automatic ID's are therefore usually
 *     only to be used on static Components. Static Components are Components
 *     that will never contain an editable value from either the user nor the
 *     program point of view. The ID String can never be changed by the
 *     Program Code in Runtime after the Component has been instantiated.
 *     <P>
 *     Almost all user interactive Components needs an ID. The ID is the
 *     reference to the Component that the Program must know to be able to
 *     communicate with the Component if the Program does not have a
 *     reference directly to the Component instance it self, but only has a
 *     reference to the Owner Dialog Panel. Other Components does not need
 *     an ID at all, since the Program will never attempt to communicate
 *     with them. Example of user interactive Components are Push Buttons
 *     ({@link GPushButton}), Text Entry Fields ({@link GTextEntry}) and
 *     List Boxes ({@link GListBox}). Examples of none-interactive Components 
 *     are Static Texts ({@link GStaticText}), Static Icons 
 *     ({@link GStaticIcon}) and Group Boxes ({@link GGroupBox}).
 *     However, non-interactive Components must have an ID if the Program
 *     needs to be able to change or query the properties of the Component
 *     during runtime. For example, if the Program need to change the text
 *     string of a Static Text Component then the Static Text Component must
 *     be defined to have an ID. In that case the Program can change the text
 *     of the Component by calling the ICA method
 *     {@link IDialogPanel#setComponentValue} on the Owner Dialog Panel.
 *     </TD>
 * </TR>
 *
 * <TR>
 * <TD><B><I>XPOS=X</I></B></TD>
 * <TD>The horizontal position of the Component. The position X is relative
 *     within the Owner Dialog Panel and starts at position #1, which is the
 *     first visible character position at the left of the Owner Dialog Panel.
 *     All horizontal positions are given in characters. However, you can give
 *     floating positions, such as "X=3.5", to place the Component at a random
 *     location.
 *     <P>
 *     <b>See:</b> {@link GDialogPanel#setComponentPos}
 *     </TD>
 * </TR>
 *
 * <TR>
 * <TD><B><I>YPOS=X</I></B></TD>
 * <TD>The vertical position of the Component. The position X is relative
 *     within the Owner Dialog Panel and starts at line #1, which is the
 *     first visible text line at the top of the Owner Dialog Panel.
 *     All vertical positions are given in lines. However, you can give
 *     floating positions, such as "Y=1.5", to place the Component at a random
 *     location.
 *     <P>
 *     <b>See:</b> {@link GDialogPanel#setComponentPos}
 *     </TD>
 * </TR>
 *
 * <TR>
 * <TD><B><I>WIDTH=X</I></B></TD>
 * <TD>Defines the width of the Component, where X is the with of the
 *     Component given in characters. You can give floating heights, such
 *     as "WIDTH=25.5". If you don't specify the width of
 *     the Component then the Component will get a default width. The
 *     default width will vary with respect to what type of Component it
 *     is about. For example, for {@link no.cba.ica.gui.dlg.TEXTENTRY} the
 *     default width will be the length of the text (given by the TEXT
 *     parameter) plus one.
 *     <P>
 *     <b>See:</b> {@link GDialogPanel#setComponentSize}
 *     </TD>
 * </TR>
 *
 * <TR>
 * <TD><B><I>HEIGHT=X</I></B></TD>
 * <TD>Defines the height of the Component, where X is the height of the
 *     Component given in character lines. Most Components does not need to
 *     be specified a height. You can give floating heights, such
 *     as "HEIGHT=1.2".
 *     <P>
 *     <b>See:</b> {@link GDialogPanel#setComponentSize}
 *     </TD>
 * </TR>
 *
 * <TR>
 * <TD><B><I>TEXT=X</I></B></TD>
 * <TD>Defines the initial text of the Component, where X is either a direct
 *     text string within quotes or a reference to any text previously defined
 *     in the Resource Table, DDS, EWM or PTT. For instance, if you want to
 *     reference a text which is defined in the Resource Table then you must write
 *     <b>TEXT=%ID</b>, where <i>ID</i> is the ID of the referenced Text
 *     Resource.
 *     <P>
 *     <b>See:</b> {@link #setText}
 *     </TD>
 * </TR>
 *
 * <TR>
 * <TD><B><I>HINT=X</I></B></TD>
 * <TD>With respect to which Dialog Component is currently active the Dialog
 *     will show a describing Hint Text in the Status Bar of the Dialog
 *     Box (if any). The Hint Text of each Component is optional. The syntax is
 *     <b>HINT=X</b>, where <i>X</i> is any text just as for the <b>TEXT=X</b>
 *     parameter.
 *     </TD>
 * </TR>
 *
 * <TR>
 * <TD><B><I>NEXT=X</I></B></TD>
 * <TD>By default, when the user press the enter key in a Dialog that
 *     does not contain any Default Push Button then the Dialog will try to
 *     activate the next Component in the positive direction with respect to
 *     which Component is currently active. To override this default behaviour
 *     you can use this parameter to make the default path different. The
 *     syntax is <b>NEXT=X</b>, where <i>X</i> is the ID of the interactive
 *     Component to automatically activate whenever the user press the <i>Tab</i>,
 *     <i>Down</i> or <i>Enter</i> in the Component that contains the
 *     <b>NEXT=X</b> parameter.
 *     </TD>
 * </TR>
 *
 * <TR>
 * <TD><B><I>PREVIOUS=X</I></B></TD>
 * <TD>Almost the same as for the <b>NEXT=X</b> parameter, but this is for
 *     changing the default path in negative direction. That is when user
 *     press the <i>Shift+Tab</i> or <i>Up</i> key.
 *     </TD>
 * </TR>
 *
 * <TR>
 * <TD><B><I>DISABLED</I></B></TD>
 * <TD>To make the Component Disabled just specify this parameter. A Disabled
 *     Component will not receive keyboard input focus and is therefore not
 *     editable from the user point of view.
 *     <P>
 *     <b>See:</b> <A HREF="#setEnabled">setEnabled()</A>
 *     </TD>
 * </TR>
 *
 * <TR>
 * <TD><B><I>GRAY</I></B></TD>
 * <TD>For backward compatibility reasons, this parameter is the same as the
 *     DISABLED parameter.
 *     </TD>
 * </TR>
 *
 * <TR>
 * <TD><B><I>TOGGLEON</I></B></TD>
 * <TD>All Components has a Toggle Flag that may be either on or off.
 *     Typically this Toggle Flag is used only by Toggle Push Buttons, but
 *     your Program can use it on any type of Component. To make the Component
 *     initially toggled on, just append this parameter to the Component
 *     definition.
 *     <P>
 *     <b>See:</b> {@link #setToggleState}
 *     </TD>
 * </TR>
 *
 * <TR>
 * <TD><B><I>MANDATORY</I></B></TD>
 * <TD>Use this parameter to make the Component Mandatory. A Mandatory Component
 *     is a Component that must have some legal data value before it can loose
 *     keyboard input focus to another Component in the positive direction.
 *     </TD>
 * </TR>
 *
 * <TR>
 * <TD><B><I>HIDDEN</I></B></TD>
 * <TD>Specifying this flag will cause the Component to be initially Hidden.
 *     <P>
 *     <b>See:</b> {@link #setVisible}
 *     </TD>
 * </TR>
 *
 * <TR>
 * <TD><B><I>OILY</I></B></TD>
 * <TD>Specify this argument if you want the component not to receive the
 *     keyboard focus when user walks through components by pressing the
 *     UP or DOWN keys. An oily component will receive keyboard focus when
 *     pressing TAB or SHIFT+TAB only, and when clicked on by mouse, of course.
 *     <P>
 *     <b>See:</b> {@link #setOily}
 *     </TD>
 * </TR>
 * </TABLE>
 *
 * @author  Leif Erik Larsen
 * @since   2000.01.23
 */
class GDialogPanel : public GWindow
{
   friend class GDialogFrame;
   friend class GTabbedPanel;

#ifdef __BCPLUSPLUS__ // This compiler does not assume a class being a "friend of it self".
   friend class GDialogPanel;
#endif

   public:

      /** Add a few vertical pixels to each row of Components, just to have some room for the borders, etc. */
      static const int COMPONENT_EXTRAFONTH;

   private:

      /** The tiny number of pixels to distance components vertically. */
      static const int COMPONENT_DISTANCE_Y;

      /** The tiny number of pixels to distance components horizontally. */
      static const int COMPONENT_DISTANCE_X;

      /**
       * This variable will be true for dialog panels that are to
       * be treated as roots for the focus manager.
       *
       * @see #isFocusCycleRoot
       */
      bool focusRoot;

      /** A pointer to the Default Push Button of the Dialog Panel (if any). */
      class GPushButton* defPushButton;

      /** A pointer to the Escape Push Button of the Dialog Panel (if any). */
      class GPushButton* escPushButton;

      /**
       * Width of the current dialog font, given in pixels.
       */
      int iDlgFontW;

      /**
       * Height of the current dialog font, given in pixels.
       */
      int iDlgFontH;

      /**
       * The current wallpaper settings of the dialog window.
       */
      class GWallPaperParams* dlgWallPaper;

      /**
       * Reference to the Dialog Box object that is managing us.
       */
      class GDialogFrame& ownerDlg;

      /**
       * Reference to the Dialog Panel object that is managing us.
       * This will be null for top level panels.
       */
      class GDialogPanel* ownerDlgPane;

      /**
       * A pointer to the icon resource of the dialog, or null if none.
       * This is a pointer to a common object, so don't try to delete
       * it. Instead, just set this pointer to null.
       */
      class GIconResource* icon;

      /**
       * The Command Palette that contains both the menubar and the toolbar,
       * or just one of them if not both has been set.
       */
      class GCommandPalette* cmdPalette;

      /**
       * The current statusbar. Normally the statusbar is located below the
       * client area of this <i>IWindowPanel</i>. To get a statusbar, use the
       * method <i>setStatBarText()</i>.
       */
      class GStatusbar* statusbar;

      /**
       * The client area is the window panel where the Components of the
       * Dialog Panel are actually contained.
       */
      class GWindow* clientArea;

      /**
       * True when the dialog destructor has been called.
       */
      bool destructorIsCalled;

      /**
       * Inner class used to manage Hot Keys of the different Components of
       * the Dialog Panel. The Dialog Panel will automatically assign a
       * Hot Key to each Component that uses <i>autoAssocCtrlToAHotKey()</i>.
       *
       * When user press one of the Hot Keys then we will call
       * <i>hotKeyAction()</i> on the associated Component. The Component
       * it self is free to override that method and do whatever it desires.
       * Most Components should, however, ignore this and let the super class
       * ({@link GWindow) handle the event. It will automatically request
       * the keyboard focus to the Component, or give the focus to the
       * Component next to it self.
       *
       * @author  Leif Erik Larsen
       * @since   2000.01.23
       */
      class HotKeyItem : public GObject
      {
         public:

            /** The Component that owns the Hot Key, or null if it hasn't been associated. */
            class GWindow* comp;

            /** The Hot Key character (uppercase). */
            char hotChar;

            /** The keyboard code of the Hot Key (KEY_*). */
            int hotKey;

         public:

            HotKeyItem ( char chr, int key );
            virtual ~HotKeyItem ();
      };

      /**
       * This is the bag of available Hot Keys. Each item in the bag will
       * be an instance of our inner class <i>HotKeyItem</i>. The Key of
       * the bag elements will be the Hot Key Character (hotChar).
       */
      GKeyBag<HotKeyItem>* hotKeys;

   private:

      /**
       * @author  Leif Erik Larsen
       * @since   2000.01.23
       * @throws  GAddComponentException in case of any error constructing
       *                                 the new dialog panel window.
       */
      GDialogPanel ( class GDialogFrame& ownerFrame,
                     const GString& name,
                     const GString& constraints,
                     class GDialogMessageHandler* msgProc,
                     const GDialogResource& params );

      GDialogPanel ( class GDialogPanel& ownerPanel,
                     class GWindow& parentWin,
                     const GString& name,
                     const GString& constraints,
                     class GDialogMessageHandler* msgProc,
                     const GDialogResource& params );

      void init ( const GDialogResource& params,
                  class GDialogMessageHandler* msgProc );

   public:

      virtual ~GDialogPanel ();

   private:

      /** Disable the copy constructor. */
      GDialogPanel ( const GDialogPanel& src ) 
         : ownerDlg(src.ownerDlg) {}

      /** Disable the assignment operator. */
      GDialogPanel& operator= ( const GDialogPanel& ) { return *this; }

   private:

      /**
       * Create the {@link GKeyBag} of all possible Hot Keys
       * that can be associated with a Component within the Dialog Panel.
       * We will make sure that none of the Hot Keys are already in use by
       * an Item in the Menu Bar (if any).
       */
      void initBagOfAvailableHotKeys ();

      /**
       * To be used by <i>initBagOfAvailableHotKeys()</i> only.
       * The specified Hot Key will not be added to the IKeyBag of available
       * Hot Keys if the Hot Key is already in use by the Menu Bar (if any).
       */
      void addAvailableHotKey ( char chr, int key );

   protected:

      virtual bool onKeyDown ( const class GKeyMessage& key );

      /**
       * @author  Leif Erik Larsen
       * @since   2004.01.24
       */
      virtual bool onTimer ( const GString& timerID, GObject* userData );

   public:

      /**
       * Auto assign a Hot Key to the specified Component. The Hot Key will be
       * the first character of the specified text that isn't already in use as
       * a Hot Key of any other Component.
       *
       * If we find that the specified Component already has a registered Hot Key
       * then we will automatically unregister the Hot Key before trying to
       * re-associate a Hot Key to the same Component. This feature is nice when
       * the Component need to be associated by another Hot Key, e.g. if the
       * Component has been given a new text. Thus, this method can be called by
       * the Component at any time, both upon initialization and upon update due
       * to text change.
       *
       * @param   comp  Reference to which Component this is about.
       * @param   text  The Component text, of where to find an unused hotkey.
       * @return  Index of hotkey within <code>text</code>, or -1 if none.
       * @see     GWindow#autoAssocAHotKey
       */
      int autoAssocCtrlToAHotKey ( class GWindow& comp, const GString& text );

      /**
       * Return true if and only if the dialog panel is part of a 
       * dialog frame that has not yet been dismissed.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.24
       */
      virtual bool isAlive () const;

      /**
       * Get a reference to the client area of the Dialog Panel.
       * The client area is the child window of the Dialog Panel, which
       * is the direct parent of the Dialog's contained components.
       */
      GWindow& getClientArea ();

      /**
       * Make sure that the Dialog Panel contains an instance of a
       * Command Palette.
       */
      void ensureCommandPalette ();

      /**
       * Get a reference to the current Command Palette of the Dialog Panel.
       * If the Dialog Panel does not have any Command Palette then we will
       * return null. In that case you can use <i>ensureCommandPalette()</i>
       * to automatically get a default Command Palette activated.
       *
       * @see     #setCommandPalette
       * @see     #ensureCommandPalette
       */
      GCommandPalette* getCommandPalette ();

      /**
       * Change the text in the Statusbar of this Window Panel.
       * If the Panel doesn't have a Statusbar already then we will create
       * one now. Remove (deactivate) the Statusbar if <i>newText</i> is
       * <i>null</i>.
       *
       * The specified text may be the ID of a Text Resource that we should
       * automatically fetch from the Resource Table of the Window Panel.
       */
      virtual void setStatbarText ( const GString& newText );

      /**
       * Get a reference to the current WallPaper settings of
       * the Dialog Panel. If the Dialog Panel as not already created an
       * instance of the wallpaper parameters class then we will create one
       * now, and return a reference to that new instanmce. Therefore,
       * in order to prevent the creation of wallpaper parameter objects that
       * are not actually needed, don't call this method if not necessary.
       */
      virtual GWallPaperParams& getWallPaperParams ();

      /**
       * Set the title text of the dialog panel.
       *
       * For top level dialogs this is the same as setting the text of our
       * owner frame.
       *
       * @see  #getOwnerPanel
       */
      virtual void setTitleText ( const GString& text );

      /**
       * Get a reference to the Dialog Frame Window object of where we
       * are contained.
       *
       * @see  #getOwnerPanel
       */
      class GDialogFrame& getOwnerFrame ();

      /**
       * Get a reference to the dialog panel object of where we are
       * contained. If we are the top-level dialog panel then we will
       * return null.
       *
       * @see     #getOwnerFrame
       */
      class GDialogPanel* getOwnerPanel ();

      /**
       * Returns true for top level dialog panels, and for dialog
       * panels that are contain in a tab of a tabbed panel.
       */
      virtual bool isFocusCycleRoot () const;

      /**
       * Overrides {@link GWindow#setFontNameSize} in order to activate
       * the same fontNameSize on the {@link #clientArea} as of the
       * dialog panel it self.
       *
       * @author  Leif Erik Larsen
       * @since   2000.10.05
       */
      virtual void setFontNameSize ( const GString& fontNameSize );

      /**
       * Set the Menu Bar Resource of the Dialog Panel.
       */
      void setMenubar ( class GMenuResource& newMenu, 
                        bool useFancy, 
                        class GResourceTable* res = null, 
                        bool topMnemonicsOn = true );

      /**
       * Set the Toolbar of the Window Panel.
       *
       * @param   tbarID The ID of which toolbar resource to look up in the
       *                 resource table as is specified in <code>rc</code>.
       * @param   res    The resource table of where to look up the toolbar
       *                 resource as is specified in <code>tbarID</code>,
       *                 or else <code>null</code> to look up in default
       *                 Resource Table which in the most cases is the Resource
       *                 Table of the owner Program.
       */
      void setToolbar ( class GToolbarResource& tb, 
                        class GResourceTable* res = null );

      /**
       * Set the specified Icon Resource on the Window Panel.
       *
       * If the Window Panel is a Top Level Window Panel then the Icon will
       * probably show in the System Menu of the Owner Frame Window. This is,
       * however, somewhat system dependent. On some systems the Icon will not
       * be visible at all.
       *
       * @param   icon   The Icon Resource of which to activate on the Window.
       */
      virtual void setIcon ( class GIconResource& icon );
      void setIcon ( const GString& iconName );

      /**
       * Get the current number of components that are contained in the
       * client area of the Dialog Panel. This does not include the
       * statusbar and the command palette (if any).
       */
      virtual int getComponentCount () const;

      /**
       * Get a reference to the value of the specified dialog Component.
       * The meaning of the returned value depends much on what type of
       * Component that is referenced by the specified name.
       *
       * <b>Mark:</b> If the specified Component doesn't exist in the Dialog
       * Panel then we will do nothing but return the default value.
       *
       * @param   id      ID of which Component to query.
       * @param   defVal  The default value to return in case of any error.
       * @return  A reference to the current value of the specified Component,
       *          or a copy of the specified default value if there are
       *          no Component with the specified name in the Dialog Panel.
       * @see     GWindow#queryValue
       */
      GString getComponentValue ( const GString& id, const GString& defVal = GString::Empty ) const;

      /**
       * Get a reference to the value of the specified dialog Component, after
       * it has been converted to an integer. The meaning of the returned value
       * depends much on what type of Component that is referenced.
       *
       * @param   id         ID of which Component to query.
       * @param   defaultVal Which value to return in case of any errors. Possible
       *                     errors can be e.g. number format error or null
       *                     reference.
       * @return  The current value of the Component, after converted to an integer.
       */
      int getComponentIntValue ( const GString& id, int defaultVal = -1 ) const;

      /**
       * Get a reference to the value of the specified dialog Component, after
       * it has been converted to a long. The meaning of the returned value
       * depends much on what type of Component that is referenced.
       *
       * @param   id         ID of which Component to query.
       * @param   defaultVal Which value to return in case of any errors. Possible
       *                     errors can be e.g. number format error or null
       *                     reference.
       * @return  The current value of the Component, after converted to an integer.
       */
      longlong getComponentLongValue ( const GString& id, longlong defaultVal = -1L ) const;

      /**
       * Get a reference to the value of the specified dialog Component, after
       * it has been converted to a boolean. The meaning of the returned value
       * depends much on what type of Component that is referenced.
       *
       * @param   id    ID of which Component to query.
       * @return  The current value of the Component, after converted to a boolean.
       */
      bool getComponentBoolValue ( const GString& id, bool defVal = false ) const;

      /**
       * Read and activate all default settings of this dialog from the
       * profile of the dialog owner program.
       *
       * This method is typically called by the user program upon
       * handling of </i>GM_QUERYPROFILE</i>. It can, however, also be used
       * upon <i>GM_INITDIALOG</i>, but for nested Dialog Panels the
       * programs are better doing it upon </i>GM_QUERYPROFILE</i> as is
       * recommended.
       */
      virtual void queryProfile ( const GString& sectName );

      /**
       * Write the current dialog settings to the profile of the program that
       * owns the Dialog Panel.
       *
       * This method is typically called by the user program upon
       * handling of </i>GM_WRITEPROFILE</i>. It can, however, also be used
       * upon <i>GM_DISMISSDIALOG</i>, but for nested Dialog Panels the
       * programs are better doing it upon </i>GM_WRITEPROFILE</i> as is
       * recommended.
       */
      virtual void writeProfile ( const GString& sectName, bool force = false );

      /**
       * Set the wallpaper of the dialog panel.
       * The wallpaper will be automatically painted when using this method.
       */
      virtual void setWallPaper ( const class GWallPaperParams& wp );

      /**
       * Read and activate the wallpaper settings of this dialog panel
       * from the profile of the dialog owner program.
       *
       * @param  section  Name of the profile section of where to read from.
       * @param  item     Name of the profile entry of which to query.
       */
      void queryProfileWallPaper ( const GString& section, const GString& item );

      /**
       * Call this method to dynamically add a Component to this dialog. New
       * Components can be dynamically added to any dialog, not only dialogs
       * created with {@link GProgram#makeDialog}.
       *
       * This method also supports adding Components after the dialog has been
       * executed. That is when the dialog window is currently active and
       * probably visible on screen (if it hasn't been made hidden by the
       * program).
       *
       * @param   params      The Component Parameters.
       * @param   constraints The initial layout constraints.
       *                      See {@link GWindow#setLayoutConstraints}.
       * @return  A reference to the added Component.
       * @throws  GAddComponentException if something goes wrong.
       */
      GWindow& addComponent ( const class GComponentParams& params, 
                              const GString& constraints = GString::Empty );

      /**
       * Add a new Component to the Dialog using the specified Component Script.
       * Also notifies the Layout Manager (if any) to add the component to the
       * container's layout using the specified constraints object.
       *
       * Use this method to dynamically add a Component to the dialog. New
       * Components can be dynamically added to any dialog, not only dialogs
       * created with {@link GProgram#makeDialog}.
       *
       * This method also supports adding Components after the dialog has been
       * executed. That is when the dialog window is currently active and
       * probably visible on screen.
       *
       * @param   script      The definition script of the new Component.
       *                      This is a string script that follows the
       *                      same syntax and functionality as of the
       *                      Resource Script facility.
       * @param   constraints The initial layout constraints.
       *                      See {@link GWindow#setLayoutConstraints}.
       * @throws  GAddComponentException if adding the Component fails.
       */
      GWindow& addComponent ( const GString& script, 
                              const GString& constraints = GString::Empty );

      /**
       * Remove the indexed Component from the Dialog Panel.
       * This method should normaly not be used. It is implemented first and
       * foremost to be used by the Dialog Editor component of the Designer
       * program, which uses the IDialogPanel class in a very special way.
       *
       * @param   index   Index of which Component to remove from the Dialog Panel.
       */
      void removeComponent ( int index );

      /**
       * Remove the specified Component from the Dialog Panel.
       * This method should normaly not be used. It is implemented first and
       * foremost to be used by the Dialog Editor component of the Designer
       * program, which uses the IDialogPanel class in a very special way.
       *
       * @param   ctrlName  ID of which Component to remove from the Dialog Panel.
       * @throws  GNoSuchComponentException if the specified Component does not
       *                                    exist in the Dialog Panel.
       */
      void removeComponent ( const GString& ctrlName );

      /**
       * Move specified Component to a new position within the Dialog Panel.
       * The position must be specified in characters (not in pixels).
       *
       * <b>Mark:</b> If the specified Component doesn't exist in the Dialog
       * Panel then we will do nothing but return.
       *
       * @param   id   ID of which Component to move.
       * @param   xPos The new horizontal position of the Component.
       * @param   yPos The new vertical position of the Component.
       * @see     #getComponentPos
       */
      void setComponentPos ( const GString& id, double xPos, double yPos );
      void setComponentPos ( const GString& id, const class GComponentPos& pt );

      /**
       * Resize specified Component.
       * The size must be specified in characters (not in pixels).
       *
       * <b>Mark:</b> If the specified Component doesn't exist in the Dialog
       * Panel then we will do nothing but return.
       *
       * @param   id     ID of which Component to resize.
       * @param   width  The new width of the Component.
       * @param   height The new height of the Component.
       * @see     #getComponentSize
       * @see     #setComponentPos
       */
      void setComponentSize ( const GString& id, double width, double height );
      void setComponentSize ( const GString& id, const class GComponentSize& newSize );

      /**
       * Get the size of the specified Dialog Component.
       * The returned size is measured in characters (not in pixels).
       *
       * @return  The Dimension of the Component, or else null if the specified
       *          Component doesn't exist in the Dialog Panel.
       * @throws  GNoSuchComponentException if the specified Component doesn't
       *                                    exist in the Dialog Panel.
       * @see     #setComponentSize
       */
      GComponentSize getComponentSize ( const GString& id );

      /**
       * Get the position of the specified Dialog Component.
       * The returned position is measured in characters (not in pixels).
       *
       * @return  The Point of the Component, or else null if the specified
       *          Component doesn't exist in the Dialog Panel.
       * @throws  GNoSuchComponentException if the specified Component doesn't
       *                                    exist in the Dialog Panel.
       * @see     #setComponentPos
       */
      GComponentPos getComponentPos ( const GString& id );

      /**
       * This method is to be called only once from PUSHBUTTON if the
       * new push button has the DEFAULT parameter from the resource script.
       *
       * @param   ctrl  The Default Push Button Component.
       * @see     #getDefaultPushButtonID
       */
      void setDefaultPushButton ( class GPushButton* ctrl );

      /**
       * This method is to be called only once from PUSHBUTTON if the
       * new push button has the ESCAPE parameter from the script.
       *
       * @param   ctrl  The Escape Push Button Component.
       * @see     #getEscapePushButtonID
       */
      void setEscapePushButton ( class GPushButton* ctrl );

      /**
       * Show or hide the specified Component.
       * This will cause <code>GM_CTRLNOTIFY</code> to be sent to the message
       * handler with <code>DN_SHOWCTRL</code> or <code>DN_HIDECTRL</code>
       * in message parameter #2.
       *
       * <b>Mark:</b> If the specified Component doesn't exist in the Dialog
       * Panel then we will do nothing but return.
       *
       * @param   id    ID of which Component to show or hide.
       * @param   show  True to show the Component, or else false to hide it.
       */
      void setComponentVisible ( const GString& id, bool show = true );

      /**
       * Convert a character-based position to corresponding pixel-based position
       * with respect to the current dialog font. A character-based position is
       * upper left at 1:1, while the pixel-based position is upper left at 0:0.
       *
       * @see     #convertToCharPosition
       */
      GPoint convertToPixelPosition ( double xPos, double yPos );

      /**
       * Convert a character-based position to corresponding pixel-based position
       * with respect to the current dialog font. A character-based position is
       * upper left at 1:1, while the pixel-based position is upper left at 0:0.
       *
       * @see     #convertToCharPosition
       * @see     #convertToCharXPosition
       * @see     #convertToCharYPosition
       */
      GPoint convertToPixelPosition ( const class GComponentPos& pos );

      /**
       * Convert a pixel-based position to corresponding character-based position
       * with respect to the current dialog font. A character-based position is
       * upper left at 1:1, while the pixel-based position is upper left at 0:0.
       *
       * @see     #convertToPixelPosition
       * @see     #convertToCharXPosition
       * @see     #convertToCharYPosition
       */
      GComponentPos convertToCharPosition ( int xpos, int ypos );

      /**
       * Convert the pixel-based X-position to its corresponding character-based
       * X-position with respect to the current dialog font.
       *
       * @see     #convertToCharPosition
       * @see     #convertToCharYPosition
       */
      double convertToCharXPosition ( int xpos );

      /**
       * Convert the pixel-based Y-position to its corresponding character-based
       * Y-position with respect to the current dialog font.
       *
       * @see     #convertToCharPosition
       * @see     #convertToCharXPosition
       */
      double convertToCharYPosition ( int ypos );

      /**
       * Convert a character-based dimension to corresponding pixel-based
       * dimension with respect to the current dialog font.
       *
       * @see     #convertToCharDimension
       * @see     #convertToPixelWidth
       * @see     #convertToPixelHeight
       */
      GDimension convertToPixelDimension ( const class GComponentSize& dim );

      /**
       * Convert a character-based dimension to corresponding pixel-based
       * dimension with respect to the current dialog font.
       *
       * @see     #convertToCharDimension
       * @see     #convertToPixelWidth
       * @see     #convertToPixelHeight
       */
      GDimension convertToPixelDimension ( double width, double height );

      /**
       * Convert the character-based width to its corresponding pixel-based
       * width, with respect to the current dialog font.
       *
       * @see     #convertToPixelDimension
       * @see     #convertToPixelHeight
       */
      int convertToPixelWidth ( double width );

      /**
       * Convert the character-based width to its corresponding pixel-based
       * width, with respect to the current dialog font.
       *
       * @see     #convertToPixelDimension
       * @see     #convertToPixelWidth
       */
      int convertToPixelHeight ( double height );

      /**
       * Convert a pixel-based dimension to corresponding character-based
       * dimension with respect to the current dialog font.
       *
       * @see     #convertToPixelDimension
       */
      GComponentSize convertToCharDimension ( double width, double height );

      /**
       * Get the average width of a single character with respect to the
       * current font of the Dialog Panel. The returned width will be
       * given in pixels.
       */
      int getFontWidth () const;

      /**
       * Get the height of a single character with respect to the current
       * font of the Dialog Panel. The returned height will be given in
       * pixels, and it will include the number of additional pixels that
       * are set aside for each line of Components to make room
       * for borders, etc.
       */
      int getFontHeight () const;

      /**
       * Set the size of the Dialog Panel.
       * The new size must be measured in characters (not in pixels).
       *
       * This method can be used on both Top Level Panels as well on Dialog
       * Component Panels. If the dialog is a Top Level Panel then we will
       * automatically adjust the size of the Frame Window so that the Client
       * Area of the Dialog Box gets as large as is requested.
       *
       * @param   width   Entry: New width of this dialog, in character cells.
       * @param   height  Entry: New height of this dialog, in character cells.
       * @see     #getDialogSize
       */
      virtual void setDialogSize ( double width, double height );

      /**
       * Set the size of the Dialog.
       * The size must be specified in characters (not in pixels).
       */
      void setDialogSize ( const class GComponentSize& newSize );

      /**
       * Get the size of the Dialog Panel.
       * The returned size is measured in characters (not in pixels).
       *
       * @see     #setDialogSize
       */
      virtual GComponentSize getDialogSize ();

      /**
       * Request the keyboard input focus to be set on the Dialog Panel. The
       * request will be traversed down on the stack of Dialog Panels until we
       * reach the Low Level Dialog Component which was the most recently focused
       * Component within the Dialog Panel.
       *
       * This method should normally not be used by user program code. It is for
       * internal usage by ICA only. Is is public as an implementation side
       * effect. To change the current focus in a Dialog Panel you should use
       * the method <i>setComponentFocus()</i> instead.
       */
      virtual void grabFocus ( bool force = false );

      /**
       * Get a reference to the Component within the Dialog Panel that contains
       * the Component that has the keyboard focus. If the current Component
       * isn't contained in the Dialog Panel then we will return null.
       */
      GWindow* getFocusComponent ();

      /**
       * Get the ID of which Component is the current Component of the dialog.
       * The current Component is the Component which has the keyboard focus.
       * If no Component has focus within the Dialog Panel then we will return
       * the specified default value.
       */
      GString getFocusComponentID ( const GString& defaultID = GString::Empty );

      /**
       * Get the ID of which is the Default Push Button of the Dialog.
       *
       * @return  The ID of the current Default button, or an empty string
       *          if there are currently no Default button defined at all.
       * @see     #setDefaultPushButton
       */
      GString getDefaultPushButtonID ();

      /**
       * get a reference to the default push button of the dialog panel or
       * any of its parents.
       */
      GPushButton* getDefaultPushButton ( bool searchUpwards = false );

      /**
       * get a reference to the default push button of the dialog panel or
       * any of its parents.
       */
      GPushButton* getEscapePushButton ( bool searchUpwards = false );

      /**
       * Get the ID of which is the Escape Push Button of the Dialog.
       *
       * @return  The ID of the current Escape button, or an empty string
       *          if there are currently no Escape button defined at all.
       * @see     #setEscapePushButton
       */
      GString getEscapePushButtonID ();

      /**
       * Test if the specified Component exist in the dialog.
       */
      bool isContainingComponent ( const GString& id ) const;

      /**
       * Get a reference to the indexed Dialog Component Object.
       *
       * @throws  GNoSuchComponentException if the specified index is out of bounds.
       */
      virtual GWindow& getComponentByIndex ( int index );

      /**
       * Get a reference to the specfied Component.
       *
       * @throws  GNoSuchComponentException if the specified Component does not
       *                                    exist in the Dialog Panel.
       */
      virtual GWindow& getComponentByID ( const GString& id );

      /**
       * Get the ID-string of the indexed Component. This method
       * will return an empty string (rathen than a null) in case the specified
       * index is illegal.
       *
       * @throws  GNoSuchComponentException if the specified index is out of bounds.
       */
      GString getComponentID ( int index ) const;

      /**
       * Test if the specified Component is currently disabled or not.
       *
       * <b>Mark:</b> If the specified Component doesn't exist in the Dialog
       * Panel then we will do nothing but return true.
       *
       * @param   id  ID of which Component to test.
       * @return  True if the specified Component is disabled, or else false.
       */
      bool isComponentDisabled ( const GString& id ) const;

      /**
       * Test if the specified Component is currently enabled or not.
       *
       * <b>Mark:</b> If the specified Component doesn't exist in the Dialog
       * Panel then we will do nothing but return false.
       *
       * @param   id  ID of which Component to test.
       * @return  True if the specified Component is enabled, or else false.
       */
      bool isComponentEnabled ( const GString& id ) const;

      /**
       * Test if the specified Component is empty or contains an illegal value.
       *
       * <b>Mark:</b> If the specified Component doesn't exist in the Dialog
       * Panel then we will do nothing but return false.
       *
       * @return  True if the specified Component is empty or if it contains an
       *          illegal value or else false.
       * @see     GWindow#isEmpty
       */
      bool isComponentEmpty ( const GString& id ) const;

      /**
       * Enable or disable the identified Component.
       *
       * <b>Mark:</b> If the specified Component doesn't exist in the Dialog
       * Panel then we will do nothing but return.
       *
       * @param   id      ID of which Component to test.
       * @param   enable  True to enable, or else false to disable the Component.
       *                  If true and the Component is both gray and disabled then
       *                  we will also ungray the Component.
       * @see     #setComponentDisabled
       */
      void setComponentEnabled ( const GString& id, bool enable = true );

      /**
       * Enable or disable the identified Component.
       *
       * <b>Mark:</b> If the specified Component doesn't exist in the Dialog
       * Panel then we will do nothing but return.
       *
       * @param   id    ID of which Component to enable.
       * @param   flag  True to disable, or else false to enable the Component.
       * @see     #setComponentEnabled
       */
      void setComponentDisabled ( const GString& id, bool flag = true );

      /**
       * Clear the content of the specified Dialog Component.
       * This is the same as setting the component value to be an empty
       * string.
       *
       * <b>Mark:</b> If the specified Component doesn't exist in the Dialog
       * Panel then we will do nothing but return.
       *
       * @param   id      ID of which Component to clear.
       * @param   notify  False to prevent the GM_CTRLCHANGED message from
       *                  being sent back to the program defined message
       *                  handler, or else false. Normally, this is to be true.
       */
      void clearComponentValue ( const GString& id, bool notify = true );

      /**
       * Change the content of the specified Component.
       * The meaning of the value depends much on what type of Component
       * that is referenced.
       *
       * <b>Mark:</b> If the specified Component doesn't exist in the Dialog
       * Panel then we will do nothing but return.
       *
       * @param   id      ID of which Component to change.
       * @param   newVal  The new value to put into the Component.
       * @param   notify  True if we shall send GM_CTRLCHANGED, or else false.
       * @see     GWindow#changeValue
       */
      void setComponentValue ( const GString& id, const char* newVal, bool notify = true );
      void setComponentValue ( const GString& id, const GString& newVal, bool notify = true );
      void setComponentValue ( const GString& id, int newVal, bool notify = true );
      void setComponentValue ( const GString& id, unsigned long newVal, bool notify = true );
      void setComponentValue ( const GString& id, longlong newVal, bool notify = true );
      void setComponentValue ( const GString& id, bool newVal, bool notify = true );

      /**
       * Set the text of the specified Component.
       *
       * This method is more useful on Components of some types than others.
       * E.g. For TEXTENTRY fields it is better to call <i>setComponentValue</i>,
       * because the main data of text entry fields is the same as the
       * main text. However, for e.g. PushButton the main data is not the
       * text value. Therefore, to change the text of a push button you
       * must call this method rather than <i>setComponentValue()</i>.
       *
       * <b>Mark:</b> If the specified Component doesn't exist in the Dialog
       * Panel then we will do nothing but return.
       *
       * @see     GWindow#setText
       */
      void setComponentText ( const GString& id, const GString& newText );

      /**
       * We override this method in order to give the user message handler
       * a chance to handle the <i>GM_COMMAND</i> message.
       */
      virtual bool executeAbstractCommand ( GAbstractCommand *cmd );

      /**
       * Get a reference to the object that has eventaully recently been given
       * to <i>dismiss()</i>. This method should not be called before
       * <i>dismiss()</i> has been called to terminate the Dialog.
       * If you do, then we will return null.
       *
       * @see     #dismiss
       */
      const GString& getDismissArgument ();

      /**
       * Activate the next editable Component starting from the current one.
       *
       * @param   skipOily  True if we shall ignore oily Components, or else false.
       *                    Typically this parameter is true if called upon
       *                    handling of the <up> or the <down> key.
       * @return  <code>true</code> if we was successfully able to activate
       *          another Component or else <code>false</code>.
       */
      virtual bool activateNextEditableComponent ( bool skipOily );

      /**
       * Activate the previous editable Component starting from the current one.
       *
       * @param   skipOily  True if we shall ignore oily Components, or else false.
       *                    Typically this parameter is true if called upon
       *                    handling of the <up> or the <down> key.
       * @return  <code>true</code> if we was successfully able to activate
       *          another Component or else <code>false</code>.
       */
      virtual bool activatePrevEditableComponent ( bool skipOily );
};

#endif // #ifndef __GLIB_DIALOGPANEL
