/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/util/GLog.h"
#include "glib/exceptions/GException.h"
#include "glib/primitives/GInteger.h"

GException::GException ( const char* msg, 
                         GError sysErr, 
                         GSymbolEngine::ThreadContext* context )
           :exception(msg),
            msg(null),
            sysErr(sysErr),
            msgFullText(null),
            context(context)
{
   setMessage(msg);

   // If context != null then we can assume we are called from the 
   // low level exception handler (which is called upon various untyped
   // C exceptions - such as dividxe-by-zero, access-violation, etc.).
   // In that case we must postpone getting the stack trace, because we 
   // are possibly called more than once and doing anything else than 
   // throwing the exception is documented by MSVC++ to possibly 
   // hurt the C++ exception handling. Thus, if context != null then 
   // skip getting the stack trace here.
   if (context == null)
      bareStackTrace = GSymbolEngine::GetStackTrace();
}

GException::~GException ()
{
}

GString GException::getStackTrace ( const GString& msg ) const
{
   if (context != null)
   {
      // Fetching the bare stack trace was postponed, so do it now.
      __DECLARE_CONST_SELF__(GException, self);
      self.bareStackTrace = GSymbolEngine::GetStackTrace(GString::Empty, context);
      self.context = null;
   }

   // ---
   GString ret(msg.length() + GString::EOL.length() + bareStackTrace.length());
   if (msg != "")
   {
      ret += msg;
      ret += GString::EOL;
   }
   
   ret += bareStackTrace;
   return ret;
}

const char* GException::what () const throw()
{
   if (msgFullText == "")
   {
      __DECLARE_CONST_SELF__(GException, self);
      self.msgFullText = toString();
   }
   return msgFullText;
}

GError GException::getSystemErrorCode () const
{
   return sysErr;
}

GString GException::getMessage () const
{
   const char* msg = this->msg;
   if (msg == null)
      return GString::Empty;
   else
      return GString(msg);
}

void GException::setMessage ( const char* msg )
{
   this->msg = msg;
}

GString GException::toString () const
{
   GString ret(256);
   GString className(typeid(*this).name());
   if (className.beginsWith("class "))
      className.remove(0, 6);
   ret += className;
   ret += ": ";
   ret += getMessage();
   return ret;
}

#ifndef GTHROW_NO_LOGGING
void GException::logForGThrow ( const char* moduleName, int lineNr ) const
{
   if (GLog::Filter(GLog::PROD))
   {
      GString fullMsg(256);
      fullMsg += typeid(*this).name();
      if (moduleName != null)
      {
         fullMsg += " thrown at ";
         fullMsg += moduleName;
         fullMsg += "#";
         fullMsg += GInteger::ToString(lineNr);
      }
      fullMsg += ": ";
      GString msgstr = getMessage();
      if (msgstr != "")
         fullMsg += msgstr;
      else
         fullMsg += "No further information.";
      GLog::Log(this, fullMsg);
   }
}
#endif
