/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_THREAD
#define __GLIB_THREAD

#include "glib/GRunnable.h"
#include "glib/primitives/GString.h"
#include "glib/util/GEventSemaphore.h"
#include "glib/gui/GWindow.h"

class GThread : public GObject, public GRunnable
{
   friend class GWindow;
   friend class Entry;

   public:
      enum Priority
      {
         PRIORITY_MIN_MINIMUM = -7,
         PRIORITY_MIN_MINUS = -6,
         PRIORITY_MIN = -5,
         PRIORITY_MIN_PLUS = -4,
         PRIORITY_MIN_MAXIMUM = -3,
         PRIORITY_NORMAL_MINIMUM = -2,
         PRIORITY_NORMAL_MINUS = -1,
         PRIORITY_NORMAL = 0,
         PRIORITY_NORMAL_PLUS = 1,
         PRIORITY_NORMAL_MAXIMUM = 2,
         PRIORITY_MAX_MINIMUM = 3,
         PRIORITY_MAX_MINUS = 4,
         PRIORITY_MAX = 5,
         PRIORITY_MAX_PLUS = 6,
         PRIORITY_MAX_MAXIMUM = 7
      };

   private:
      const GString name;
      const int stacksize;
      bool isrunning;
      GRunnable* runner;
      int threadid;
      Priority prio;
      int suspendCount;
      GEventSemaphore eventSemWaitForGUI;
      GEventSemaphore eventSemWaitForThreadEntry;
      GEventSemaphore eventSemWaitForThreadCreator;
      GEventSemaphore eventSemWaitForExit;
      GEventSemaphore eventSemWaitForInterrupt;

   protected:
      HAB hAB_Thread;
      HMQ hMQ_Thread;

   private:
      static GThread* MainThread;

   private:
      class UMParams : public GObject
      {
         friend class GThread;
         friend class GWindow;
         GThread& postingThread;
         bool autoNotify;
         UMParams ( GThread& postingThread, bool autoNotify );
         virtual ~UMParams ();
      };

   public:
      explicit GThread ( const GString& name = GString::Empty, 
                         int stacksize = 262144,
                         GRunnable* runner = null );
      virtual ~GThread ();

   private:
      GThread ( const GThread& ); //!< Disable the copy constructor.
      void operator= ( const GThread& ); //!< Disable the assignment operator.

   public:
      static GThread& GetCurrentThread ();
      static int GetCurrentThreadID ();
      const GString& getName () const;
      int getThreadID () const;
      void guiUserMessageHandlerHasFinished ();
      void interrupt ();
      bool isMainThread () const;
      virtual bool isRunning () const;
      void resume ();
      void sendGuiUserMessage ( GWindow& win, GUserMessage& um, bool autoNotify = true );
      void setPriority ( GThread::Priority prio );
      static void Sleep ( int msec );
      virtual void start ();
      void suspend ();
      virtual void waitUntilTheThreadHasExited ();

   private:
      void activateCurrentPriority ();
      static GArray<GThread>& GetThreadObjects ();
      static void GTHREADENTRY OSThreadEntry ( void* arg );
      void threadEntry ();

   protected:
      virtual void run ();

   public:
      HAB getHAB () const;
      HMQ getHMQ () const;
};

#endif
