/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include <process.h>
#include "glib/util/GLog.h"
#include "glib/exceptions/GThreadStartException.h"
#include "glib/exceptions/GIllegalStateException.h"
#include "glib/exceptions/GIllegalArgumentException.h"
#include "glib/exceptions/GSystemException.h"
#include "glib/GProgram.h"
#include "glib/sys/GSystem.h"
#include "glib/gui/event/GUserMessage.h"

/**
 * \class GThread
 * A thread class that is directly ispired by the Thread class of
 * the Java language.
 *
 * @author  Leif Erik Larsen
 * @since   1999.11.10
 */

/**
 * \class GThread::UMParams
 * This class is for internal usage in GLib only.
 * It is used to pass some extended information from 
 * {@link #sendGuiUserMessage} to the handling window object.
 * 
 * @author  Leif Erik Larsen
 * @since   2004.02.01
 */

/**
 * \enum GThread::Priority
 * These are the supported ID's for setting and getting the
 * runtime priority of the thread.
 *
 * @author  Leif Erik Larsen
 * @since   2000.10.06
 */

/** 
 * \var GThread::name
 * The thread ID string. 
 */

/** 
 * \var GThread::stacksize
 * The stack size (in bytes) of the thread. 
 */

/** 
 * \var GThread::isrunning
 * True while the thread is actually running. 
 */

/** 
 * \var GThread::runner
 * The runnable of which to call by {@link #run}. 
 */

/** 
 * \var GThread::threadid
 * The system dependent thread ID (TID). -1 means "thread not created", 
 * 0 means "current thread", else it is the true thread ID. 
 * In Win32 this is not the hThread, but the threadID.
 */

/** 
 * \var GThread::prio
 * The priority of the thread. 
 */

/**
 * \var GThread::suspendCount
 * This variable is used to count the number of times {@link #suspend}
 * has been called with respect to how many times {@link #resume}
 * has been called.
 */

/** 
 * \var GThread::eventSemWaitForGUI
 * Semaphore used by {@link #sendGuiUserMessage}. 
 */

/**
 * \var GThread::eventSemWaitForThreadEntry
 * This semaphore is used to make sure that {@link #start} does not
 * return before the thread entry function has actually been
 * called by the new thread.
 */

/** 
 * \var GThread::eventSemWaitForExit
 * Semaphore used by {@link #waitUntilTheThreadHasExited}. 
 */

/** 
 * \var GThread::eventSemWaitForInterrupt
 * Semaphore used by {@link #sleep}. 
 */

/**
 * @author  Leif Erik Larsen
 * @since   1999.11.10
 */
GThread::GThread ( const GString& name, int stacksize, GRunnable* runner )
        :name(name),
         stacksize(stacksize > 16384 ? stacksize : 16384),
         isrunning(false),
         runner(runner),
         threadid(-1),
         hAB_Thread(null),
         hMQ_Thread(null),
         prio(GThread::PRIORITY_NORMAL),
         suspendCount(0)
{
   // Make sure that only one thread accesses the vector of
   // existing threads while we are adding our self to it.
   GArray<GThread>& allThreads = GThread::GetThreadObjects();
   GObject::Synchronizer synch(allThreads);
   allThreads.add(this, false);

   // Assume the very first instance of this class is to be assigned 
   // to {@link GThread::MainThread}.
   if (GThread::MainThread == null)
   {
      GThread::MainThread = this;
      // We must fetch the Thread-ID manually, because the main thread is 
      // already running. It was created by the OS, not by us.
      isrunning = true;
      threadid = GThread::GetCurrentThreadID();
   }
}

/**
 * If the thread is still running when this destructor is called
 * then we will automatically {@link #interrupt} it and wait for 
 * it to exit before returning.
 *
 * @author  Leif Erik Larsen
 * @since   1999.11.10
 */
GThread::~GThread ()
{
   if (isRunning())
   {
      if (GLog::Filter(GLog::TEST))
         GLog::Log(this, "Thread '%s' is still running, so wait for it to exit (%s:%d)", GVArgs(getName()).add(__FUNCTION__).add(__LINE__));
      interrupt();
      waitUntilTheThreadHasExited();
      if (GLog::Filter(GLog::TEST))
         GLog::Log(this, "Thread '%s' has exited. Continue destruction of GThread.", GVArgs(getName()).add(__FUNCTION__).add(__LINE__));
   }

   // Make sure that only one thread accesses the vector of
   // existing threads while we are removing our self from it.
   GArray<GThread>& allThreads = GThread::GetThreadObjects();
   GObject::Synchronizer synch(allThreads);
   int idx = allThreads.indexOf(*this);
   if (idx >= 0)
      allThreads.remove(idx);
}

/**
 * Get a reference to the static vector containing a reference 
 * to each instance of the GThread class.
 *
 * @author  Leif Erik Larsen
 * @since   2005.01.20
 */
GArray<GThread>& GThread::GetThreadObjects ()
{
   static GArray<GThread>* ThreadObjects = null;
   if (ThreadObjects == null)
      ThreadObjects = new GArray<GThread>(16);
   return *ThreadObjects;
}

/**
 * Get the thread object of the calling thread.
 *
 * @author  Leif Erik Larsen
 * @since   2004.01.03
 * @throws  GIllegalStateException if the calling thread is 
 *               unknown to GLib, and therefore is not a GThread.
 */
GThread& GThread::GetCurrentThread ()
{
   // Make sure that only one thread accesses the vector of
   // existing threads while we are searching it.
   // TODO: Speed optimize by using GHashtable instead of GArray.
   GArray<GThread>& allThreads = GThread::GetThreadObjects();
   GObject::Synchronizer synch(allThreads);
   int tid = GThread::GetCurrentThreadID();
   for (int i=0, count=allThreads.getCount(); i<count; i++)
   {
      GThread& t = allThreads[i];
      if (t.getThreadID() == tid && t.isRunning())
         return t;
   }
   gthrow_(GIllegalStateException("Current thread is not a GThread!"));
}

const GString& GThread::getName () const
{
   return name;
}

/**
 * Get the ID of this thread, or -1 if this thread is currently
 * not alive (running).
 *
 * One typical usage of this method is in conjunction with the
 * static method {@link GThread#getCurrentThreadID} to test
 * if the current/active/calling thread is the same thread
 * as the thread of this thread object. For example as shown
 * in the below code:
 *
 * <pre>
 * if (GThread::getCurrentThreadID() != threadObj->geThreadID())
 *    gthrow_(GIllegalStateException(GStringf("To be called by the \"%s\" thread only!", threadObj->getName())));
 * </pre>
 *
 * @author  Leif Erik Larsen
 * @since   2000.10.12
 */
int GThread::getThreadID () const
{
   return threadid;
}

/**
 * Get the thread ID of the calling thread.
 *
 * @author  Leif Erik Larsen
 * @since   2000.10.11
 */
int GThread::GetCurrentThreadID ()
{
   TIB *tib; // Thread Info Block.
   ::DosGetInfoBlocks(&tib, null);
   int tid = tib->tib_ptib2->tib2_ultid;
   return tid;
}

/**
 * Return true if this is the main program thread.
 * The main program thread is the thread that is used to 
 * call the main entry method {@link GProgram#mainStart}.
 * On most programs this is also the one and only GUI thread.
 *
 * @author  Leif Erik Larsen
 * @since   2004.05.14
 */
bool GThread::isMainThread () const
{
   if (this == GThread::MainThread)
      return true;
   else
      return false;
}

/**
 * Set the runtime priority of the thread.
 *
 * @author  Leif Erik Larsen
 * @since   2000.10.06
 * @throws  GIllegalArgumentException if the specified priority is not
 *                    within the range defined by the enumeration type.
 */
void GThread::setPriority ( GThread::Priority prio )
{
   switch (prio)
   {
      case PRIORITY_MIN_MINIMUM:
      case PRIORITY_MIN_MINUS:
      case PRIORITY_MIN:
      case PRIORITY_MIN_PLUS:
      case PRIORITY_MIN_MAXIMUM:
      case PRIORITY_NORMAL_MINIMUM:
      case PRIORITY_NORMAL_MINUS:
      case PRIORITY_NORMAL:
      case PRIORITY_NORMAL_PLUS:
      case PRIORITY_NORMAL_MAXIMUM:
      case PRIORITY_MAX_MINIMUM:
      case PRIORITY_MAX_MINUS:
      case PRIORITY_MAX:
      case PRIORITY_MAX_PLUS:
      case PRIORITY_MAX_MAXIMUM:
           this->prio = prio;
           activateCurrentPriority();
           break;

      default:
           gthrow_(GIllegalArgumentException(GString("GThread::setPriority(): Invalid priority: %d", GVArgs(prio))));
   }
}

void GThread::activateCurrentPriority ()
{
   if (!isRunning() || threadid == -1)
      return;

   ULONG prioClass;
   LONG prioDelta;
   switch (prio)
   {
      case PRIORITY_MIN_MINIMUM:    prioClass = PRTYC_IDLETIME; prioDelta = -31; break;
      case PRIORITY_MIN_MINUS:      prioClass = PRTYC_IDLETIME; prioDelta = -15; break;
      case PRIORITY_MIN:            prioClass = PRTYC_IDLETIME; prioDelta = 0; break;
      case PRIORITY_MIN_PLUS:       prioClass = PRTYC_IDLETIME; prioDelta = +15; break;
      case PRIORITY_MIN_MAXIMUM:    prioClass = PRTYC_IDLETIME; prioDelta = +31; break;

      case PRIORITY_NORMAL_MINIMUM: prioClass = PRTYC_REGULAR; prioDelta = -31; break;
      case PRIORITY_NORMAL_MINUS:   prioClass = PRTYC_REGULAR; prioDelta = -15; break;
      case PRIORITY_NORMAL:         prioClass = PRTYC_REGULAR; prioDelta = 0; break;
      case PRIORITY_NORMAL_PLUS:    prioClass = PRTYC_REGULAR; prioDelta = +15; break;
      case PRIORITY_NORMAL_MAXIMUM: prioClass = PRTYC_REGULAR; prioDelta = +31; break;

      case PRIORITY_MAX_MINIMUM:    prioClass = PRTYC_TIMECRITICAL; prioDelta = -31; break;
      case PRIORITY_MAX_MINUS:      prioClass = PRTYC_TIMECRITICAL; prioDelta = -15; break;
      case PRIORITY_MAX:            prioClass = PRTYC_TIMECRITICAL; prioDelta = 0; break;
      case PRIORITY_MAX_PLUS:       prioClass = PRTYC_TIMECRITICAL; prioDelta = +15; break;
      case PRIORITY_MAX_MAXIMUM:    prioClass = PRTYC_TIMECRITICAL; prioDelta = +31; break;

      default:                      prioClass = PRTYC_REGULAR; prioDelta = 0; break;
   }
   DosSetPriority(PRTYS_THREAD, prioClass, prioDelta, threadid);
}

GThread::UMParams::UMParams ( GThread& postingThread, bool autoNotify )
                  :postingThread(postingThread),
                   autoNotify(autoNotify)
{
}

GThread::UMParams::~UMParams ()
{
}

/**
 * Send the specified user message to the specified window and wait 
 * for the window message handler to call 
 * {@link #guiUserMessageHandlerHasFinished}.
 *
 * The first message parameter will be the specified use message,
 * and the second message parameter will be a pointer to
 * a temporary instance of a private class; {@link #UMParams}.
 *
 * @author  Leif Erik Larsen
 * @since   2004.02.01
 * @param   win  The window of where to send the user message.
 * @param   um   The user defined message of which to send.
 *               If the specified window is an instance of
 *               {@link GDialogPanel} then the user message 
 *               is allowed to be of type {@link GDialogMessage},
 *               in which case the user defined dialog message handler
 *               ({@link GDialogMessageHandler#handleDialogMessage}) 
 *               will be called with the specified dialog message.
 * @param   autoNotify If this parameter is true (which is default) 
 *               then the window message handler will automatically 
 *               call {@link #guiUserMessageHandlerHasFinished} when
 *               the user message has been handled by the GUI.
 *               Else the GUI-code must manually notify us by that 
 *               method when the calling thread should no longer wait.
 * @see     GWindow#GM_USERMESSAGE_FROM_THREAD
 * @see     #guiUserMessageHandlerHasFinished
 */
void GThread::sendGuiUserMessage ( GWindow& win, GUserMessage& um, bool autoNotify )
{
   UMParams ump(*this, autoNotify);
   GWindowMessage::Param1 p1 = GWindowMessage::Param1(&um);
   GWindowMessage::Param2 p2 = GWindowMessage::Param2(&ump);
   eventSemWaitForGUI.reset();
   win.postMessage(GWindow::GM_USERMESSAGE_FROM_THREAD, p1, p2);
   eventSemWaitForGUI.wait();
}

/**
 * The window that receives the message posted by 
 * {@link #sendGuiUserMessage} is responsible for calling this method 
 * as soon as the GUI thread has finished handling the message, so 
 * that we (the Thread object) can continue our work afterwards.
 *
 * @see #sendGuiUserMessage
 */
void GThread::guiUserMessageHandlerHasFinished ()
{
   eventSemWaitForGUI.notifyAll();
}

/**
 * Return true if and only if the thread has not yet returned
 * from its {@link #run} method.
 *
 * @author  Leif Erik Larsen
 * @since   2000.10.06
 */
bool GThread::isRunning () const
{
   return isrunning;
}

/**
 * Wait for the thread to exit.
 *
 * @author  Leif Erik Larsen
 * @since   2004.01.03
 */
void GThread::waitUntilTheThreadHasExited ()
{
   if (isrunning)
      eventSemWaitForExit.wait();
}

/**
 * Interrupt the thread if it is running. That is, if the thread 
 * represented by this object is actually running and is about to 
 * {@link #sleep} then we will interrupt it so that the {@link #sleep}
 * statement returns immediately.
 *
 * @author  Leif Erik Larsen
 * @since   2004.01.03
 */
void GThread::interrupt ()
{
   eventSemWaitForInterrupt.notifyAll();
}

/**
 * This static method will cause the calling thread to sleep for the 
 * specified number of milliseconds, or until it is interrupted
 * by some other thread.
 *
 * @see    #interrupt
 */
void GThread::Sleep ( int millis )
{
   GThread& thread = GThread::GetCurrentThread();
   thread.eventSemWaitForInterrupt.reset();
   thread.eventSemWaitForInterrupt.wait(millis);
}

/**
 * Temporarily suspend the thread until {@link #resume} is issued.
 *
 * Subsequent calls to this method nest, so there must be as many
 * calls to {@link #resume} as there were to {@link #suspend}
 * before the thread actually resumes execution.
 *
 * @author  Leif Erik Larsen
 * @since   2000.10.06
 * @see     #resume
 */
void GThread::suspend ()
{
   GObject::Synchronizer synch(*this);
   if (isrunning)
   {
      if (suspendCount++ == 0)
         ::DosSuspendThread(threadid);
   }
}

/**
 * Let the thread continue after it has been previously
 * suspended by {@link #suspend}.
 *
 * Subsequent calls to {@link #suspend} nest, so there must be as
 * many calls to this method as there were to {@link #suspend}
 * before the thread actually resumes execution.
 *
 * @author  Leif Erik Larsen
 * @since   2000.10.06
 * @see     #suspend
 */
void GThread::resume ()
{
   GObject::Synchronizer synch(*this);
   if (isrunning && suspendCount > 0)
   {
      if (--suspendCount == 0)
         ::DosResumeThread(threadid);
   }
}

/**
 * The run() method is the main entry point of the thread.
 * It will be called by the new thread after start() has been called.
 */
void GThread::run ()
{
   if (runner != null)
      runner->run();
}

/** 
 * Get the OS/2 "Anchor Block Handle" (HAB) for this thread.
 * The returned handle is valid only while the thread is running.
 * This method is available on OS/2 only.
 *
 * @author  Leif Erik Larsen
 * @since   2005.01.16
 */
HAB GThread::getHAB () const
{
   return hAB_Thread;
}

/** 
 * Get the OS/2 "Message Queue Handle" (HMQ) for this thread.
 * The returned handle is valid only while the thread is running.
 * This method is available on OS/2 only.
 *
 * @author  Leif Erik Larsen
 * @since   2005.01.16
 */
HMQ GThread::getHMQ () const
{
   return hMQ_Thread;
}

/**
 * @author  Leif Erik Larsen
 * @since   2005.01.20
 */
void GThread::threadEntry ()
{
   int tid = GThread::GetCurrentThreadID();

   // Make sure every thread has a GUI event queue. This is needed in order 
   // to make sure that most GLib API's might be called by any thread and 
   // not only by the main thread. One example is GvfsLocal::removeFile(), 
   // which in some cases will call a WPS-related method that needs the 
   // calling thread to have a GUI event queue.
   hAB_Thread = ::WinInitialize(0);
   hMQ_Thread = ::WinCreateMsgQueue(hAB_Thread, 0);
   ::WinCancelShutdown(hMQ_Thread, TRUE);

   // Reset the semaphore so that any new threads
   // calling {@link #waitUntilTheThreadHasExited} will wait.
   eventSemWaitForExit.reset();

   // Wait until the hThread variable has been set.
   eventSemWaitForThreadCreator.wait();

   // Now the thread is running
   isrunning = true;

   // Get the Thread ID of the thread.
   threadid = tid;
   suspendCount = 0;

   // Activate the thread priority as is possibly requested
   // by recent call to {@link #setPriority}.
   activateCurrentPriority();

   // Notify the thread that waits for us to start, in start().
   eventSemWaitForThreadEntry.notifyAll();

   // Call the user/program defined code.
   // Protect the call in a low-level and system-dependent exception
   // handler so that a log-statement is written in case the thread traps.
   try {
      bool ok = true;
      GTRY(e, true) {
         run();
      } GCATCH(e) {
         ok = false;
      } GENDCATCH();
      if (!ok)
         gthrow_(GException(GString("System trap in thread '%s'.", GVArgs(name))));
   } catch (GException& e) {
      GString msg("Unhandled exception in thread '%s'.%sMessage is: %s", GVArgs(name).add(GString::EOL).add(e.what()));
      GString stackTrace = e.getStackTrace(msg);
      GLog::Log(this, stackTrace);
   } catch (std::exception& e) {
      GString msg("Unhandled exception in thread '%s'.%sMessage is: %s", GVArgs(name).add(GString::EOL).add(e.what()));
      GLog::Log(this, msg);
   } catch (APIRET& rc) {
      GString msg("Unhandled exception in thread '%s'.%sMessage is: %s (APIRET=%d)", GVArgs(name).add(GString::EOL).add(GSystem::GetApiRetString(rc)).add(rc));
      GLog::Log(this, msg);
   } catch (...) {
      GString msg("Unhandled exception of unknown type in thread '%s'.", GVArgs(name));
      GLog::Log(this, msg);
   }

   ::WinDestroyMsgQueue(hMQ_Thread);
   ::WinTerminate(hAB_Thread);
   hMQ_Thread = null;
   hAB_Thread = null;
}

void GTHREADENTRY GThread::OSThreadEntry ( void* arg )
{
   // Run the thread.
   GThread* t = (GThread*) arg;
   t->threadEntry();

   // Now that the thread is no longer running (from the user code point
   // of view) we must clear some variables.
   t->isrunning = false;
   t->threadid = -1;
   
   // Notify anyone waiting for us to exit.
   t->eventSemWaitForExit.notifyAll();

   // Terminate the thread.
   _endthread();
}

/**
 * Start the thread.
 *
 * The new thread will call {@link #run} automatically.
 *
 * @throws   GThreadStartException if the thread could not be started.
 */
void GThread::start ()
{
   if (isRunning())
      gthrow_(GIllegalStateException("Thread already running"));

   eventSemWaitForThreadEntry.reset();

   int tid = _beginthread(GThread::OSThreadEntry, null, stacksize, this);
   if (tid == -1)
      gthrow_(GThreadStartException(GString("Failed to start thread: \"%s\"", GVArgs(name))));

   // Notify the thread that it is ok to run.
   if (GLog::Filter(GLog::DEBUG))
      GLog::Log(this, "Notify thread '%s' that it can start (%s:%d)", GVArgs(name).add(__FUNCTION__).add(__LINE__));
   eventSemWaitForThreadCreator.notifyAll();

   // Wait until the thread has actually started.
   if (GLog::Filter(GLog::DEBUG))
      GLog::Log(this, "Wait until thread '%s' has actually started (%s:%d)", GVArgs(name).add(__FUNCTION__).add(__LINE__));
   eventSemWaitForThreadEntry.wait();

   // ---
   if (GLog::Filter(GLog::DEBUG))
      GLog::Log(this, "The thread starter for thread '%s' now returns (%s:%d)", GVArgs(name).add(__FUNCTION__).add(__LINE__));
   return;
}
