/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include <ctype.h>
#include <stdio.h>
#include "glib/util/GSearchPath.h"
#include "glib/vfs/GFile.h"
#include "glib/GProgram.h"
#include "glib/GEnvironment.h"

GEnvironment::GEnvironment ()
             :envBag(256, -3, true) // ignoreCase=true
{
   // Initialize our bag of environment variables.
   for (int c1=0; _environ[c1]; setEnv(_environ[c1++]));
}

GEnvironment::~GEnvironment ()
{
}

int GEnvironment::getVarCount () const
{
   return envBag.getCount();
}

const GString& GEnvironment::getKey ( int index ) const
{
   return envBag.getKey(index);
}

const char* GEnvironment::getEnv ( const char *envName, const char* defaultVal ) const
{
   const GString* val = envBag.get(envName);
   if (val == null)
      return defaultVal;
   else
      return val->cstring();
}

const GString& GEnvironment::getEnv ( int index ) const
{
   return envBag.getIndexedItem(index);
}

void GEnvironment::setEnv ( const GString& envScript )
{
   if (envScript == "")
      return;

   // Parse the variable name.
   const char* ptrStart = envScript.cstring();
   const char* ptr = strchr((char*) ptrStart, '=');
   if (ptr == null)
      return;
   size_t envNameLen = size_t(ptr - ptrStart);
   char *envName = new char[envNameLen + 1];
   memcpy(envName, ptrStart, envNameLen);
   envName[envNameLen] = '\0';

   // String blank ends and change to UPPERCASE letters only.
   GString temp(envName);
   temp.trim().toUpperCase();
   strcpy(envName, temp.cstring());

   ptrStart += envNameLen + 1; // Jump to next character behind the equal sign
   while (isspace(*ptrStart))
      ptrStart++;

   // Parse the variable value.
   const char* macroStart = null;
   GString buff(1024);
   for (ptr = ptrStart; *ptr; ptr++)
   {
      if (*ptr == '%')
      {
         if (macroStart == null)
         {
            macroStart = ptr + 1;
         }
         else
         {
            if (macroStart == ptr)
            {
               buff += '%';
               macroStart = ptr + 1;
            }
            else
            {
               size_t macroLen = size_t(ptr - macroStart);
               char* macroName = new char[macroLen + 1];
               memcpy(macroName, macroStart, macroLen);
               macroName[macroLen] = '\0';
               const char* macroVal = getEnv(macroName);
               if (macroVal == null)
                  buff += '%';
               const char *copyFrom = (macroVal == null ? macroName : macroVal);
               while (*copyFrom)
                  buff += *copyFrom++;
               delete macroName;
               if (macroVal == null)
                  macroStart = ptr + 1;
               else
                  macroStart = null;
            }
         }
      }
      else
      if (macroStart == null)
         buff += *ptr;
   }

   if (macroStart != null)
   {
      buff += '%';
      while (*macroStart)
         buff += *macroStart++;
   }

   // ---
   if (buff == "")
   {
      envBag.remove(envName);
   }
   else
   {
      GString* s = new GString(buff);
      if (envBag.containsKey(envName))
         envBag.update(envName, s);
      else
         envBag.put(envName, s);
   }

   // ---
   delete envName;
}

GString GEnvironment::makeString ( char eq, char eol, char end1, char end2 ) const
{
   GString buff(4096);
   int num = getVarCount();
   for (int i=0; i<num; i++)
   {
      buff += getKey(i);
      buff += eq;
      buff += getEnv(i);
      buff += eol;
   }
   buff += end1;
   buff += end2;
   return buff;
}

bool GEnvironment::locateFileInEnv ( const GString& searchPath, const GString& objToFind, GFile& obj ) const
{
   GFile file(objToFind);

   // First, try to find the object in the "Current Directory", but only
   // if the given file does not contain a drive or directory specification.
   bool driveOrDir = (file.containsDrive() || file.containsDirectory());
   if (driveOrDir)
   {
      if (GFile::LocateExecutableFileInDir(GString::Empty, objToFind, obj))
         return true;
   }
   else
   {
      try {
         GString dir = GFile::GetCurrentDir();
         if (GFile::LocateExecutableFileInDir(dir, objToFind, obj))
            return true;
      } catch (APIRET& /*rc*/) {
         // Not critical. Just ignore this.
      }
   }

   // Search the specified path only if the specified object doesn't include
   // a drive or directory specification.
   if (driveOrDir)
      return false;

   // Search for the object in the path.
   bool ret = false;
   GSearchPath spath(searchPath);
   const int num = spath.getCount();
   for (int i=0; i<num && !ret; i++)
   {
      const GString& nextDir = spath.getIndexedDir(i);
      if (GFile::LocateExecutableFileInDir(nextDir, objToFind, obj))
         ret = true;
   }

   // We didn't find the object in the specified path.
   return ret;
}

GString GEnvironment::which ( const GString& prog ) const
{
   GFile obj;
   GString path = getEnv("PATH", "");
   if (locateFileInEnv(path, prog, obj))
      return obj.getFullPath();
   else
      return GString::Empty;
}


