/* bigint1.c */

/* Test bigint functions.  Note that these functions are for internal
   use by the C library only. */

#undef NDEBUG
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <assert.h>
#include <emx/bigint.h>

#define NW      128
#define ZNW     (2*NW+1)

void clear (_bi_bigint *b)
{
  int i;

  b->n = -1;
  for (i = 0; i < NW; ++i)
    b->v[i] = 0xffffffff;
}


static void usage (void)
{
  puts ("Usage: bigint1 [-f] [-i <iterations>] [-v]");
  exit (1);
}


int main (int argc, char *argv[])
{
  _BI_DECLARE (x, NW);
  _BI_DECLARE (y, NW);
  _BI_DECLARE (z, ZNW);
  _BI_DECLARE (r, NW);
  _BI_DECLARE (x0, NW);
  _BI_DECLARE (y0, NW);
  _BI_DECLARE (r0, NW);
  _bi_word w;
  int c, i, iter, niter = 0, forever = 0, quiet = 0;

  while ((c = getopt (argc, argv, "fi:q")) != -1)
    switch (c)
      {
      case 'f':
        forever = 1;
        break;
      case 'i':
        niter = atoi (optarg);
        if (niter == 0)
          usage ();
        break;
      case 'q':
        quiet = 1;
        break;
      default:
        usage ();
      }
  if (optind != argc)
    usage ();

  _BI_INIT (x);
  _BI_INIT (y);
  _BI_INIT (z);
  _BI_INIT (r);
  _BI_INIT (x0);
  _BI_INIT (y0);
  _BI_INIT (r0);

  puts ("Testing with special numbers...");

  /* Test _BI_WORDS(). */

  assert (_BI_WORDS (0) == 0);
  assert (_BI_WORDS (1) == 1);
  assert (_BI_WORDS (_BI_WORDSIZE - 1) == 1);
  assert (_BI_WORDS (_BI_WORDSIZE + 0) == 1);
  assert (_BI_WORDS (_BI_WORDSIZE + 1) == 2);

  /* Test _bi_set_w(). */

  clear (&x);
  assert (_bi_set_w (&x, 0, 0) == 0);
  assert (x.n == 0);
  assert (_bi_set_w (&x, 0, 0x12345678) == 1);
  assert (_bi_set_w (&x, 1, 0x12345678) == 0);
  assert (x.n == 1);
  assert (x.v[0] == 0x12345678);
  assert (_bi_set_w (&x, 1, 0) == 0);
  assert (x.n == 0);

  /* Test _bi_set_d(). */

  clear (&y);
  assert (_bi_set_d (&y, 0, 0) == 0);
  assert (y.n == 0);
  assert (_bi_set_d (&y, 0, 0x12345678) == 1);
  assert (_bi_set_d (&y, 1, 0x12345678) == 0);
  assert (y.n == 1);
  assert (y.v[0] == 0x12345678);
  assert (_bi_set_d (&y, 1, 0x123456789abcdefULL) == 1);
  assert (_bi_set_d (&y, 2, 0x123456789abcdefULL) == 0);
  assert (y.n == 2);
  assert (y.v[0] == 0x89abcdef);
  assert (y.v[1] == 0x01234567);
  assert (_bi_set_d (&y, 1, 0) == 0);
  assert (y.n == 0);

  /* Test _bi_set_b(). */

  clear (&y);
  _bi_set_w (&x, NW, 0);
  assert (_bi_set_b (&y, 0, &x) == 0);
  assert (y.n == 0);

  clear (&y);
  _bi_set_w (&x, NW, 0x12345678);
  assert (_bi_set_b (&y, 0, &x) == 1);
  assert (_bi_set_b (&y, 1, &x) == 0);
  assert (y.n == 1);
  assert (y.v[0] == 0x12345678);

  clear (&y);
  _bi_set_d (&x, NW, 0x123456789abcdef0ULL);
  assert (_bi_set_b (&y, 1, &x) == 1);
  assert (_bi_set_b (&y, 2, &x) == 0);
  assert (y.n == 2);
  assert (y.v[0] == 0x9abcdef0);
  assert (y.v[1] == 0x12345678);

  /* Test _bi_fls(). */

  assert (_bi_set_w (&x, NW, 0) == 0);
  assert (_bi_fls (&x) == 0);
  assert (_bi_set_w (&x, NW, 0x00000001) == 0);
  assert (_bi_fls (&x) == 1);
  assert (_bi_set_w (&x, NW, 0x00000002) == 0);
  assert (_bi_fls (&x) == 2);
  assert (_bi_set_w (&x, NW, 0x00000003) == 0);
  assert (_bi_fls (&x) == 2);
  assert (_bi_set_w (&x, NW, 0x00000004) == 0);
  assert (_bi_fls (&x) == 3);
  assert (_bi_set_w (&x, NW, 0x12345678) == 0);
  assert (_bi_fls (&x) == 29);
  assert (_bi_set_w (&x, NW, 0x80000000) == 0);
  assert (_bi_fls (&x) == 32);
  assert (_bi_set_w (&x, NW, 0xffffffff) == 0);
  assert (_bi_fls (&x) == 32);
  assert (_bi_set_d (&x, NW, 0x0000123456789abcULL) == 0);
  assert (_bi_fls (&x) == 45);

  /* Test _bi_cmp_bb(). */

  _bi_set_w (&x, NW, 0);
  _bi_set_w (&y, NW, 0);
  assert (_bi_cmp_bb (&x, &y) == 0);
  _bi_set_w (&y, NW, 1);
  assert (_bi_cmp_bb (&x, &y) == -1);
  assert (_bi_cmp_bb (&y, &x) == 1);
  _bi_set_w (&x, NW, 1);
  assert (_bi_cmp_bb (&x, &y) == 0);
  _bi_set_w (&y, NW, 2);
  assert (_bi_cmp_bb (&x, &y) == -1);
  assert (_bi_cmp_bb (&y, &x) == 1);

  _bi_set_d (&x, NW, 0x8977affe4711deafULL);
  _bi_set_d (&y, NW, 0x8977affe4711deafULL);
  assert (_bi_cmp_bb (&x, &y) == 0);
  _bi_set_d (&y, NW, 0x8977affe4711deaeULL);
  assert (_bi_cmp_bb (&x, &y) == 1);
  assert (_bi_cmp_bb (&y, &x) == -1);
  _bi_set_d (&x, NW, 0x8977afff4711deffULL);
  assert (_bi_cmp_bb (&x, &y) == 1);
  assert (_bi_cmp_bb (&y, &x) == -1);

  /* Test _bi_cmp_pow2(). */

  _bi_set_w (&x, NW, 0);
  assert (_bi_cmp_pow2 (&x, -1) == -1); /* 0 < 0.5 */
  assert (_bi_cmp_pow2 (&x, 0) == -1); /* 0 < 1 */
  assert (_bi_cmp_pow2 (&x, 1) == -1);
  assert (_bi_cmp_pow2 (&x, 32) == -1);
  assert (_bi_cmp_pow2 (&x, 33) == -1);

  _bi_set_w (&x, NW, 0x0001000);
  assert (_bi_cmp_pow2 (&x, -1) == 1);
  assert (_bi_cmp_pow2 (&x, 0) == 1);
  assert (_bi_cmp_pow2 (&x, 11) == 1);
  assert (_bi_cmp_pow2 (&x, 12) == 0);
  assert (_bi_cmp_pow2 (&x, 13) == -1);

  _bi_set_w (&x, NW, 0x0001001);
  assert (_bi_cmp_pow2 (&x, -1) == 1);
  assert (_bi_cmp_pow2 (&x, 0) == 1);
  assert (_bi_cmp_pow2 (&x, 11) == 1);
  assert (_bi_cmp_pow2 (&x, 12) == 1);
  assert (_bi_cmp_pow2 (&x, 13) == -1);

  _bi_set_d (&x, NW, 0x0010000000000000ULL);
  assert (_bi_cmp_pow2 (&x, -1) == 1);
  assert (_bi_cmp_pow2 (&x, 0) == 1);
  assert (_bi_cmp_pow2 (&x, 51) == 1);
  assert (_bi_cmp_pow2 (&x, 52) == 0);
  assert (_bi_cmp_pow2 (&x, 53) == -1);

  _bi_set_d (&x, NW, 0x0010000000000100ULL);
  assert (_bi_cmp_pow2 (&x, -1) == 1);
  assert (_bi_cmp_pow2 (&x, 0) == 1);
  assert (_bi_cmp_pow2 (&x, 51) == 1);
  assert (_bi_cmp_pow2 (&x, 52) == 1);
  assert (_bi_cmp_pow2 (&x, 53) == -1);

  _bi_set_d (&x, NW, 0x0018000000000000ULL);
  assert (_bi_cmp_pow2 (&x, -1) == 1);
  assert (_bi_cmp_pow2 (&x, 0) == 1);
  assert (_bi_cmp_pow2 (&x, 51) == 1);
  assert (_bi_cmp_pow2 (&x, 52) == 1);
  assert (_bi_cmp_pow2 (&x, 53) == -1);

  /* Test _bi_add_bb(). */

  clear (&z);
  _bi_set_w (&x, NW, 0);
  _bi_set_w (&y, NW, 0);
  assert (_bi_add_bb (&z, 0, &x, &y) == 0);
  assert (z.n == 0);
  assert (_bi_add_bb (&z, NW, &x, &y) == 0);
  assert (z.n == 0);

  _bi_set_w (&y, NW, 0x12345678);
  assert (_bi_add_bb (&z, 0, &x, &y) == 1);
  assert (_bi_add_bb (&z, 1, &x, &y) == 0);
  assert (z.n == 1);
  assert (z.v[0] == 0x12345678);

  _bi_set_w (&x, NW, 0xedcba987);
  assert (_bi_add_bb (&z, 1, &x, &y) == 0);
  assert (z.n == 1);
  assert (z.v[0] == 0xffffffff);

  _bi_set_w (&x, NW, 0xedcba988);
  assert (_bi_add_bb (&z, 1, &x, &y) == 1);
  assert (_bi_add_bb (&z, 2, &x, &y) == 0);
  assert (z.n == 2);
  assert (z.v[0] == 0);
  assert (z.v[1] == 1);

  _bi_set_w (&x, NW, 0xedcba98a);
  assert (_bi_add_bb (&z, NW, &x, &y) == 0);
  assert (z.n == 2);
  assert (z.v[0] == 2);
  assert (z.v[1] == 1);

  _bi_set_d (&x, NW, 0xffffffffffffffffULL);
  _bi_set_d (&y, NW, 0xbad993a78ecc3a93ULL);
  assert (_bi_add_bb (&z, 2, &x, &y) == 1);
  assert (_bi_add_bb (&x, 3, &x, &y) == 0);
  assert (x.n == 3);
  assert (x.v[0] == 0x8ecc3a92);
  assert (x.v[1] == 0xbad993a7);
  assert (x.v[2] == 1);

  /* Test _bi_sub_mul_bw() with a factor of 1 (these test were made
     for _bi_sub_b() which has been deleted since). */

  _bi_set_w (&x, NW, 0);
  _bi_set_w (&y, NW, 0);
  assert (_bi_sub_mul_bw (&x, &y, 1) == 0);
  assert (x.n == 0);

  _bi_set_w (&y, NW, 1);
  assert (_bi_sub_mul_bw (&x, &y, 1) == 1);
  assert (x.n == 0);

  _bi_set_w (&x, NW, 1);
  _bi_set_w (&y, NW, 1);
  assert (_bi_sub_mul_bw (&x, &y, 1) == 0);
  assert (x.n == 0);

  _bi_set_w (&x, NW, 17);
  _bi_set_w (&y, NW, 42);
  assert (_bi_sub_mul_bw (&x, &y, 1) == 1);
  assert (x.n == 1);
  assert (x.v[0] == 17);

  _bi_set_w (&x, NW, 42);
  _bi_set_w (&y, NW, 17);
  assert (_bi_sub_mul_bw (&x, &y, 1) == 0);
  assert (x.n == 1);
  assert (x.v[0] == 25);

  _bi_set_w (&x, NW, 42);
  _bi_set_w (&y, NW, 0);
  assert (_bi_sub_mul_bw (&x, &y, 1) == 0);
  assert (x.n == 1);
  assert (x.v[0] == 42);

  _bi_set_d (&x, NW, 0x0000001200000000ULL);
  _bi_set_w (&y, NW, 1);
  assert (_bi_sub_mul_bw (&x, &y, 1) == 0);
  assert (x.n == 2);
  assert (x.v[0] == 0xffffffff);
  assert (x.v[1] == 0x00000011);

  _bi_set_d (&x, NW, 0xd676a34c234ff23bULL);
  _bi_set_d (&y, NW, 0xd676a34c234ff23cULL);
  assert (_bi_sub_mul_bw (&x, &y, 1) == 1);
  assert (x.n == 2);
  assert (x.v[0] == 0x234ff23b);
  assert (x.v[1] == 0xd676a34c);

  assert (_bi_sub_mul_bw (&y, &x, 1) == 0);
  assert (y.n == 1);
  assert (y.v[0] == 1);

  /* Test _bi_sub_mul_bw(). */

  _bi_set_d (&x, NW, 0x31ade97bb4a1220fULL);
  _bi_set_d (&y, NW, 0x02ec1ccb0aa0110fULL);
  assert (_bi_sub_mul_bw (&x, &y, 18) == 1);
  assert (_bi_sub_mul_bw (&x, &y, 17) == 0);
  assert (x.n == 1);
  assert (x.v[0] == 0x00000010);

  _bi_set_d (&x, NW, 0x0000123456789abcULL);
  _bi_set_d (&y, NW, 0x0000000035ffbac7ULL);
  assert (_bi_sub_mul_bw (&x, &y, 0x564e) == 1);
  assert (_bi_sub_mul_bw (&x, &y, 0x564d) == 0);
  assert (x.n == 1);
  assert (x.v[0] == 0x2fce92e1);

  _bi_set_d (&x, NW, 0x0000000200000000ULL);
  _bi_set_d (&y, NW, 0x00000000ffffffffULL);
  assert (_bi_sub_mul_bw (&x, &y, 3) == 1);
  assert (x.n == 2);
  assert (x.v[0] == 0x00000000);
  assert (x.v[1] == 0x00000002);
  assert (_bi_sub_mul_bw (&x, &y, 2) == 0);
  assert (x.n == 1);
  assert (x.v[0] == 0x00000002);

  /* Test _bi_shl_w(). */

  assert (_bi_shl_w (&x, 0, 0, 100) == 0);
  assert (_bi_shl_w (&x, 1, 3, 0) == 0);
  assert (x.n == 1);
  assert (x.v[0] == 3);
  assert (_bi_shl_w (&x, 1, 3, 1) == 0);
  assert (x.n == 1);
  assert (x.v[0] == 6);
  assert (_bi_shl_w (&x, 1, 17, 28) == 1);
  assert (_bi_shl_w (&x, 1, 17, 27) == 0);
  assert (x.n == 1);
  assert (x.v[0] == 0x88000000);
  assert (_bi_shl_w (&x, 2, 17, 28) == 0);
  assert (x.n == 2);
  assert (x.v[0] == 0x10000000);
  assert (x.v[1] == 0x00000001);

  /* Test _bi_shl_b(). */

  _bi_set_w (&x, NW, 0x72ccd93a);
  assert (_bi_shl_b (&y, 0, &x, 0) == 1);
  assert (_bi_shl_b (&y, 1, &x, 1) == 0);
  assert (y.n == 1);
  assert (y.v[0] == 0xe599b274);
  assert (_bi_shl_b (&y, 1, &x, 2) == 1);
  assert (_bi_shl_b (&y, 2, &x, 2) == 0);
  assert (y.n == 2);
  assert (y.v[0] == 0xcb3364e8);
  assert (y.v[1] == 0x00000001);

  /* Test _bi_shr_b(). */

  _bi_set_w (&x, NW, 0x72ccd93b);
  assert (_bi_shr_b (&x, NW, &x, 1) == 0);
  assert (x.n == 1);
  assert (x.v[0] == 0x39666c9d);
  assert (_bi_shr_b (&x, NW, &x, 11) == 0);
  assert (x.n == 1);
  assert (x.v[0] == 0x00072ccd);
  assert (_bi_shr_b (&x, NW, &x, 20) == 0);
  assert (x.n == 0);
  assert (_bi_shr_b (&x, NW, &x, 1) == 0);
  assert (x.n == 0);
  _bi_set_d (&x, NW, 0x123456789abcdef0ULL);
  assert (_bi_shr_b (&x, NW, &x, 1) == 0);
  assert (x.n == 2);
  assert (x.v[0] == 0x4d5e6f78);
  assert (x.v[1] == 0x091a2b3c);
  assert (_bi_shr_b (&x, NW, &x, 32) == 0);
  assert (x.n == 1);
  assert (x.v[0] == 0x091a2b3c);
  _bi_set_d (&x, NW, 0x123456789abcdef0ULL);
  assert (_bi_shr_b (&x, NW, &x, 33) == 0);
  assert (x.n == 1);
  assert (x.v[0] == 0x091a2b3c);

  x.n = 4;
  x.v[0] = 0x1234beef;
  x.v[1] = 0x5678face;
  x.v[2] = 0x9876dead;
  x.v[3] = 0xecce3141;
  assert (_bi_shr_b (&x, NW, &x, 80) == 0);
  assert (x.n == 2);
  assert (x.v[0] == 0x31419876);
  assert (x.v[1] == 0x0000ecce);

  /* Test _bi_shl_b(). */

  _bi_set_d (&x, NW, 0x7fade9b3141aa718ULL);
  assert (_bi_shl_b (&x, NW, &x, 96) == 0);
  assert (x.n == 5);
  assert (x.v[0] == 0x00000000);
  assert (x.v[1] == 0x00000000);
  assert (x.v[2] == 0x00000000);
  assert (x.v[3] == 0x141aa718);
  assert (x.v[4] == 0x7fade9b3);

  _bi_set_d (&x, NW, 0x7fade9b3141aa718ULL);
  assert (_bi_shl_b (&x, NW, &x, 101) == 0);
  assert (x.n == 6);
  assert (x.v[0] == 0x00000000);
  assert (x.v[1] == 0x00000000);
  assert (x.v[2] == 0x00000000);
  assert (x.v[3] == 0x8354e300);
  assert (x.v[4] == 0xf5bd3662);
  assert (x.v[5] == 0x0000000f);

  /* Test _bi_mul_bw(). */

  _bi_set_w (&x, NW, 0xca6dd3b5);
  assert (_bi_mul_bw (&x, 2, &x, 0xff7ed3ea) == 0);
  assert (x.n == 2);
  assert (x.v[0] == 0x0bf7b272);
  assert (x.v[1] == 0xca07af81);

  /* Test _bi_mul_bb(). */

  clear (&z);
  _bi_set_w (&x, NW, 0);
  _bi_set_w (&y, NW, 0);
  assert (_bi_mul_bb (&z, 0, &x, &y) == 0);
  assert (z.n == 0);

  clear (&z);
  _bi_set_w (&x, NW, 17);
  _bi_set_w (&y, NW, 0);
  assert (_bi_mul_bb (&z, 1, &x, &y) == 0);
  assert (z.n == 0);

  clear (&z);
  _bi_set_w (&x, NW, 0);
  _bi_set_w (&y, NW, 42);
  assert (_bi_mul_bb (&z, 1, &x, &y) == 0);
  assert (z.n == 0);

  clear (&z);
  _bi_set_w (&x, NW, 1);
  _bi_set_w (&y, NW, 42);
  assert (_bi_mul_bb (&z, 2, &x, &y) == 0);
  assert (z.n == 1);
  assert (z.v[0] == 42);

  clear (&z);
  _bi_set_d (&x, NW, 0xf74e99c6ae0bbff3ULL);
  _bi_set_d (&y, NW, 0xc30fa414c2edb0a9ULL);
  assert (_bi_mul_bb (&z, 4, &x, &y) == 0);
  assert (z.n == 4);
  assert (z.v[0] == 0xd3afc76b);
  assert (z.v[1] == 0xb18fcf23);
  assert (z.v[2] == 0xc22d9fe1);
  assert (z.v[3] == 0xbc6ffb3f);

  /* Test _bi_hdiv_rem_b(). */

  _bi_set_w (&x, NW, 0);
  _bi_set_w (&y, NW, 1);
  assert (_bi_hdiv_rem_b (&x, &y) == 0);
  assert (x.n == 0);

  _bi_set_w (&x, NW, 47112001);
  _bi_set_w (&y, NW,   876339);
  assert (_bi_hdiv_rem_b (&x, &y) == 53);
  assert (x.n == 1);
  assert (x.v[0] == 666034);

  _bi_set_d (&x, NW,  471120333301ULL);
  _bi_set_w (&y, NW,    1270558314);
  assert (_bi_hdiv_rem_b (&x, &y) == 370);
  assert (x.n == 1);
  assert (x.v[0] == 1013757121);

  _bi_set_d (&x, NW, 471120333301ULL);
  _bi_set_d (&y, NW, 471120333302ULL);
  assert (_bi_hdiv_rem_b (&x, &y) == 0);
  assert (x.n == 2);
  assert (x.v[0] == 0xb0f5c9f5);
  assert (x.v[1] == 0x0000006d);

  _bi_set_d (&x, NW, 471120333301ULL);
  _bi_set_d (&y, NW, 471120333301ULL);
  assert (_bi_hdiv_rem_b (&x, &y) == 1);
  assert (x.n == 0);

  _bi_set_d (&x, NW, 471120333302ULL);
  _bi_set_d (&y, NW, 471120333301ULL);
  assert (_bi_hdiv_rem_b (&x, &y) == 1);
  assert (x.n == 1);
  assert (x.v[0] == 1);

  x.n = 21;
  x.v[ 0] = 0xd687d378;
  x.v[ 1] = 0x6766efdb;
  x.v[ 2] = 0xc7ecff5c;
  x.v[ 3] = 0x76f3f9d1;
  x.v[ 4] = 0x05087df1;
  x.v[ 5] = 0x2aa39fdd;
  x.v[ 6] = 0x62704f7f;
  x.v[ 7] = 0x7c25d5ce;
  x.v[ 8] = 0x13ecd974;
  x.v[ 9] = 0x89372a18;
  x.v[10] = 0x1d91e2bd;
  x.v[11] = 0xc5dc0881;
  x.v[12] = 0xef57fe72;
  x.v[13] = 0x76eb2054;
  x.v[14] = 0xcba88cab;
  x.v[15] = 0x3cab72e0;
  x.v[16] = 0x679b216e;
  x.v[17] = 0xc348b55f;
  x.v[18] = 0x62de5db2;
  x.v[19] = 0x3530bcd4;
  x.v[20] = 0x00000048;

  y.n = 21;
  y.v[ 0] = 0x9432daf5;
  y.v[ 1] = 0x683a58f4;
  y.v[ 2] = 0x2557c25d;
  y.v[ 3] = 0xe004fd33;
  y.v[ 4] = 0x969e1d52;
  y.v[ 5] = 0x5e8e0801;
  y.v[ 6] = 0xb6c24a70;
  y.v[ 7] = 0xcb51ea2e;
  y.v[ 8] = 0x2fcdcaeb;
  y.v[ 9] = 0xa1987da5;
  y.v[10] = 0x3e652db8;
  y.v[11] = 0xf79b112d;
  y.v[12] = 0x580a52af;
  y.v[13] = 0xd0b4aeed;
  y.v[14] = 0x6a8a7f66;
  y.v[15] = 0xd7701384;
  y.v[16] = 0x68b24c51;
  y.v[17] = 0x7fb2b806;
  y.v[18] = 0x50187e22;
  y.v[19] = 0x06a617d4;
  y.v[20] = 0x00000009;

  assert (_bi_hdiv_rem_b (&x, &y) == 7);
  assert (x.n == 21);
  assert (x.v[ 0] == 0xc923d6c5);
  assert (x.v[ 1] == 0x8dce812b);
  assert (x.v[ 2] == 0xc286aece);
  assert (x.v[ 3] == 0x56d10d6b);
  assert (x.v[ 4] == 0xe6b5b0ad);
  assert (x.v[ 5] == 0x94c167d1);
  assert (x.v[ 6] == 0x6320466c);
  assert (x.v[ 7] == 0xece86e87);
  assert (x.v[ 8] == 0xc54c4d01);
  assert (x.v[ 9] == 0x1e0bba93);
  assert (x.v[10] == 0x68cda2b1);
  assert (x.v[11] == 0x009e9044);
  assert (x.v[12] == 0x870fbba3);
  assert (x.v[13] == 0xc1fa57d7);
  assert (x.v[14] == 0xe1df10db);
  assert (x.v[15] == 0x589aea41);
  assert (x.v[16] == 0x8abb0b31);
  assert (x.v[17] == 0x4565ad32);
  assert (x.v[18] == 0x3232eac1);
  assert (x.v[19] == 0x06a61606);
  assert (x.v[20] == 0x00000009);

  /* Test _bi_wdiv_rem_pow2(). */

  _bi_set_w (&x, NW, 0);
  assert (_bi_wdiv_rem_pow2 (&x, 1) == 0);
  assert (x.n == 0);
  assert (_bi_wdiv_rem_pow2 (&x, 17) == 0);
  assert (x.n == 0);

  _bi_set_w (&x, NW, 0x1974a849);
  assert (_bi_wdiv_rem_pow2 (&x, 1) == 0x0cba5424);
  assert (x.n == 1);
  assert (x.v[0] == 0x00000001);

  _bi_set_w (&x, NW, 0x1974a849);
  assert (_bi_wdiv_rem_pow2 (&x, 12) == 0x0001974a);
  assert (x.n == 1);
  assert (x.v[0] == 0x00000849);

  _bi_set_d (&x, NW, 0xabcd19ef74a9aa49ULL);
  assert (_bi_wdiv_rem_pow2 (&x, 44) == 0x000abcd1);
  assert (x.n == 2);
  assert (x.v[0] == 0x74a9aa49);
  assert (x.v[1] == 0x000009ef);

  _bi_set_d (&x, NW, 0xabcd19ef74a9aa49ULL);
  assert (_bi_wdiv_rem_pow2 (&x, 77) == 0);
  assert (x.n == 2);
  assert (x.v[0] == 0x74a9aa49);
  assert (x.v[1] == 0xabcd19ef);

  /* Test _bi_div_rem_pow2(). */

  _bi_set_w (&x, NW, 0);
  clear (&z);
  assert (_bi_div_rem_pow2 (&x, &z, NW, 0) == 0);
  assert (x.n == 0);
  assert (z.n == 0);

  clear (&z);
  assert (_bi_div_rem_pow2 (&x, &z, NW, 1) == 0);
  assert (x.n == 0);
  assert (z.n == 0);

  clear (&z);
  assert (_bi_div_rem_pow2 (&x, &z, NW, 63) == 0);
  assert (x.n == 0);
  assert (z.n == 0);

  _bi_set_w (&x, NW, 0x12345678);
  clear (&z);
  assert (_bi_div_rem_pow2 (&x, &z, NW, 0) == 0);
  assert (x.n == 0);
  assert (z.n == 1);
  assert (z.v[0] == 0x12345678);

  _bi_set_w (&x, NW, 0x12345678);
  clear (&z);
  assert (_bi_div_rem_pow2 (&x, &z, NW, 1) == 0);
  assert (x.n == 0);
  assert (z.n == 1);
  assert (z.v[0] == 0x091a2b3c);

  _bi_set_w (&x, NW, 0x12345678);
  clear (&z);
  assert (_bi_div_rem_pow2 (&x, &z, NW, 4) == 0);
  assert (x.n == 1);
  assert (x.v[0] == 0x00000008);
  assert (z.n == 1);
  assert (z.v[0] == 0x01234567);

  _bi_set_w (&x, NW, 0x12345678);
  clear (&z);
  assert (_bi_div_rem_pow2 (&x, &z, NW, 28) == 0);
  assert (x.n == 1);
  assert (x.v[0] == 0x02345678);
  assert (z.n == 1);
  assert (z.v[0] == 0x00000001);

  _bi_set_w (&x, NW, 0x12345678);
  clear (&z);
  assert (_bi_div_rem_pow2 (&x, &z, NW, 32) == 0);
  assert (x.n == 1);
  assert (x.v[0] == 0x12345678);
  assert (z.n == 0);

  _bi_set_w (&x, NW, 0x12345678);
  clear (&z);
  assert (_bi_div_rem_pow2 (&x, &z, NW, 33) == 0);
  assert (x.n == 1);
  assert (x.v[0] == 0x12345678);
  assert (z.n == 0);

  _bi_set_w (&x, NW, 0x12345678);
  clear (&z);
  assert (_bi_div_rem_pow2 (&x, &z, NW, 66) == 0);
  assert (x.n == 1);
  assert (x.v[0] == 0x12345678);
  assert (z.n == 0);

  _bi_set_d (&x, NW, 0x9876543210abcdefULL);
  clear (&z);
  assert (_bi_div_rem_pow2 (&x, &z, NW, 0) == 0);
  assert (x.n == 0);
  assert (z.n == 2);
  assert (z.v[0] == 0x10abcdef);
  assert (z.v[1] == 0x98765432);

  _bi_set_d (&x, NW, 0x9876543210abcdefULL);
  clear (&z);
  assert (_bi_div_rem_pow2 (&x, &z, NW, 4) == 0);
  assert (x.n == 1);
  assert (x.v[0] == 0x0000000f);
  assert (z.n == 2);
  assert (z.v[0] == 0x210abcde);
  assert (z.v[1] == 0x09876543);

  _bi_set_d (&x, NW, 0x9876543210abcdefULL);
  clear (&z);
  assert (_bi_div_rem_pow2 (&x, &z, NW, 32) == 0);
  assert (x.n == 1);
  assert (x.v[0] == 0x10abcdef);
  assert (z.n == 1);
  assert (z.v[0] == 0x98765432);

  _bi_set_d (&x, NW, 0x9876543210abcdefULL);
  clear (&z);
  assert (_bi_div_rem_pow2 (&x, &z, NW, 40) == 0);
  assert (x.n == 2);
  assert (x.v[0] == 0x10abcdef);
  assert (x.v[1] == 0x00000032);
  assert (z.n == 1);
  assert (z.v[0] == 0x00987654);

  /* Test _bi_div_rem_bw(). */

  _bi_set_w (&x, NW, 0);
  assert (_bi_div_rem_bw (&x, NW, &w, &x, 0) == 1);

  _bi_set_w (&x, NW, 0);
  assert (_bi_div_rem_bw (&x, NW, &w, &x, 17) == 0);
  assert (x.n == 0);
  assert (w == 0);

  w = 0;
  _bi_set_d (&x, NW, 0xabcd19ef74a9aa49ULL);
  assert (_bi_div_rem_bw (&x, 2, &w, &x, 0x77665544) == 0);
  assert (x.n == 2);
  assert (x.v[0] == 0x705a0b8e);
  assert (x.v[1] == 0x000000001);
  assert (w == 0x1a547291);

  w = 0xffffffff;
  _bi_set_d (&x, NW, 0xabcd19ef74a9aa49ULL);
  assert (_bi_div_rem_bw (&x, 2, &w, &x, 0xbbbbbbbb) == 0);
  assert (x.n == 1);
  assert (x.v[0] == 0xea463aa4);
  assert (w == 0x64bb087d);

  w = 0;
  _bi_set_d (&x, NW, 0xabcd19ef74a9aa49ULL);
  assert (_bi_mul_bw (&x, NW, &x, 0x87654321) == 0);
  _bi_set_w (&y, NW, 0x12345678);
  assert (_bi_add_bb (&x, NW, &x, &y) == 0);
  assert (x.n == 3);
  assert (_bi_div_rem_bw (&x, 3, &w, &x, 0x87654321) == 0);
  assert (x.n == 2);
  assert (x.v[0] == 0x74a9aa49);
  assert (x.v[1] == 0xabcd19ef);
  assert (w == 0x12345678);

  /* Test _bi_div_rem_bb(). */

  x.n = 21;
  x.v[ 0] = 0xd687d378;
  x.v[ 1] = 0x6766efdb;
  x.v[ 2] = 0xc7ecff5c;
  x.v[ 3] = 0x76f3f9d1;
  x.v[ 4] = 0x05087df1;
  x.v[ 5] = 0x2aa39fdd;
  x.v[ 6] = 0x62704f7f;
  x.v[ 7] = 0x7c25d5ce;
  x.v[ 8] = 0x13ecd974;
  x.v[ 9] = 0x89372a18;
  x.v[10] = 0x1d91e2bd;
  x.v[11] = 0xc5dc0881;
  x.v[12] = 0xef57fe72;
  x.v[13] = 0x76eb2054;
  x.v[14] = 0xcba88cab;
  x.v[15] = 0x3cab72e0;
  x.v[16] = 0x679b216e;
  x.v[17] = 0xc348b55f;
  x.v[18] = 0x62de5db2;
  x.v[19] = 0x3530bcd4;
  x.v[20] = 0x00000048;
  x.v[21] = 0xf0f0f0f0;         /* guard word */

  y.n = 21;
  y.v[ 0] = 0x9432daf5;
  y.v[ 1] = 0x683a58f4;
  y.v[ 2] = 0x2557c25d;
  y.v[ 3] = 0xe004fd33;
  y.v[ 4] = 0x969e1d52;
  y.v[ 5] = 0x5e8e0801;
  y.v[ 6] = 0xb6c24a70;
  y.v[ 7] = 0xcb51ea2e;
  y.v[ 8] = 0x2fcdcaeb;
  y.v[ 9] = 0xa1987da5;
  y.v[10] = 0x3e652db8;
  y.v[11] = 0xf79b112d;
  y.v[12] = 0x580a52af;
  y.v[13] = 0xd0b4aeed;
  y.v[14] = 0x6a8a7f66;
  y.v[15] = 0xd7701384;
  y.v[16] = 0x68b24c51;
  y.v[17] = 0x7fb2b806;
  y.v[18] = 0x50187e22;
  y.v[19] = 0x06a617d4;
  y.v[20] = 0x00000009;
  y.v[21] = 0xf0f0f0f0;         /* guard word */

  clear (&z);
  assert (_bi_div_rem_bb (&x, NW, &z, NW, &y) == 0);
  assert (z.n == 1);
  assert (z.v[0] == 7);
  assert (x.n == 21);
  assert (x.v[ 0] == 0xc923d6c5);
  assert (x.v[ 1] == 0x8dce812b);
  assert (x.v[ 2] == 0xc286aece);
  assert (x.v[ 3] == 0x56d10d6b);
  assert (x.v[ 4] == 0xe6b5b0ad);
  assert (x.v[ 5] == 0x94c167d1);
  assert (x.v[ 6] == 0x6320466c);
  assert (x.v[ 7] == 0xece86e87);
  assert (x.v[ 8] == 0xc54c4d01);
  assert (x.v[ 9] == 0x1e0bba93);
  assert (x.v[10] == 0x68cda2b1);
  assert (x.v[11] == 0x009e9044);
  assert (x.v[12] == 0x870fbba3);
  assert (x.v[13] == 0xc1fa57d7);
  assert (x.v[14] == 0xe1df10db);
  assert (x.v[15] == 0x589aea41);
  assert (x.v[16] == 0x8abb0b31);
  assert (x.v[17] == 0x4565ad32);
  assert (x.v[18] == 0x3232eac1);
  assert (x.v[19] == 0x06a61606);
  assert (x.v[20] == 0x00000009);

  x.n = 7;
  x.v[0] = 0x8edefc5d;
  x.v[1] = 0x32103ef7;
  x.v[2] = 0x083b2c31;
  x.v[3] = 0x29688485;
  x.v[4] = 0x68c8d705;
  x.v[5] = 0xfe773899;
  x.v[6] = 0x00251d9c;
  x.v[7] = 0xf0f0f0f0;          /* guard word */

  y.n = 3;
  y.v[0] = 0x1ac33b2b;
  y.v[1] = 0xbb8a9348;
  y.v[2] = 0x0000004e;
  y.v[3] = 0xf0f0f0f0;          /* guard word */

  clear (&z);
  assert (_bi_div_rem_bb (&x, NW, &z, NW, &y) == 0);
  assert (z.n == 5);
  assert (z.v[0] == 0x600f451f);
  assert (z.v[1] == 0xe666cad7);
  assert (z.v[2] == 0xb3141aff);
  assert (z.v[3] == 0x9dead64b);
  assert (z.v[4] == 0x000078ae);
  assert (x.n == 3);
  assert (x.v[0] == 0x1ac33b28);
  assert (x.v[1] == 0xbb8a9348);
  assert (x.v[2] == 0x0000004e);

  x.n = 7;
  x.v[0] = 0xef588e1d;
  x.v[1] = 0x4c352a21;
  x.v[2] = 0x1e46f5d8;
  x.v[3] = 0x5dd19d3f;
  x.v[4] = 0xa27b3cad;
  x.v[5] = 0xd37ffe90;
  x.v[6] = 0x000000f6;

  y.n = 3;
  y.v[0] = 0xffffffff;
  y.v[1] = 0xffffffff;
  y.v[2] = 0x000fffff;

  clear (&z);
  assert (_bi_div_rem_bb (&x, NW, &z, NW, &y) == 0);
  assert (z.n == 4);
  assert (z.v[0] == 0x10a771e3);
  assert (z.v[1] == 0xb3cad5de);
  assert (z.v[2] == 0xffe90a27);
  assert (z.v[3] == 0x000f6d37);
  assert (x.n == 0);

  /* This test has u[0] == v1 in _bi_div_estimate(). */

  _bi_set_d (&x, NW, 0x06ea878ccb5ca3a0ULL);
  _bi_shl_b (&x, NW, &x, 512);
  y.n = 17;
  y.v[ 0] = 0x8a898f4d;
  y.v[ 1] = 0x4075e8eb;
  y.v[ 2] = 0xd6ff1701;
  y.v[ 3] = 0xfec39152;
  y.v[ 4] = 0x3628270c;
  y.v[ 5] = 0xda581d76;
  y.v[ 6] = 0xb4310289;
  y.v[ 7] = 0x8eabe460;
  y.v[ 8] = 0xad50ca82;
  y.v[ 9] = 0x07519967;
  y.v[10] = 0x76fe9f23;
  y.v[11] = 0x39e8b585;
  y.v[12] = 0x3dddc64b;
  y.v[13] = 0x4bc4938a;
  y.v[14] = 0xe51d4634;
  y.v[15] = 0x543c665a;
  y.v[16] = 0x00000037;
  clear(&z);
  assert (_bi_div_rem_bb (&x, NW, &z, NW, &y) == 0);
  assert (z.n == 2);
  assert (z.v[0] == 0xffffffff);
  assert (z.v[1] == 0x001fffff);
  assert (x.n == 17);
  assert (x.v[ 0] == 0x8a898f4d);
  assert (x.v[ 1] == 0x56d5e8eb);
  assert (x.v[ 2] == 0xb98dc5cf);
  assert (x.v[ 3] == 0x1e9b8295);
  assert (x.v[ 4] == 0x0bcd472a);
  assert (x.v[ 5] == 0xf8b84504);
  assert (x.v[ 6] == 0x056a3d84);
  assert (x.v[ 7] == 0x3d70995d);
  assert (x.v[ 8] == 0x213a4462);
  assert (x.v[ 9] == 0xb6ffc3eb);
  assert (x.v[10] == 0x4a08f509);
  assert (x.v[11] == 0x5587cb52);
  assert (x.v[12] == 0x8d2ee677);
  assert (x.v[13] == 0x825d5673);
  assert (x.v[14] == 0x73d58a7b);
  assert (x.v[15] == 0x8db2edc8);
  assert (x.v[16] == 0x0000002e);

  x.n = 19;
  x.v[ 0] = 0x00000000;
  x.v[ 1] = 0x00000000;
  x.v[ 2] = 0x00000000;
  x.v[ 3] = 0x00000000;
  x.v[ 4] = 0x00000000;
  x.v[ 5] = 0x00000000;
  x.v[ 6] = 0x00000000;
  x.v[ 7] = 0x00000000;
  x.v[ 8] = 0x00000000;
  x.v[ 9] = 0x00000000;
  x.v[10] = 0x00000000;
  x.v[11] = 0x00000000;
  x.v[12] = 0x00000000;
  x.v[13] = 0x00000000;
  x.v[14] = 0x00000000;
  x.v[15] = 0x00000000;
  x.v[16] = 0x7d2b8000;
  x.v[17] = 0xe294eebc;
  x.v[18] = 0x00000010;
  y.n = 17;
  y.v[ 0] = 0x39d6dafd;
  y.v[ 1] = 0x5fdda70d;
  y.v[ 2] = 0xe4c72a0e;
  y.v[ 3] = 0xfb75cb3e;
  y.v[ 4] = 0x380756b9;
  y.v[ 5] = 0x111fed9a;
  y.v[ 6] = 0xeba7328e;
  y.v[ 7] = 0x51a89017;
  y.v[ 8] = 0x223e68be;
  y.v[ 9] = 0xde37861e;
  y.v[10] = 0x83a28484;
  y.v[11] = 0x61232ad7;
  y.v[12] = 0x0a711da8;
  y.v[13] = 0xfaec3481;
  y.v[14] = 0x5c7865ac;
  y.v[15] = 0xa775e3e9;
  y.v[16] = 0x00008714;
  clear(&z);
  assert (_bi_div_rem_bb (&x, NW, &z, NW, &y) == 0);
  assert (z.n == 2);
  assert (z.v[0] == 0xffffffff);
  assert (z.v[1] == 0x001fffff);
  assert (x.n == 17);
  assert (x.v[ 0] == 0x39d6dafd);
  assert (x.v[ 1] == 0x003da70d);
  assert (x.v[ 2] == 0x031fef33);
  assert (x.v[ 3] == 0xb9a9cf8a);
  assert (x.v[ 4] == 0xd02abdd4);
  assert (x.v[ 5] == 0x39e07ee0);
  assert (x.v[ 6] == 0x386031a3);
  assert (x.v[ 7] == 0xffe66c1a);
  assert (x.v[ 8] == 0x1f40f3d7);
  assert (x.v[ 9] == 0xc66d510c);
  assert (x.v[10] == 0xbfde3cb7);
  assert (x.v[11] == 0xd08763e6);
  assert (x.v[12] == 0xaf80a957);
  assert (x.v[13] == 0x45e0101b);
  assert (x.v[14] == 0xcc571789);
  assert (x.v[15] == 0xf1d68662);
  assert (x.v[16] == 0x00007807);

  puts ("Testing with random numbers...");
  for (iter = 1; forever || iter <= niter; ++iter)
    {
      if (!quiet && iter % 10000 == 0)
        printf ("Iterations: %d\n", iter);
      x0.n = random () % NW;
      do
        {
          y0.n = 1 + random () % (NW - 1);
        } while (y0.n == 0);
      r0.n = random () % (y0.n + 1);
      for (i = 0; i < x0.n; ++i)
        x0.v[i] = random ();
      for (i = 0; i < y0.n; ++i)
        y0.v[i] = random ();
      while (x0.n != 0 && x0.v[x0.n-1] == 0)
        x0.v[x0.n-1] = random ();
      while (y0.v[y0.n-1] == 0)
        y0.v[y0.n-1] = random ();
      switch (i & 7)
        {
        case 0:
          _bi_set_b (&r0, NW, &y0);
          _bi_set_w (&z, NW, 1);
          _bi_sub_mul_bw (&r0, &z, 1);
          break;

        case 1:
          if (y0.n == 1 && y0.v[0] == 1)
            _bi_set_w (&r0, NW, 0);
          else
            _bi_set_w (&r0, NW, 1);
          break;

        case 2:
          _bi_set_w (&r0, NW, 0);
          break;

        default:
          for (i = 0; i < r0.n; ++i)
            r0.v[i] = random ();
          while (r0.n != 0 && (r0.v[r0.n-1] == 0
                               || _bi_cmp_bb (&r0, &y0) >= 0))
            r0.v[r0.n-1] = random ();
          break;
        }

      assert (_bi_mul_bb (&z, ZNW, &x0, &y0) == 0);
      assert (_bi_add_bb (&z, ZNW, &z, &r0) == 0);

      clear (&x);
      assert (_bi_div_rem_bb (&z, ZNW, &x, NW, &y0) == 0);
      assert (_bi_cmp_bb (&x, &x0) == 0);
      assert (_bi_cmp_bb (&z, &r0) == 0);
    }


  return 0;
}
