/*
     Copyright (c) 2000,2010 Michael Dillon

     Permission is hereby granted, free of charge, to any person obtaining a copy
     of this software and associated documentation files (the "Software"), to deal
     in the Software without restriction, including without limitation the rights
     to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
     copies of the Software, and to permit persons to whom the Software is
     furnished to do so, subject to the following conditions:
     
     The above copyright notice and this permission notice shall be included in
     all copies or substantial portions of the Software.
     
     THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
     IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
     FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
     AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
     LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
     OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
     THE SOFTWARE.
*/
#include <iostream>
#include <fstream>
#include <iomanip>
#include <cstdlib>
#include <cstdio>
#include <cerrno>
#include <cctype>
#include <ctime>
#include <cstring>

using namespace std;

#if defined( _MSC_VER )
# include <search.h>
#endif

/* Force Pack alignment */
#pragma pack( 1 )

#include "cstats.h"
#include "lord.h"   /* LORD Functions          */
#include "lord2.h"  /* LORD II Functions       */
#include "pteos.h"  /* Planets: TEOS Functions */
#include "avln85.h" /* Avalon v0.85 Functions  */
#include "utilfunc.h"

int main( int argc, char *argv[] )
{
        PRGSCAN prgScan;

        cout << PRG_FULLNAME << " v" << PRG_VERSION << ".  Game Statistics Generator." << endl;
        cout << "Copyright (C) 2000 " << PRG_AUTHOR << ". All Rights Reserved." << endl;        cout << endl;

        if ( argc < 3 ) {
                Usage();
                return ( EXIT_FAILURE );
        }

        ProcessCommandLine( argc, argv, &prgScan );

        if ( prgScan.Program )
        {
                cout << "Processing player data from "
                        << PRGNAMES[ prgScan.Program - 1 ]
                        << "..." << endl;
                ProcessPlayers( &prgScan );
        }
        else
        {
                cerr << "Invalid game or path specified." << endl;
                return ( EXIT_FAILURE );
        }

        return ( EXIT_SUCCESS );
}

void Usage( void )
{
        cerr << "Usage: " << PRG_NAME << " <game> <path>" << endl;
        cerr << "Examples: " << endl;
        cerr << "          " << PRG_NAME << " -lord c:\\lord\\       ( Windows, DOS, OS/2 )" << endl;
        cerr << "          " << PRG_NAME << " -lord /usr/bbs/lord/ ( Linux, *NIX )" << endl;
        cerr << endl;
        cerr << "Games Supported: " << endl;
        cerr << "        -lord       L.O.R.D." << endl;
        cerr << "        -teos       Planets: The Exploration Of Space" << endl;
        cerr << "        -lord2      L.O.R.D. 2" << endl;
        cerr << "        -avln85     Avalon - Live The Epic v0.85" << endl;
        cerr << endl;
        cerr << "Path must be where the game you specify is located." << endl;
        cerr << endl;
        cerr << "NOTE: Only one game can be specified at any given call, if" << endl;
        cerr << "      more than one is specified then the first one is used." << endl;
}

void ProcessCommandLine( int argc, char *argv[], LPPRGSCAN prgScan )
{
        int                     idx, prgIdx;
        unsigned long int       Prg = 0L;
        char                    PrgPath[ CSTATS_MAX_PATH ] = "\x0";
        int                     SizeFlags = ( sizeof( PRGFLAGS ) / 10 );

        memset( prgScan, 0, sizeof( PRGSCAN ) );

        for ( idx = 1; idx < argc; idx++ )
        {
                for ( prgIdx = 0; prgIdx < SizeFlags; prgIdx++ )
                {
                        if ( ( stricmp( argv[ idx ], PRGFLAGS[ prgIdx ] ) == 0 ) && ( Prg == 0L ) )
                                Prg = (long)prgIdx + 1;
                }
                if ( argv[ idx ][ 0 ] != '-' )
                {
                        if ( strlen( PrgPath ) == 0 )
                                strncpy( PrgPath, argv[ idx ], CSTATS_MAX_PATH );
                }
        }

        if ( Prg > 0 && strlen( PrgPath ) > 0 )
        {
                strcpy( prgScan->PrgPath, PrgPath );
                prgScan->Program = Prg;
        }
}

void ProcessPlayers( LPPRGSCAN prgScan )
{
//        FNDATA  lordFuncs = { lordHTML, lordCompare, lordDelChk, lordAssign, lordOutput };
//        FNDATA  lord2Funcs = { l2HTML, l2Compare, l2DelChk, l2Assign, l2Output };
//        FNDATA  teosFuncs = { teosHTML, teosCompare, teosDelChk, teosAssign, teosOutput };
//        FNDATA  avln85Funcs = { avln85HTML, avln85Compare, avln85DelChk, avln85Assign, avln85Output };
        switch ( prgScan->Program )
        {
                case 1 : /* LORD */
                        ProcessPlayerData<lord_player_info>(
                                prgScan, "player.dat", lordHTML, lordCompare,
                                lordDelChk, lordAssign, lordOutput  );
                        break;
                case 2 : /* LORD 2 */
                        ProcessPlayerData<l2_player>(
                                prgScan, "trader.dat", l2HTML, l2Compare,
                                l2DelChk, l2Assign, l2Output );
                        break;
                case 3 : /* Planets: TEOS */
                        ProcessPlayerData<teos_player>(
                                prgScan, "trader.dat", teosHTML, teosCompare,
                                teosDelChk, teosAssign, teosOutput );
                        break;
                case 4 : /* Avalon - Live The Epic v0.85 */
                        ProcessPlayerData<avln_UserRecord>(
                                prgScan, "player.dat", avln85HTML, avln85Compare,
                                avln85DelChk, avln85Assign, avln85Output );
                        break;
        }
}

//template <class OBJ> int ProcessPlayerData( PRGSCAN *prgScan, const char *szFileName, FNDATA *fnData )
template <class OBJ> int ProcessPlayerData(
        PRGSCAN *prgScan,
        const char *szFileName,
        void ( *htmlOutput )( int ),
        int  ( *Compare )( const void *, const void * ),
        int  ( *DeleteChk )( OBJ * ),
        void ( *Assign )( SORT *, OBJ * ),
        void ( *Output )( PRGSCAN *, OBJ * ) )
{
        ifstream                *datFile = new ifstream;
        char                    szPrgFile[ CSTATS_MAX_PATH ] = "\x0";
//        char                    *object = new char[ objSize + 20 ];
        OBJ                     *object = new OBJ;
        size_t                  objSize = sizeof( OBJ );
        unsigned long int       fSize;
        clock_t                 BeginProc;
        clock_t                 EndProc;
        double                  ProcTime;
        unsigned long int       delOmitted = 0L;
        unsigned long int       curRecord = 0L;
        unsigned long int       curPos = 0L;
        SORT                    plyrSort[ 8192 ];
        unsigned long int       idx;

        if ( !datFile || !object )
        {
                cerr << "Failure to allocate the file stream: " << strerror( errno ) << endl;
                if ( object )
                        delete object;
                return ( 1 );
        }

        if ( !object )
        {
                cerr << "Failure to allocate the object required: " << strerror( errno ) << endl;
                delete datFile;
                return ( 1 );
        }

        FixPath( prgScan->PrgPath );

        if ( strcspn( prgScan->PrgPath, "<>|" ) != strlen( prgScan->PrgPath ) )
        {
                cerr << "Invalid Path... contained piping characters." << endl;
                delete datFile;
                delete object;
                return ( 2 );
        }

        strcpy( szPrgFile, prgScan->PrgPath );
        strcat( szPrgFile, szFileName );

        datFile->open( szPrgFile, ios::binary | ios::in );

        if ( datFile->fail() )
        {
                cerr << "Error while opening " << szPrgFile << endl;
                delete datFile;
                delete object;
                return ( 3 );
        }

        fSize = FileSize( datFile );

        BeginProc = clock();

        if ( ( fSize % objSize ) != 0 )
        {
                cerr << "File size does not match appropriate size for player data... aborting" << endl;
                delete datFile;
                delete object;
                return ( 4 );
        }

        cout << "Processing " << ( fSize / objSize ) << " player records. " << endl;

#ifdef DEBUG
        printf( "ProcessPlayerData(): size of structure: %d\n", objSize );
#endif
        while ( !datFile->eof() )
        {
                memset( ( void * )object, 0, objSize );
                datFile->read( ( char * )object, objSize );
#ifdef DEBUG
                printf( "ProcessPlayerData(): gcount(): %d\n", datFile->gcount() );
#endif
                if ( datFile->gcount() == objSize )
                {
//                        if ( fnData->DeleteChk( object ) )
                        if ( DeleteChk( object ) )
                        {
                                delOmitted++;
                        }
                        else
                        {
                                if ( curRecord < ( sizeof( plyrSort ) / sizeof( SORT ) ) ) {
//                                        fnData->Assign( &plyrSort[ curRecord ], ( void * )object );
                                        Assign( &plyrSort[ curRecord ], object );
                                        plyrSort[ curRecord++ ].Record = curPos;
                                }
                                else {
                                        cerr << "Warning excessive records detected, over " << ( sizeof( plyrSort ) / sizeof( SORT ) ) << " entries!" << endl;
                                }
                        }
                        curPos++;
                }
                else if ( datFile->gcount() > 0 && datFile->gcount() != objSize )
                {
                        cerr << "Player data corrupted on read... Possible Invalid File" << endl;
                }
        }

        datFile->clear();

        datFile->seekg( 0L, ios::beg );

        cout << "Omitted a total of " << delOmitted << " records that were invalid or marked deleted." << endl;

//        qsort( ( void * )plyrSort, ( size_t )curRecord, sizeof( SORT ), fnData->Compare );
        qsort( ( void * )plyrSort, ( size_t )curRecord, sizeof( SORT ), Compare );

        cout << endl;

//        fnData->htmlOutput( 0 ); /* Begin the HTML Code */
        htmlOutput( 0 );

        for ( idx = 0; idx < curRecord; idx++ )
        {
//                LoadBinFileRec( ( void * )object, objSize, plyrSort[ idx ].Record, datFile );
                LoadBinFileRec<OBJ>( object, plyrSort[ idx ].Record, datFile );
//                fnData->Output( prgScan, ( void * )object );
                Output( prgScan, object );
        }

//        fnData->htmlOutput( 1 ); /* End the HTML Code */
        htmlOutput( 1 );

        EndProc = clock();

        ProcTime = ( ( (double)EndProc - (double)BeginProc ) / CLOCKS_PER_SEC );
        cout << "Processing required " << ProcTime << " seconds to complete. ("
                << (double)( fSize / objSize ) / ProcTime << " records/sec)" << endl;

        datFile->close();

        delete datFile;
        delete object;

        return ( 0 );
}

void htmlStatHeader( ofstream *file, const char *szTitle )
{
        *file << "<html>" << endl;
        *file << "<head>" << endl;
        *file << "<title>" << szTitle << "</title>" << endl;
        *file << "<meta name=\"GENERATOR\" content=\"" << PRG_NAME << " " << PRG_VERSION << "\">" << endl;
        *file << "</head>" << endl;
        *file << "<body bgcolor=\"#000000\" text=\"#ffffff\">" << endl;
        *file << "<center>" << endl;
        *file << "<h1>" << szTitle << "</h1>" << endl;
        *file << "<table width=\"95%\" cellspacing=\"0\" cellpadding=\"2\" border=\"0\">" << endl;
        file->flush();
}

void htmlStatClose( ofstream *file )
{
        char strTime[ 1024 ];
        time_t curTime;
        struct tm *today;
        curTime = time( NULL );
        today = gmtime( &curTime );

        strftime( strTime, 1024, "%A, %B %d, %Y  %H:%M:%S UTC", today );
        *file << "</table>" << endl;
        *file << "</center>" << endl;
        *file << "<hr>" << endl;
        *file << "<br>" << endl << "<br>" << endl;
        *file << "Created by " << PRG_FULLNAME << " " << PRG_VERSION << " on " << strTime << "." << endl;
        *file << "<br>" << endl;
        *file << PRG_FULLNAME << " Copyright &copy; 2000 " << PRG_AUTHOR << ".  All Rights Reserved. " << endl;
        *file << "</body>" << endl;
        *file << "</html>" << endl;
        file->flush();
}

void htmlSetColor( ofstream *file, signed long int color )
{
        char hexval[ 10 ];
        *file << "<font color=\"#";
        sprintf( hexval, "%06lx", color );
        *file << hexval;
        *file << "\">";
        file->flush();
}

void FixPath( char *path )
{
        const char APPEND_DIRSEPARATOR[ ] = { DIRSEPARATOR, '\0' };
        if ( path[ strlen( path ) - 1 ] != DIRSEPARATOR )
                strncat( path, APPEND_DIRSEPARATOR, 1 );
}
