program Recurse;

{$APPTYPE CONSOLE}

{$R 'recurse.res' 'recurse.rc'}

uses
  Windows, Classes, SysUtils;

var
  FStartDir, FCommand: string;
  FShowDir, FShowCmd, FTest, FMatching, FDoReplace: Boolean;
  FMinDepth, FMaxDepth: Integer;
  FInclude, FExclude, FIgnore: TStringList;

function GetLastErrorStr(const iError: Integer): string;
var
  Buffer: array[0..MAX_PATH] of Char;
  nSize: DWORD;
begin
  Result := '';
  nSize := MAX_PATH - 1;
  if FormatMessage(FORMAT_MESSAGE_IGNORE_INSERTS or FORMAT_MESSAGE_FROM_SYSTEM,
                   nil, iError, 0, Buffer, nSize, nil) <> 0 then
    Result := Buffer
  else
    Result := 'Error ' + IntToStr(GetLastError) +
              ' calling FormatMessage using error #' + IntToStr(iError);
  Result := Trim(Result); // Discard trailing CRLF that is often attached.
end;

function GetYNEsc: Boolean; // Accept Y/N/Esc input from the keyboard
var
  ev: TInputRecord;
  dw: DWORD;
  bDone: Boolean;
begin
  Result := False;
  bDone := False;
  repeat
    ReadConsoleInput(GetStdhandle(STD_INPUT_HANDLE), ev, 1, dw);
    if ev.EventType = KEY_EVENT then begin
      if ev.Event.KeyEvent.bKeyDown then begin
        if (ev.Event.KeyEvent.wVirtualKeyCode = VK_ESCAPE) or
           (Pos(ev.Event.KeyEvent.AsciiChar, 'Nn') > 0) then begin
          WriteLn('N');
          bDone := True;
        end else if Pos(ev.Event.KeyEvent.AsciiChar, 'Yy') > 0 then begin
          Result := True;
          WriteLn('Y');
          bDone := True;
        end;
      end;
    end;
  until bDone;
  WriteLn('');
end;

function ProcessDir(const ADir: string; const ADepth: Integer): Boolean;
var
  bDoCmd: Boolean;
  sPath, sCmd, sMatch, sItem: String;
  si: TStartupInfo;
  ProcInfo: TProcessInformation;
  dw: DWORD;
  slSubs: TStringList;
  SR: TSearchRec;
  s: Integer;
  procedure FindMatch(const AKind: String; AList: TStringList);
  var
    m: Integer;
  begin
    // Don't override a higher-precedence match (while sMatch = '')
    m := 0;
    while (sMatch = '') and (m < AList.Count) do
      if Pos(AList[m], sPath) > 0 then begin
        sMatch := AKind;
        sItem := AList[m];
      end;
  end;
begin
  // Only return False if we want to quit completely (blow the ballast):
  Result := True;
  // Check for a directory substring match:
  bDoCmd := True;
  if FMatching then begin
    sPath := IncludeTrailingBackslash(UpperCase(ADir));
    sMatch := '';
    FindMatch('ignore', FIgnore);
    FindMatch('exclude', FExclude);
    FindMatch('include', FInclude);
    if FShowDir then
      if sMatch <> '' then
        WriteLn(Format('%s (%s=%s)', [ADir, sMatch, sItem]))
      else
        WriteLn(ADir);
    if sMatch = 'ignore' then // Skip the command AND subdirs on ignore
      Exit
    else if sMatch = 'exclude' then // Skip the command on exclude
      bDoCmd := False
    else if FInclude.Count > 0 then // Skip the command if an include match is
      bDoCmd := sMatch = 'include'; // required but not made
  end else if FShowDir then
    WriteLn(ADir);
  // Failure to reach Depth trumps a match:
  if ADepth < FMinDepth then
    bDoCmd := False;
  // Issue the command:
  if bDoCmd then begin
    if FDoReplace then
      sCmd := StringReplace(FCommand, '%dir%', ADir, [rfIgnoreCase, rfReplaceAll])
    else
      sCmd := FCommand;
    if FShowCmd then
      WriteLn(sCmd);
    if not FTest then begin
      FillChar(si, SizeOf(si), 0);
      si.cb := SizeOf(si);
      si.dwFlags := STARTF_USESHOWWINDOW;
      si.wShowWindow := SW_SHOWNORMAL;
      if CreateProcess(nil, PChar(sCmd), nil, nil, False, NORMAL_PRIORITY_CLASS,
                       nil, PChar(ADir), si, ProcInfo) then begin
        // Wait for the program to finish:
        {dw := } WaitForSingleObject(ProcInfo.hProcess, INFINITE);
        // (We don't really care why it ended, just that it's done)
        CloseHandle(ProcInfo.hProcess);
        CloseHandle(ProcInfo.hThread);
      end else begin
        // It failed to launch.  What do you want to do?
        dw := GetLastError;
        if FDoReplace then // No need to show dir if part of the command...
          WriteLn(Format('Command [%s] failed with the following error (%d):',
                         [sCmd, dw]))
        else
          WriteLn(Format('Command [%s] in directory %s failed with the following error (%d):',
                         [sCmd, QuotedStr(ADir), dw]));
        WriteLn(GetLastErrorStr(dw));
        Write('Continue (Y/N)? ');
        Result := GetYNEsc;
        if not Result then
          Exit;
      end;
    end; // if not FTest
  end; // if bDoCmd
  // Now recurse subdirs if not already at our depth limit:
  if (FMaxDepth > 0) and (ADepth >= FMaxDepth) then
    Exit;
  sPath := IncludeTrailingBackslash(ADir);
  slSubs := TStringList.Create;
  try
    slSubs.Sorted := True;
    if FindFirst(sPath + '*.*', faAnyFile, SR) = 0 then begin
      repeat
        if ((SR.Attr and faDirectory) <> 0) and (SR.Name[1] <> '.') then
          slSubs.Add(SR.Name);
      until FindNext(SR) <> 0;
      FindClose(SR);
    end;
    s := 0;
    while Result and (s < slSubs.Count) do begin
      Result := ProcessDir(sPath + slSubs[s], ADepth + 1);
      Inc(s);
    end;
  finally
    slSubs.Free;
  end;
end;

function ParseMatchList(const AValue: String; AList: TStringList): Boolean;
var
  sFile: String;
begin
  { The OS is kind enough to remove any "internal" quoting for us, e.g.
    /INCLUDE="Program Files";foo -> /INCLUDE=Program Files;foo
    /EXCLUDE=@"C:\Program Files\x.txt" -> /EXCLUDE=@C:\Program Files\x.txt }
  if AValue <> '' then begin
    if AValue[1] = '@' then begin
      sFile := Copy(AValue, 2, MaxInt);
      if FileExists(sFile) then begin
        AList.LoadFromFile(sFile);
        AList.Text := UpperCase(AList.Text);
      end;
    end else begin
      AList.Delimiter := ';';
      AList.StrictDelimiter := True; // Finally introduced in BDS2006!
      AList.DelimitedText := UpperCase(AValue);
    end;
  end;
  Result := AList.Count > 0;
  if not Result then begin
    WriteLn('Invalid directory match list: ' + AValue);
    WriteLn('');
  end;
end;

function ReQuote(const S: string): string;
begin
  { If there are spaces in a command line argument, it means it was given to
    us quoted.  Put the quotes back on before presenting it to the command
    processor again: }
  if (Pos(' ', S) > 0) and (AnsiDequotedStr(S, '"') = S) then
    Result := AnsiQuotedStr(S, '"')
  else
    Result := S;
end;

function ParseArgs: Boolean;
var
  slArgs: TStringList;
  p, x: Integer;
  sArg, sUp, sValue: String;
begin
  Result := False;
  slArgs := TStringList.Create;
  try
    // Gather all the parameters and see if they should come from a file:
    for p := 1 to ParamCount do
      slArgs.Add(ParamStr(p));
    if (slArgs.Count = 1) and (slArgs[0][1] = '@') then begin
      sArg := Copy(slArgs[1], 2, MaxInt);
      slArgs.Clear;
      if FileExists(sArg) then
        slArgs.LoadFromFile(sArg)
      else begin
        WriteLn('File not found: ' + sArg);
        WriteLn('');
      end;
    end;
    // Now see what they really want:
    for p := 0 to slArgs.Count - 1 do begin
      sArg := slArgs[p];
      if Result then begin
        // Any remaining items are command arguments:
        FCommand := FCommand + ' ' + ReQuote(sArg);
      end else if sArg[1] = '/' then begin
        // It's an option:
        if FStartDir <> '' then
          Break; // Options are supposed to be first!
        sUp := UpperCase(sArg);
        x := Pos('=', sArg);
        if x > 0 then begin
          Delete(sUp, x, MaxInt);
          sValue := Copy(sArg, x + 1, MaxInt);
        end else
          sValue := '';
        if sUp = '/SHOWDIR' then
          FShowDir := True
        else if sUp = '/SHOWCMD' then
          FShowCmd := True
        else if sUp = '/TEST' then
          FTest := True
        else if (sUp = '/MINDEPTH') or (sUp = '/MAXDEPTH') then begin
          if (x = 0) or not TryStrToInt(sValue, x) then begin
            WriteLn('Invalid or missing ' + sUp + ' value');
            WriteLn('');
            Break;
          end;
          if sUp = '/MAXDEPTH' then
            FMaxDepth := x
          else
            FMinDepth := x;
        end else if sUp = '/INCLUDE' then begin
          if not ParseMatchList(sValue, FInclude) then
            Break;
        end else if sUp = '/EXCLUDE' then begin
          if not ParseMatchList(sValue, FExclude) then
            Break;
        end else if sUp = '/IGNORE' then begin
          if not ParseMatchList(sValue, FIgnore) then
            Break;
        end else begin
          WriteLn('Unrecognized option: ' + sArg);
          WriteLn('');
          Break;
        end;
      end else if FStartDir = '' then begin
        // StartDir is the first non-option argument:
        FStartDir := ExcludeTrailingBackslash(sArg);
        if not DirectoryExists(FStartDir) then begin
          WriteLn('Starting directory not found: ' + FStartDir);
          WriteLn('');
          Break;
        end;
        if Copy(FStartDir, 1, 2) <> '\\' then // If not a UNC name...
          FStartDir := ExpandFileName(FStartDir); // expand .. etc.
      end else begin
        // This must be the command!
        FCommand := ReQuote(sArg);
        { Watch for built-in shell commands like copy, dir, etc.
          I exclude things like del or attrib that support their own recursion.
          Anything else can be put in a batch file anyway, e.g. foo.bat %dir% }
        if Pos('~' + LowerCase(FCommand) + '~',
               '~copy~dir~echo~move~rd~ren~') > 0 then
          FCommand := 'cmd /c ' + FCommand;
        // We now have everything we need!
        Result := True;
      end;
    end;
    FMatching := FInclude.Count + FExclude.Count + FIgnore.Count > 0;
    FDoReplace := Pos('%dir%', LowerCase(FCommand)) > 0;
  finally
    slArgs.Free;
  end;
end; // ParseArgs

{----- Main Procedure ---------------------------------------------------------}

var
  rs: TResourceStream;
begin
  // Initialize global variables:
  FStartDir := '';
  FCommand := '';
  FMinDepth := 0;
  FMaxDepth := 0;
  FTest := False;
  FShowDir := False;
  FShowCmd := False;
  FMatching := False;
  FDoReplace := False;
  FInclude := TStringList.Create;
  FExclude := TStringList.Create;
  FIgnore  := TStringList.Create;
  try
    WriteLn('');
    if ParseArgs then
      ProcessDir(FStartDir, 0)
    else begin
      // Print the usage instructions if the arguments were wrong/missing:
      rs := TResourceStream.Create(hInstance, 'USAGETXT', RT_RCDATA);
      FInclude.LoadFromStream(rs);
      rs.free;
      WriteLn(Trim(FInclude.Text));
    end;
  finally
    FInclude.Free;
    FExclude.Free;
    FIgnore.Free;
  end;
end.
