package jp.co.sra.smalltalk;

import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.image.ColorModel;

/**
 * StOpaqueImage class
 * 
 *  @author    nisinaka
 *  @created   2004/01/15 (by nisinaka)
 *  @updated   N/A
 *  @version   8.9
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: StOpaqueImage.java,v 8.11 2008/02/20 06:33:18 nisinaka Exp $
 */
public class StOpaqueImage extends StObject implements StDisplayable {

	protected StImage figure;
	protected StImage shape;
	protected StImage image;

	/**
	 * Create a new instance of StOpaqueImage and initialize it.
	 * 
	 * @param figure jp.co.sra.smalltalk.StImage
	 * @param shape jp.co.sra.smalltalk.StImage
	 * @category Instance creation
	 */
	public StOpaqueImage(StImage figure, StImage shape) {
		this.figure_shape_(figure, shape);
	}

	/**
	 * Answer the receiver's figure.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category accessing
	 */
	public StImage figure() {
		return figure;
	}

	/**
	 * Answer the receiver's shape.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category accessing
	 */
	public StImage shape() {
		return shape;
	}

	/**
	 * Answer the receiver's extent.
	 * 
	 * @return java.awt.Dimension
	 * @category accessing
	 */
	public Dimension extent() {
		return new Dimension(this.figure().width(), this.figure().height());
	}

	/**
	 * Set the receiver's figure and shape.
	 * 
	 * @param figureImage jp.co.sra.smalltalk.StImage
	 * @param shapeImage jp.co.sra.smalltalk.StImage
	 * @category accessing
	 */
	protected void figure_shape_(StImage figureImage, StImage shapeImage) {
		if (figureImage.extent().equals(shapeImage.extent()) == false) {
			throw SmalltalkException.Error("figure and shape sizes don't match");
		}
		if (shapeImage.colorModel() != StImage.MonoMaskColorModel()) {
			throw SmalltalkException.Error("shape must have MonoMaskColorModel");
		}

		figure = figureImage;
		shape = shapeImage;
	}

	/**
	 * Answer the receiver as StImage.
	 *
	 * @return jp.co.sra.smalltalk.StImage
	 * @see jp.co.sra.smalltalk.StDisplayable#asImage()
	 * @category converting
	 */
	public StImage asImage() {
		if (image == null) {
			int width = this.figure().width();
			int height = this.figure().height();
			StImage anImage = new StImage(width, height, ColorModel.getRGBdefault());
			for (int y = 0; y < height; y++) {
				for (int x = 0; x < width; x++) {
					byte[] bytes = (byte[]) this.shape().image().getRaster().getDataElements(x, y, null);
					int pixel = (bytes[0] > 0) ? this.figure().getPixel(x, y) : 0;
					anImage.setPixel(x, y, pixel);
				}
			}
			image = anImage;
		}
		return image;
	}

	/**
	 * Answer the bounding box of the receiver.
	 *
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.smalltalk.StDisplayable#bounds()
	 * @category bounds accessing
	 */
	public Rectangle bounds() {
		return this.figure().bounds();
	}

	/**
	 * Display the receiver on the graphics.
	 *
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		this.asImage().displayOn_(aGraphics);
	}

	/**
	 * Display the receiver on the graphics at the specified point.
	 *
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_at_(java.awt.Graphics, java.awt.Point)
	 * @category displaying
	 */
	public void displayOn_at_(Graphics aGraphics, Point aPoint) {
		this.asImage().displayOn_at_(aGraphics, aPoint);
	}

}
