package jp.co.sra.smalltalk;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.StringWriter;
import java.io.Writer;
import java.lang.reflect.Array;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Vector;

/**
 * StObject class
 * 
 *  @author    nisinaka
 *  @created   1998/08/05 (by nisinaka)
 *  @updated   2002/11/26 (by nisinaka)
 *  @updated   2005/03/11 (by nisinaka)
 *  @version   8.9
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: StObject.java,v 8.14 2008/02/20 06:33:18 nisinaka Exp $
 */
public class StObject implements Cloneable, DependentListener {

	public static final String _System = "StPL";
	public static final String _Version = "8.9";
	public static final String _Date = "2008/02/20";

	/** The dependents fields. */
	private static HashMap DependentsFields = new HashMap();

	/**
	 * Answer the current Name of the system.
	 *
	 * @return java.lang.String
	 * @category Copyright
	 */
	public static String System() {
		return _System;
	}

	/**
	 * Answer the current Version of the system.
	 *
	 * @return java.lang.String
	 * @category Copyright
	 */
	public static String Version() {
		return _Version;
	}

	/**
	 * Answer the current Date of the system.
	 *
	 * @return java.lang.String
	 * @category Copyright
	 */
	public static String Date() {
		return _Date;
	}

	/**
	 * Create a new instance of aClass without arguments.
	 * 
	 * @return java.lang.Object
	 * @param aClass java.lang.Class
	 * @throws SmalltalkException if failed to create a new instance.
	 * @category Instance creation
	 */
	public static Object _New(Class aClass) {
		try {
			return aClass.newInstance();
		} catch (Exception e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Create a new instance of aClass with arguments. Please refer to
	 * StObjectTestExamples.Example1() for the usage.
	 * 
	 * @return java.lang.Object
	 * @param aClass java.lang.Class
	 * @param arguments java.lang.Object[]
	 * @throws SmalltalkException if failed to create a new instance.
	 * @category Instance creation
	 */
	public static Object _New(Class aClass, Object[] arguments) {
		Class[] parameterTypes = _ParameterTypesOf(arguments);
		Constructor aConstructor = null;

		try {
			aConstructor = aClass.getConstructor(parameterTypes);
		} catch (NoSuchMethodException e) {
			int arraySize = parameterTypes.length - 1;
			Class[] anArray = new Class[arraySize];
			for (int i = 0; i < arraySize; i++) {
				anArray[i] = parameterTypes[i];
			}
			Vector aList = _ListOfPossibleParameterTypes(anArray, parameterTypes[arraySize]);
			for (int i = 0; i < aList.size(); i++) {
				Class[] aPossibleParameterTypes = (Class[]) aList.elementAt(i);
				try {
					aConstructor = aClass.getConstructor(aPossibleParameterTypes);
				} catch (NoSuchMethodException e2) {
					// Just ignore the exception and try another parameter types.
				}
			}

			if (aConstructor == null) {
				throw SmalltalkException.MessageNotUnderstood("Constructor of " + aClass.getName());
			}
		}

		try {
			return aConstructor.newInstance(arguments);
		} catch (Exception e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Create a new instance of aClass with an argument.
	 * 
	 * @return java.lang.Object
	 * @param aClass java.lang.Class
	 * @param argument java.lang.Object
	 * @category Instance creation
	 */
	public static Object _New(Class aClass, Object argument) {
		Object[] arguments = { argument };
		return _New(aClass, arguments);
	}

	/**
	 * Create a new instance of aClass with two arguments.
	 * 
	 * @return java.lang.Object
	 * @param aClass java.lang.Class
	 * @param argument1 java.lang.Object
	 * @param argument2 java.lang.Object
	 * @category Instance creation
	 */
	public static Object _New(Class aClass, Object argument1, Object argument2) {
		Object[] arguments = { argument1, argument2 };
		return _New(aClass, arguments);
	}

	/**
	 * Create an object based on the contents of aStream.
	 * 
	 * @return jp.co.sra.smalltalk.StObject
	 * @param aStream java.io.InputStream
	 * @throws Exception DOCUMENT ME!
	 * @category Instance creation
	 */
	public static StObject ReadFrom_(InputStream aStream) throws Exception {
		ObjectInputStream aObjectInputStream = new ObjectInputStream(aStream);
		return (StObject) aObjectInputStream.readObject();
	}

	/**
	 * Find a class instance variable and return its value. Please refer to
	 * StObjectTestExamples.Example4() for the usage..
	 * 
	 * @return java.lang.Object
	 * @param aClass java.lang.Class
	 * @param variableName java.lang.String
	 * @throws SmalltalkException if the variable with the specified name does not exist.
	 * @category Accessing
	 */
	public static Object _ClassInstanceVariable(Class aClass, String variableName) {
		String className = aClass.getName();
		int index = className.lastIndexOf('.');
		if (index != -1) {
			className = className.substring(index + 1);
		}

		String fieldName = className + "_" + variableName;
		try {
			return aClass.getField(fieldName).get(null);
		} catch (NoSuchFieldException e) {
			if (aClass == Object.class) {
				throw new SmalltalkException(e);
			}
			return _ClassInstanceVariable(aClass.getSuperclass(), variableName);
		} catch (Exception e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Find a class instance variable and set its value. Please refer to
	 * StObjectTestExamples.Example5() for the usage..
	 * 
	 * @param aClass java.lang.Class
	 * @param variableName java.lang.String
	 * @param anObject DOCUMENT ME!
	 * @throws SmalltalkException if the variable with the specified name does not exist.
	 * @category Accessing
	 */
	public static void _ClassInstanceVariable(Class aClass, String variableName, Object anObject) {
		String className = aClass.getName();
		int index = className.lastIndexOf('.');
		if (index != -1) {
			className = className.substring(index + 1);
		}

		String fieldName = className + "_" + variableName;
		Field aField = null;
		try {
			aField = aClass.getField(fieldName);
			aField.set(null, anObject);
		} catch (NoSuchFieldException e) {
			if (aClass == Object.class) {
				throw new SmalltalkException(e);
			}
			_ClassInstanceVariable(aClass.getSuperclass(), variableName, anObject);
		} catch (Exception e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Make an array of class which contains parameter types of arguments. This
	 * is used to get an appropriate parameter types which correspond to the
	 * arguments.  The parameter types can be used as an argument for
	 * Class.getConstructor() or Class.getMethod(). Please refer to
	 * StObjectTestExamples.Example2() for the usage.
	 * 
	 * @return java.lang.Class[]
	 * @param arguments java.lang.Object[]
	 * @throws jp.co.sra.smalltalk.SmalltalkException DOCUMENT ME!
	 * @category Accessing
	 */
	protected static Class[] _ParameterTypesOf(Object[] arguments) {
		int numberOfArguments = arguments.length;
		Class[] parameterTypes = new Class[numberOfArguments];
		for (int i = 0; i < numberOfArguments; i++) {
			Class aClass;
			try {
				// You cannot use the return value of getClass()
				// for Class.getMethod() or Class.getConstructor()
				// if the type of object is a primitive one. So
				// check for the primitive types. If there is a
				// "TYPE" field, then use its value.
				aClass = (Class) arguments[i].getClass().getField("TYPE").get(null);
			} catch (NoSuchFieldException e) {
				// If there is no "TYPE" field, use the return
				// value of getClass().
				aClass = arguments[i].getClass();
			} catch (IllegalAccessException e) {
				throw new jp.co.sra.smalltalk.SmalltalkException(e);
			}

			parameterTypes[i] = aClass;
		}

		return parameterTypes;
	}

	/**
	 * Send the message indicated by "selector" to aClass. The "selector" is
	 * the method name.
	 * 
	 * @return java.lang.Object
	 * @param aClass java.lang.Class
	 * @param selector java.lang.String
	 * @exception NoSuchMethodException if a matching method is not found.
	 * @exception IllegalAccessException if the underlying method is inaccessible.
	 * @exception IllegalArgumentException if the number of actual and formal parameters differ, or if an unwrapping conversion fails.
	 * @exception InvocationTargetException if the underlying method throws an exception.
	 * @category Message handling
	 */
	public static Object _Perform(Class aClass, String selector) throws NoSuchMethodException, IllegalAccessException, IllegalArgumentException, InvocationTargetException {
		Object theObject = null;
		Method theMethod = aClass.getMethod(selector, (Class[]) null);
		Object[] argumentList = null;
		theObject = theMethod.invoke(aClass, argumentList);
		return theObject;
	}

	/**
	 * Send the message indicated by "selector" and "argument" to aClass. The
	 * "selector" is the method name and its argument is "argument".
	 * 
	 * @return java.lang.Object
	 * @param aClass java.lang.Class
	 * @param selector java.lang.String
	 * @param argument java.lang.Object
	 * @exception NoSuchMethodException if the method cannot be found.
	 * @category Message handling
	 */
	public static Object _PerformWith(Class aClass, String selector, Object argument) throws NoSuchMethodException {
		Object[] arguments = { argument };
		return _PerformWithArguments(aClass, selector, arguments);
	}

	/**
	 * Send the message indicated by "selector" and "argument" to aClass. The
	 * "selector" is the method name and its argument is "argument".
	 * 
	 * @return java.lang.Object
	 * @param aClass java.lang.Class
	 * @param selector java.lang.String
	 * @param argument1 java.lang.Object
	 * @param argument2 java.lang.Object
	 * @exception NoSuchMethodException if the method cannot be found.
	 * @category Message handling
	 */
	public static Object _PerformWith(Class aClass, String selector, Object argument1, Object argument2) throws NoSuchMethodException {
		Object[] arguments = { argument1, argument2 };
		return _PerformWithArguments(aClass, selector, arguments);
	}

	/**
	 * Send the message indicated by "selector" and "arguments" to aClass. The
	 * "selector" is the method name and its arguments are the elements of
	 * "arguments".
	 * 
	 * @return java.lang.Object
	 * @param aClass java.lang.Class
	 * @param selector java.lang.String
	 * @param arguments java.lang.Object[]
	 * @exception NoSuchMethodException if the method cannot be found.
	 * @throws SmalltalkException DOCUMENT ME!
	 * @category Message handling
	 */
	public static Object _PerformWithArguments(Class aClass, String selector, Object[] arguments) throws NoSuchMethodException {
		Object theResult = null;
		Method theMethod = _SearchMethod(aClass, selector, _ParameterTypesOf(arguments));
		try {
			theResult = theMethod.invoke(aClass, arguments);
		} catch (Exception e) {
			throw new SmalltalkException(e);
		}

		return theResult;
	}

	/**
	 * Answer a method which parameter types do not much exactly.
	 * 
	 * @return java.lang.reflect.Method
	 * @param aClass java.lang.Class
	 * @param selector java.lang.String
	 * @param parameterTypes java.lang.Class[]
	 * @exception NoSuchMethodException if the method cannot be found.
	 * @category Message handling
	 */
	protected static Method _SearchMethod(Class aClass, String selector, Class[] parameterTypes) throws NoSuchMethodException {
		try {
			return aClass.getMethod(selector, parameterTypes);
		} catch (NoSuchMethodException e) {
			int arraySize = parameterTypes.length - 1;
			Class[] anArray = new Class[arraySize];
			for (int i = 0; i < arraySize; i++) {
				anArray[i] = parameterTypes[i];
			}
			Vector aList = _ListOfPossibleParameterTypes(anArray, parameterTypes[arraySize]);
			for (int i = 0; i < aList.size(); i++) {
				Class[] aPossibleParameterTypes = (Class[]) aList.elementAt(i);
				try {
					return aClass.getMethod(selector, aPossibleParameterTypes);
				} catch (NoSuchMethodException e2) {
					// Just ignore the exception and try another parameter types.
				}
			}
			throw e;
		}
	}

	/**
	 * Answer the list of possible parameter types. This is used to get a list
	 * of possible parameter types.  This process will be done recursively so
	 * you need to divide parameter types into two: the last one and the rest.
	 * Please refer to StObjectTestExamples.Example3() for the usage..
	 * 
	 * @return java.util.Vector
	 * @param parameterTypes java.lang.Class[]
	 * @param tailClass java.lang.Class
	 * @category Message handling
	 */
	protected static Vector _ListOfPossibleParameterTypes(Class[] parameterTypes, Class tailClass) {
		Vector possibleParameterTypes = new Vector();

		// Make a list of tailClass class and its superclasses.
		Vector listOfClasses = new Vector();
		Class aClass = tailClass;
		do {
			listOfClasses.addElement(aClass);
			aClass = aClass.getSuperclass();
		} while (aClass != null);

		// Make a list of possible parameter types.
		if (parameterTypes.length == 0) {
			// There is no list of head parameter types.
			// You just need to create a list of parameter types.
			for (int i = 0; i < listOfClasses.size(); i++) {
				Class[] anArray = { (Class) listOfClasses.elementAt(i) };
				possibleParameterTypes.addElement(anArray);
			}
		} else {
			// There is a list of head parameter types.
			// You need to make a list of possible parameter types with them.
			Vector listOfParameterTypes;
			int arraySize = parameterTypes.length - 1;
			Class[] anArray = new Class[arraySize];
			for (int i = 0; i < arraySize; i++) {
				anArray[i] = parameterTypes[i];
			}
			listOfParameterTypes = _ListOfPossibleParameterTypes(anArray, parameterTypes[arraySize]);

			// Append each class of "listOfClasses" as a last class of parameter,
			// and add it to the list of possible parameter types.
			arraySize = parameterTypes.length + 1;
			for (int i = 0; i < listOfParameterTypes.size(); i++) {
				Class[] anElement = (Class[]) listOfParameterTypes.elementAt(i);
				for (int j = 0; j < listOfClasses.size(); j++) {
					anArray = new Class[arraySize];
					for (int k = 0; k < (arraySize - 1); k++) {
						anArray[k] = anElement[k];
					}
					anArray[arraySize - 1] = (Class) listOfClasses.elementAt(j);
					possibleParameterTypes.addElement(anArray);
				}
			}
		}

		return possibleParameterTypes;
	}

	/**
	 * Replacement for the # notation of Smalltalk. :-)
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @param aString java.lang.String
	 * @category Symbol utilities
	 */
	public static StSymbol $(String aString) {
		return StSymbol.Intern_(aString);
	}

	/**
	 * Answer a string which corresponds to the key according to the current message catalogs.
	 * 
	 * @param key java.lang.String
	 * @return java.lang.String
	 * @category String utilities
	 */
	public static String $String(String key) {
		return new StUserMessage(key).toString();
	}

	/**
	 * Answer a string which corresponds to the key according to the current message catalogs.
	 * 
	 * @param key java.lang.String
	 * @param defaultString java.lang.String
	 * @return java.lang.String
	 * @category String utilities
	 */
	public static String $String(String key, String defaultString) {
		return new StUserMessage(key, defaultString).toString();
	}

	/**
	 * Answer a string which corresponds to the key according to the current message catalogs.
	 * 
	 * @param key java.lang.String
	 * @param defaultString java.lang.String
	 * @param argument java.lang.Object
	 * @return java.lang.String
	 * @category String utilities
	 */
	public static String $String(String key, String defaultString, Object argument) {
		return new StUserMessage(key, defaultString, argument).toString();
	}

	/**
	 * Answer a string which corresponds to the key according to the current message catalogs.
	 * 
	 * @param key java.lang.String
	 * @param defaultString java.lang.String
	 * @param argument1 java.lang.Object
	 * @param argument2 java.lang.Object
	 * @return java.lang.String
	 * @category String utilities
	 */
	public static String $String(String key, String defaultString, Object argument1, Object argument2) {
		return new StUserMessage(key, defaultString, argument1, argument2).toString();
	}

	/**
	 * Answer a string which corresponds to the key according to the current message catalogs.
	 * 
	 * @param key java.lang.String
	 * @param defaultString java.lang.String
	 * @return java.lang.String
	 * @category String utilities
	 */
	public static String $String(String key, String defaultString, Object[] arguments) {
		return new StUserMessage(key, defaultString, arguments).toString();
	}

	/**
	 * Remove references to objects that may refer to the receiver.
	 * 
	 * @category initialize-release
	 */
	public void release() {
		this.breakDependents();
	}

	/**
	 * The receiver is changed and broadcast it to my dependents.
	 * 
	 * @category changing
	 */
	public void changed() {
		this.changed_(new DependentEvent(this, null, null));
	}

	/**
	 * The receiver is changed.  Broadcast it to my dependents.
	 * 
	 * @param evt jp.co.sra.smalltalk.DependentEvent
	 * @category changing
	 */
	public void changed_(DependentEvent evt) {
		DependentListener[] dependents = this.dependents();
		for (int i = 0; i < dependents.length; i++) {
			dependents[i].update_(evt);
		}
	}

	/**
	 * The receiver is changed and broadcast it to my dependents.
	 * 
	 * @param anAspect jp.co.sra.smalltalk.StSymbol
	 * @category changing
	 */
	public void changed_(StSymbol anAspect) {
		this.changed_(new DependentEvent(this, anAspect, null));
	}

	/**
	 * The receiver is changed.  Broadcast it to my dependents.
	 * 
	 * @param anAspect jp.co.sra.smalltalk.StSymbol
	 * @param aParameter java.lang.Object
	 * @category changing
	 */
	public void changed_with_(StSymbol anAspect, Object aParameter) {
		this.changed_(new DependentEvent(this, anAspect, aParameter));
	}

	/**
	 * Receive a change notice from an object of whom the receiver is a
	 * dependent.  The argument evt.getAspect() is typically a Symbol that
	 * indicates what change has occurred.
	 * 
	 * @param evt jp.co.sra.smalltalk.DependentEvent
	 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent evt) {
		// Do nothing as a default.
	}

	/**
	 * Answer a copy of receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StObject
	 * @category copying
	 */
	public StObject copy() {
		return this.shallowCopy().postCopy();
	}

	/**
	 * Answer a copy of the receiver which shares the receiver's instance
	 * variables.
	 * 
	 * @return jp.co.sra.smalltalk.StObject
	 * @throws SmalltalkException if failed
	 * @category copying
	 */
	public StObject shallowCopy() {
		try {
			return (StObject) super.clone();
		} catch (CloneNotSupportedException e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Do something after a shallow copy.  Subclasses may add functionality,
	 * but they should always do 'super.postCopy()' first.
	 * 
	 * @return jp.co.sra.smalltalk.StObject
	 * @category copying
	 */
	public StObject postCopy() {
		return this;
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws IOException if failed.
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write("a ");
		aWriter.write(this.getClass().getName());
	}

	/**
	 * Print my string representation on aStream.
	 * 
	 * @param aStream java.io.OutputStream
	 * @throws IOException if failed
	 * @category printing
	 */
	public void printOn_(OutputStream aStream) throws IOException {
		BufferedWriter aWriter = new BufferedWriter(new OutputStreamWriter(aStream));
		this.printOn_(aWriter);
		aWriter.flush();
	}

	/**
	 * Answer printable string of this object.
	 * 
	 * @return java.lang.String
	 * @throws SmalltalkException DOCUMENT ME!
	 * @category printing
	 */
	public String printString() {
		try {
			StringWriter aWriter = new StringWriter();
			this.printOn_(aWriter);
			return aWriter.toString();
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Answer printable string of this object.
	 * 
	 * @return java.lang.String
	 * @see java.lang.Object#toString()
	 * @category printing
	 */
	public String toString() {
		return this.printString();
	}

	/**
	 * Print my storable string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws IOException if failed.
	 * @category printing
	 */
	public void storeOn_(Writer aWriter) throws IOException {
		aWriter.write("(");
		aWriter.write(this.getClass().getName());
		aWriter.write(" new)");
	}

	/**
	 * Print my storable string representation on aStream.
	 * 
	 * @param aStream java.io.OutputStream
	 * @throws IOException if failed.
	 * @category printing
	 */
	public void storeOn_(OutputStream aStream) throws IOException {
		BufferedWriter aWriter = new BufferedWriter(new OutputStreamWriter(aStream));
		this.printOn_(aWriter);
		aWriter.flush();
	}

	/**
	 * Answer storable string of this object.
	 * 
	 * @return java.lang.String
	 * @throws SmalltalkException if failed.
	 * @category printing
	 */
	public String storeString() {
		try {
			StringWriter aWriter = new StringWriter();
			this.storeOn_(aWriter);
			return aWriter.toString();
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Answer the Smalltalk compatible class name.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category class membership
	 */
	public StSymbol _className() {
		String className = this.getClass().getName();
		return $(className.substring(className.lastIndexOf('.') + 1));
	}

	/**
	 * Please refer to the corresponding method in Smalltalk.
	 * 
	 * @return java.lang.Class
	 * @category class membership
	 */
	public Class species() {
		return this.getClass();
	}

	/**
	 * Answer true if the receiver can accept a method specified with "selector".
	 * 
	 * @return boolean
	 * @param selector java.lang.String
	 * @category class membership
	 */
	public boolean respondsTo_(String selector) {
		Method[] methods = this.getClass().getDeclaredMethods();
		for (int i = 0; i < methods.length; i++) {
			if (methods[i].getName().equals(selector)) {
				return true;
			}
		}
		return false;
	}

	/**
	 * Answer an array of the objects that are dependent on the receiver.
	 *
	 * @return java.lang.Object[]
	 * @category dependents access
	 */
	public synchronized DependentListener[] dependents() {
		if (this.myDependents() == null) {
			return new DependentListener[0];
		}

		DependentListener[] dependents = new DependentListener[this.myDependents().size()];
		this.myDependents().toArray(dependents);
		return dependents;
	}

	/**
	 * Add anObject as a dependent of the receiver.
	 * 
	 * @param anObject jp.co.sra.smalltalk.DependentListener
	 * @category dependents access
	 */
	public void addDependent_(DependentListener anObject) {
		this.addDependentListener(anObject);
	}

	/**
	 * Add anObject as a dependent listener of the receiver.
	 * 
	 * @param anObject jp.co.sra.smalltalk.DependentListener
	 * @category dependents access
	 */
	public synchronized void addDependentListener(DependentListener anObject) {
		Vector myDependentsCopy;
		if (this.myDependents() == null) {
			myDependentsCopy = new Vector();
		} else {
			myDependentsCopy = (Vector) this.myDependents().clone();
		}

		myDependentsCopy.addElement(anObject);
		this.myDependents_(myDependentsCopy);
	}

	/**
	 * Remove anObject from the receiver's dependents.
	 * 
	 * @param anObject jp.co.sra.smalltalk.DependentListener
	 * @category dependents access
	 */
	public void removeDependent_(DependentListener anObject) {
		this.removeDependentListener(anObject);
	}

	/**
	 * Remove anObject from the receiver's dependents.
	 * 
	 * @param anObject jp.co.sra.smalltalk.DependentListener
	 * @category dependents access
	 */
	public synchronized void removeDependentListener(DependentListener anObject) {
		if (this.myDependents() != null) {
			int indx = this.myDependents().indexOf(anObject);
			if (indx >= 0) {
				Vector myDependentsCopy = (Vector) this.myDependents().clone();
				myDependentsCopy.removeElementAt(indx);
				this.myDependents_(myDependentsCopy);
			}
		}
	}

	/**
	 * Arrange to receive a message with aSelector when anAspect changes at
	 * anObject.
	 * 
	 * @param anAspect jp.co.sra.smalltalk.StSymbol
	 * @param anObject jp.co.sra.smalltalk.StObject
	 * @param aSelector java.lang.String
	 * @category dependents access
	 */
	public void expressInterestIn_for_sendBack_(StSymbol anAspect, StObject anObject, String aSelector) {
		StDependencyTransformer dt = new StDependencyTransformer();
		dt.setReceiver_aspect_selector_(anObject, anAspect, aSelector);
		if (this.myDependents() == null || this.myDependents().contains(dt) == false) {
			this.addDependent_(dt);
		}
	}

	/**
	 * Undo a send of expressInterestIn_for_sendBack_().
	 * 
	 * @param anAspect jp.co.sra.smalltalk.StSymbol
	 * @param anObject jp.co.sra.smalltalk.StObject
	 * @category dependents access
	 */
	public void retractInterestIn_for_(StSymbol anAspect, StObject anObject) {
		Object[] dependents = this.dependents();
		for (int i = 0; i < dependents.length; i++) {
			if (dependents[i] instanceof StDependencyTransformer) {
				StDependencyTransformer dt = (StDependencyTransformer) dependents[i];
				if (dt.matches_forAspect_(anObject, anAspect)) {
					this.removeDependent_(dt);
				}
			}
		}
	}

	/**
	 * Print out the error message
	 * 
	 * @param errorMessage java.lang.String
	 * @category error handling
	 */
	public void error_(String errorMessage) {
		System.err.println(errorMessage);
	}

	/**
	 * Just output the error message "shouldNotImplement".
	 * 
	 * @category error handling
	 */
	public void shouldNotImplement() {
		this.error_("shouldNotImplement");
	}

	/**
	 * Just output the error message "subclassResponsibility".
	 * 
	 * @throws SmalltalkException to notify the runtime exception.
	 * @category error handling
	 */
	public void subclassResponsibility() throws SmalltalkException {
		this.error_("subclassResponsibility");
	}

	/**
	 * Throw an exception which indicate an improper integer was used as an
	 * index.
	 * 
	 * @param index int
	 * @throws SmalltalkException to notify the runtime exception.
	 * @category private
	 */
	public Object subscriptBoundsError_(int index) {
		throw new SmalltalkException("subscriptBoundsError: " + index);
	}

	/**
	 * Send the message indicated by "selector" to this object. The "selector"
	 * is the method name.
	 * 
	 * @return dependents java.lang.Object
	 * @param aSymbol java.lang.String
	 * @exception NoSuchMethodException if a matching method is not found.
	 * @exception IllegalAccessException if the underlying method is inaccessible.
	 * @exception IllegalArgumentException if the number of actual and formal parameters differ, or if an unwrapping conversion fails.
	 * @exception InvocationTargetException if the underlying method throws an exception.
	 * @category message handling
	 */
	public Object perform_(String aSymbol) throws NoSuchMethodException, IllegalAccessException, IllegalArgumentException, InvocationTargetException {
		Object returnObject = null;
		Class thisClass = this.getClass();
		Class[] parameterTypes = null;
		Method method = thisClass.getMethod(aSymbol, parameterTypes);
		Object[] argumentList = null;
		returnObject = method.invoke(this, argumentList);
		return returnObject;
	}

	/**
	 * Send the message indicated by "selector" and "argument" to this object.
	 * The "selector" is the method name and its argument is "argument".
	 * 
	 * @return java.lang.Object
	 * @param selector java.lang.String
	 * @param argument java.lang.Object
	 * @exception NoSuchMethodException if the method cannot be found.
	 * @exception IllegalAccessException if the underlying method is inaccessible.
	 * @exception IllegalArgumentException if the number of actual and formal parameters differ, or if an unwrapping conversion fails.
	 * @exception InvocationTargetException if the underlying method throws an exception.
	 * @category message handling
	 */
	public Object perform_with_(String selector, Object argument) throws NoSuchMethodException, IllegalAccessException, IllegalArgumentException, InvocationTargetException {
		Object[] arguments = { argument };
		return this.perform_withArguments_(selector, arguments);
	}

	/**
	 * Send the message indicated by "selector" and "argument" to this object.
	 * The "selector" is the method name and its arguments are "argument1" and
	 * "argument2".
	 * 
	 * @return java.lang.Object
	 * @param selector java.lang.String
	 * @param argument1 java.lang.Object
	 * @param argument2 java.lang.Object
	 * @exception NoSuchMethodException if the method cannot be found.
	 * @exception IllegalAccessException if the underlying method is inaccessible.
	 * @exception IllegalArgumentException if the number of actual and formal parameters differ, or if an unwrapping conversion fails.
	 * @exception InvocationTargetException if the underlying method throws an exception.
	 * @category message handling
	 */
	public Object perform_with_with_(String selector, Object argument1, Object argument2) throws NoSuchMethodException, IllegalAccessException, IllegalArgumentException, InvocationTargetException {
		Object[] arguments = { argument1, argument2 };
		return this.perform_withArguments_(selector, arguments);
	}

	/**
	 * Send the message indicated by "selector" and "arguments" to this object.
	 * The "selector" is the method name and its arguments are the elements of
	 * "arguments".
	 * 
	 * @return java.lang.Object
	 * @param selector java.lang.String
	 * @param arguments java.lang.Object[]
	 * @exception NoSuchMethodException if the method cannot be found.
	 * @exception IllegalAccessException if the underlying method is inaccessible.
	 * @exception IllegalArgumentException if the number of actual and formal parameters differ, or if an unwrapping conversion fails.
	 * @exception InvocationTargetException if the underlying method throws an exception.
	 * @category message handling
	 */
	public Object perform_withArguments_(String selector, Object[] arguments) throws NoSuchMethodException, IllegalAccessException, IllegalArgumentException, InvocationTargetException {
		Object theResult = null;
		Method theMethod = this._searchMethod(selector, arguments);
		theResult = theMethod.invoke(this, arguments);
		return theResult;
	}

	/**
	 * Answer the method, which name is "selector" and its arguments can be "arguments".
	 * 
	 * @return java.lang.reflect.Method
	 * @param selector java.lang.String
	 * @param arguments java.lang.Object[]
	 * @exception NoSuchMethodException if the method cannot be found
	 * @category message handling
	 */
	protected Method _searchMethod(String selector, Object[] arguments) throws NoSuchMethodException {
		return _SearchMethod(this.getClass(), selector, _ParameterTypesOf(arguments));
	}

	/**
	 * Answer the receiver's dependents.
	 * 
	 * @return java.util.Vector
	 * @category private
	 */
	public Vector myDependents() {
		return (Vector) DependentsFields.get(this);
	}

	/**
	 * Set the receiver's dependents.
	 * 
	 * @param dependents java.util.Vector
	 * @category private
	 */
	public void myDependents_(Vector dependents) {
		if (dependents == null || dependents.isEmpty()) {
			DependentsFields.remove(this);
		} else {
			DependentsFields.put(this, dependents);
		}
	}

	/**
	 * Clear all of the receiver's dependents.
	 * 
	 * @category private
	 */
	public void breakDependents() {
		this.myDependents_(null);
	}

	/**
	 * Input all the public fileds from a ObjectInputStream.   Please refer to
	 * StObjectTestExamples.Example8() for the usage..
	 * 
	 * @param in java.io.ObjectInputStream
	 * @throws SmalltalkException DOCUMENT ME!
	 * @category private
	 */
	private void _getFileds(ObjectInputStream in) {
		Class aClass = this.getClass();
		Field[] aFields = aClass.getFields();
		try {
			for (int i = 0; i < Array.getLength(aFields); i++) {
				aFields[i].set(this, in.readObject());
			}
		} catch (Exception e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Output all the public fileds to a ObjectOutputStream.
	 * Please refer to StObjectTestExamples.Example8() for the usage..
	 * 
	 * @param out java.io.ObjectOutputStream
	 * @throws SmalltalkException DOCUMENT ME!
	 * @category private
	 */
	private void _putFileds(ObjectOutputStream out) {
		Class aClass = this.getClass();
		Field[] aFields = aClass.getFields();
		try {
			for (int i = 0; i < Array.getLength(aFields); i++) {
				out.writeObject(aFields[i].get(this));
			}
		} catch (Exception e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * ReadObject is called to restore the state of the StObject from an input stream.   
	 * It is responsible for implementing the interface serializable.
	 * 
	 * @param in java.io.ObjectInputStream
	 * @category private reading
	 */
	private void readObject(ObjectInputStream in) {
		this._getFileds(in);
	}

	/**
	 * WriteObject is called to output the filed of the StObject to a  output stream. 
	 * It is responsible for implementing the interface serializable.
	 * 
	 * @param out java.io.ObjectOutputStream
	 * @category private writing
	 */
	private void writeObject(ObjectOutputStream out) {
		this._putFileds(out);
	}

}