package jp.co.sra.smalltalk;

import java.io.IOException;
import java.io.Writer;
import java.lang.IndexOutOfBoundsException;

/**
 * StInterval class
 * 
 *  @author    NISHIHARA Satoshi
 *  @created   2000/01/13 (by NISHIHARA Satoshi)
 *  @updated   2000/01/21 (by NISHIHARA Satoshi)
 *  @version   8.9
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: StInterval.java,v 8.10 2008/02/20 06:33:18 nisinaka Exp $
 */
public class StInterval extends StObject {
	/** initial number in the progression. */
	private double start;

	/** last number in the progression. */
	private double stop;

	/** increment number for determining the next number in the progression. */
	private double step = 1.0;

	/**
	 * Create a new instance of <code>StInterval</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public StInterval() {
		super();
	}

	/**
	 * Create a new instance of <code>StInterval</code> and initialize it.
	 * 
	 * @param startNumber double
	 * @param stopNumber double
	 * @category Instance creation
	 */
	public StInterval(double startNumber, double stopNumber) {
		super();
		this.setFrom_to_by_(startNumber, stopNumber, 1);
	}

	/**
	 * Create a new instance of <code>StInterval</code> and initialize it.
	 * 
	 * @param startNumber double
	 * @param stopNumber double
	 * @param stepNumber double
	 * @category Instance creation
	 */
	public StInterval(double startNumber, double stopNumber, double stepNumber) {
		super();
		this.setFrom_to_by_(startNumber, stopNumber, stepNumber);
	}

	/**
	 * Create a new instance of <code>StInterval</code> and initialize it.
	 * 
	 * @param startNumber double
	 * @param stopNumber double
	 * @return jp.co.sra.smalltalk.StInterval
	 * @category Instance creation
	 */
	public static StInterval From_to_(double startNumber, double stopNumber) {
		StInterval interval = new StInterval();
		interval.setFrom_to_by_(startNumber, stopNumber, 1);
		return interval;
	}

	/**
	 * Create a new instance of <code>StInterval</code> and initialize it.
	 * 
	 * @param startNumber double
	 * @param stopNumber double
	 * @param stepNumber DOCUMENT ME!
	 * @return jp.co.sra.smalltalk.StInterval
	 * @category Instance creation
	 */
	public static StInterval From_to_by_(double startNumber, double stopNumber, double stepNumber) {
		StInterval interval = new StInterval();
		interval.setFrom_to_by_(startNumber, stopNumber, stepNumber);
		return interval;
	}

	/**
	 * Computes the number of the index of this progression.
	 * 
	 * @param index int
	 * @return double
	 * @throws java.lang.IndexOutOfBoundsException
	 * @category accessing
	 */
	public double at_(int index) {
		if ((index >= 1) && (index <= this.size())) {
			return start + (step * (index - 1));
		}

		throw new IndexOutOfBoundsException("size: " + this.size() + " index: " + index);
	}

	/**
	 * Enumerate all elements of the receiver and evaluate the StBlockClosure.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return double[]
	 * @category enumerating
	 */
	public double[] collect_(StBlockClosure aBlock) {
		double[] result = new double[this.size()];
		double nextValue = start;

		for (int i = 0; i < result.length; i++) {
			result[i] = ((Double) (aBlock.value_(new Double(nextValue)))).doubleValue();
			nextValue += step;
		}

		return result;
	}

	/**
	 * Enumerate all elements of the receiver and evaluate the StBlockClosure.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public void do_(StBlockClosure aBlock) {
		double value = start;

		if (step < 0) {
			while (stop <= value) {
				aBlock.value_(new Double(value));
				value += step;
			}
		} else {
			while (stop >= value) {
				aBlock.value_(new Double(value));
				value += step;
			}
		}
	}

	/**
	 * Answer true if the Object is equal to the receiver, otherwise false.
	 * 
	 * @param other java.lang.Object
	 * @return boolean
	 * @see java.lang.Object#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object other) {
		if (other.getClass() == this.getClass()) {
			StInterval aStInterval = (StInterval) other;

			return ((this.start == (aStInterval.start)) && (this.step == (aStInterval.increment())) && (this.size() == aStInterval.size()));
		}

		return false;
	}

	/**
	 * Initinal number of this progression.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double first() {
		return start;
	}

	// Hash Function

	/**
	 * Computes a hash code for this object.
	 * 
	 * @return int
	 * @see java.lang.Object#hashCode()
	 * @category comparing
	 */
	public int hashCode() {
		long bitsStart = Double.doubleToLongBits(start);
		int startHash = (int) (bitsStart ^ (bitsStart >> 32));
		long bitsStop = Double.doubleToLongBits(stop);
		int stopHash = (int) (bitsStop ^ (bitsStop >> 32));

		return (((startHash >> 2) | stopHash) >> 1 | this.size());
	}

	/**
	 * Increment number of this progression.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double increment() {
		return step;
	}

	/**
	 * Computes the last number of this progression.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double last() {
		return stop - ((stop - start) % step);
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write("(" + start + " to: " + stop);
		if (step != 1) {
			aWriter.write(" by: " + step);
		}
		aWriter.write(")");
	}

	/**
	 * Enumerate all elements of the receiver and evaluate the StBlockClosure.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public void reverseDo_(StBlockClosure aBlock) {
		double value = stop;

		if (step < 0) {
			while (start >= value) {
				aBlock.value_(new Double(value));
				value -= step;
			}
		} else {
			while (start <= value) {
				aBlock.value_(new Double(value));
				value -= step;
			}
		}
	}

	/**
	 * Computes the size of this progression.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int size() {
		int size = 0;

		if (step < 0) {
			if (!(start < stop)) {
				size = (int) ((stop - start) / step) + 1;
			}
		} else {
			if (!(stop < start)) {
				size = (int) ((stop - start) / step) + 1;
			}
		}

		return size;
	}

	/**
	 * Get initial number in the progression.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double start() {
		return this.start;
	}

	/**
	 * Set initial number in the progression.
	 * 
	 * @param startValue
	 * @return double
	 * @category accessing
	 */
	public double start_(double startValue) {
		return this.start = startValue;
	}

	/**
	 * Get increment number for determining the next number in the progression.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double step() {
		return this.step;
	}

	/**
	 * Set increment number for determining  the next number in the
	 * progression.
	 * 
	 * @param stepValue
	 * @return double
	 * @category accessing
	 */
	public double step_(double stepValue) {
		return this.step = stepValue;
	}

	/**
	 * Get last number in the progression.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double stop() {
		return this.stop;
	}

	/**
	 * Set last number in the progression.
	 * 
	 * @param stopValue
	 * @return double
	 * @category accessing
	 */
	public double stop_(double stopValue) {
		return this.stop = stopValue;
	}

	/**
	 * Store my string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#storeOn_(java.io.Writer)
	 * @category printing
	 */
	public void storeOn_(Writer aWriter) throws IOException {
		this.printOn_(aWriter);
	}

	/**
	 * Set the receiver's parameter values.
	 * 
	 * @param startNumber double
	 * @param stopNumber double
	 * @param stepNumber double
	 * @return jp.co.sra.smalltalk.StInterval
	 * @category private
	 */
	private StInterval setFrom_to_by_(double startNumber, double stopNumber, double stepNumber) {
		this.start = startNumber;
		this.stop = stopNumber;
		this.step = stepNumber;

		return this;
	}
}
