package jp.co.sra.smalltalk;

import java.awt.BorderLayout;
import java.awt.Canvas;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.awt.image.DirectColorModel;
import java.awt.image.IndexColorModel;
import java.awt.image.PixelGrabber;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.io.Writer;
import java.util.Hashtable;

/**
 * StImage class
 * 
 *  @author    Hirotsugu Kondo
 *  @created   1998/11/10 (by Hirotsugu Kondo)
 *  @updated   2000/04/05 (by Mitsuhiro Asada)
 *  @updated   2002/04/17 (by nisinaka)
 *  @updated   2004/01/14 (by nisinaka)
 *  @version   8.9
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: StImage.java,v 8.18 2008/02/20 06:33:18 nisinaka Exp $
 */
public class StImage extends StObject implements StDisplayable, Serializable {

	// Serialize version
	private static final long serialVersionUID = -7647941984469789820L;

	// Constants
	public static final int WriteZeros = 0;
	public static final int And = 1;
	public static final int Over = 3;
	public static final int Erase = 4;
	public static final int Reverse = 6;
	public static final int Under = 7;
	public static final int ReverseUnder = 13;
	public static final int WriteOnes = 15;
	public static final int Paint = 16;

	protected static IndexColorModel MonoMaskColorModel;

	protected transient BufferedImage image;
	protected transient int rowByteSize;
	protected transient byte[] bits;

	/**
	 * Create a new instance of StImage.
	 * 
	 * @param width int
	 * @param height int
	 * @category Instance creation
	 */
	public StImage(int width, int height) {
		this(width, height, ColorModel.getRGBdefault());

		int[] pixels = new int[width * height];
		for (int i = 0; i < pixels.length; i++) {
			pixels[i] = -1;
		}
		this.setPixels(pixels);
	}

	/**
	 * Create a new instance of StImage and initialize it.
	 * 
	 * @param width int
	 * @param height int
	 * @param colorModel java.awt.image.ColorModel
	 * @category Instance creation
	 */
	public StImage(int width, int height, ColorModel colorModel) {
		image = new BufferedImage(colorModel, colorModel.createCompatibleWritableRaster(width, height), false, null);
		rowByteSize = (width * this.bitsPerPixel() + 31) >> 5 << 2;
	}

	/**
	 * Create a new instance of StImage and initialize it.
	 * 
	 * @param width int
	 * @param height int
	 * @param bits byte[]
	 * @category Instance creation
	 */
	public StImage(int width, int height, byte[] bits) {
		this(width, height, bits, ColorModel.getRGBdefault());
	}

	/**
	 * Create a new instance of StImage and initialize it.
	 * 
	 * @param width int
	 * @param height int
	 * @param byteArray byte[]
	 * @param colorModel java.awt.image.ColorModel
	 * @category Instance creation
	 */
	public StImage(int width, int height, byte[] byteArray, ColorModel colorModel) {
		this(width, height, colorModel);

		if (byteArray != null) {
			if ((rowByteSize * height) > byteArray.length) {
				throw SmalltalkException.Error("Bitmap array has incorrect size.");
			}

			// Set the pixels.
			for (int y = 0; y < height; y++) {
				for (int x = 0; x < width; x++) {
					this.setPixel(x, y, this._getPixelFromBits(x, y, byteArray));
				}
			}

			// Set the bit array.
			bits = byteArray;
		}
	}

	/**
	 * Create a new instance of StImage and initialize it.
	 * 
	 * @param width int
	 * @param height int
	 * @param pixels int[]
	 * @category Instance creation
	 */
	public StImage(int width, int height, int[] pixels) {
		this(width, height, pixels, ColorModel.getRGBdefault());
	}

	/**
	 * Create a new instance of StImage and initialize it.
	 * 
	 * @param width int
	 * @param height int
	 * @param pixels int[]
	 * @param colorModel java.awt.image.ColorModel
	 * @category Instance creation
	 */
	public StImage(int width, int height, int[] pixels, ColorModel colorModel) {
		this(width, height, colorModel);
		this.setPixels(pixels);
	}

	/**
	 * Create a new instance of StImage and initialize it.
	 * 
	 * @param width int
	 * @param height int
	 * @param byteArray jp.co.sra.smalltalk.StByteArray
	 * @param colorModel java.awt.image.ColorModel
	 * @category Instance creation
	 */
	public StImage(int width, int height, StByteArray byteArray, ColorModel colorModel) {
		this(width, height, byteArray._asBytes(), colorModel);
	}

	/**
	 * Create a new instance of StImage and initialize it with the image.
	 * 
	 * @param originalImage java.awt.Image
	 * @category Instance creation
	 */
	public StImage(Image originalImage) {
		if (originalImage instanceof BufferedImage) {
			BufferedImage bufferedImage = (BufferedImage) originalImage;
			image = bufferedImage.getSubimage(0, 0, bufferedImage.getWidth(), bufferedImage.getHeight());
		} else {
			Component medium = new Canvas();
			int width = originalImage.getWidth(medium);
			int height = originalImage.getHeight(medium);
			int[] pixels = new int[width * height];
			PixelGrabber grabber = new PixelGrabber(originalImage, 0, 0, width, height, pixels, 0, width);
			try {
				grabber.grabPixels();
			} catch (InterruptedException e) {
			}
			image = (new StImage(width, height, pixels)).image();
			this.setPixels(pixels);
		}

		rowByteSize = (this.width() * this.bitsPerPixel() + 31) >> 5 << 2;
	}

	/**
	 * Create a new instance of StImage.
	 * 
	 * @param extentPoint java.awt.Point
	 * @category Instance creation
	 */
	public StImage(Point extentPoint) {
		this(extentPoint.x, extentPoint.y);
	}

	/**
	 * Create a new instance of StImage.
	 * 
	 * @param extent java.awt.Dimension
	 * @category Instance creation
	 */
	public StImage(Dimension extent) {
		this(extent.width, extent.height);
	}

	/**
	 * Answer the mono mask color model.
	 * 
	 * @return java.awt.image.IndexColorModel
	 * @category Constants of color model
	 */
	public static IndexColorModel MonoMaskColorModel() {
		if (MonoMaskColorModel == null) {
			byte[] bytes = { (byte) 0xFF, (byte) 0 };
			MonoMaskColorModel = new IndexColorModel(1, 2, bytes, bytes, bytes);
		}
		return MonoMaskColorModel;
	}

	/**
	 * Answer an image from user.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Utilities
	 */
	public static StImage _FromUser() {
		SystemInterface StPLInterface = SystemInterface.Current();
		int[] pixels;
		int[] dimension = new int[2];
		synchronized (StPLInterface) {
			pixels = StPLInterface.utilImageFromUser(dimension);
		}

		if (pixels != null) {
			return new StImage(dimension[0], dimension[1], pixels);
		} else {
			return null;
		}
	}

	/**
	 * Answer an image of specified area of screen.
	 * 
	 * @param rect java.awt.Rectangle
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Utilities
	 */
	public static StImage _OfArea(Rectangle rect) {
		SystemInterface StPLInterface = SystemInterface.Current();
		int[] pixels;
		synchronized (StPLInterface) {
			pixels = StPLInterface.utilImageOfArea(rect.x, rect.y, rect.width, rect.height);
		}

		if (pixels != null) {
			return new StImage(rect.width, rect.height, pixels);
		} else {
			return null;
		}
	}

	/**
	 * Answer an image of whole screen.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Utilities
	 */
	public static StImage _OfWholeScreen() {
		SystemInterface StPLInterface = SystemInterface.Current();
		int[] pixels;
		int[] dimension = new int[2];
		synchronized (StPLInterface) {
			pixels = StPLInterface.utilImageOfWholeScreen(dimension);
		}

		if (pixels != null) {
			return new StImage(dimension[0], dimension[1], pixels);
		} else {
			return null;
		}
	}

	/**
	 * Answer the index of the nearest color in the index color model.
	 * This method could be used instead of 'indexOfPaintNearest:' of ColorPalette.
	 * 
	 * @param aColor java.awt.Color
	 * @param anIndexColorModel java.awt.image.IndexColorModel
	 * @return int
	 * @category Utilities
	 */
	public static int _IndexOfNearestColor(Color aColor, IndexColorModel anIndexColorModel) {
		int fit = StColorValue._MaxDistanceSquared() + 1;
		int fitIndex = -1;
		for (int index = 0; index < anIndexColorModel.getMapSize(); index++) {
			int pixelRGB = anIndexColorModel.getRGB(index);
			Color newColor = new Color(pixelRGB);
			Integer d = StColorValue._DistanceSquaredFrom_ifLessThan_(newColor, aColor, fit);
			if (d != null) {
				fit = d.intValue();
				fitIndex = index;
			}
		}
		return fitIndex;
	}

	/**
	 * Display the image.
	 * 
	 * @param anImage jp.co.sra.smalltalk.StImage
	 * @category Utilities
	 */
	public static void Display_(StImage anImage) {
		Frame aFrame = new Frame();
		aFrame.setTitle("Display");
		aFrame.setLayout(new BorderLayout());
		aFrame.addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				e.getWindow().dispose();
			}
		});

		final StImage _image = anImage;
		Canvas newCanvas = new Canvas() {
			public void paint(Graphics g) {
				_image.displayOn_(g);
				return;
			}
		};

		newCanvas.setBounds(0, 0, anImage.width(), anImage.height());
		aFrame.add(newCanvas);
		aFrame.pack();
		StApplicationModel._ShowAtMousePoint(aFrame);
	}

	/**
	 * Answer the number of the bits for one pixel.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int bitsPerPixel() {
		return this.colorModel().getPixelSize();
	}

	/**
	 * Answer my current color model.
	 * 
	 * @return java.awt.image.ColorModel
	 * @category accessing
	 */
	public ColorModel colorModel() {
		return this.image().getColorModel();
	}

	/**
	 * Answer extent of this image.
	 * 
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point extent() {
		return new Point(this.width(), this.height());
	}

	/**
	 * Answer the height of this image.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int height() {
		return this.image().getHeight();
	}

	/**
	 * Answer the image object.
	 * 
	 * @return java.awt.image.BufferedImage
	 * @category accessing
	 */
	public BufferedImage image() {
		return image;
	}

	/**
	 * Answer the width of this image.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int width() {
		return this.image().getWidth();
	}

	/**
	 * Answer the bounds of this image.
	 * 
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.smalltalk.StDisplayable#bounds()
	 * @category bounds accessing
	 */
	public Rectangle bounds() {
		return new Rectangle(0, 0, this.width(), this.height());
	}

	/**
	 * Answer the pixel at the specified zero-based 2D index.
	 * 
	 * @param point java.awt.Point
	 * @return int
	 * @deprecated use getPixel(int x, int y) instead.
	 * @category bits accessing
	 */
	public int atPoint_(Point point) {
		return this.getPixel(point.x, point.y);
	}

	/**
	 * Set a pixel value at the specified zero-based 2D index.
	 * 
	 * @param point java.awt.Point
	 * @param pixelValue int
	 * @deprecated use setPixel(int x, int y, int pixel) instead.
	 * @category bits accessing
	 */
	public void atPoint_put_(Point point, int pixelValue) {
		this.setPixel(point.x, point.y, pixelValue);
	}

	/**
	 * Answer the pixel value at the specified point.
	 * 
	 * @param x int
	 * @param y int
	 * @return int
	 * @category bits accessing
	 */
	public int getPixel(int x, int y) {
		if (x >= this.width() || y >= this.height()) {
			throw new ArrayIndexOutOfBoundsException("Index out of Image Bounds Exception at (" + x + ", " + y + ")");
		}
		return this.image().getRGB(x, y);
	}

	/**
	 * Answer all of the pixel values.
	 * 
	 * @return int[]
	 * @category bits accessing
	 */
	public int[] getPixels() {
		int width = this.width();
		int height = this.height();
		return this.image().getRGB(0, 0, width, height, null, 0, width);
	}

	/**
	 * Answer the pixel value at the specified point in bits.
	 * 
	 * @param x int
	 * @param y int
	 * @param bits byte[]
	 * @return int
	 * @category bits accessing
	 */
	protected int _getPixelFromBits(int x, int y, byte[] bits) {
		int bitsPerPixel = this.bitsPerPixel();

		if (bitsPerPixel == 32) {
			return this._getPixel32FromBits(x, y, bits);
		} else if (bitsPerPixel == 24) {
			return this._getPixel24FromBits(x, y, bits);
		} else if (bitsPerPixel == 8) {
			return this._getPixel8FromBits(x, y, bits);
		} else if (bitsPerPixel == 1) {
			return this._getPixel1FromBits(x, y, bits);
		}

		throw new SmalltalkException("can't handle an " + bitsPerPixel + "-color image");
	}

	/**
	 * Answer the pixel value as a 1-bit color image at the specified point in bits.
	 * 
	 * @param x int
	 * @param y int
	 * @param bits int[]
	 * @return int
	 * @category bits accessing
	 */
	protected int _getPixel1FromBits(int x, int y, byte[] bits) {
		int pixel = bits[(y * rowByteSize) + (x >> 3)] & 0xff;
		pixel = pixel >> (7 - (x & 7));
		pixel = pixel & 1;
		return this.image().getColorModel().getRGB(pixel);
	}

	/**
	 * Answer the pixel value as an 8-bit color image at the specified point in bits.
	 * 
	 * @param x int
	 * @param y int
	 * @param bits int[]
	 * @return int
	 * @category bits accessing
	 */
	protected int _getPixel8FromBits(int x, int y, byte[] bits) {
		int pixel = bits[y * rowByteSize + x] & 0xff;
		return this.image().getColorModel().getRGB(pixel);
	}

	/**
	 * Answer the pixel value as a 24-bit color image at the specified point in bits.
	 * 
	 * @param x int
	 * @param y int
	 * @param bits int[]
	 * @return int
	 * @category bits accessing
	 */
	protected int _getPixel24FromBits(int x, int y, byte[] bits) {
		int byteIndex = (y * rowByteSize) + x + x + x;
		int r = bits[byteIndex + 0] & 0xff;
		int g = bits[byteIndex + 1] & 0xff;
		int b = bits[byteIndex + 2] & 0xff;
		return this.image().getColorModel().getRGB((new Color(r, g, b)).getRGB());
	}

	/**
	 * Answer the pixel value as a 32-bit color image at the specified point in bits.
	 * 
	 * @param x int
	 * @param y int
	 * @param bits byte[]
	 * @return int
	 * @category bits accessing
	 */
	protected int _getPixel32FromBits(int x, int y, byte[] bits) {
		int byteIndex = (y * rowByteSize) + x + x + x + x;
		int a = 0xff - (bits[byteIndex + 0] & 0xff);
		int r = bits[byteIndex + 1] & 0xff;
		int g = bits[byteIndex + 2] & 0xff;
		int b = bits[byteIndex + 3] & 0xff;
		return this.image().getColorModel().getRGB((new Color(r, g, b, a)).getRGB());
	}

	/**
	 * Set the pixel value to the specified point.
	 * 
	 * @param x int
	 * @param y int
	 * @param pixel int
	 * @category bits accessing
	 */
	public void setPixel(int x, int y, int pixel) {
		this.image().setRGB(x, y, pixel);
		this._flushBits();
	}

	/**
	 * Set the specified pixel values.
	 * 
	 * @param pixels int[]
	 * @category bits accessing
	 */
	public void setPixels(int[] pixels) {
		int width = this.width();
		int height = this.height();
		if (pixels == null) {
			pixels = new int[width * height];
		} else if ((width * height) > pixels.length) {
			throw SmalltalkException.Error("Pixel array has incorrect size.");
		}
		this.image().setRGB(0, 0, width, height, pixels, 0, width);
		this._flushBits();
	}

	/**
	 * Set the pixel value at the specified point in bits.
	 * 
	 * @param x int
	 * @param y int
	 * @param bits byte[]
	 * @return int
	 * @category bits accessing
	 */
	protected int _setPixelIntoBits(int x, int y, byte[] bits) {
		int bitsPerPixel = this.bitsPerPixel();

		if (bitsPerPixel == 32) {
			return this._setPixel32IntoBits(x, y, bits);
		} else if (bitsPerPixel == 24) {
			return this._setPixel24IntoBits(x, y, bits);
		} else if (bitsPerPixel == 8) {
			return this._setPixel8IntoBits(x, y, bits);
		} else if (bitsPerPixel == 1) {
			return this._setPixel1IntoBits(x, y, bits);
		}

		throw new SmalltalkException("can't handle an " + bitsPerPixel + "-color image");
	}

	/**
	 * Set the pixel value as a 1-bit color image at the specified point in bits.
	 * 
	 * @param x int
	 * @param y int
	 * @param bits byte[]
	 * @return int
	 * @category bits accessing
	 */
	protected int _setPixel1IntoBits(int x, int y, byte[] bits) {
		ColorModel colorModel = this.image().getColorModel();
		int pixelValue = colorModel.getRGB(this.getPixel(x, y) & 0xff);
		int byteIndex = (y * rowByteSize) + (x >> 3);
		int whichBit = 7 - (x & 7);
		int value = (pixelValue == 0) ? (bits[byteIndex] & (-1 - (1 << whichBit))) : bits[byteIndex] | (1 << whichBit);
		bits[byteIndex] = (byte) value;
		return pixelValue;
	}

	/**
	 * Set the pixel value as an 8-bit color image at the specified point in bits.
	 * 
	 * @param x int
	 * @param y int
	 * @param bits byte[]
	 * @return int
	 * @category bits accessing
	 */
	protected int _setPixel8IntoBits(int x, int y, byte[] bits) {
		IndexColorModel colorModel = (IndexColorModel) this.image().getColorModel();
		int pixelValue = this.getPixel(x, y);
		int transparentPixel = colorModel.getTransparentPixel();
		if (transparentPixel >= 0 && colorModel.getRGB(transparentPixel) == pixelValue) {
			pixelValue = transparentPixel;
		} else {
			pixelValue = _IndexOfNearestColor(new Color(pixelValue), colorModel);
		}
		bits[y * rowByteSize + x] = (byte) (pixelValue & 0xFF);
		return pixelValue;
	}

	/**
	 * Set the pixel value as a 24-bit color image at the specified point in bits.
	 * 
	 * @param x int
	 * @param y int
	 * @param bits byte[]
	 * @return int
	 * @category bits accessing
	 */
	protected int _setPixel24IntoBits(int x, int y, byte[] bits) {
		ColorModel colorModel = this.image().getColorModel();
		int pixelValue = this.getPixel(x, y);
		int byteIndex = (y * rowByteSize) + x + x + x;
		bits[byteIndex + 0] = (byte) colorModel.getRed(pixelValue);
		bits[byteIndex + 1] = (byte) colorModel.getGreen(pixelValue);
		bits[byteIndex + 2] = (byte) colorModel.getBlue(pixelValue);
		return pixelValue;
	}

	/**
	 * Set the pixel value as a 32-bit color image at the specified point in bits.
	 * 
	 * @param x int
	 * @param y int
	 * @param bits byte[]
	 * @return int
	 * @category bits accessing
	 */
	protected int _setPixel32IntoBits(int x, int y, byte[] bits) {
		ColorModel colorModel = this.image().getColorModel();
		int pixelValue = this.getPixel(x, y);
		int byteIndex = (y * rowByteSize) + x + x + x + x;
		bits[byteIndex + 0] = (byte) (0xff - colorModel.getAlpha(pixelValue));
		bits[byteIndex + 1] = (byte) colorModel.getRed(pixelValue);
		bits[byteIndex + 2] = (byte) colorModel.getGreen(pixelValue);
		bits[byteIndex + 3] = (byte) colorModel.getBlue(pixelValue);
		return pixelValue;
	}

	/**
	 * Answer the value at aPoint according to this image interpretation.
	 * 
	 * @param point java.awt.Point
	 * @return java.awt.Color
	 * @category bits accessing
	 */
	public Color valueAtPoint_(Point point) {
		return new Color(this.getPixel(point.x, point.y), this.colorModel().hasAlpha());
	}

	/**
	 * Set the value at aPoint according to this image interpretation.
	 * 
	 * @param point java.awt.Point
	 * @param color java.awt.Color
	 * @category bits accessing
	 */
	public void valueAtPoint_put_(Point point, Color color) {
		this.setPixel(point.x, point.y, color.getRGB());
	}

	/**
	 * Answer my current byte array.
	 * 
	 * @return byte[]
	 * @category bits accessing
	 */
	public byte[] getBits() {
		byte[] bits = new byte[this.bits().length];
		System.arraycopy(this.bits(), 0, bits, 0, bits.length);
		return bits;
	}

	/**
	 * Display the receiver on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		this.displayOn_at_(aGraphics, new Point(0, 0));
	}

	/**
	 * Display the receiver on the graphics at the specified point.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_at_(java.awt.Graphics, java.awt.Point)
	 * @category displaying
	 */
	public void displayOn_at_(Graphics aGraphics, Point aPoint) {
		aGraphics.drawImage(this.image(), aPoint.x, aPoint.y, null);
	}

	/**
	 * Create a copy of the StImage.
	 * 
	 * @return java.lang.Object
	 * @see java.lang.Object#clone()
	 * @category copying
	 */
	public Object clone() {
		return new StImage(this.width(), this.height(), this.getPixels(), this.image().getColorModel());
	}

	/**
	 * Answer a new Image with the same extent, depth as the this image.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category copying
	 */
	public StImage copyEmpty() {
		return this.copyEmpty_(this.extent());
	}

	/**
	 * Answer a new Image with the same extent, depth as the this image.
	 * 
	 * @param newExtent java.awt.Point
	 * @return jp.co.sra.smalltalk.StImage
	 * @category copying
	 */
	public StImage copyEmpty_(Point newExtent) {
		return new StImage(newExtent.x, newExtent.y);
	}

	/**
	 * Answer the receiver as StImage.
	 *
	 * @return jp.co.sra.smalltalk.StImage
	 * @see jp.co.sra.smalltalk.StDisplayable#asImage()
	 * @category converting
	 */
	public StImage asImage() {
		return this;
	}

	/**
	 * Answer image representing my contents using the palette aPalette.
	 * 
	 * @param aPalette java.awt.image.ColorModel
	 * @return jp.co.sra.smalltalk.StImage
	 * @category converting
	 */
	public StImage convertToPalette_(IndexColorModel aPalette) {
		return this._convertToPalette_RenderedByNearistPaint(aPalette);
	}

	/**
	 * Answer image representing my contents using the palette aPalette. Use
	 * ErrorDiffusionImageRender to render the new image. ErrorDiffusion
	 * generate halftones using the Floyd-Steinberg error diffusion algorithm.
	 * 
	 * @param aPalette java.awt.image.IndexColorModel
	 * @return jp.co.sra.smalltalk.StImage
	 * @category converting
	 */
	public StImage _convertToPalette_RenderedByErrorDiffusion(IndexColorModel aPalette) {
		int width = this.width();
		int height = this.height();
		StImage newImage = new StImage(width, height, aPalette);

		Hashtable colorDictionary = new Hashtable();
		for (int y = 0; y < height; y++) {
			for (int x = 0; x < width; x++) {
				Integer pixel = new Integer(this.getPixel(x, y));
				Color currentColor = new Color(pixel.intValue());
				if (colorDictionary.containsKey(pixel)) {
					newImage.setPixel(x, y, ((Integer) colorDictionary.get(pixel)).intValue());
				} else {
					int fit = StColorValue._MaxDistanceSquared() + 1;
					int fitRGB = 0xFF000000;
					for (int j = 0; j < aPalette.getMapSize(); j++) {
						int pixelRGB = aPalette.getRGB(j);
						Color newColor = new Color(pixelRGB);
						Integer d = StColorValue._DistanceSquaredFrom_ifLessThan_(newColor, currentColor, fit);
						if (d != null) {
							fit = d.intValue();
							fitRGB = pixelRGB;
						}
					}
					colorDictionary.put(pixel, new Integer(fitRGB));
					newImage.setPixel(x, y, fitRGB);
				}

				Color actualColor = new Color(this.getPixel(x, y));
				float redError = (currentColor.getRed() - actualColor.getRed()) / 16.0f;
				float greenError = (currentColor.getGreen() - actualColor.getGreen()) / 16.0f;
				float blueError = (currentColor.getBlue() - actualColor.getBlue()) / 16.0f;
				if (x < (width - 1)) {
					Color rightColor = new Color(this.getPixel(x + 1, y));
					rightColor = new Color(Math.min(255, Math.max(0, rightColor.getRed() + (int) (redError * 7))), Math.min(255, Math.max(0, rightColor.getGreen() + (int) (greenError * 7))), Math.min(255, Math.max(0, rightColor.getBlue()
							+ (int) (blueError * 7))));
					newImage.setPixel(x + 1, y, rightColor.getRGB());
				}

				if (y < (height - 1)) {
					if (x > 0) {
						Color leftDownColor = new Color(this.getPixel(x - 1, y + 1));
						leftDownColor = new Color(Math.min(255, Math.max(0, leftDownColor.getRed() + (int) (redError * 3))), Math.min(255, Math.max(0, leftDownColor.getGreen() + (int) (greenError * 3))), Math.min(255, Math.max(0, leftDownColor.getBlue()
								+ (int) (blueError * 3))));
						newImage.setPixel(x - 1, y + 1, leftDownColor.getRGB());
					}

					Color centerDownColor = new Color(this.getPixel(x, y + 1));
					centerDownColor = new Color(Math.min(255, Math.max(0, centerDownColor.getRed() + (int) (redError * 5))), Math.min(255, Math.max(0, centerDownColor.getGreen() + (int) (greenError * 5))), Math.min(255, Math.max(0, centerDownColor
							.getBlue()
							+ (int) (blueError * 5))));
					newImage.setPixel(x, y + 1, centerDownColor.getRGB());

					if (x < (newImage.width() - 1)) {
						Color rightDownColor = new Color(this.getPixel(x + 1, y + 1));
						rightDownColor = new Color(Math.min(255, Math.max(0, rightDownColor.getRed() + (int) redError)), Math.min(255, Math.max(0, rightDownColor.getGreen() + (int) greenError)), Math.min(255, Math.max(0, rightDownColor.getBlue()
								+ (int) blueError)));
						newImage.setPixel(x + 1, y + 1, rightDownColor.getRGB());
					}
				}
			}
		}

		return newImage;
	}

	/**
	 * Answer image representing my contents using the palette aPalette. Use
	 * NearistPatintImageRender to render the new image.
	 * 
	 * @param aPalette java.awt.image.IndexColorModel
	 * @return jp.co.sra.smalltalk.StImage
	 * @category converting
	 */
	public StImage _convertToPalette_RenderedByNearistPaint(IndexColorModel aPalette) {
		int width = this.width();
		int height = this.height();
		StImage newImage = new StImage(width, height, aPalette);

		Hashtable colorDictionary = new Hashtable();
		for (int y = 0; y < height; y++) {
			for (int x = 0; x < width; x++) {
				Integer pixel = new Integer(this.getPixel(x, y));
				if (colorDictionary.containsKey(pixel)) {
					newImage.setPixel(x, y, ((Integer) colorDictionary.get(pixel)).intValue());
				} else {
					int fit = StColorValue._MaxDistanceSquared() + 1;
					int fitRGB = 0xFF000000;
					Color oldColor = new Color(pixel.intValue());
					for (int j = 0; j < aPalette.getMapSize(); j++) {
						int pixelRGB = aPalette.getRGB(j);
						Color newColor = new Color(pixelRGB);
						Integer d = StColorValue._DistanceSquaredFrom_ifLessThan_(newColor, oldColor, fit);
						if (d != null) {
							fit = d.intValue();
							fitRGB = pixelRGB;
						}
					}
					colorDictionary.put(pixel, new Integer(fitRGB));
					newImage.setPixel(x, y, fitRGB);
				}
			}
		}

		return newImage;
	}

	/**
	 * Print my storable string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @exception IOException The exception description.
	 * @see jp.co.sra.smalltalk.StObject#storeOn_(java.io.Writer)
	 * @category printing
	 */
	public void storeOn_(Writer aWriter) throws IOException {
		aWriter.write("(Image extent: ");
		aWriter.write(String.valueOf(this.width()));
		aWriter.write('@');
		aWriter.write(String.valueOf(this.height()));
		aWriter.write(" depth: ");
		aWriter.write(String.valueOf(this.bitsPerPixel()));
		aWriter.write(" bitsPerPixel: ");
		aWriter.write(String.valueOf(this.bitsPerPixel()));
		aWriter.write(" palette: ");
		this.storePaletteOn_(aWriter);
		aWriter.write(" usingBits: ");
		(new StByteArray(this.bits())).storeOn_(aWriter);
		aWriter.write(')');
	}

	/**
	 * Print the storable string representation of my palette on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @exception java.io.IOException
	 * @category printing
	 */
	protected void storePaletteOn_(Writer aWriter) throws IOException {
		ColorModel colorModel = this.image().getColorModel();

		if (colorModel instanceof DirectColorModel) {
			aWriter.write("(FixedPalette redShift: ");
			aWriter.write(String.valueOf((int) (Math.log(((DirectColorModel) colorModel).getRedMask() / 255) / Math.log(2))));
			aWriter.write(" redMask: 255 greenShift: ");
			aWriter.write(String.valueOf((int) (Math.log(((DirectColorModel) colorModel).getGreenMask() / 255) / Math.log(2))));
			aWriter.write(" greenMask: 255 blueShift: ");
			aWriter.write(String.valueOf((int) (Math.log(((DirectColorModel) colorModel).getBlueMask() / 255) / Math.log(2))));
			aWriter.write(" blueMask: 255)");
		} else if (colorModel instanceof IndexColorModel) {
			IndexColorModel indexColorModel = (IndexColorModel) colorModel;
			int map_size = indexColorModel.getMapSize();
			aWriter.write("(MappedPalette withColors: ");
			aWriter.write("((Array new: " + map_size + ") ");
			for (int i = 0; i < map_size; i++) {
				aWriter.write("at: " + (i + 1) + " put: ");
				aWriter.write(StColorValue._StoreStringOf(new Color(indexColorModel.getRGB(i))));
				aWriter.write("; ");
			}
			aWriter.write(" yourself))");
		} else {
			throw new SmalltalkException("The color model [" + colorModel.getClass() + "] is not supported yet.");
		}
	}

	/**
	 * Copy into the destination rectangle destRectangle in the receiver  those
	 * bits in the sourceImage starting at position sourcePt, according the
	 * combination rule rule.
	 * 
	 * @param destRectangle java.awt.Rectangle
	 * @param sourcePt java.awt.Point
	 * @param sourceImage jp.co.sra.smalltalk.StImage
	 * @param combinationRule int
	 * @return jp.co.sra.smalltalk.StImage
	 * @category bit processing
	 */
	public StImage copy_from_in_rule_(Rectangle destRectangle, Point sourcePt, StImage sourceImage, int combinationRule) {
		if (this.bounds().contains(destRectangle.x, destRectangle.y) == false) {
			return null;
		}
		if (this.bounds().contains(destRectangle.x + destRectangle.width - 1, destRectangle.y + destRectangle.height - 1) == false) {
			return null;
		}
		if (sourceImage.bounds().contains(sourcePt.x, sourcePt.y) == false) {
			return null;
		}
		if (sourceImage.bounds().contains(sourcePt.x + destRectangle.width - 1, sourcePt.y + destRectangle.height - 1) == false) {
			return null;
		}

		for (int y = 0; y < destRectangle.height; y++) {
			for (int x = 0; x < destRectangle.width; x++) {
				int srcX = sourcePt.x + x;
				int srcY = sourcePt.y + y;
				int dstX = destRectangle.x + x;
				int dstY = destRectangle.y + y;
				int newPaint = this._combinationedColorFrom_to_in_(sourceImage.getPixel(srcX, srcY), this.getPixel(dstX, dstY), combinationRule);
				this.setPixel(dstX, dstY, newPaint);
			}
		}

		return this;
	}

	/**
	 * Tile the destination rectangle destRectangle in the receiver with copies
	 * of the sourceImage, according to the combination rule rule. The
	 * sourcePt in the sourceImage is aligned with 0&064;0 in the receiver.
	 * 
	 * @param destRectangle java.awt.Rectangle
	 * @param sourcePt java.awt.Point
	 * @param sourceImage jp.co.sra.smalltalk.StImage
	 * @param combinationRule int
	 * @return jp.co.sra.smalltalk.StImage
	 * @category bit processing
	 */
	public StImage tile_from_in_rule_(Rectangle destRectangle, Point sourcePt, StImage sourceImage, int combinationRule) {
		if (this.bounds().contains(destRectangle.x, destRectangle.y) == false) {
			return null;
		}
		if (this.bounds().contains(destRectangle.x + destRectangle.width - 1, destRectangle.y + destRectangle.height - 1) == false) {
			return null;
		}

		int srcImageWidth = sourceImage.width();
		int srcImageHeight = sourceImage.height();
		for (int y = 0; y < destRectangle.height; y++) {
			for (int x = 0; x < destRectangle.width; x++) {
				int srcX = (sourcePt.x + x) % srcImageWidth;
				int srcY = (sourcePt.y + y) % srcImageHeight;
				int dstX = destRectangle.x + x;
				int dstY = destRectangle.y + y;
				int newPaint = this._combinationedColorFrom_to_in_(sourceImage.getPixel(srcX, srcY), this.getPixel(dstX, dstY), combinationRule);
				this.setPixel(dstX, dstY, newPaint);
			}
		}

		return this;
	}

	/**
	 * Answer the combination of the colors regarding the rule.
	 * 
	 * @param srcColor int
	 * @param dstColor int
	 * @param combinationRule int
	 * @return int
	 * @category private
	 */
	protected int _combinationedColorFrom_to_in_(int srcColor, int dstColor, int combinationRule) {
		switch (combinationRule) {
			case WriteZeros:
				return 0xFFFFFFFF;
			case And:
				return 0xFF000000 | (srcColor | dstColor);
			case 2:
				return 0xFF000000 | (srcColor | ~dstColor);
			case Over:
				return 0xFF000000 | srcColor;
			case Erase:
				return 0xFF000000 | (~srcColor | dstColor);
			case 5:
				return 0xFF000000 | dstColor;
			case Reverse:
				return 0xFF000000 | ~(~srcColor ^ ~dstColor);
			case Under:
				return 0xFF000000 | (srcColor & dstColor);
			case 8:
				return 0xFF000000 | ~(srcColor & dstColor);
			case 9:
				return 0xFF000000 | (~srcColor ^ ~dstColor);
			case 10:
				return 0xFF000000 | ~dstColor;
			case 11:
				return 0xFF000000 | ~(~srcColor | dstColor);
			case 12:
				return 0xFF000000 | ~srcColor;
			case ReverseUnder:
				return 0xFF000000 | ~(srcColor | ~dstColor);
			case 14:
				return 0xFF000000 | ~(srcColor | dstColor);
			case WriteOnes:
				return 0xFF000000;
		}

		throw new SmalltalkException("unknown combination rule - " + combinationRule);
	}

	/**
	 * Fill newImage with a copy of the receiver rotated clockwise by quads * 90 degrees.
	 * quads = 0 means unchanged, 1 means clockwise 90 degrees, and so on.
	 * Answer newImage.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @param quads int
	 * @param newImage jp.co.sra.smalltalk.StImage
	 * @category image processing
	 */
	public StImage rotateByQuadrants_to_(int quads, StImage newImage) {
		if (newImage == this) {
			return ((StImage) this.copy()).rotateByQuadrants_to_(quads, newImage);
		}

		int angle = quads & 3; // take angle mod 360 degrees
		if (angle == 0) {
			return newImage.copy_from_in_rule_(this.bounds(), new Point(0, 0), this, StImage.Over);
		}

		Graphics2D graphics = null;
		try {
			graphics = (Graphics2D) newImage.image().getGraphics();
			if (angle == 1) {
				graphics.translate(newImage.width(), 0);
			} else if (angle == 2) {
				graphics.translate(newImage.width(), newImage.height());
			} else {
				// angle = 3
				graphics.translate(0, newImage.height());
			}
			AffineTransform transform = AffineTransform.getRotateInstance(angle * Math.PI / 2);
			graphics.drawImage(this.image(), transform, null);
		} finally {
			if (graphics != null) {
				graphics.dispose();
			}
		}

		return newImage;
	}

	/**
	 * Answer a copied image rotated clockwise by angle in units of 90 degrees.
	 * quads = 0 means unchanged, 1 means clockwise 90 degrees, and so on.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @param quads int
	 * @category image processing
	 */
	public StImage rotatedByQuadrants_(int quads) {
		StImage newImage = (quads % 2 == 0) ? this.copyEmpty() : this.copyEmpty_(new Point(this.extent().y, this.extent().x));
		return this.rotateByQuadrants_to_(quads, newImage);
	}

	/**
	 * Display the image on a window.
	 * 
	 * @category utilities
	 */
	public void _display() {
		StImage.Display_(this);
	}

	/**
	 * Answer the pixel at location (x, y).
	 * 
	 * @param x int
	 * @param y int
	 * @return int
	 * @deprecated use getPixel(int x, int y) instead.
	 * @category private
	 */
	public int atX_y_(int x, int y) {
		return this.getPixel(x, y);
	}

	/**
	 * Set the pixel at location (x, y).
	 * 
	 * @param x int
	 * @param y int
	 * @param pixelValue int
	 * @deprecated use setPixel(int x, int y, int pixel) instead.
	 * @category private
	 */
	public void atX_y_put_(int x, int y, int pixelValue) {
		this.setPixel(x, y, pixelValue);
	}

	/**
	 * Flush the bit array.
	 * 
	 * @category private-bit processing
	 */
	protected void _flushBits() {
		bits = null;
	}

	/**
	 * Answer the bitmap contents of this image.
	 * 
	 * @return byte[]
	 * @category private-bit processing
	 */
	protected byte[] bits() {
		if (bits == null) {
			int width = this.width();
			int height = this.height();
			bits = new byte[rowByteSize * height];
			for (int y = 0; y < height; y++) {
				for (int x = 0; x < width; x++) {
					this._setPixelIntoBits(x, y, bits);
				}
			}
		}

		return bits;
	}

	/**
	 * Answer resized image with (x, y).
	 * 
	 * @param scaleX double
	 * @param scaleY double
	 * @return java.awt.Point
	 * @category private-image processing
	 */
	public Point scaledExtent_(double scaleX, double scaleY) {
		Double x = new Double((this.width() * 1.0) / scaleX);
		Double y = new Double((this.height() * 1.0) / scaleY);
		return new Point(x.intValue(), y.intValue());
	}

	/**
	 * Restores this object from a object stream (i.e., deserializes it).
	 * 
	 * @param stream java.io.ObjectInputStream
	 * @exception IOException
	 * @exception ClassNotFoundException
	 * @category private-serializing
	 */
	private void readObject(ObjectInputStream stream) throws IOException, ClassNotFoundException {
		stream.defaultReadObject();

		// read width, height and pixels.
		int width = stream.readInt();
		int height = stream.readInt();
		int[] pixels = (int[]) stream.readObject();

		// read ColorModel.
		String colorModelName = stream.readUTF();
		ColorModel colorModel = null;
		if (colorModelName.equals("DirectColorModel")) {
			int pixelSize = stream.readInt();
			int redMask = stream.readInt();
			int greenMask = stream.readInt();
			int blueMask = stream.readInt();
			int alphaMask = stream.readInt();
			colorModel = new DirectColorModel(pixelSize, redMask, greenMask, blueMask, alphaMask);
		} else if (colorModelName.equals("IndexColorModel")) {
			int pixelSize = stream.readInt();
			int mapSize = stream.readInt();
			byte[] r = new byte[mapSize];
			byte[] g = new byte[mapSize];
			byte[] b = new byte[mapSize];
			byte[] a = new byte[mapSize];
			for (int i = 0; i < mapSize; i++) {
				r[i] = (byte) stream.read();
			}
			for (int i = 0; i < mapSize; i++) {
				g[i] = (byte) stream.read();
			}
			for (int i = 0; i < mapSize; i++) {
				b[i] = (byte) stream.read();
			}
			for (int i = 0; i < mapSize; i++) {
				a[i] = (byte) stream.read();
			}
			colorModel = new IndexColorModel(pixelSize, mapSize, r, g, b, a);
		} else {
			throw new IOException("The color model [" + colorModel.getClass() + "] is not supported yet.");
		}

		// create image.
		image = new BufferedImage(colorModel, colorModel.createCompatibleWritableRaster(width, height), false, null);
		rowByteSize = (width * this.bitsPerPixel() + 31) >> 5 << 2;
		this.setPixels(pixels);
	}

	/**
	 * Writes this object out to a stream (i.e., serializes it).
	 * 
	 * @param stream java.io.ObjectOutputStream
	 * @exception IOException
	 * @exception ClassNotFoundException
	 * @category private-serializing
	 */
	private void writeObject(ObjectOutputStream stream) throws IOException, ClassNotFoundException {
		stream.defaultWriteObject();

		// write width, height and pixels.
		stream.writeInt(this.width());
		stream.writeInt(this.height());
		stream.writeObject(this.getPixels());

		// write ColorModel.
		ColorModel colorModel = this.image().getColorModel();
		if (colorModel instanceof DirectColorModel) {
			DirectColorModel directColorModel = (DirectColorModel) colorModel;
			stream.writeUTF("DirectColorModel");
			stream.writeInt(directColorModel.getPixelSize());
			stream.writeInt(directColorModel.getRedMask());
			stream.writeInt(directColorModel.getGreenMask());
			stream.writeInt(directColorModel.getBlueMask());
			stream.writeInt(directColorModel.getAlphaMask());
		} else if (colorModel instanceof IndexColorModel) {
			IndexColorModel indexColorModel = (IndexColorModel) colorModel;
			stream.writeUTF("IndexColorModel");
			stream.writeInt(indexColorModel.getPixelSize());
			stream.writeInt(indexColorModel.getMapSize());
			byte[] bytes = new byte[indexColorModel.getMapSize()];
			indexColorModel.getReds(bytes);
			stream.write(bytes, 0, bytes.length);
			indexColorModel.getGreens(bytes);
			stream.write(bytes, 0, bytes.length);
			indexColorModel.getBlues(bytes);
			stream.write(bytes, 0, bytes.length);
			indexColorModel.getAlphas(bytes);
			stream.write(bytes, 0, bytes.length);
		} else {
			throw new IOException("The color model [" + colorModel.getClass() + "] is not supported yet.");
		}
	}
}
