package jp.co.sra.smalltalk;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.font.FontRenderContext;
import java.awt.font.LineBreakMeasurer;
import java.awt.font.TextAttribute;
import java.awt.font.TextLayout;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.text.AttributedString;
import java.text.BreakIterator;
import java.util.ArrayList;

/**
 * StComposedText class
 * 
 *  @author    nisinaka
 *  @created   2002/11/20 (by nisinaka)
 *  @updated   2005/09/07 (by nisinaka)
 *  @version   8.9
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: StComposedText.java,v 8.10 2008/02/20 06:33:18 nisinaka Exp $
 */
public class StComposedText extends StObject implements StDisplayable {

	protected String string;
	protected Font font;
	protected int compositionWidth;
	protected int compositionHeight;
	protected boolean fitWidth;
	protected TextLayout[] layouts;
	protected int alignment;

	public static final int DefaultCompositionWidth = Integer.MAX_VALUE;
	public static final int LeftFlush = 0;
	public static final int RightFlush = 1;
	public static final int Centered = 2;

	/**
	 * Answer the default font.
	 *
	 * @return java.awt.Font
	 * @category Defaults
	 */
	public static Font DefaultFont() {
		String name = System.getProperty("default.font.name", "Dialog");

		int style = Font.PLAIN;
		if (System.getProperty("default.font.style") != null) {
			try {
				style = Integer.parseInt(System.getProperty("default.font.style"));
			} catch (NumberFormatException e) {
			}
		}

		int size = 12;
		if (System.getProperty("default.font.size") != null) {
			try {
				size = Integer.parseInt(System.getProperty("default.font.size"));
			} catch (NumberFormatException e) {
			}
		}

		return new Font(name, style, size);
	}

	/**
	 * Answer the default FontRenderContext.
	 *
	 * @return java.awt.font.FontRenderContext
	 * @category Defaults
	 */
	public static FontRenderContext DefaultFontRenderContext() {
		return new FontRenderContext(null, false, false);
	}

	/**
	 * Create a new instance of StComposedText with a string and the default font.
	 *
	 * @param aString java.lang.String
	 * @category Instance creation
	 */
	public StComposedText(String aString) {
		this(aString, null);
	}

	/**
	 * Create a new instance of StComposedText and initialize it.
	 *
	 * @param aString java.lang.String
	 * @param width int
	 * @category Instance creation
	 */
	public StComposedText(String aString, int width) {
		this();
		this.compositionWidth_string_font_fitWidth_(width, aString, null, false);
	}

	/**
	 * Create a new instance of StComposedText and initialize it.
	 *
	 * @param aString java.lang.String
	 * @param aFont java.awt.Font
	 * @category Instance creation
	 */
	public StComposedText(String aString, Font aFont) {
		this();
		this.compositionWidth_string_font_fitWidth_(DefaultCompositionWidth, aString, aFont, true);
	}

	/**
	 * Create a new instance of StComposedText and initialize it.
	 *
	 * @param aString java.lang.String
	 * @param aFont java.awt.Font
	 * @param width int
	 * @category Instance creation
	 */
	public StComposedText(String aString, Font aFont, int width) {
		this();
		this.compositionWidth_string_font_fitWidth_(width, aString, aFont, false);
	}

	/**
	 * Create a new instance of StComposedText and initialize it.
	 * 
	 * @category Instance creation
	 */
	protected StComposedText() {
		string = null;
		font = null;
		compositionWidth = -1;
		compositionHeight = -1;
		fitWidth = true;
		layouts = null;
	}

	/**
	 * Answer the current string.
	 *
	 * @return java.lang.String
	 * @category accessing
	 */
	public String string() {
		return string;
	}

	/**
	 * Set the new string.
	 *
	 * @param newString java.lang.String
	 * @category accessing
	 */
	public void string_(String newString) {
		string = newString;
		this.composeAll();
	}

	/**
	 * Answer the current font.
	 *
	 * @return java.awt.Font
	 * @category accessing
	 */
	public Font font() {
		if (font == null) {
			return DefaultFont();
		}
		return font;
	}

	/**
	 * Set the new font.
	 *
	 * @param newFont java.awt.Font
	 * @category accessing
	 */
	public void font_(Font newFont) {
		Font current = this.font();
		font = newFont;
		if (current.equals(this.font()) == false) {
			this.composeAll();
		}
	}

	/**
	 * Answer the number of lines of text in the receiver.
	 *
	 * @return int
	 * @category accessing
	 */
	public int numberOfLines() {
		return this.layouts().length;
	}

	/**
	 * Answer the layouts.  Initialize it if necessary.
	 *
	 * @return java.awt.font.TextLayout[]
	 * @category accessing
	 */
	protected TextLayout[] layouts() {
		if (layouts == null) {
			this.composeAll();
		}
		return layouts;
	}

	/**
	 * Compose the text from the current string and the font.
	 *
	 * @return int
	 * @category accessing
	 */
	protected int composeAll() {
		if (this.string() == null || this.string().length() == 0) {
			layouts = new TextLayout[0];
			this.setHeight_(0);
			return 0;
		}

		float formatWidth = this.width();
		ArrayList layoutCollection = new ArrayList();
		BufferedReader reader = null;
		try {
			reader = new BufferedReader(new StringReader(this.string()));
			String line;
			while ((line = reader.readLine()) != null) {
				int length = line.length();
				if (length == 0) {
					// AttributedString does not allow to add an attribute to a 0-length string.
					// Don't worry about the width.  TextLayout#getVisibleAdvance() will return 0.0.
					line = " ";
					length = 1;
				}
				AttributedString anAttributedString = new AttributedString(line);
				anAttributedString.addAttribute(TextAttribute.FONT, this.font());
				LineBreakMeasurer aLineBreakMeasurer = new LineBreakMeasurer(anAttributedString.getIterator(), BreakIterator.getWordInstance(), DefaultFontRenderContext());
				aLineBreakMeasurer.setPosition(0);
				while (aLineBreakMeasurer.getPosition() < length) {
					layoutCollection.add(aLineBreakMeasurer.nextLayout(formatWidth));
				}
			}
		} catch (IOException e) {
			this.setHeight_(0);
			return 0;
		} finally {
			if (reader != null) {
				try {
					reader.close();
				} catch (IOException e) {
					this.setHeight_(0);
					return 0;
				}
			}
		}

		layouts = (TextLayout[]) layoutCollection.toArray(new TextLayout[layoutCollection.size()]);
		float maximumRightX = layouts[0].getAdvance();
		float height = layouts[0].getAscent() + layouts[0].getDescent();
		for (int i = 1; i < layouts.length; i++) {
			maximumRightX = Math.max(maximumRightX, layouts[i].getVisibleAdvance());
			height += layouts[i].getLeading() + layouts[i].getAscent() + layouts[i].getDescent();
		}
		this.setHeight_((int) height);

		return (int) maximumRightX;
	}

	/**
	 * Ansewr the width of the receiver's bounding box.
	 *
	 * @return int
	 * @category display box accessing
	 */
	public int width() {
		return compositionWidth;
	}

	/**
	 * Ansewr the height of the receiver's bounding box.
	 *
	 * @return int
	 * @category display box accessing
	 */
	public int height() {
		if (compositionHeight < 0) {
			this.updateCompositionHeight();
		}
		return compositionHeight;
	}

	/**
	 * Answer the bounding box of the receiver.
	 *
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.smalltalk.StDisplayable#bounds()
	 * @category display box accessing
	 */
	public Rectangle bounds() {
		return new Rectangle(0, 0, this.width(), this.height());
	}

	/**
	 * Answer the extent around the receiver.
	 *
	 * @return java.awt.Dimension
	 * @category display box accessing
	 */
	public Dimension extent() {
		return new Dimension(this.width(), this.height());
	}

	/**
	 * Set the width for composition.
	 * Make sure the composition width in not negative.
	 *
	 * @param newCompositionWidth int
	 * @category display box accessing
	 */
	protected void setCompositionWidth_(int newCompositionWidth) {
		compositionWidth = Math.max(newCompositionWidth, 0);
	}

	/**
	 * Set the height of the receiver's bounding box.
	 *
	 * @param newHeight int
	 * @category display box accessing
	 */
	protected void setHeight_(int newHeight) {
		compositionHeight = newHeight;
	}

	/**
	 * Set the alignment for the style with which the receiver displays its text
	 * so that the characters in each of text begin on an even border.
	 * 
	 * @category justification and spacing
	 */
	public void leftFlush() {
		this.setAlignment_(LeftFlush);
	}

	/**
	 * Set the alignment for the style with which the receiver displays its text
	 * so that the characters in each of text end on an even border but the beginning of each line does not.
	 * 
	 * @category justification and spacing
	 */
	public void rightFlush() {
		this.setAlignment_(RightFlush);
	}

	/**
	 * Set the alignment for the style with which the receiver displays its text
	 * so that text is centered in the composition width.
	 * 
	 * @category justification and spacing
	 */
	public void centered() {
		this.setAlignment_(Centered);
	}

	/**
	 * Set the receiver's alignment.
	 * 
	 * @param anAlignment int
	 * @category justification and spacing
	 */
	protected void setAlignment_(int anAlignment) {
		alignment = anAlignment;
	}

	/**
	 * Display the receiver on aGraphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		this.displayOn_at_(aGraphics, new Point(0, 0));
	}

	/**
	 * Display the receiver on the graphics at the specified point.
	 *
	 * @param aGraphics java.awt.Graphics
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_at_(java.awt.Graphics, java.awt.Point)
	 * @category displaying
	 */
	public void displayOn_at_(Graphics aGraphics, Point aPoint) {
		Graphics2D aGraphics2D = (Graphics2D) aGraphics;
		int x = aPoint.x;
		int y = aPoint.y;
		TextLayout[] layouts = this.layouts();
		for (int i = 0; i < layouts.length; i++) {
			y += layouts[i].getAscent();

			float offset = 0;
			if (alignment != LeftFlush) {
				float margin = this.width() - layouts[i].getVisibleAdvance();
				if (alignment == RightFlush) {
					offset = margin;
				} else if (alignment == Centered) {
					offset = margin / 2;
				}
			}

			layouts[i].draw(aGraphics2D, x + offset, y);

			y += layouts[i].getDescent() + layouts[i].getLeading();
		}
	}

	/**
	 * Answer true if the receiver is composed, otherwise false.
	 *
	 * @return boolean
	 * @category testing
	 */
	public boolean isComposed() {
		return (layouts != null);
	}

	/**
	 * Convert to a StImage.
	 *
	 * @return java.awt.Image
	 * @see jp.co.sra.smalltalk.StDisplayable#asImage()
	 * @category converting
	 */
	public StImage asImage() {
		return this.asImage(Color.black, null);
	}

	/**
	 * Convert to StImage drawn with the specified colors.
	 * 
	 * @param foregroundColor java.awt.Color
	 * @param backgroundColor java.awt.Color
	 * @return java.awt.Image
	 * @category converting
	 */
	public StImage asImage(Color foregroundColor, Color backgroundColor) {
		int width = this.width();
		int height = this.height();
		int[] pixels = new int[width * height];
		for (int i = 0; i < pixels.length; i++) {
			pixels[i] = 0;
		}
		StImage anImage = new StImage(width, height, pixels);
		Graphics2D aGraphics = null;
		try {
			aGraphics = (Graphics2D) anImage.image().getGraphics();
			if (backgroundColor != null) {
				aGraphics.setColor(backgroundColor);
				aGraphics.fillRect(0, 0, width, height);
			}
			if (foregroundColor != null) {
				aGraphics.setColor(foregroundColor);
			}
			this.displayOn_(aGraphics);
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
				aGraphics = null;
			}
		}
		return anImage;
	}

	/**
	 * Convert to an image.
	 *
	 * @return java.awt.Image
	 * @category converting
	 */
	public Image toImage() {
		return this.asImage().image();
	}

	/**
	 * Initialize the receiver.
	 *
	 * @param newWidth int
	 * @param newString java.lang.String
	 * @param newFont java.awt.Font
	 * @param aBoolean boolean
	 * @category private
	 */
	protected void compositionWidth_string_font_fitWidth_(int newWidth, String newString, Font newFont, boolean aBoolean) {
		fitWidth = aBoolean;
		string = newString;
		font = newFont;
		this.setCompositionWidth_(newWidth);

		if (aBoolean) {
			compositionWidth = this.composeAll();
		} else {
			this.composeAll();
		}
	}

	/**
	 * Update the height of the receiver's bounding box.
	 * 
	 * @category private
	 */
	protected void updateCompositionHeight() {

	}

}
