package jp.co.sra.qt4jun;

import java.awt.Rectangle;
import java.awt.Dimension;
import java.awt.Image;
import java.awt.image.BufferedImage;
import java.awt.image.ColorModel;
import java.awt.image.WritableRaster;
import java.io.File;
import quicktime.Errors;
import quicktime.QTException;
import quicktime.QTSession;
import quicktime.io.IOConstants;
import quicktime.io.OpenMovieFile;
import quicktime.io.QTFile;
import quicktime.qd.Pict;
import quicktime.qd.PixMap;
import quicktime.qd.QDConstants;
import quicktime.qd.QDGraphics;
import quicktime.qd.QDRect;
import quicktime.std.StdQTConstants;
import quicktime.std.StdQTException;
import quicktime.std.movies.Movie;
import quicktime.std.movies.TimeInfo;
import quicktime.util.EndianOrder;
import quicktime.util.RawEncodedImage;

/**
 * JunQTJMovie class
 * 
 *  @author    Hoshi Takanori
 *  @created   2005/01/30 (by Hoshi Takanori)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunQTJMovie.java,v 8.11 2008/02/20 06:33:17 nisinaka Exp $
 */
public class JunQTJMovie implements JunQTMovie {
	protected Movie movie;

	static {
		try {
			QTSession.open();
		} catch (QTException e) {
			throw new Error(e.getMessage());
		}

		String version = Integer.toHexString(QTSession.getQTVersion());
		System.out.println("JunQTJMovie: QuickTime version " + version);
	}

	/**
	 * answer true if getMovieFile() works.
	 *
	 * @return boolean
	 * @category Instance creation
	 */
	static boolean canGetMovieFile() {
		return true;
	}

	/**
	 * open the standard file preview dialog and get the movie filename.
	 *
	 * @return java.lang.String
	 * @category Instance creation
	 */
	static String getMovieFile() {
		QTFile file;
		try {
			file = QTFile.standardGetFilePreview(QTFile.kStandardQTFileTypes);
		} catch (QTException e) {
			if (e.errorCode() == Errors.userCanceledErr) {
				return null;
			} else {
				throw new Error(e.getMessage());
			}
		}
		if (file != null) {
			return file.getPath();
		} else {
			return null;
		}
	}

	/**
	 * open the movie.
	 *
	 * @param filename java.lang.String
	 * @return jp.co.sra.qt4jun.JunQTMovie
	 * @category Instance creation
	 */
	static JunQTMovie openMovie(String filename) {
		Movie movie;
		try {
			QTFile file = new QTFile(filename);
			OpenMovieFile movieFile = OpenMovieFile.asRead(file);
			movie = Movie.fromFile(movieFile);
		} catch (QTException e) {
			throw new Error(e.getMessage());
		}
		if (movie != null) {
			return new JunQTJMovie(movie);
		} else {
			return null;
		}
	}

	/**
	 * constructor.
	 *
	 * @param movie quicktime.std.movies.Movie
	 * @category Instance creation
	 */
	public JunQTJMovie(Movie movie) {
		this.movie = movie;
	}

	/**
	 * finalize.
	 *
	 * @category initialize-release
	 */
	protected void finalize() {
		this.release();
	}

	/**
	 * release the movie.
	 *
	 * @category initialize-release
	 */
	public void release() {
		movie = null;
	}

	/**
	 * set the movie to active/deactive.
	 *
	 * @param active boolean
	 * @category accessing
	 */
	public void setActive(boolean active) {
		try {
			movie.setActive(active);
		} catch (StdQTException e) {
			throw new Error(e.getMessage());
		}
	}

	/**
	 * get the movie box.
	 *
	 * @return java.awt.Rectangle
	 * @category accessing
	 */
	public Rectangle getBox() {
		QDRect r;
		try {
			r = movie.getBox();
		} catch (StdQTException e) {
			throw new Error(e.getMessage());
		}
		return new Rectangle(r.getX(), r.getY(), r.getWidth(), r.getHeight());
	}

	/**
	 * set the movie box.
	 *
	 * @param box java.awt.Rectangle
	 * @category accessing
	 */
	public void setBox(Rectangle box) {
		/*
		try {
			movie.setBox(new QDRect(box.x, box.y, box.width, box.height));
		} catch (StdQTException e) {
			throw new Error(e.getMessage());
		}
		*/
	}

	/**
	 * get the duration.
	 *
	 * @return int
	 * @category accessing
	 */
	public int getDuration() {
		try {
			return movie.getDuration();
		} catch (StdQTException e) {
			throw new Error(e.getMessage());
		}
	}

	/**
	 * get the next interesting time.
	 *
	 * @param time int
	 * @param rate int
	 * @return int
	 * @category accessing
	 */
	public int getNextInterestingTime(int time, int rate) {
		int[] mediaTypes = { StdQTConstants.visualMediaCharacteristic };
		try {
			TimeInfo info = movie.getNextInterestingTime(
					StdQTConstants.nextTimeStep, mediaTypes, time, rate);
			return info.time;
		} catch (StdQTException e) {
			throw new Error(e.getMessage());
		}
	}

	/**
	 * select portions of the movie.
	 *
	 * @param start int
	 * @param duration int
	 * @category accessing
	 */
	public void setSelection(int start, int duration) {
		try {
			movie.setSelection(start, duration);
		} catch (StdQTException e) {
			throw new Error(e.getMessage());
		}
	}

	/**
	 * get the time scale.
	 *
	 * @return int
	 * @category accessing
	 */
	public int getTimeScale() {
		try {
			return movie.getTimeScale();
		} catch (StdQTException e) {
			throw new Error(e.getMessage());
		}
	}

	/**
	 * get the time value.
	 *
	 * @return int
	 * @category accessing
	 */
	public int getTimeValue() {
		try {
			return movie.getTime();
		} catch (StdQTException e) {
			throw new Error(e.getMessage());
		}
	}

	/**
	 * set the time value.
	 *
	 * @param time int
	 * @category accessing
	 */
	public void setTimeValue(int time) {
		try {
			movie.setTimeValue(time);
		} catch (StdQTException e) {
			throw new Error(e.getMessage());
		}
	}

	/**
	 * get the volume.
	 *
	 * @return double
	 * @category accessing
	 */
	public double getVolume() {
		try {
			return movie.getVolume();
		} catch (StdQTException e) {
			throw new Error(e.getMessage());
		}
	}

	/**
	 * set the volume.
	 *
	 * @param volume double
	 * @category accessing
	 */
	public void setVolume(double volume) {
		try {
			movie.setVolume((float) volume);
		} catch (StdQTException e) {
			throw new Error(e.getMessage());
		}
	}

	/**
	 * get the image at the given time value.
	 *
	 * @param time int
	 * @param extent java.awt.Dimension
	 * @return java.awt.Image
	 * @category converting
	 */
	public Image getImageAt(int time, Dimension extent) {
		try {
			Pict pict = movie.getPict(time);
			QDRect r = new QDRect(0, 0, extent.width, extent.height);
			QDGraphics g;
			int pixelFormat = QDGraphics.kDefaultPixelFormat;
			if (EndianOrder.isNativeLittleEndian()) {
				pixelFormat = QDConstants.k32BGRAPixelFormat;
			}
			g = new QDGraphics(pixelFormat, r);
			pict.draw(g, r);
			PixMap pm = g.getPixMap();
			RawEncodedImage raw = pm.getPixelData();
			int rowSize = pm.getRowBytes() / 4;
			int[] pixels = new int[rowSize * extent.height];
			raw.copyToArray(0, pixels, 0, pixels.length);
			ColorModel cm = ColorModel.getRGBdefault();
			WritableRaster raster = cm.createCompatibleWritableRaster(extent.width, extent.height);
			BufferedImage image = new BufferedImage(cm, raster, false, null);
			image.setRGB(0, 0, extent.width, extent.height, pixels, 0, rowSize);
			return image;
		} catch (QTException e) {
			throw new Error(e.getMessage());
		}
	}

	/**
	 * get the sound data from the movie.
	 *
	 * @param start int
	 * @param duration int
	 * @return byte[]
	 * @category converting
	 */
	public byte[] getSoundData(int start, int duration) {
		return null;	// this._interface().convertMovieToSoundData(movie, start, duration);
	}

	/**
	 * copy the selected portions of the movie.
	 *
	 * @return jp.co.sra.qt4jun.JunQTMovie
	 * @category copying
	 */
	public JunQTMovie copySelection() {
		Movie newMovie;
		try {
			newMovie = movie.copySelection();
		} catch (QTException e) {
			throw new Error(e.getMessage());
		}
		if (newMovie != null) {
			return new JunQTJMovie(newMovie);
		} else {
			return null;
		}
	}

	/**
	 * flatten the movie to file.
	 *
	 * @param filename java.io.File
	 * @category writing
	 */
	public void flattenToFile(File filename) {
		try {
			QTFile file = new QTFile(filename);
			movie.flatten(StdQTConstants.flattenAddMovieToDataFork, file,
					StdQTConstants.kMoviePlayer, IOConstants.smSystemScript,
					StdQTConstants.createMovieFileDeleteCurFile |
							StdQTConstants.createMovieFileDontCreateResFile,
					StdQTConstants.movieInDataForkResID, null);
		} catch (QTException e) {
			throw new Error(e.getMessage());
		}
	}

	/**
	 * go to beginning of the movie.
	 *
	 * @category playing
	 */
	public void goToBeginning() {
		try {
			movie.goToBeginning();
		} catch (StdQTException e) {
			throw new Error(e.getMessage());
		}
	}

	/**
	 * go to end of the movie.
	 *
	 * @category playing
	 */
	public void goToEnd() {
		try {
			movie.goToEnd();
		} catch (StdQTException e) {
			throw new Error(e.getMessage());
		}
	}

	/**
	 * is the movie done or not?
	 *
	 * @return boolean
	 * @category playing
	 */
	public boolean isDone() {
		try {
			return movie.isDone();
		} catch (StdQTException e) {
			throw new Error(e.getMessage());
		}
	}

	/**
	 * start the movie.
	 *
	 * @category playing
	 */
	public void start() {
		try {
			movie.start();
		} catch (StdQTException e) {
			throw new Error(e.getMessage());
		}
	}

	/**
	 * stop the movie.
	 *
	 * @category playing
	 */
	public void stop() {
		try {
			movie.stop();
		} catch (StdQTException e) {
			throw new Error(e.getMessage());
		}
	}

	/**
	 * do movie task.
	 *
	 * @category playing
	 */
	public void task() {
		/*
		try {
			movie.task(0);
		} catch (QTException e) {
			throw new Error(e.getMessage());
		}
		*/
	}

	/**
	 * update the movie.
	 *
	 * @category playing
	 */
	public void update() {
		try {
			movie.update();
		} catch (StdQTException e) {
			throw new Error(e.getMessage());
		}
	}

	/**
	 * set the movie port.
	 *
	 * @param port jp.co.sra.qt4jun.JunQTPort
	 * @category private
	 */
	public void setPort(JunQTPort port) {
		((JunQTJPort) port).setMovie(movie);
	}
}
