package jp.co.sra.jun.topology.elements;

/*
 * @(#)JunEdge.java        1.0 98/8/6
 */
import java.io.*;
import jp.co.sra.jun.geometry.abstracts.JunCurve;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.goodies.lisp.*;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext;
import jp.co.sra.jun.topology.abstracts.*;
import jp.co.sra.smalltalk.*;

/**
 * JunEdge class
 * 
 *  @author    ASTI Shanghai
 *  @created   UNKNOWN
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunEdge.java,v 8.11 2008/02/20 06:33:01 nisinaka Exp $
 */
public class JunEdge extends JunTopologicalElement {
	/** The start vertex of the edge. */
	protected JunVertex startVertex = null;

	/** The end vertex of the edge. */
	protected JunVertex endVertex = null;

	/** The left loop of the edge. */
	protected JunLoop leftLoop = null;

	/** The right loop of the edge. */
	protected JunLoop rightLoop = null;

	/** The start right edge of the edge. */
	protected JunEdge startRightEdge = null;

	/** The end left edge of the edge. */
	protected JunEdge endLeftEdge = null;

	/** The geometry which corresponds to the edge. */
	protected JunCurve curve = null;

	/**
	 * Create a new <code>JunEdge</code> with two vertexes.
	 * 
	 * @param startVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @param endVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @return jp.co.sra.jun.topology.elements.JunEdge
	 * @category Instance creation
	 */
	public static final JunEdge StartVertex_endVertex_(JunVertex startVertex, JunVertex endVertex) {
		JunEdge anEdge = new JunEdge();
		anEdge.startVertex_(startVertex);
		anEdge.endVertex_(endVertex);

		return anEdge;
	}

	/**
	 * Convert the corresponding geometry as a JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.topology.abstracts.JunTopologicalElement#asJunOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		return ((Jun3dLine) this.curve()).asJunOpenGL3dObject();
	}

	/**
	 * Convert the receiver as a proxy in aBody.
	 * 
	 * @param aBody jp.co.sra.jun.topology.elements.JunBody
	 * @return jp.co.sra.jun.topology.abstracts.JunTopologicalElementProxy
	 * @category converting
	 */
	public JunTopologicalElementProxy asProxyIn_(JunBody aBody) {
		return this.asProxyIn_advise_(aBody, null);
	}

	/**
	 * Convert the receiver as a proxy in aBody.
	 * 
	 * @param aBody jp.co.sra.jun.topology.elements.JunBody
	 * @param aTopologicalElementProxy jp.co.sra.jun.topology.abstracts.JunTopologicalElementProxy
	 * @return jp.co.sra.jun.topology.abstracts.JunTopologicalElementProxy
	 * @see jp.co.sra.jun.topology.abstracts.JunTopologicalElementOrProxy#asProxyIn_advise_(jp.co.sra.jun.topology.elements.JunBody, jp.co.sra.jun.topology.abstracts.JunTopologicalElementProxy)
	 * @category converting
	 */
	public JunTopologicalElementProxy asProxyIn_advise_(JunBody aBody, JunTopologicalElementProxy aTopologicalElementProxy) {
		JunTopologicalElementProxy proxy = aBody.proxyForEdge_(this);
		if (proxy != null) {
			return proxy;
		}
		return aBody.addEdge_as_(this, (JunEdgeProxy) aTopologicalElementProxy);
	}

	/**
	 * Convert the corresponding geometry as a wireframed JunOpenGL3dObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.topology.abstracts.JunTopologicalElement#asWireframedOpenGL3dObject()
	 * @category converting
	 */
	public JunOpenGL3dObject asWireframedOpenGL3dObject() {
		return ((Jun3dLine) this.curve()).asJunOpenGL3dObject();
	}

	/**
	 * Answer the geometry which correspnds to the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.abstracts.JunCurve
	 * @category accessing
	 */
	public JunCurve basicCurve() {
		return curve;
	}

	/**
	 * Answer the bounding box of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dBoundingBox
	 * @category bounds accessing
	 */
	public Jun3dBoundingBox boundingBox() {
		return ((Jun3dLine) this.curve()).boundingBox();
	}

	/**
	 * Answer the commom loop with the edge.
	 * 
	 * @param anEdge jp.co.sra.jun.topology.elements.JunEdge
	 * @return jp.co.sra.jun.topology.elements.JunLoop
	 * @category private
	 */
	public JunLoop commonLoopWith_(JunEdge anEdge) {
		if (leftLoop == anEdge.leftLoop) {
			return leftLoop;
		}
		if (leftLoop == anEdge.rightLoop) {
			return leftLoop;
		}
		if (rightLoop == anEdge.leftLoop) {
			return rightLoop;
		}
		if (rightLoop == anEdge.rightLoop) {
			return rightLoop;
		}
		return null;
	}

	/**
	 * Answer the geometry which corresponds to the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.abstracts.JunCurve
	 * @category accessing
	 */
	public JunCurve curve() {
		if (this.basicCurve() == null) {
			return this.defaultCurve();
		} else {
			return this.basicCurve();
		}
	}

	/**
	 * Set the curve as a corresponding geometry.
	 * 
	 * @param aCurve jp.co.sra.jun.geometry.abstracts.JunCurve
	 * @category accessing
	 */
	public void curve_(JunCurve aCurve) {
		this.setCurve_(aCurve);
	}

	/**
	 * Answer the curve created from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @return jp.co.sra.jun.geometry.abstracts.JunCurve
	 * @category lisp support
	 */
	public JunCurve curveFromList_(JunLispList aList) {
		throw SmalltalkException.ShouldNotImplement();
	}

	/**
	 * Answer the default curve of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.abstracts.JunCurve
	 * @category defaults
	 */
	public JunCurve defaultCurve() {
		return new Jun3dLine(startVertex.point(), endVertex.point());
	}

	/**
	 * Detect the loop which meets the condition specified with the block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param errorBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.jun.topology.elements.JunLoop
	 * @category enumerating
	 */
	public JunLoop detectLoop_ifNone_(StBlockClosure aBlock, StBlockClosure errorBlock) {
		if (((Boolean) aBlock.value_(leftLoop)).booleanValue() == true) {
			return leftLoop;
		}
		if (((Boolean) aBlock.value_(rightLoop)).booleanValue() == true) {
			return rightLoop;
		}
		return (JunLoop) errorBlock.value();
	}

	/**
	 * Evaluate the block with an opposite vertex, a focused edge, and a
	 * focused vertex.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object directionVertex_vertexEdgeVertexDo_(JunVertex aVertex, StBlockClosure aBlock) {
		JunVertex focusedVertex;
		JunEdge focusedEdge;
		JunVertex oppositeVertex;
		focusedEdge = this;
		focusedVertex = aVertex;
		oppositeVertex = focusedEdge.oppositeVertexOfVertex_(focusedVertex);
		while (focusedEdge != null) {
			Object result = aBlock.value_value_value_(oppositeVertex, focusedEdge, focusedVertex);
			if (result != null) {
				return result;
			}
			focusedEdge = focusedEdge.nextEdgeWithVertex_(focusedVertex);
			oppositeVertex = focusedVertex;
			focusedVertex = focusedEdge.oppositeVertexOfVertex_(focusedVertex);
			if ((this == focusedEdge) && (aVertex == focusedVertex)) {
				return null;
			}
		}

		return null;
	}

	/**
	 * Answer the receiver's direction vertex on the loop.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @return jp.co.sra.jun.topology.elements.JunVertex
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category accessing
	 */
	public JunVertex directionVertexOnLoop_(JunLoop aLoop) {
		if (aLoop == this.leftLoop()) {
			return this.endVertex();
		}
		if (aLoop == this.rightLoop()) {
			return this.startVertex();
		}

		throw new SmalltalkException("invalid argument");
	}

	/**
	 * Answer the end left edge of the receiver.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunEdge
	 * @category accessing
	 */
	public JunEdge endLeftEdge() {
		return endLeftEdge;
	}

	/**
	 * Set the end left edge of the receiver.
	 * 
	 * @param anEdge jp.co.sra.jun.topology.elements.JunEdge
	 * @category accessing
	 */
	public void endLeftEdge_(JunEdge anEdge) {
		endLeftEdge = anEdge;
	}

	/**
	 * Answer the end vertex of the receiver.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunVertex
	 * @category accessing
	 */
	public JunVertex endVertex() {
		return endVertex;
	}

	/**
	 * Set the end vertex of the receiver.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @category accessing
	 */
	public void endVertex_(JunVertex aVertex) {
		endVertex = aVertex;
	}

	/**
	 * Forget every instance variables.
	 * 
	 * @category private
	 */
	public void forget() {
		startVertex = null;
		endVertex = null;
		leftLoop = null;
		rightLoop = null;
		endLeftEdge = null;
		startRightEdge = null;
		curve = null;
	}

	/**
	 * Initialize the receiver with the lisp list.
	 * 
	 * @param aLispList jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @param loopArray jp.co.sra.jun.topology.elements.JunLoop[]
	 * @param edgeArray jp.co.sra.jun.topology.elements.JunEdge[]
	 * @param vertexArray jp.co.sra.jun.topology.elements.JunVertex[]
	 * @category lisp support
	 */
	public void fromLispList_forLoops_edges_vertexes_(JunLispCons aLispList, final JunLoop[] loopArray, final JunEdge[] edgeArray, final JunVertex[] vertexArray) {
		final JunEdge self = this;
		((JunLispCons) aLispList.tail()).do_(new StBlockClosure() {
			public Object value_(Object anObject) {
				JunLispCons tuple = (JunLispCons) anObject;
				int index = (tuple.tail() == JunLispList.NullList()) ? (-1) : ((Integer) tuple.tail()).intValue() - 1;
				if (tuple.head() == $("startVertex")) {
					startVertex = vertexArray[index];
					return null;
				}
				if (tuple.head() == $("endVertex")) {
					endVertex = vertexArray[index];
					return null;
				}
				if (tuple.head() == $("leftLoop")) {
					leftLoop = loopArray[index];
					return null;
				}
				if (tuple.head() == $("rightLoop")) {
					rightLoop = loopArray[index];
					return null;
				}
				if (tuple.head() == $("endLeftEdge")) {
					endLeftEdge = edgeArray[index];
					return null;
				}
				if (tuple.head() == $("startRightEdge")) {
					startRightEdge = edgeArray[index];
					return null;
				}
				if (tuple.head() == $("curve")) {
					if (tuple.tail() == JunLispList.NullList()) {
						curve = null;
					} else {
						curve = (JunCurve) self.curveFromList_((JunLispList) tuple.tail());
					}
				}
				return null;
			}
		});
	}

	/**
	 * Answer true if the receiver includes the loop.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @return boolean
	 * @category testing
	 */
	public boolean includesLoop_(JunLoop aLoop) {
		return (aLoop == leftLoop) || (aLoop == rightLoop);
	}

	/**
	 * Answer true if the receiver includes the vertex.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @return boolean
	 * @category testing
	 */
	public boolean includesVertex_(JunVertex aVertex) {
		return (aVertex == startVertex) || (aVertex == endVertex);
	}

	/**
	 * Answer true if the receiver is killed.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isKilled() {
		return startVertex == null;
	}

	/**
	 * Answer the left loop of the receiver.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunLoop
	 * @category accessing
	 */
	public JunLoop leftLoop() {
		return leftLoop;
	}

	/**
	 * Set the left loop of the receiver.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @category accessing
	 */
	public void leftLoop_(JunLoop aLoop) {
		leftLoop = aLoop;
	}

	/**
	 * Answer the representative loop of the receiver.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunLoop
	 * @category accessing
	 */
	public JunLoop loop() {
		return this.leftLoop();
	}

	/**
	 * Answer the loop which starts from the vertex.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @return jp.co.sra.jun.topology.elements.JunLoop
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category accessing
	 */
	public JunLoop loopFromVertex_(JunVertex aVertex) {
		if (aVertex == this.endVertex()) {
			return this.rightLoop();
		}
		if (aVertex == this.startVertex()) {
			return this.leftLoop();
		}

		throw new SmalltalkException("invalid argument");
	}

	/**
	 * Answer the loop which starts from aVertex1 and ends with aVertex2.
	 * 
	 * @param aVertex1 jp.co.sra.jun.topology.elements.JunVertex
	 * @param aVertex2 jp.co.sra.jun.topology.elements.JunVertex
	 * @return jp.co.sra.jun.topology.elements.JunLoop
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category accessing
	 */
	public JunLoop loopFromVertex_toVertex_(JunVertex aVertex1, JunVertex aVertex2) {
		if ((aVertex1 == this.startVertex()) && (aVertex2 == this.endVertex())) {
			return this.leftLoop();
		}
		if ((aVertex1 == this.endVertex()) && (aVertex2 == this.startVertex())) {
			return this.rightLoop();
		}

		throw new SmalltalkException("invalid argument");
	}

	/**
	 * Evaluate the block for all of the loops.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category enumerating
	 */
	public void loopsDo_(StBlockClosure aBlock) {
		aBlock.value_(this.leftLoop());
		if (this.leftLoop() != this.rightLoop()) {
			aBlock.value_(this.rightLoop());
		}
	}

	/**
	 * Answer the loop which ends with the vertex.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @return jp.co.sra.jun.topology.elements.JunLoop
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category accessing
	 */
	public JunLoop loopToVertex_(JunVertex aVertex) {
		if (aVertex == this.endVertex()) {
			return this.leftLoop();
		}
		if (aVertex == this.startVertex()) {
			return this.rightLoop();
		}

		throw new SmalltalkException("invalid argument");
	}

	/**
	 * Answer the negative direction vertex on the loop.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @return jp.co.sra.jun.topology.elements.JunVertex
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category accessing
	 */
	public JunVertex negativeDirectionVertexOnLoop_(JunLoop aLoop) {
		if (aLoop == this.leftLoop()) {
			return this.startVertex();
		}
		if (aLoop == this.rightLoop()) {
			return this.endVertex();
		}

		throw new SmalltalkException("invalid argument");
	}

	/**
	 * Answer the next edge with the vertex.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @return jp.co.sra.jun.topology.elements.JunEdge
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category accessing
	 */
	public JunEdge nextEdgeWithVertex_(JunVertex aVertex) {
		if (aVertex.equals(this.endVertex())) {
			return this.endLeftEdge();
		}
		if (aVertex.equals(this.startVertex())) {
			return this.startRightEdge();
		}

		throw new SmalltalkException("invalid argument");
	}

	/**
	 * Evaluate the block with with a vertex on a loop.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category accessing
	 */
	public Object onLoop_vertexEdgeVertexDo_(JunLoop aLoop, StBlockClosure aBlock) {
		if (aLoop == this.leftLoop()) {
			return this.directionVertex_vertexEdgeVertexDo_(this.endVertex(), aBlock);
		}
		if (aLoop == this.rightLoop()) {
			return this.directionVertex_vertexEdgeVertexDo_(this.startVertex(), aBlock);
		}

		throw new SmalltalkException("invalid argument");
	}

	/**
	 * Answer the opposite loop of the loop.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @return jp.co.sra.jun.topology.elements.JunLoop
	 * @category accessing
	 */
	public JunLoop oppositeLoopOfLoop_(JunLoop aLoop) {
		if (aLoop == this.leftLoop()) {
			return this.rightLoop();
		}
		if (aLoop == this.rightLoop()) {
			return this.leftLoop();
		}

		return null;
	}

	/**
	 * Answer the opposite vertex of the vertex.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @return jp.co.sra.jun.topology.elements.JunVertex
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category accessing
	 */
	public JunVertex oppositeVertexOfVertex_(JunVertex aVertex) {
		if (aVertex.equals(this.endVertex())) {
			return this.startVertex();
		}
		if (aVertex.equals(this.startVertex())) {
			return this.endVertex();
		}

		throw new SmalltalkException("invalid argument");
	}

	/**
	 * Answer the previous edge with the vertex.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @return jp.co.sra.jun.topology.elements.JunEdge
	 * @category accessing
	 */
	public JunEdge prevEdgeWithVertex_(JunVertex aVertex) {
		JunEdge current = this;
		JunVertex vertex = this.oppositeVertexOfVertex_(aVertex);
		JunEdge next;
		do {
			next = current.nextEdgeWithVertex_(vertex);
			if ((this == next) && (vertex == aVertex)) {
				return current;
			}
			vertex = next.oppositeVertexOfVertex_(vertex);
			current = next;
		} while (true);
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws java.io.IOException
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write("Edge (");
		startVertex.printOn_(aWriter);
		aWriter.write(" -> ");
		endVertex.printOn_(aWriter);
		aWriter.write(")");
	}

	/**
	 * Render the receiver on the rendering context.
	 * 
	 * @param aRenderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category displaying
	 */
	public void renderOn_(JunOpenGLRenderingContext aRenderingContext) {
		throw new SmalltalkException("something wrong with the Smalltalk implementation.");
	}

	/**
	 * Repatch the receiver.
	 * 
	 * @category accessing
	 */
	public void repatch() {
		if (this.basicCurve() == null) {
			return;
		}
		this.curve_(null);
	}

	/**
	 * Answer the right loop of the receiver.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunLoop
	 * @category accessing
	 */
	public JunLoop rightLoop() {
		return rightLoop;
	}

	/**
	 * Set a right loop of the receiver.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @category accessing
	 */
	public void rightLoop_(JunLoop aLoop) {
		rightLoop = aLoop;
	}

	/**
	 * Set a geometry which corresponds to the receiver.
	 * 
	 * @param aCurve jp.co.sra.jun.geometry.abstracts.JunCurve
	 * @category accessing
	 */
	public void setCurve_(JunCurve aCurve) {
		curve = aCurve;
	}

	/**
	 * Set a loop starts from the vertex.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category private
	 */
	public void setLoop_fromVertex_(JunLoop aLoop, JunVertex aVertex) {
		if (aVertex == endVertex) {
			this.rightLoop_(aLoop);
			return;
		}
		if (aVertex == startVertex) {
			this.leftLoop_(aLoop);
			return;
		}

		throw new SmalltalkException("invalid argument");
	}

	/**
	 * Set a loop which starts from aVertex1 and ends with aVertex2.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @param aVertex1 jp.co.sra.jun.topology.elements.JunVertex
	 * @param aVertex2 jp.co.sra.jun.topology.elements.JunVertex
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category private
	 */
	public void setLoop_fromVertex_toVertex_(JunLoop aLoop, JunVertex aVertex1, JunVertex aVertex2) {
		if ((aVertex1 == startVertex) && (aVertex2 == endVertex)) {
			this.leftLoop_(aLoop);
			return;
		}
		if ((aVertex1 == endVertex) && (aVertex2 == startVertex)) {
			this.rightLoop_(aLoop);
			return;
		}

		throw new SmalltalkException("invalid argument");
	}

	/**
	 * Set a loop instead of another loop.
	 * 
	 * @param aLoop1 jp.co.sra.jun.topology.elements.JunLoop
	 * @param aLoop2 jp.co.sra.jun.topology.elements.JunLoop
	 * @category private
	 */
	public void setLoop_insteadOf_(JunLoop aLoop1, JunLoop aLoop2) {
		if (rightLoop == aLoop2) {
			this.rightLoop_(aLoop1);
		}
		if (leftLoop == aLoop2) {
			this.leftLoop_(aLoop1);
		}
	}

	/**
	 * Set a loop which ends with the vertex.
	 * 
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category private
	 */
	public void setLoop_toVertex_(JunLoop aLoop, JunVertex aVertex) {
		if (aVertex == this.endVertex()) {
			this.leftLoop_(aLoop);
			return;
		}
		if (aVertex == this.startVertex()) {
			this.rightLoop_(aLoop);
			return;
		}

		throw new SmalltalkException("invalid argument");
	}

	/**
	 * Set a next edge on the loop.
	 * 
	 * @param anEdge jp.co.sra.jun.topology.elements.JunEdge
	 * @param aLoop jp.co.sra.jun.topology.elements.JunLoop
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category private
	 */
	public void setNextEdge_onLoop_(JunEdge anEdge, JunLoop aLoop) {
		if (leftLoop == aLoop) {
			this.endLeftEdge_(anEdge);
			return;
		}
		if (rightLoop == aLoop) {
			this.startRightEdge_(anEdge);
			return;
		}

		throw new SmalltalkException("invalid argument");
	}

	/**
	 * Set a next edge with the vertex.
	 * 
	 * @param anEdge DOCUMENT ME!
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category private
	 */
	public void setNextEdge_withVertex_(JunEdge anEdge, JunVertex aVertex) {
		if (aVertex == this.endVertex()) {
			this.endLeftEdge_(anEdge);
			return;
		}
		if (aVertex == startVertex) {
			this.startRightEdge_(anEdge);
			return;
		}

		throw new SmalltalkException("invalid argument");
	}

	/**
	 * Set a previous edge with the vertex.
	 * 
	 * @param anEdge jp.co.sra.jun.topology.elements.JunEdge
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @category private
	 */
	public void setPrevEdge_withVertex_(JunEdge anEdge, JunVertex aVertex) {
		JunEdge prev = this.prevEdgeWithVertex_(aVertex);
		prev.setNextEdge_withVertex_(this, aVertex);
	}

	/**
	 * Set aVertex1 instead of aVertex2.
	 * 
	 * @param aVertex1 jp.co.sra.jun.topology.elements.JunVertex
	 * @param aVertex2 jp.co.sra.jun.topology.elements.JunVertex
	 * @category private
	 */
	public void setVertex_insteadOf_(JunVertex aVertex1, JunVertex aVertex2) {
		if (aVertex2 == startVertex) {
			startVertex = aVertex1;
		}
		if (aVertex2 == endVertex) {
			endVertex = aVertex1;
		}
	}

	/**
	 * Answer the start right edge of the receiver.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunEdge
	 * @category accessing
	 */
	public JunEdge startRightEdge() {
		return startRightEdge;
	}

	/**
	 * Set a start right edge of the receiver.
	 * 
	 * @param anEdge jp.co.sra.jun.topology.elements.JunEdge
	 * @category accessing
	 */
	public void startRightEdge_(JunEdge anEdge) {
		startRightEdge = anEdge;
	}

	/**
	 * Answer the start vertex of the receiver.
	 * 
	 * @return jp.co.sra.jun.topology.elements.JunVertex
	 * @category accessing
	 */
	public JunVertex startVertex() {
		return startVertex;
	}

	/**
	 * Set a start vertex of the receiver.
	 * 
	 * @param aVertex jp.co.sra.jun.topology.elements.JunVertex
	 * @category accessing
	 */
	public void startVertex_(JunVertex aVertex) {
		startVertex = aVertex;
	}

	/**
	 * Answer the lisp list which represents the receiver.
	 * 
	 * @param loopArray jp.co.sra.jun.topology.elements.JunLoop[]
	 * @param edgeArray jp.co.sra.jun.topology.elements.JunEdge[]
	 * @param vertexArray jp.co.sra.jun.topology.elements.JunVertex[]
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @see jp.co.sra.jun.topology.abstracts.JunTopologicalElement#toLispListForLoops_edges_vertexes_(jp.co.sra.jun.topology.elements.JunLoop[], jp.co.sra.jun.topology.elements.JunEdge[], jp.co.sra.jun.topology.elements.JunVertex[])
	 * @category lisp support
	 */
	public JunLispList toLispListForLoops_edges_vertexes_(JunLoop[] loopArray, JunEdge[] edgeArray, JunVertex[] vertexArray) {
		int indexOfStartVertex = -1;
		int indexOfEndVertex = -1;
		int indexOfLeftLoop = -1;
		int indexOfRightLoop = -1;
		int indexOfEndLeftEdge = -1;
		int indexOfStartRightEdge = -1;

		for (int i = 0; i < vertexArray.length; i++) {
			if (vertexArray[i] == startVertex) {
				indexOfStartVertex = i;
			}
			if (vertexArray[i] == endVertex) {
				indexOfEndVertex = i;
			}
		}
		for (int i = 0; i < loopArray.length; i++) {
			if (loopArray[i] == leftLoop) {
				indexOfLeftLoop = i;
			}
			if (loopArray[i] == rightLoop) {
				indexOfRightLoop = i;
			}
		}
		for (int i = 0; i < edgeArray.length; i++) {
			if (edgeArray[i] == endLeftEdge) {
				indexOfEndLeftEdge = i;
			}
			if (edgeArray[i] == startRightEdge) {
				indexOfStartRightEdge = i;
			}
		}

		JunLispCons list = this.lispCons();
		list.head_(this.kindName());
		list.add_(JunLispCons.Head_tail_($("startVertex"), new Integer(indexOfStartVertex + 1)));
		list.add_(JunLispCons.Head_tail_($("endVertex"), new Integer(indexOfEndVertex + 1)));
		list.add_(JunLispCons.Head_tail_($("leftLoop"), new Integer(indexOfLeftLoop + 1)));
		list.add_(JunLispCons.Head_tail_($("rightLoop"), new Integer(indexOfRightLoop + 1)));
		list.add_(JunLispCons.Head_tail_($("endLeftEdge"), new Integer(indexOfEndLeftEdge + 1)));
		list.add_(JunLispCons.Head_tail_($("startRightEdge"), new Integer(indexOfStartRightEdge + 1)));
		list.add_(JunLispCons.Head_tail_($("curve"), this.lispNil()));
		return list;
	}
}
