package jp.co.sra.jun.terrain.support;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dPolygon;
import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunTerrainTriangle class
 * 
 *  @author    nisinaka
 *  @created   2002/01/22 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunTerrainTriangle.java,v 8.11 2008/02/20 06:33:00 nisinaka Exp $
 */
public class JunTerrainTriangle extends JunAbstractObject {
	protected Jun3dPoint p1;
	protected Jun3dPoint p2;
	protected Jun3dPoint p3;
	protected double a;
	protected double b;
	protected double c;
	protected double d;
	protected Jun2dBoundingBox bounding2dBox;

	/**
	 * Create a new instance of JunTrerrainTriangle and initialize it.
	 * 
	 * @param aJun3dPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJun3dPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJun3dPoint3 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 */
	public JunTerrainTriangle(Jun3dPoint aJun3dPoint1, Jun3dPoint aJun3dPoint2, Jun3dPoint aJun3dPoint3) {
		super();
		Jun2dPoint p1 = new Jun2dPoint(aJun3dPoint1.x(), aJun3dPoint1.y());
		Jun2dPoint p2 = new Jun2dPoint(aJun3dPoint2.x(), aJun3dPoint2.y());
		Jun2dPoint p3 = new Jun2dPoint(aJun3dPoint3.x(), aJun3dPoint3.y());

		if (p2.minus_(p1).product_(p3.minus_(p1)) > (-1.0d - 12)) {
			this.setP1_p2_p3_(aJun3dPoint1, aJun3dPoint2, aJun3dPoint3);
		} else {
			this.setP1_p2_p3_(aJun3dPoint3, aJun3dPoint2, aJun3dPoint1);
		}
	}

	/**
	 * Convert to a JunOpenGL3dPolygon.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 */
	public JunOpenGL3dObject asJunOpenGL3dObject() {
		return new JunOpenGL3dPolygon(new Jun3dPoint[] { p1, p2, p3 });
	}

	/**
	 * Answer my bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dBoundingBox
	 */
	public Jun2dBoundingBox bounding2dBox() {
		if (bounding2dBox == null) {
			bounding2dBox = this.preferredBounding2dBox();
		}

		return bounding2dBox;
	}

	/**
	 * Answer true if the specified point is contained.
	 * 
	 * @param aJun2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * 
	 * @return boolean
	 */
	public boolean contains2dPoint_(Jun2dPoint aJun2dPoint) {
		if (this.bounding2dBox().containsOrTouchesPoint_(aJun2dPoint) == false) {
			return false;
		}

		Jun2dPoint v1 = new Jun2dPoint(p1.x() - aJun2dPoint.x(), p1.y() - aJun2dPoint.y());
		Jun2dPoint v2 = new Jun2dPoint(p2.x() - aJun2dPoint.x(), p2.y() - aJun2dPoint.y());
		Jun2dPoint v3 = new Jun2dPoint(p3.x() - aJun2dPoint.x(), p3.y() - aJun2dPoint.y());

		return ((v1.product_(v2) > (-1.0d - 12)) && (v2.product_(v3) > (-1.0d - 12)) && v3.product_(v1) > (-1.0d - 12));
	}

	/**
	 * Answer my current normal vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 */
	public Jun3dPoint normalVector() {
		return this.p2().minus_(this.p1()).product_(this.p3().minus_(this.p1()));
	}

	/**
	 * Answer my first point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 */
	public Jun3dPoint p1() {
		return p1;
	}

	/**
	 * Answer my second point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 */
	public Jun3dPoint p2() {
		return p2;
	}

	/**
	 * Answer my third point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 */
	public Jun3dPoint p3() {
		return p3;
	}

	/**
	 * Anser my preferred bounding box.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dBoundingBox
	 */
	public Jun2dBoundingBox preferredBounding2dBox() {
		Jun3dPoint origin = p1.min_(p2).min_(p3);
		Jun3dPoint corner = p1.max_(p2).max_(p3);

		return Jun2dBoundingBox.Origin_corner_(new Jun2dPoint(origin.x(), origin.y()), new Jun2dPoint(corner.x(), corner.y()));
	}

	/**
	 * Anser the z-coordinate at the specified 2D point.
	 * 
	 * @param aJun2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * 
	 * @return double
	 */
	public double zAt2dPoint_(Jun2dPoint aJun2dPoint) {
		if (Math.abs(c) < (1.0d - 12)) {
			return Math.max(Math.max(p1.z(), p2.z()), p3.z());
		}

		return (((a * aJun2dPoint.x()) + (b * aJun2dPoint.y()) + d) * -1) / c;
	}

	/**
	 * Set the three points.
	 * 
	 * @param aJun3dPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJun3dPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJun3dPoint3 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 */
	protected void setP1_p2_p3_(Jun3dPoint aJun3dPoint1, Jun3dPoint aJun3dPoint2, Jun3dPoint aJun3dPoint3) {
		p1 = aJun3dPoint1;
		p2 = aJun3dPoint2;
		p3 = aJun3dPoint3;

		Jun3dPoint normalVector = p2.minus_(p1).product_(p3.minus_(p1));
		a = normalVector.x();
		b = normalVector.y();
		c = normalVector.z();
		d = normalVector.dotProduct_(p1) * -1;
	}
}
