package jp.co.sra.jun.system.support;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.awt.Image;
import java.awt.Rectangle;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.TreeMap;

import jp.co.sra.smalltalk.DependentListener;
import jp.co.sra.smalltalk.StComposedText;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StMessageCatalog;
import jp.co.sra.smalltalk.StModel;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StUserMessage;
import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.SystemInterface;
import jp.co.sra.smalltalk.SystemResourceSupport;
import jp.co.sra.smalltalk.menu.MenuEvent;
import jp.co.sra.smalltalk.menu.MenuListener;
import jp.co.sra.smalltalk.menu.StMenu;
import jp.co.sra.smalltalk.menu.StRadioButtonGroup;
import jp.co.sra.smalltalk.menu.StRadioButtonMenuItem;

import jp.co.sra.gl4jun.GLjInterface;
import jp.co.sra.gl4jun.GLjRenderingMode;

import jp.co.sra.qt4jun.JunQTInterface;

/**
 * JunSystem class
 * 
 *  @author    nisinaka
 *  @created   1998/11/16 (by nisinaka)
 *  @updated   2002/11/21 (by nisinaka)
 *  @updated   2005/02/24 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun682 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunSystem.java,v 8.23 2008/02/20 06:32:59 nisinaka Exp $
 */
public class JunSystem extends StModel {

	public static final String _System = "Jun";
	public static final String _Version = "699";
	public static final String _Date = "2008/02/20";

	public static final StMenu DefaultRenderingModeMenu;
	public static final StMenu DefaultLanguageModeMenu;

	protected static final String _MessageCatalogName = "jp.co.sra.jun.system.support.MessageCatalog";
	protected static final StValueHolder DefaultLanguageModeHolder = new StValueHolder();
	protected static int DefaultRenderingMode;

	protected static long _UniqueNumberGenerationTime = Long.MIN_VALUE;

	static {
		DefaultRenderingModeMenu = CreateDefaultRenderingModeMenu();
		SetDefaultRenderingMode(GLjRenderingMode.DOUBLE_BUFFER);

		DefaultLanguageModeMenu = CreateDefaultLanguageModeMenu();
		SetDefaultLanguageMode(Locale.getDefault());
	}

	/**
	 * Answer a generated unique number.
	 * 
	 * @return long
	 * @category Accessing
	 */
	public static long GenerateUniqueNumber() {
		long currentTime;
		do {
			currentTime = System.currentTimeMillis();
		} while (_UniqueNumberGenerationTime >= currentTime);
		_UniqueNumberGenerationTime = currentTime;

		SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMddHHmmssSSS");
		return Long.parseLong(dateFormat.format(new Date(_UniqueNumberGenerationTime))) * 100;
	}

	/**
	 * Answer the platform name.
	 * For example, "Windows XP", "Windows 2000", "Mac OS X", "Linux".
	 * 
	 * @return java.lang.String
	 * @category Accessing
	 */
	public static String PlatformName() {
		return System.getProperty("os.name");
	}

	/**
	 * Answer true if the platform is Mac OS X, otherwise false.
	 * 
	 * @return boolean
	 * @category Accessing
	 */
	public static boolean IsMacOSX() {
		return PlatformName().toLowerCase().matches(".*mac.*os.*x.*");
	}

	/**
	 * Answer the current Name of the system.
	 *
	 * @return java.lang.String
	 * @category Copyright
	 */
	public static final String System() {
		return $String(_System);
	}

	/**
	 * Answer the current Version of the system.
	 *
	 * @return java.lang.String
	 * @category Copyright
	 */
	public static final String Version() {
		return _Version;
	}

	/**
	 * Answer the string of the copyright.
	 * 
	 * @return java.lang.String
	 * @category Copyright
	 */
	public static final String Copyright() {
		StringWriter sw = new StringWriter();
		PrintWriter pw = null;
		try {
			pw = new PrintWriter(sw);
			pw.println("\"Jun for Java\" package is being published as open source software under the terms of \"The modified BSD license\".");
			pw.println();
			pw.println(CopyrightSRA());
			pw.println(CopyrightIPA());
			pw.println(CopyrightKTL());
			pw.println();
			pw.println("All rights reserved.");
			pw.println();
			pw.println(JavaTM());
		} finally {
			if (pw != null) {
				pw.flush();
				pw.close();
			}
		}

		return sw.toString();
	}

	/**
	 * Answer the string of the IPA copyright.
	 * 
	 * @return java.lang.String
	 * @category Copyright
	 */
	public static final String CopyrightIPA() {
		return "Copyright (C) 1999-2005 Information-technology Promotion Agency, Japan (IPA)";
	}

	/**
	 * Answer the string of the KTL copyright.
	 * 
	 * @return java.lang.String
	 * @category Copyright
	 */
	public static final String CopyrightKTL() {
		int thisYear = Calendar.getInstance().get(Calendar.YEAR);
		return "Copyright (C) 2001-" + thisYear + " SRA/KTL (SRA Key Technology Laboratory, Inc.)";
	}

	/**
	 * Answer the string of the SRA copyright.
	 * 
	 * @return java.lang.String
	 * @category Copyright
	 */
	public static final String CopyrightSRA() {
		int thisYear = Calendar.getInstance().get(Calendar.YEAR);
		return "Copyright (C) 1999-" + thisYear + " SRA (Software Research Associates, Inc.)";
	}

	/**
	 * Answer the current Date of the system.
	 *
	 * @return java.lang.String
	 * @category Copyright
	 */
	public static final String Date() {
		return _Date;
	}

	/**
	 * Answer the string of default base name.
	 *
	 * @return java.lang.String
	 * @category Defaults
	 */
	public static String DefaultBaseName() {
		SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMddHHmmss");
		return dateFormat.format(new Date());
	}

	/**
	 * Answer the default font name.
	 * 
	 * @return java.lang.String
	 * @category Defaults
	 */
	public static String DefaultFontName() {
		String fontName = System.getProperty("default.font.name");
		return (fontName != null) ? fontName : "Dialog";
	}

	/**
	 * Answer the default extension for a text file.
	 *
	 * @return java.lang.String
	 * @category Defaults
	 */
	public static String DefaultTextExtension() {
		return "txt";
	}

	/**
	 * Answer the default extensions for text files.
	 *
	 * @return java.lang.String[]
	 * @category Defaults
	 */
	public static String[] DefaultTextExtensions() {
		return new String[] { "txt", "java", "html", "htm", "bat" };
	}

	/**
	 * Answer the default extension patterns for text files.
	 *
	 * @return java.lang.String[]
	 * @category Defaults
	 */
	public static String[] DefaultTextExtensionPatterns() {
		return DefaultExtensionPatterns(DefaultTextExtensions());
	}

	/**
	 * Answer the default extension for an image file.
	 *
	 * @return java.lang.String
	 * @category Defaults
	 */
	public static String DefaultImageExtension() {
		return "jpg";
	}

	/**
	 * Answer the default extensions for image files.
	 *
	 * @return java.lang.String[]
	 * @category Defaults
	 */
	public static String[] DefaultImageExtensions() {
		return new String[] { "jpg", "jpeg", "png", "bmp", "gif" };
	}

	/**
	 * Answer the default extension patterns for image files.
	 *
	 * @return java.lang.String[]
	 * @category Defaults
	 */
	public static String[] DefaultImageExtensionPatterns() {
		return DefaultExtensionPatterns(DefaultImageExtensions());
	}

	/**
	 * Answer the default extension for a movie file.
	 *
	 * @return java.lang.String
	 * @category Defaults
	 */
	public static String DefaultMovieExtension() {
		return "mov";
	}

	/**
	 * Answer the default extensions for movie files.
	 *
	 * @return java.lang.String[]
	 * @category Defaults
	 */
	public static String[] DefaultMovieExtensions() {
		return new String[] { "mov", "mp4", "avi", "mpg", "mpeg" };
	}

	/**
	 * Answer the default extension patterns for movie files.
	 *
	 * @return java.lang.String[]
	 * @category Defaults
	 */
	public static String[] DefaultMovieExtensionPatterns() {
		return DefaultExtensionPatterns(DefaultMovieExtensions());
	}

	/**
	 * Answer the default extension for a sound file.
	 *
	 * @return java.lang.String
	 * @category Defaults
	 */
	public static String DefaultSoundExtension() {
		return "mov";
	}

	/**
	 * Answer the default extensions for sound files.
	 *
	 * @return java.lang.String[]
	 * @category Defaults
	 */
	public static String[] DefaultSoundExtensions() {
		return new String[] { "mp3", "wav" };
	}

	/**
	 * Answer the default extension patterns for sound files.
	 *
	 * @return java.lang.String[]
	 * @category Defaults
	 */
	public static String[] DefaultSoundExtensionPatterns() {
		return DefaultExtensionPatterns(DefaultSoundExtensions());
	}

	/**
	 * Answer the default extension for a body file.
	 *
	 * @return java.lang.String
	 * @category Defaults
	 */
	public static String DefaultBodyExtension() {
		return "lst";
	}

	/**
	 * Answer the default extensions for body files.
	 *
	 * @return java.lang.String[]
	 * @category Defaults
	 */
	public static String[] DefaultBodyExtensions() {
		return new String[] { "lst", "wrl" };
	}

	/**
	 * Answer the default extension patterns for body files.
	 *
	 * @return java.lang.String[]
	 * @category Defaults
	 */
	public static String[] DefaultBodyExtensionPatterns() {
		return DefaultExtensionPatterns(DefaultBodyExtensions());
	}

	/**
	 * Answer the default extensions for extra files.
	 *
	 * @return java.lang.String[]
	 * @category Defaults
	 */
	public static String[] DefaultExtraExtensions() {
		return new String[] { "pdf", "doc", "xls", "ppt" };
	}

	/**
	 * Answer the default extension patterns for extra files.
	 *
	 * @return java.lang.String[]
	 * @category Defaults
	 */
	public static String[] DefaultExtraExtensionPatterns() {
		return DefaultExtensionPatterns(DefaultExtraExtensions());
	}

	/**
	 * Answer the default extension patterns for the specified extensions.
	 * 
	 * @param extensions java.lang.String[]
	 * @return java.lang.String[]
	 * @category Defaults
	 */
	protected static String[] DefaultExtensionPatterns(String[] extensions) {
		int size = extensions.length;
		String[] patterns = new String[size * 2];
		for (int i = 0; i < size; i++) {
			patterns[i] = "*." + extensions[i].toLowerCase();
			patterns[i + size] = "*." + extensions[i].toUpperCase();
		}
		return patterns;
	}

	/**
	 * Answer the fullname of the current JunSystem.
	 *
	 * @return java.lang.String
	 * @category Copyright
	 */
	public static String FullName() {
		return JunSystem.System() + " for Java (" + JunSystem.Version() + " with " + StObject.System() + StObject.Version() + ")";
	}

	/**
	 * Return Jun jni version string
	 * 
	 * @return string
	 * @category Copyright
	 */
	public static final String getJniVersion() {
		return GLjInterface.getVersion();
	}

	/**
	 * Answer the system information.
	 *
	 * @return java.lang.String
	 * @category Information
	 */
	public static final String Information() {
		String indent = "    ";
		StringWriter sw = new StringWriter();
		PrintWriter pw = new PrintWriter(sw);

		try {
			pw.println("Versions");
			pw.println(indent + JunSystem.System() + " : " + JunSystem.Version() + " (" + JunSystem.Date() + ")");
			pw.println(indent + StObject.System() + " : " + StObject.Version() + " (" + StObject.Date() + ")");
			pw.println();

			pw.println("JNI library versions");
			try {
				GLjInterface.Current();
				pw.println(indent + GLjInterface.libraryName + " : " + JunSystem.getJniVersion());
			} catch (Throwable e) {
				pw.println(indent + GLjInterface.libraryName + " : can not find");
			}
			try {
				JunQTInterface.Interface();
				pw.println(indent + JunQTInterface.libraryName + " : " + JunQTInterface.GetVersion());
			} catch (Throwable e) {
				pw.println(indent + JunQTInterface.libraryName + " : can not find");
			}
			try {
				SystemInterface.Current();
				pw.println(indent + SystemInterface.libraryName + " : " + SystemInterface.getVersion());
			} catch (Throwable e) {
				pw.println(indent + SystemInterface.libraryName + " : can not find");
			}
			pw.println();

			pw.println("Java Properties");
			TreeMap map = new TreeMap();
			map.putAll(System.getProperties());
			String lineSeparator = (String) map.get("line.separator");
			if (lineSeparator != null && lineSeparator.length() > 0) {
				String newLineSeparator = "";
				for (int i = 0; i < lineSeparator.length(); i++) {
					char c = lineSeparator.charAt(i);
					switch (c) {
						case '\n':
							newLineSeparator += "\\n";
							break;
						case '\r':
							newLineSeparator += "\\r";
							break;
						case '\f':
							newLineSeparator += "\\f";
							break;
						default:
							newLineSeparator += String.valueOf(c);
					}
				}
				map.put("line.separator", newLineSeparator);
			}
			Object[] keys = map.keySet().toArray();
			for (int i = 0; i < keys.length; i++) {
				pw.println(indent + keys[i].toString() + " : " + map.get(keys[i]).toString());
			}
		} finally {
			pw.flush();
			pw.close();
		}

		return sw.toString();
	}

	/**
	 * Answer the string of the Java trademark legend.
	 * 
	 * @return java.lang.String
	 * @category Copyright
	 */
	public static final String JavaTM() {
		return "Java and all Java-based trademarks and logos are trademarks or registered trademarks of Sun Microsystems, Inc. in the U.S. or other countries, and are used under license. ";
	}

	/**
	 * Answer the string of the license term.
	 * 
	 * @return java.lang.String
	 * @category License
	 */
	public static final String License() {
		StringWriter sw = new StringWriter();
		PrintWriter pw = null;

		try {
			pw = new PrintWriter(sw);
			pw.println(CopyrightSRA());
			pw.println(CopyrightIPA());
			pw.println(CopyrightKTL());
			pw.println("All rights reserved.");
			pw.println();
			pw.println("Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:");
			pw.println();
			pw.println("1. Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer. ");
			pw.println();
			pw.println("2. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution. ");
			pw.println();
			pw.println("3. The name of the author may not be used to endorse or promote products derived from this software without specific prior written permission. ");
			pw.println();
			pw
					.println("THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.");
			pw.println();
			pw.println(JavaTM());
		} finally {
			if (pw != null) {
				pw.flush();
				pw.close();
			}
		}

		return sw.toString();
	}

	/**
	 * Answer a logo.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @category Logo
	 */
	public static final StImage Logo() {
		Font font = new Font(JunSystem.DefaultFontName(), Font.BOLD, 12);

		StComposedText fullName = new StComposedText(JunSystem.FullName(), font);
		StComposedText date = new StComposedText(JunSystem.Date(), font);
		StImage logoImage = new StImage(JunSystem.LogoImage());

		int horizontalMargin = 20;
		int verticalMargin = 6;

		int width = logoImage.width() + horizontalMargin * 2;
		int height = logoImage.height() + fullName.height() + date.height() + verticalMargin * 2;
		StImage aboutImage = new StImage(width, height);

		Graphics gc = null;
		try {
			gc = aboutImage.image().getGraphics();
			gc.setColor(Color.white);
			gc.fillRect(0, 0, aboutImage.width(), aboutImage.height());

			gc.drawImage(fullName.toImage(), horizontalMargin, verticalMargin, null);
			gc.drawImage(logoImage.image(), horizontalMargin, verticalMargin + fullName.height(), null);
			gc.drawImage(date.toImage(), aboutImage.width() - date.width() - horizontalMargin, verticalMargin + fullName.height() + logoImage.height(), null);

			aboutImage.image().flush();
		} finally {
			if (gc != null) {
				gc.dispose();
				gc = null;
			}
		}

		return aboutImage;
	}

	/**
	 * Create a Jun logo image.
	 * 
	 * @return java.awt.Image
	 * @category Logo
	 */
	public static final Image LogoImage() {
		return SystemResourceSupport.createImage("/jp/co/sra/jun/system/support/LogoImage.jpg");
	}

	/**
	 * Answer the project name string.
	 * 
	 * @return java.lang.String
	 * @category Logo
	 */
	public static final String ProjectNameString() {
		return JunSystem.System() + " for Java Project";
	}

	/**
	 * Answer the screen bounding box.
	 * 
	 * @return java.awt.Rectangle
	 * @category Screen
	 */
	public static Rectangle ScreenBoundingBox() {
		Rectangle[] bounds = ScreenBounds();
		return (bounds == null || bounds.length == 0) ? null : bounds[0];
	}

	/**
	 * Answer the screen bounds.
	 * 
	 * @return java.awt.Rectangle
	 * @category Screen
	 */
	public static Rectangle[] ScreenBounds() {
		GraphicsEnvironment graphicsEnvironment = GraphicsEnvironment.getLocalGraphicsEnvironment();
		GraphicsDevice[] screenDevices = graphicsEnvironment.getScreenDevices();
		if (screenDevices == null) {
			return null;
		}

		Rectangle[] bounds = new Rectangle[screenDevices.length];
		for (int i = 0; i < screenDevices.length; i++) {
			bounds[i] = screenDevices[i].getDefaultConfiguration().getBounds();
		}
		return bounds;
	}

	/**
	 * Answer the current message catalog.
	 * 
	 * @return jp.co.sra.smalltalk.StMessageCatalog
	 * @category Message catalog
	 */
	public static StMessageCatalog GetMessageCatalog() {
		if (GetDefaultLanguageMode() == null) {
			return StMessageCatalog.DefaultMessageCatalog;
		}

		return StMessageCatalog.With(_MessageCatalogName, GetDefaultLanguageMode());
	}

	/**
	 * Answer the message catalog for the specified locale.
	 * 
	 * @param aLocale java.util.Locale
	 * @return jp.co.sra.smalltalk.StMessageCatalog
	 * @category Message catalog
	 */
	private static StMessageCatalog GetMessageCatalogFor(Locale aLocale) {
		if (aLocale == null) {
			return StMessageCatalog.DefaultMessageCatalog;
		}

		try {
			return StMessageCatalog.With(_MessageCatalogName, aLocale);
		} catch (MissingResourceException e) {
			System.err.println(e.getMessage());
		}

		if (Locale.getDefault().equals(aLocale) == false) {
			try {
				aLocale = Locale.getDefault();
				StMessageCatalog aMessageCatalog = StMessageCatalog.With(_MessageCatalogName, aLocale);
				System.err.println("Use the message catalog with the default locale " + aLocale);
				return aMessageCatalog;
			} catch (MissingResourceException e) {
				System.err.println(e.getMessage());
			}
		}

		System.err.println("Use the default message catalog.");
		return StMessageCatalog.DefaultMessageCatalog;
	}

	/**
	 * Answer the current default rendering mode.
	 * 
	 * @return int
	 * @category Rendering mode
	 */
	public static int GetDefaultRenderingMode() {
		return DefaultRenderingMode;
	}

	/**
	 * Return default rendering mode.
	 * 
	 * @return int
	 * @deprecated since Jun500, use GetDefaultRenderingMode()
	 * @category Rendering mode
	 */
	public static int getDefaultRenderingMode() {
		return GetDefaultRenderingMode();
	}

	/**
	 * Set the default rendering mode.
	 * 
	 * @param mode int
	 * @category Rendering mode
	 */
	public static void SetDefaultRenderingMode(int mode) {
		DefaultRenderingMode = mode;
		UpdateDefaultRenderingModeMenu();
	}

	/**
	 * Set default rendering mode.
	 * 
	 * @param mode int
	 * @deprecated since Jun500, use SetDefaultRenderingMode(int)
	 * @category Rendering mode
	 */
	public static void setDefaultRenderingMode(int mode) {
		SetDefaultRenderingMode(mode);
	}

	/**
	 * Update the "Default Rendering Mode" menu.
	 * 
	 * @category Rendering mode
	 */
	public static void UpdateDefaultRenderingModeMenu() {
		StSymbol key = null;
		switch (GetDefaultRenderingMode()) {
			case GLjRenderingMode.NOTHING:
				key = $("renderNothing");
				break;
			case GLjRenderingMode.SINGLE_BUFFER:
				key = $("renderSingleBuffer");
				break;
			case GLjRenderingMode.IMAGE:
				key = $("renderImage");
				break;
			case GLjRenderingMode.DIRECT:
				key = $("renderDirect");
				break;
			case GLjRenderingMode.DOUBLE_BUFFER:
				key = $("renderDoubleBuffer");
				break;
		}
		if (key == null) {
			return;
		}

		StRadioButtonMenuItem aMenuItem = (StRadioButtonMenuItem) DefaultRenderingModeMenu.atNameKey_(key);
		if (aMenuItem != null && aMenuItem.isSelected() == false) {
			aMenuItem.beOn();
		}
	}

	/**
	 * Create a "Default Rendering Mode" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category Rendering mode
	 */
	protected static StMenu CreateDefaultRenderingModeMenu() {
		StRadioButtonGroup aGroup = new StRadioButtonGroup();
		StMenu aMenu = new StMenu(new StUserMessage("Default Rendering Mode"));
		aMenu.add(new StRadioButtonMenuItem(new StUserMessage("Nothing"), $("renderNothing"), aGroup, new MenuListener() {
			public void menuPerformed(MenuEvent aMenuEvent) {
				setDefaultRenderingMode(GLjRenderingMode.NOTHING);
			}
		}));
		aMenu.add(new StRadioButtonMenuItem(new StUserMessage("Single Buffer"), $("renderSingleBuffer"), aGroup, new MenuListener() {
			public void menuPerformed(MenuEvent aMenuEvent) {
				setDefaultRenderingMode(GLjRenderingMode.SINGLE_BUFFER);
			}
		}));
		aMenu.add(new StRadioButtonMenuItem(new StUserMessage("Image"), $("renderImage"), aGroup, new MenuListener() {
			public void menuPerformed(MenuEvent aMenuEvent) {
				setDefaultRenderingMode(GLjRenderingMode.IMAGE);
			}
		}));
		aMenu.add(new StRadioButtonMenuItem(new StUserMessage("Direct"), $("renderDirect"), aGroup, new MenuListener() {
			public void menuPerformed(MenuEvent aMenuEvent) {
				setDefaultRenderingMode(GLjRenderingMode.DIRECT);
			}
		}));
		aMenu.add(new StRadioButtonMenuItem(new StUserMessage("Double Buffer"), $("renderDoubleBuffer"), aGroup, new MenuListener() {
			public void menuPerformed(MenuEvent aMenuEvent) {
				setDefaultRenderingMode(GLjRenderingMode.DOUBLE_BUFFER);
			}
		}));
		return aMenu;
	}

	/**
	 * Answer the current default language mode.
	 * 
	 * @return java.util.Locale
	 * @category Language mode
	 */
	public static Locale GetDefaultLanguageMode() {
		return (Locale) DefaultLanguageModeHolder.value();
	}

	/**
	 * Set the new default language mode.
	 * 
	 * @param aLocale java.util.Locale
	 * @category Language mode
	 */
	public static void SetDefaultLanguageMode(Locale aLocale) {
		if (GetDefaultLanguageMode() == aLocale) {
			return;
		}

		StMessageCatalog aMessageCatalog = GetMessageCatalogFor(aLocale);
		StMessageCatalog.GivePriorityTo(aMessageCatalog);

		DefaultLanguageModeHolder.setValue_(aMessageCatalog.getLocale());
		UpdateDefaultLanguageModeMenu();
		DefaultLanguageModeHolder.changed_($("value"));
	}

	/**
	 * Add a dependent listener on the default language mode holder.
	 * 
	 * @param aDependentListener jp.co.sra.smalltalk.DependentListener
	 * @category Language mode
	 */
	public static void AddDependentListenerOnDefaultLanguageMode(DependentListener aDependentListener) {
		DefaultLanguageModeHolder.addDependentListener(aDependentListener);
	}

	/**
	 * Set the new locale.
	 * 
	 * @param aLocale java.util.Locale
	 * @deprecated since Jun500, use SetDefaultLanguageMode
	 * @category Language mode
	 */
	public static void SetLocale(Locale aLocale) {
		SetDefaultLanguageMode(aLocale);
	}

	/**
	 * Answer true if the current default language mode is the specified one.
	 * 
	 * @param aLocale java.util.Locale
	 * @return boolean
	 * @category Language mode
	 */
	public static boolean IsDefaultLanguageMode(Locale aLocale) {
		if (GetDefaultLanguageMode() == aLocale) {
			return true;
		}
		if (GetDefaultLanguageMode() == null || aLocale == null) {
			return false;
		}
		return GetDefaultLanguageMode().getLanguage().equals(aLocale.getLanguage());
	}

	/**
	 * Update the "Default Language Mode" menu.
	 * 
	 * @category Language mode
	 */
	public static void UpdateDefaultLanguageModeMenu() {
		StSymbol key = null;
		if (IsDefaultLanguageMode(Locale.JAPAN)) {
			key = $("languageJapanese");
		} else {
			key = $("languageEnglish");
		}

		StRadioButtonMenuItem aMenuItem = (StRadioButtonMenuItem) DefaultLanguageModeMenu.atNameKey_(key);
		if (aMenuItem != null && aMenuItem.isSelected() == false) {
			aMenuItem.beOn();
		}
	}

	/**
	 * Create a "Default Language Mode" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @category Language mode
	 */
	protected static StMenu CreateDefaultLanguageModeMenu() {
		StRadioButtonGroup aGroup = new StRadioButtonGroup();
		StMenu aMenu = new StMenu(new StUserMessage("Default Language Mode"));
		aMenu.add(new StRadioButtonMenuItem(new StUserMessage("Japanese"), $("languageJapanese"), aGroup, new MenuListener() {
			public void menuPerformed(MenuEvent aMenuEvent) {
				SetDefaultLanguageMode(Locale.JAPAN);
			}
		}));
		aMenu.add(new StRadioButtonMenuItem(new StUserMessage("English"), $("languageEnglish"), aGroup, new MenuListener() {
			public void menuPerformed(MenuEvent aMenuEvent) {
				SetDefaultLanguageMode(null);
			}
		}));
		return aMenu;
	}

}
