package jp.co.sra.jun.opengl.rotation;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.PopupMenu;
import java.awt.Rectangle;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StController;
import jp.co.sra.smalltalk.StModel;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.curves.Jun2dLine;
import jp.co.sra.jun.system.framework.JunAbstractViewCanvas;

/**
 * JunOpenGLRotationViewAwt class
 * 
 *  @author    nisinaka
 *  @created   1998/12/09 (by nisinaka)
 *  @updated   2000/01/20 (by MATSUDA Ryouichi)
 *  @updated   2004/09/21 (by nisinaka)
 *  @updated   2005/03/03 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun519 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGLRotationViewAwt.java,v 8.10 2008/02/20 06:32:49 nisinaka Exp $
 */
public class JunOpenGLRotationViewAwt extends JunAbstractViewCanvas implements JunOpenGLRotationView {

	protected static final Color _DarkGreenColor = new Color(0, 127, 0);
	protected static final Color _RoyalBlueColor = new Color(127, 127, 255);
	protected PopupMenu _popupMenu;

	/**
	 * Create a new instance of JunOpenGLRotationViewAwt and initialize it.
	 *
	 * @param aModel jp.co.sra.jun.opengl.rotation.JunOpenGLRotationModel
	 */
	public JunOpenGLRotationViewAwt(JunOpenGLRotationModel aModel) {
		super(aModel);
	}

	/**
	 * Answer the de-scaled vertex at the specified Point.
	 * 
	 * @param aPoint java.awt.Point
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 */
	public Jun2dPoint descaledPoint_(Point aPoint) {
		Point centerPoint = this.centerPoint();

		return this.getModel().descaledPoint_(new Point(aPoint.x - centerPoint.x, aPoint.y - centerPoint.y));
	}

	/**
	 * Display the receiver on the Graphics.
	 * 
	 * @param graphics java.awt.Graphics
	 */
	public void displayOn_(final Graphics graphics) {
		if (this.isShowing() == false) {
			return;
		}

		this.displayAxesOn_(graphics);
		this.getModel().edgesDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				Jun2dLine edge = (Jun2dLine) anObject;
				graphics.setColor(JunOpenGLRotationViewAwt.this.edgeColorFor_(edge));

				Point fromPoint = JunOpenGLRotationViewAwt.this.scaledPoint_((Jun2dPoint) edge.from());
				Point toPoint = JunOpenGLRotationViewAwt.this.scaledPoint_((Jun2dPoint) edge.to());
				graphics.drawLine(fromPoint.x, fromPoint.y, toPoint.x, toPoint.y);

				return null;
			}
		});
		this.getModel().verticesDo_(new StBlockClosure() {
			public Object value_(Object anObject) {
				Jun2dPoint vertex = (Jun2dPoint) anObject;
				graphics.setColor(JunOpenGLRotationViewAwt.this.vertexColorFor_(vertex));

				Point atPoint = JunOpenGLRotationViewAwt.this.scaledPoint_(vertex);
				graphics.fillRect(atPoint.x - 1, atPoint.y - 1, 3, 3);

				return null;
			}
		});
	}

	/**
	 * model property accessing.
	 * 
	 * @return jp.co.sra.jun.opengl.rotation.JunOpenGLRotationModel
	 */
	public JunOpenGLRotationModel getModel() {
		return (JunOpenGLRotationModel) this.model();
	}

	/**
	 * Answer the scaled point of the Jun2dPoint.
	 * 
	 * @param aJun2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * 
	 * @return java.awt.Point
	 */
	public Point scaledPoint_(Jun2dPoint aJun2dPoint) {
		Point centerPoint = this.centerPoint();
		Point scaledPoint = this.getModel().scaledPoint_(aJun2dPoint);
		scaledPoint.x += centerPoint.x;
		scaledPoint.y += centerPoint.y;

		return scaledPoint;
	}

	/**
	 * Answer the scaled point of the vertex with the specified index.
	 * 
	 * @param anIndex int
	 * 
	 * @return java.awt.Point
	 */
	public Point scaledVertexAt_(int anIndex) {
		return this.scaledPoint_(this.getModel().vertexAt_(anIndex));
	}

	/**
	 * Answer the index of the vertex at the specified point. If there is no
	 * such vertex, return -1.
	 * 
	 * @param aPoint java.awt.Point
	 * 
	 * @return int
	 */
	public int scaledVertexIndex_(Point aPoint) {
		int size = this.getModel().numVertices();

		for (int i = 0; i < size; i++) {
			Point anotherPoint = this.scaledVertexAt_(i);

			if ((Math.abs(aPoint.x - anotherPoint.x) < 4) && (Math.abs(aPoint.y - anotherPoint.y) < 4)) {
				return i;
			}
		}

		return -1;
	}

	/**
	 * model property accessing.
	 * 
	 * @param aModel jp.co.sra.jun.opengl.rotation.JunOpenGLRotationModel
	 */
	public void setModel(JunOpenGLRotationModel aModel) {
		this.model_(aModel);
	}

	/**
	 * Build this component.
	 */
	protected void buildComponent() {
		this.setSize(192, 192);
	}

	/**
	 * Answer the center point of the receiver's view.
	 * 
	 * @return java.awt.Point
	 */
	protected Point centerPoint() {
		Rectangle box = new Rectangle(0, 0, this.getSize().width, this.getSize().height);

		if (this.getModel().swapXY()) {
			return new Point(box.x + (box.width / 4), box.y + (box.height / 2));
		} else {
			return new Point(box.x + (box.width / 2), box.y + ((box.height * 3) / 4));
		}
	}

	/**
	 * Answer my default controller.
	 * 
	 * @return jp.co.sra.smalltalk.StController
	 */
	protected StController defaultController() {
		return new JunOpenGLRotationController();
	}

	/**
	 * Answer a default model.
	 * 
	 * @return jp.co.sra.smalltalk.StModel
	 */
	protected StModel defaultModel() {
		return new JunOpenGLRotationModel();
	}

	/**
	 * Show axes on the receiver's view.
	 * 
	 * @param graphics java.awt.Graphics
	 */
	protected void displayAxesOn_(Graphics graphics) {
		JunOpenGLRotationModel model = this.getModel();
		Rectangle box = new Rectangle(0, 0, this.getSize().width, this.getSize().height);
		graphics.clipRect(box.x, box.y, box.width, box.height);
		graphics.setColor(Color.white);
		graphics.fillRect(box.x, box.y, box.width, box.height);

		Point center = this.centerPoint();
		int boxLeft = box.x;
		int boxRight = box.x + box.width;
		int boxTop = box.y;
		int boxBottom = box.y + box.height;

		if (Double.isNaN(model.grid()) == false) {
			graphics.setColor(this.gridColor());

			double scaledGrid = model.grid() * model.scale();
			int x;
			int y;

			for (int i = 1;; i++) {
				boolean flag = false;
				int delta = (int) (i * scaledGrid);

				if ((x = center.x - delta) >= boxLeft) {
					graphics.drawLine(x, boxTop, x, boxBottom);
					flag = true;
				}

				if ((x = center.x + delta) <= boxRight) {
					graphics.drawLine(x, boxTop, x, boxBottom);
					flag = true;
				}

				if ((y = center.y - delta) >= boxTop) {
					graphics.drawLine(boxLeft, y, boxRight, y);
					flag = true;
				}

				if ((y = center.y + delta) <= boxBottom) {
					graphics.drawLine(boxLeft, y, boxRight, y);
					flag = true;
				}

				if (flag == false) {
					break;
				}
			}
		}

		if (this.getModel().showXAxis() == true) {
			graphics.setColor(this.xAxisColor());
			graphics.drawLine(boxLeft, center.y, boxRight, center.y);
		}

		if (this.getModel().showYAxis() == true) {
			graphics.setColor(this.yAxisColor());
			graphics.drawLine(center.x, boxTop, center.x, boxBottom);
		}
	}

	/**
	 * Answer the color for an edge.
	 * 
	 * @return java.awt.Color
	 */
	protected Color edgeColor() {
		return Color.black;
	}

	/**
	 * Answer the color for the edge.
	 * 
	 * @param edge jp.co.sra.jun.geometry.curves.Jun2dLine
	 * 
	 * @return java.awt.Color
	 */
	protected Color edgeColorFor_(Jun2dLine edge) {
		if (edge.to().equals(this.getModel().firstVertex())) {
			return this.loopEdgeColor();
		}

		if (edge.from().equals(this.getModel().lastVertex())) {
			return this.loopEdgeColor();
		}

		return this.edgeColor();
	}

	/**
	 * Answer the color for a first vertex.
	 * 
	 * @return java.awt.Color
	 */
	protected Color firstVertexColor() {
		return _RoyalBlueColor;
	}

	/**
	 * Answer the color for the grid.
	 * 
	 * @return java.awt.Color
	 */
	protected Color gridColor() {
		return Color.lightGray;
	}

	/**
	 * Answer the color for a last vertex.
	 * 
	 * @return java.awt.Color
	 */
	protected Color lastVertexColor() {
		return _DarkGreenColor;
	}

	/**
	 * Answer the color for a loop edge.
	 * 
	 * @return java.awt.Color
	 */
	protected Color loopEdgeColor() {
		return Color.gray;
	}

	/**
	 * Answer the color for a selected vertex.
	 * 
	 * @return java.awt.Color
	 */
	protected Color selectedVertexColor() {
		return Color.red;
	}

	/**
	 * Answer the color for a vertex.
	 * 
	 * @return java.awt.Color
	 */
	protected Color vertexColor() {
		return Color.black;
	}

	/**
	 * Answer the color for the edge.
	 * 
	 * @param vertex jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * 
	 * @return java.awt.Color
	 */
	protected Color vertexColorFor_(Jun2dPoint vertex) {
		JunOpenGLRotationModel model = this.getModel();

		if (model.vertexSelected_(vertex)) {
			return this.selectedVertexColor();
		}

		if (vertex.equals(model.firstVertex())) {
			return this.firstVertexColor();
		}

		if (vertex.equals(model.lastVertex())) {
			return this.lastVertexColor();
		}

		return this.vertexColor();
	}

	/**
	 * Answer the color for X-axis.
	 * 
	 * @return java.awt.Color
	 */
	protected Color xAxisColor() {
		if (this.getModel().swapXY()) {
			return _DarkGreenColor;
		} else {
			return Color.blue;
		}
	}

	/**
	 * Answer the color for Y-axis.
	 * 
	 * @return java.awt.Color
	 */
	protected Color yAxisColor() {
		if (this.getModel().swapXY()) {
			return Color.blue;
		} else {
			return _DarkGreenColor;
		}
	}

}
