package jp.co.sra.jun.opengl.projection;

import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext;

/**
 * JunOpenGLPerspectiveProjection class
 * 
 *  @author    MATSUDA Ryouichi
 *  @created   1998/10/29 (by MATSUDA Ryouichi)
 *  @updated   1999/08/04 (by nisinaka)
 *  @updated   2004/10/22 (by nisinaka)
 *  @updated   2006/10/13 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun618 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGLPerspectiveProjection.java,v 8.11 2008/02/20 06:32:48 nisinaka Exp $
 */
public class JunOpenGLPerspectiveProjection extends JunOpenGLProjection {

	protected JunAngle fovy;
	protected double zoomHeight;

	public static StSymbol Type = $("perspectiveProjection");
	public static JunAngle DefaultFovy = JunAngle.FromDeg_(15);

	/**
	 * Create a new instance of JunOpenGLPerspectiveProjection.
	 * 
	 * @category Instance creation
	 */
	public JunOpenGLPerspectiveProjection() {
		super();
		this.setEyePoint_sightPoint_upVector_fovy_near_far_(DefaultEyePoint, DefaultSightPoint, DefaultUpVector, DefaultFovy, DefaultNear, DefaultFar);
	}

	/**
	 * Create a new instance of JunOpenGLPerspectiveProjection and initialize it.
	 * 
	 * @param eyePoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param sightPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param upVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param fovy jp.co.sra.jun.geometry.basic.JunAngle
	 * @param near double
	 * @param far double
	 * @category Instance creation
	 */
	public JunOpenGLPerspectiveProjection(Jun3dPoint eyePoint, Jun3dPoint sightPoint, Jun3dPoint upVector, JunAngle fovy, double near, double far) {
		super();
		this.setEyePoint_sightPoint_upVector_fovy_near_far_(eyePoint, sightPoint, upVector, fovy, near, far);
	}

	/**
	 * Create a new instance of JunOpenGLPerspectiveProjection and initialize it.
	 * 
	 * @param eyePoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param sightPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param upVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param fovy double
	 * @param near double
	 * @param far double
	 * @category Instance creation
	 */
	public JunOpenGLPerspectiveProjection(Jun3dPoint eyePoint, Jun3dPoint sightPoint, Jun3dPoint upVector, double fovy, double near, double far) {
		super();
		this.setEyePoint_sightPoint_upVector_fovy_near_far_(eyePoint, sightPoint, upVector, fovy, near, far);
	}

	/**
	 * Create a new instance of JunOpenGLPerspectiveProjection and initialize it.
	 * 
	 * @param eyePoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param sightPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param upVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param fovy jp.co.sra.jun.geometry.basic.JunAngle
	 * @param near double
	 * @param far double
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLPerspectiveProjection
	 * @category Instance creation
	 * @deprecated since Jun497, use the constructor
	 */
	public static JunOpenGLPerspectiveProjection EyePoint_sightPoint_upVector_fovy_near_far_(Jun3dPoint eyePoint, Jun3dPoint sightPoint, Jun3dPoint upVector, JunAngle fovy, double near, double far) {
		return new JunOpenGLPerspectiveProjection(eyePoint, sightPoint, upVector, fovy, near, far);
	}

	/**
	 * Create a new instance of JunOpenGLPerspectiveProjection and initialize it.
	 * 
	 * @param eyePoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param sightPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param upVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param fovy double
	 * @param near double
	 * @param far double
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLPerspectiveProjection
	 * @category Instance creation
	 * @deprecated since Jun497, use the constructor
	 */
	public static JunOpenGLPerspectiveProjection EyePoint_sightPoint_upVector_fovy_near_far_(Jun3dPoint eyePoint, Jun3dPoint sightPoint, Jun3dPoint upVector, double fovy, double near, double far) {
		return new JunOpenGLPerspectiveProjection(eyePoint, sightPoint, upVector, fovy, near, far);
	}

	/**
	 * Create a new default projection and answer it.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLProjection
	 * @category Defaults
	 * @deprecated since Jun497, use the constructor.
	 */
	public static JunOpenGLProjection Default() {
		return new JunOpenGLPerspectiveProjection();
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		zoomHeight = Double.NaN;
	}

	/**
	 * Answer my projection type as StSymbol
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#type()
	 * @category accessing
	 */
	public StSymbol type() {
		return Type;
	}

	/**
	 * Answer my current fovy.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @category accessing
	 */
	public JunAngle fovy() {
		return fovy;
	}

	/**
	 * Answer my regular height.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double regularHeight() {
		return this.fovy().div_(2).tan() * this.distance() * 2;
	}

	/**
	 * Convert to a paralalel projection.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLParallelProjection
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#asParallelProjection()
	 * @category converting
	 */
	public JunOpenGLParallelProjection asParallelProjection() {
		double height = this.fovy().div_(2).tan() * this.distance() * 2;
		return new JunOpenGLParallelProjection(this.eyePoint(), this.sightPoint(), this.upVector(), height, this.near(), this.far());
	}

	/**
	 * Convert to a perspective projection.
	 * 
	 * @return jp.co.sra.jun.opengl.projection.JunOpenGLPerspectiveProjection
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#asPerspectiveProjection()
	 * @category converting
	 */
	public JunOpenGLPerspectiveProjection asPerspectiveProjection() {
		return (JunOpenGLPerspectiveProjection) this.copy();
	}

	/**
	 * Convert to a 3D Transformation.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#asTransformation()
	 * @category converting
	 */
	public Jun3dTransformation asTransformation() {
		Jun3dPoint transformedSightPoint = new Jun3dPoint(0, 0, 0);
		Jun3dPoint transformedEyePoint = new Jun3dPoint(0, 0, -1);
		Jun3dPoint transformedUpVector = new Jun3dPoint(0, 1, 0);
		Jun3dPoint transformedRightVector = new Jun3dPoint(1, 0, 0);
		double scale = this.distance() * Math.abs(this.fovy().div_(2).tan());
		Jun3dPoint[][] alignPoints = new Jun3dPoint[][] {
				{ this.sightPoint(), transformedSightPoint },
				{ this.eyePoint(), transformedEyePoint },
				{ this.sightPoint().plus_(this.unitUpVector().multipliedBy_(scale)), transformedUpVector },
				{ this.sightPoint().plus_(this.unitRightVector().multipliedBy_(scale)), transformedRightVector } };
		Jun3dTransformation transformation = Jun3dTransformation.AlignPoints_(alignPoints);
		return transformation.product_(Jun3dTransformation.Perspective_(transformedEyePoint));
	}

	/**
	 * Convert to a 3D transformation for eye.
	 * 
	 * @return jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#asEyeTransformation()
	 * @category converting
	 */
	public Jun3dTransformation asEyeTransformation() {
		return this.asParallelProjection().asEyeTransformation();
	}

	/**
	 * Project on a RenderingContext.
	 * 
	 * @param aRenderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @category projection
	 */
	public void projectOn_(JunOpenGLRenderingContext aRenderingContext) {
		aRenderingContext.perspective_(this);
	}

	/**
	 * Answer true if I am a perspective projection.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#isPerspective()
	 * @category testing
	 */
	public boolean isPerspective() {
		return true;
	}

	/**
	 * Pan the projection.
	 * 
	 * @param factor double
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#pan_(double)
	 * @category zooming
	 */
	public void pan_(double factor) {
		this.zoom_(1 / factor);
	}

	/**
	 * Zoom the projection
	 * 
	 * @param factor double
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#zoom_(double)
	 * @category zooming
	 */
	public void zoom_(double factor) {
		fovy = fovy.div_(factor);
		double min = 0.001;
		double max = 180 - min;
		if (fovy.deg() > max) {
			fovy = JunAngle.FromDeg_(max);
		}
		if (fovy.deg() < min) {
			fovy = JunAngle.FromDeg_(min);
		}
		zoomHeight = fovy.dividedBy_(2).tan() * this.distance() * 2;
	}

	/**
	 * Answer my current zoom height.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#zoomHeight()
	 * @category zooming
	 */
	public double zoomHeight() {
		if (Double.isNaN(zoomHeight)) {
			this.zoom_(1);
		}
		return zoomHeight;
	}

	/**
	 * Set my new zoom height.
	 * 
	 * @param aNumber double
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#zoomHeight_(double)
	 * @category zooming
	 */
	public void zoomHeight_(double aNumber) {
		this.setZoomHeight_(aNumber);
		fovy = JunAngle.FromRad_(Math.atan(this.zoomHeight() / 2 / this.distance()) * 2);
		double min = 0.001;
		double max = 180 - min;
		if (fovy.deg() > max) {
			fovy = JunAngle.FromDeg_(max);
		}
		if (fovy.deg() < min) {
			fovy = JunAngle.FromDeg_(min);
		}
	}

	/**
	 * Translate the Jun2dPoint with the depth to a Jun3dPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aNumber double
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#translateTo3dPointFromPoint_depth_(jp.co.sra.jun.geometry.basic.Jun2dPoint, double)
	 * @category utilities
	 */
	public Jun3dPoint translateTo3dPointFromPoint_depth_(Jun2dPoint aPoint, double aNumber) {
		double scale = aNumber * Math.abs(this.fovy().div_(2).tan());
		Jun3dPoint forward = this.unitSightVector().multipliedBy_(aNumber);
		Jun3dPoint right = this.unitRightVector().multipliedBy_(aPoint.x() * scale);
		Jun3dPoint up = this.unitUpVector().multipliedBy_(aPoint.y() * scale);
		return eyePoint.plus_(forward).plus_(right).plus_(up);
	}

	/**
	 * Set the parameters.
	 * 
	 * @param a3dPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param a3dPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param a3dPoint3 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJunAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @param aNumber1 double
	 * @param aNumber2 double
	 * @category private
	 */
	protected void setEyePoint_sightPoint_upVector_fovy_near_far_(Jun3dPoint a3dPoint1, Jun3dPoint a3dPoint2, Jun3dPoint a3dPoint3, JunAngle aJunAngle, double aNumber1, double aNumber2) {
		eyePoint = a3dPoint1;
		sightPoint = a3dPoint2;
		upVector = a3dPoint3;
		fovy = aJunAngle;
		near = aNumber1;
		far = aNumber2;
		this.normalizeUpVector();
	}

	/**
	 * Set the parameters.
	 * 
	 * @param a3dPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param a3dPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param a3dPoint3 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJunAngle double
	 * @param aNumber1 double
	 * @param aNumber2 double
	 * @category private
	 */
	protected void setEyePoint_sightPoint_upVector_fovy_near_far_(Jun3dPoint a3dPoint1, Jun3dPoint a3dPoint2, Jun3dPoint a3dPoint3, double aJunAngle, double aNumber1, double aNumber2) {
		this.setEyePoint_sightPoint_upVector_fovy_near_far_(a3dPoint1, a3dPoint2, a3dPoint3, JunAngle.FromDeg_(aJunAngle), aNumber1, aNumber2);
	}

	/**
	 * Set the zoom height in private.
	 * 
	 * @param aNumber double
	 * @see jp.co.sra.jun.opengl.projection.JunOpenGLProjection#setZoomHeight_(double)
	 * @category private
	 */
	protected void setZoomHeight_(double aNumber) {
		zoomHeight = Math.max(aNumber, 0);
	}

}
