package jp.co.sra.jun.opengl.objects;

import java.awt.Color;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.Writer;
import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;
import jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext;

/**
 * JunOpenGL3dVertex class
 * 
 *  @author    nisinaka
 *  @created   1998/10/21 (by nisinaka)
 *  @updated   2001/11/20 (by nisinaka)
 *  @updated   2004/05/14 (by nisinaka)
 *  @updated   2004/10/19 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun582 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGL3dVertex.java,v 8.11 2008/02/20 06:32:35 nisinaka Exp $
 */
public class JunOpenGL3dVertex extends JunOpenGL3dPrimitiveObject {
	/** The point which corresponds to the vertex. */
	protected Jun3dPoint point;

	/** The vertex size. */
	protected float size;

	/**
	 * Create a new instance of JunOpenGL3dVertex and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunOpenGL3dVertex() {
		super();
	}

	/**
	 * Create a new instance of JunOpenGL3dVertex and initialize it with the Jun3dPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category Instance creation
	 */
	public JunOpenGL3dVertex(Jun3dPoint aPoint) {
		this();
		this.point_(aPoint);
	}

	/**
	 * Create a new instance of JunOpenGL3dVertex and initialize it with the Jun3dPoint and the Color.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aColor java.awt.Color
	 * @category Instance creation
	 */
	public JunOpenGL3dVertex(Jun3dPoint aPoint, Color aColor) {
		this();
		this.point_(aPoint);
		this.paint_(aColor);
	}

	/**
	 * Create a new instance of JunOpenGL3dVertex and initialize it with the Jun3dPoint ,the Color and the alpha.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aColor java.awt.Color
	 * @param aNumber float
	 * @category Instance creation
	 */
	public JunOpenGL3dVertex(Jun3dPoint aPoint, Color aColor, float aNumber) {
		this();
		this.point_(aPoint);
		this.paint_(aColor);
		this.alpha_(aNumber);
	}

	/**
	 * Create a new instance of JunOpenGL3dObject and initialize it with the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunOpenGL3dVertex(JunLispList aList) {
		this();
		this.fromLispList(aList);
	}

	/**
	 * Create a new instance of JunOpenGL3dVertex and initialize it with the Jun3dPoint.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dVertex
	 * @category Instance creation
	 */
	public static JunOpenGL3dVertex Point_(Jun3dPoint aPoint) {
		return new JunOpenGL3dVertex(aPoint);
	}

	/**
	 * Create a new instance of JunOpenGL3dVertex and initialize it with the Jun3dPoint and the Color.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aColor java.awt.Color
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dVertex
	 * @category Instance creation
	 */
	public static JunOpenGL3dVertex Point_paint_(Jun3dPoint aPoint, Color aColor) {
		return new JunOpenGL3dVertex(aPoint, aColor);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		point = this.defaultPoint();
		size = this.defaultSize();
	}

	/**
	 * Answer my current point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public Jun3dPoint point() {
		return point;
	}

	/**
	 * Set my new point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	public void point_(Jun3dPoint aPoint) {
		point = aPoint;
		this.flushBounds();
	}

	/**
	 * Set my new point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public void point_(Jun2dPoint aPoint) {
		this.point_(new Jun3dPoint(aPoint.x(), aPoint.y(), 0));
	}

	/**
	 * Answer my current size.
	 * 
	 * @return float
	 * @category accessing
	 */
	public float size() {
		return size;
	}

	/**
	 * Set my new size.
	 * 
	 * @param aNumber float
	 * @category accessing
	 */
	public void size_(float aNumber) {
		size = aNumber;
	}

	/**
	 * Render the OpenGL object on a rendering context.
	 * 
	 * @param aRenderingContext jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#renderOn_(jp.co.sra.jun.opengl.support.JunOpenGLRenderingContext)
	 * @category rendering
	 */
	public void renderOn_(JunOpenGLRenderingContext aRenderingContext) {
		if (this.hasPaint()) {
			aRenderingContext.paint_(this.paint());
		}

		if (this.hasAlpha()) {
			aRenderingContext.alpha_(this.alpha());
		}

		aRenderingContext.pointSize_(this.size());
		aRenderingContext.displayPoint_(this.point());
	}

	/**
	 * Enumerate each geometries and evaluate the block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#geometriesDo_(jp.co.sra.smalltalk.StBlockClosure)
	 * @category enumerating
	 */
	public Object geometriesDo_(StBlockClosure aBlock) {
		return aBlock.value_(point);
	}

	/**
	 * Enumerate every points and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#pointsDo_(jp.co.sra.smalltalk.StBlockClosure)
	 * @category enumerating
	 */
	public void pointsDo_(StBlockClosure aBlock) {
		aBlock.value_(this.point());
	}

	/**
	 * Answer the new JunOpen3dObject transformed with aTransformation.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#transform_(jp.co.sra.jun.geometry.transformations.Jun3dTransformation)
	 * @category transforming
	 */
	public JunOpenGL3dObject transform_(Jun3dTransformation aTransformation) {
		JunOpenGL3dVertex transformedObject = (JunOpenGL3dVertex) this.copy();
		transformedObject.point_((Jun3dPoint) point.transform_(aTransformation));
		return transformedObject;
	}

	/**
	 * Anser the reversed JunOpenGL3dObject of the receiver.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#reversed()
	 * @category converting
	 */
	public JunOpenGL3dObject reversed() {
		return (JunOpenGL3dVertex) this.copy();
	}

	/**
	 * Answer the number of polygons.
	 * 
	 * @return int
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#numberOfPolygons()
	 * @category utilities
	 */
	public int numberOfPolygons() {
		return 0;
	}

	/**
	 * Answer the StSymbol which represents the kind of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#kindName()
	 * @category lisp support
	 */
	public StSymbol kindName() {
		return $("Spot");
	}

	/**
	 * Convert the receiver as JunLispCons.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#toLispList()
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = super.toLispList();
		list.add_(this.pointToLispList());
		list.add_(this.sizeToLispList());
		return list;
	}

	/**
	 * Convert the receiver's point as a JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList pointToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("point"));
		list.tail_(this.point());
		return list;
	}

	/**
	 * Convert the receiver's size as a JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList sizeToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("size"));
		list.tail_(new Float(this.size()));
		return list;
	}

	/**
	 * Get my attributes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#fromLispList(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	public void fromLispList(JunLispList aList) {
		super.fromLispList(aList);
		this.pointFromLispList(aList);
		this.sizeFromLispList(aList);
		this.transformationFromLispList(aList);
	}

	/**
	 * Get my point from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void pointFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("point")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.point_((Jun3dPoint) list.tail());
	}

	/**
	 * Get my size from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void sizeFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("size")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		this.size_(((Number) list.tail()).floatValue());
	}

	/**
	 * Write my VRML1.0 string on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws SmalltalkException DOCUMENT ME!
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#vrml10On_(java.io.Writer)
	 * @category vrml support
	 */
	public void vrml10On_(Writer aWriter) {
		try {
			BufferedWriter bw = (aWriter instanceof BufferedWriter) ? (BufferedWriter) aWriter : new BufferedWriter(aWriter);
			bw.write("DEF ");
			bw.write(this.legalName());
			bw.newLine();
			bw.write("Separator {");
			bw.newLine();
			this.vrml10ColorOn_(bw);
			this.vrml10PointOn_(bw);
			bw.write("} #Separator");
			bw.newLine();
			bw.flush();
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Write my point as VRML1.0 on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws SmalltalkException DOCUMENT ME!
	 * @category vrml support
	 */
	protected void vrml10PointOn_(Writer aWriter) {
		try {
			BufferedWriter bw = (aWriter instanceof BufferedWriter) ? (BufferedWriter) aWriter : new BufferedWriter(aWriter);
			bw.write("\t");
			bw.write("Coordinate3 {");
			bw.newLine();
			bw.write("\t\t");
			bw.write("point [");
			bw.newLine();

			Jun3dPoint point = this.point;
			bw.write("\t\t\t");
			bw.write(String.valueOf((float) point.x()));
			bw.write(" ");
			bw.write(String.valueOf((float) point.y()));
			bw.write(" ");
			bw.write(String.valueOf((float) point.z()));
			bw.write(" #1");
			bw.newLine();
			bw.write("\t\t");
			bw.write("] #point");
			bw.newLine();
			bw.write("\t");
			bw.write("} #Coordinate3");
			bw.newLine();
			bw.write("\t");
			bw.write(this.vrml10IndexedSetName());
			bw.write(" {");
			bw.newLine();
			bw.write("\t\t");
			bw.write("coordIndex [");
			bw.newLine();
			bw.write("\t\t\t");
			bw.write("1, -1");
			bw.newLine();
			bw.write("\t\t");
			bw.write("] #coordIndex");
			bw.newLine();
			bw.write("\t");
			bw.write("} #");
			bw.write(this.vrml10IndexedSetName());
			bw.newLine();
			bw.flush();
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Write my geoemtry as VRML2.0 to the writer.
	 * 
	 * @param pw java.io.Writer
	 * @param leader java.lang.String
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#vrml20GeometryOn_(java.io.PrintWriter, java.lang.String)
	 * @category vrml support
	 */
	protected void vrml20GeometryOn_(PrintWriter pw, String leader) {
		pw.println(leader + "geometry " + this.vrml20IndexedSetName() + " {");

		pw.println(leader + INDENT + "coord Coordinate {");
		pw.print(leader + INDENT + INDENT + "point [ ");
		pw.print((float) this.point().x());
		pw.print(' ');
		pw.print((float) this.point().y());
		pw.print(' ');
		pw.print((float) this.point().z());
		pw.println(" ]");
		pw.println(leader + INDENT + "} #coord");

		pw.println(leader + "} #geometry");
	}

	/**
	 * Answer the name of my indexed set node.
	 * 
	 * @return String
	 * @category vrml support
	 */
	protected String vrml10IndexedSetName() {
		return "PointSet";
	}

	/**
	 * Answer the name of my indexed set node.
	 * 
	 * @return String
	 * @category vrml support
	 */
	protected String vrml20IndexedSetName() {
		return "PointSet";
	}

}
