package jp.co.sra.jun.opengl.objects;

import java.awt.Color;
import java.io.PrintWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.ListIterator;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;
import jp.co.sra.jun.opengl.picking.JunOpenGLPickingObjects;

/**
 * JunOpenGL3dPatchedObject class
 * 
 *  @author    nisinaka
 *  @created   2004/06/15 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGL3dPatchedObject.java,v 8.13 2008/02/20 06:32:35 nisinaka Exp $
 */
public class JunOpenGL3dPatchedObject extends JunOpenGL3dCompoundObject implements JunOpenGLPickingObjects {

	protected ArrayList points;
	protected ArrayList colors;
	protected ArrayList normalVectors;
	protected ArrayList patches;

	public static class Vertex {
		int pointIndex;
		int colorIndex;
		int vectorIndex;

		/**
		 * Create a new instance of <code>Vertex</code> and initialize it.
		 * 
		 * @param pointIndex int
		 * @param colorIndex int
		 * @param vectorIndex int
		 * @category Instance creation
		 */
		public Vertex(int pointIndex, int colorIndex, int vectorIndex) {
			this.pointIndex = pointIndex;
			this.colorIndex = colorIndex;
			this.vectorIndex = vectorIndex;
		}

		/**
		 * Create a new instance of <code>Vertex</code> and initialize it.
		 * 
		 * @param pointIndex int
		 * @category Instance creation
		 */
		public Vertex(int pointIndex) {
			this(pointIndex, -1, -1);
		}

		/**
		 * Create a new instance of <code>Vertex</code> and initialize it.
		 * 
		 * @param aVertex jp.co.sra.jun.opengl.objects.JunOpenGL3dPatchedObject.Vertex
		 * @category Instance creation
		 */
		public Vertex(Vertex aVertex) {
			this(aVertex.pointIndex, aVertex.colorIndex, aVertex.vectorIndex);
		}
	}

	/**
	 * Create a new instance of JunOpenGL3dPatchedObject.
	 * 
	 * @category Instance creation
	 */
	public JunOpenGL3dPatchedObject() {
		super();
	}

	/**
	 * Create a new instance of JunOpenGL3dPatchedObject and initialize it with polygons.
	 * 
	 * @param anArrayOfPolygons jp.co.sra.jun.opengl.objects.JunOpenGL3dPolygon[]
	 * @category Instance creation
	 */
	public JunOpenGL3dPatchedObject(JunOpenGL3dPolygon[] anArrayOfPolygons) {
		super();
		for (int i = 0; i < anArrayOfPolygons.length; i++) {
			this.add_(anArrayOfPolygons[i]);
		}
	}

	/**
	 * Create a new instance of JunopenGL3dPatchedObject and initialize it with a lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunOpenGL3dPatchedObject(JunLispList aList) {
		super();
		this.fromLispList(aList);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		points = new ArrayList();
		colors = new ArrayList();
		normalVectors = new ArrayList();
		patches = new ArrayList();
	}

	/**
	 * Answer the point at the specified index.
	 * 
	 * @param index int
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	protected Jun3dPoint pointAt(int index) {
		return (Jun3dPoint) points.get(index);
	}

	/**
	 * Answer the index of the point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return int
	 * @category accessing
	 */
	protected int indexOfPoint(Jun3dPoint aPoint) {
		int index = points.indexOf(aPoint);
		if (index < 0) {
			index = points.size();
			points.add(aPoint);
		}
		return index;
	}

	/**
	 * Answer the color at the specified index.
	 * 
	 * @param index int
	 * @return java.awt.Color
	 * @category accessing
	 */
	protected Color colorAt(int index) {
		return (Color) colors.get(index);
	}

	/**
	 * Answer the index of the color.
	 * 
	 * @param aColor java.awt.Color
	 * @return int
	 * @category accessing
	 */
	protected int indexOfColor(Color aColor) {
		if (aColor == null) {
			return -1;
		}

		int index = colors.indexOf(aColor);
		if (index < 0) {
			index = colors.size();
			colors.add(aColor);
		}
		return index;
	}

	/**
	 * Answer the normal vector at the specified index.
	 * 
	 * @param index int
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category accessing
	 */
	protected Jun3dPoint normalVectorAt(int index) {
		return (Jun3dPoint) normalVectors.get(index);
	}

	/**
	 * Answer the index of the normal vector.
	 * 
	 * @param aVector jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return int
	 * @category accessing
	 */
	protected int indexOfNormalVector(Jun3dPoint aVector) {
		if (aVector == null) {
			return -1;
		}

		int index = normalVectors.indexOf(aVector);
		if (index < 0) {
			index = normalVectors.size();
			normalVectors.add(aVector);
		}
		return index;
	}

	/**
	 * Answer my current components as ArrayList.
	 * 
	 * @return java.util.ArrayList
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject#_components()
	 * @category accessing
	 */
	protected ArrayList _components() {
		if (components == null) {
			components = new ArrayList();
			this.flushBounds();

			for (int i = 0; i < patches.size(); i++) {
				Vertex[] patch = (Vertex[]) patches.get(i);
				Jun3dPoint[] anArrayOfPoints = new Jun3dPoint[patch.length];
				ArrayList aListOfColors = new ArrayList();
				ArrayList aListOfNormalVectors = new ArrayList();
				for (int j = 0; j < patch.length; j++) {
					anArrayOfPoints[j] = this.pointAt(patch[j].pointIndex);
					if (patch[j].colorIndex >= 0) {
						aListOfColors.add(this.colorAt(patch[j].colorIndex));
					}
					if (patch[j].vectorIndex >= 0) {
						aListOfNormalVectors.add(this.normalVectorAt(patch[j].vectorIndex));
					}
				}
				JunOpenGL3dPolygon polygon = new JunOpenGL3dPolygon(anArrayOfPoints);
				if (aListOfColors.isEmpty() == false) {
					polygon.colors_((Color[]) aListOfColors.toArray(new Color[aListOfColors.size()]));
				}
				if (aListOfNormalVectors.isEmpty() == false) {
					polygon.normalVectors_((Jun3dPoint[]) aListOfNormalVectors.toArray(new Jun3dPoint[aListOfNormalVectors.size()]));
				}

				components.add(polygon);
			}
		}
		return components;
	}

	/**
	 * Add a polygon.
	 * 
	 * @param aPolygon jp.co.sra.jun.opengl.objects.JunOpenGL3dPolygon
	 * @category adding
	 */
	public void add_(JunOpenGL3dPolygon aPolygon) {
		Jun3dPoint[] anArrayOfPoints = aPolygon.vertexes();
		Color[] anArrayOfColors = aPolygon.hasColors() ? aPolygon.colors() : null;
		Jun3dPoint[] anArrayOfVectors = aPolygon.hasNormalVectors() ? aPolygon.normalVectors() : null;

		Vertex[] patch = new Vertex[anArrayOfPoints.length];
		for (int i = 0; i < anArrayOfPoints.length; i++) {
			Jun3dPoint aPoint = anArrayOfPoints[i];
			Color aColor = null;
			if (anArrayOfColors != null && i < anArrayOfColors.length && anArrayOfColors[i] != null) {
				aColor = anArrayOfColors[i];
			}
			Jun3dPoint aVector = null;
			if (anArrayOfVectors != null && i < anArrayOfVectors.length && anArrayOfVectors[i] != null) {
				aVector = anArrayOfVectors[i];
			}
			patch[i] = new Vertex(this.indexOfPoint(aPoint), this.indexOfColor(aColor), this.indexOfNormalVector(aVector));
		}
		patches.add(patch);
		this.flushComponents();
	}

	/**
	 * Answer the new JunOpenGL3dCompoundObject transformed with aTransformation.
	 * 
	 * @param aTransformation jp.co.sra.jun.geometry.transformations.Jun3dTransformation
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#transform_(jp.co.sra.jun.geometry.transformations.Jun3dTransformation)
	 * @category transforming 
	 */
	public JunOpenGL3dObject transform_(Jun3dTransformation aTransformation) {
		JunOpenGL3dPatchedObject transformedPatchedObject = new JunOpenGL3dPatchedObject();
		this._copyInto(transformedPatchedObject);

		boolean isComposed = (components != null);

		JunOpenGL3dObject[] originalComponents = this.components();
		for (int i = 0; i < originalComponents.length; i++) {
			transformedPatchedObject.add_((JunOpenGL3dPolygon) (originalComponents[i].transform_(aTransformation)));
		}

		if (isComposed) {
			JunOpenGL3dObject[] transformedComponents = transformedPatchedObject.components();
			for (int i = 0; i < transformedComponents.length; i++) {
				originalComponents[i]._copyInto(transformedComponents[i]);
			}
		}

		return transformedPatchedObject;
	}

	/**
	 * Answer true if the receiver is a compound body, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject#isCompoundBody()
	 * @category testing
	 */
	public boolean isCompoundBody() {
		return false;
	}

	/**
	 * Answer true if the receiver is a patched body, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject#isPatchedBody()
	 * @category testing
	 */
	public boolean isPatchedBody() {
		return true;
	}

	/**
	 * Answer true if the receiver is a skinning body, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject#isSkinningBody()
	 * @category testing
	 */
	public boolean isSkinningBody() {
		return false;
	}

	/**
	 * Answer true if the receiver has colors, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean hasColors() {
		return !colors.isEmpty();
	}

	/**
	 * Answer true if the receiver has normal vectors, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean hasNormalVectors() {
		return !normalVectors.isEmpty();
	}

	/**
	 * Do an extra copy of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StObject
	 * @see jp.co.sra.smalltalk.StObject#postCopy()
	 * @category copying
	 */
	public StObject postCopy() {
		super.postCopy();
		points = (ArrayList) points.clone();
		colors = (ArrayList) colors.clone();
		normalVectors = (ArrayList) normalVectors.clone();
		patches = (ArrayList) patches.clone();
		return this;
	}

	/**
	 * Convert the receiver to a JunOpenGL3dCompoundObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject#asCompoundObject()
	 * @category converting
	 */
	public JunOpenGL3dCompoundObject asCompoundObject() {
		return new JunOpenGL3dCompoundObject(this.components());
	}

	/**
	 * Answer the reversed JunOpenGL3dObject of the receiver.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#reversed()
	 * @category converting
	 */
	public JunOpenGL3dObject reversed() {
		JunOpenGL3dPatchedObject patchedObject = (JunOpenGL3dPatchedObject) this.copy();
		patchedObject.patches = new ArrayList(patches.size());
		for (int i = 0; i < patches.size(); i++) {
			Vertex[] patch = (Vertex[]) patches.get(i);
			Vertex[] reversedPatch = new Vertex[patch.length];
			for (int j = 0; j < patch.length; j++) {
				reversedPatch[patch.length - j - 1] = new Vertex(patch[j]);
			}
			patchedObject.patches.add(reversedPatch);
		}
		return patchedObject;
	}

	/**
	 * Answer the number of polygons.
	 * 
	 * @return int
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#numberOfPolygons()
	 * @category utilities
	 */
	public int numberOfPolygons() {
		return patches.size();
	}

	/**
	 * Answer the number of primitives.
	 * 
	 * @return int
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#numberOfPrimitives()
	 * @category utilities
	 */
	public int numberOfPrimitives() {
		return patches.size();
	}

	/**
	 * Answer the StSymbol which represents the kind of the receiver.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#kindName()
	 * @category lisp support
	 */
	public StSymbol kindName() {
		return $("PatchedBody");
	}

	/**
	 * Convert the receiver as JunLispCons.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#toLispList()
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = this.lispCons();
		list.head_(this.kindName());

		if (this.hasName()) {
			list.add_(this.nameToLispList());
		}
		if (this.hasColor()) {
			list.add_(this.colorToLispList());
		}
		if (this.hasTexture()) {
			list.add_(this.textureToLispList());
		}
		list.add_(this.pointsToLispList());
		list.add_(this.patchesToLispList());
		if (this.hasColors()) {
			list.add_(this.colorsToLispList());
			list.add_(this.colorIndexesToLispList());
		}
		if (this.hasNormalVectors()) {
			list.add_(this.normalVectorsToLispList());
			list.add_(this.normalVectorIndexesToLispList());
		}

		return list;
	}

	/**
	 * Convert the receiver's points as JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList pointsToLispList() {
		return new JunLispCons($("points"), JunLispCons.List_(points));
	}

	/**
	 * Convert the receiver's patches as JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList patchesToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("patches"));
		for (int i = 0; i < patches.size(); i++) {
			Vertex[] patch = (Vertex[]) patches.get(i);
			int[] indexes = new int[patch.length];
			for (int j = 0; j < patch.length; j++) {
				indexes[j] = patch[j].pointIndex + 1;
			}
			list.add_(JunLispCons.List_(indexes));
		}
		return list;
	}

	/**
	 * Convert the receiver's colors as JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList colorsToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("colors"));
		for (int i = 0; i < colors.size(); i++) {
			Color aColor = this.colorAt(i);
			list.add_(JunLispCons.List_(new float[] { aColor.getRed() / 255f, aColor.getGreen() / 255f, aColor.getBlue() / 255f, aColor.getAlpha() / 255f }));
		}
		return list;
	}

	/**
	 * Convert the receiver's color indexes as JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList colorIndexesToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("colorIndexes"));
		for (int i = 0; i < patches.size(); i++) {
			Vertex[] patch = (Vertex[]) patches.get(i);
			ArrayList indexes = new ArrayList();
			for (int j = 0; j < patch.length; j++) {
				if (patch[j].colorIndex >= 0) {
					indexes.add(new Integer(patch[j].colorIndex + 1));
				}
			}
			list.add_(JunLispCons.List_(indexes));
		}
		return list;
	}

	/**
	 * Convert the receiver's normal vectors as JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList normalVectorsToLispList() {
		return new JunLispCons($("normalVectors"), JunLispCons.List_(normalVectors));
	}

	/**
	 * Convert the receiver's normal vector indexes as JunLispList.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected JunLispList normalVectorIndexesToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("normalVectorIndexes"));
		for (int i = 0; i < patches.size(); i++) {
			Vertex[] patch = (Vertex[]) patches.get(i);
			ArrayList indexes = new ArrayList();
			for (int j = 0; j < patch.length; j++) {
				if (patch[j].vectorIndex >= 0) {
					indexes.add(new Integer(patch[j].vectorIndex + 1));
				}
			}
			list.add_(JunLispCons.List_(indexes));
		}
		return list;
	}

	/**
	 * Get my attributes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#fromLispList(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	public void fromLispList(JunLispList aList) {
		this.nameFromLispList(aList);
		this.colorFromLispList(aList);
		this.textureFromLispList(aList);
		this.pointsFromLispList(aList);
		this.patchesFromLispList(aList);
		this.colorsFromLispList(aList);
		this.colorIndexesFromLispList(aList);
		this.normalVectorsFromLispList(aList);
		this.normalVectorIndexesFromLispList(aList);
		this.transformationFromLispList(aList);
	}

	/**
	 * Get my points from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void pointsFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("points")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		points = new ArrayList();
		for (Object obj = list.tail(); obj instanceof JunLispCons; obj = ((JunLispCons) obj).tail()) {
			Jun3dPoint aPoint = (Jun3dPoint) ((JunLispCons) obj).head();
			points.add(aPoint);
		}
		this.flushComponents();
	}

	/**
	 * Get my patches from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void patchesFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("patches")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		patches = new ArrayList();
		for (Object obj = list.tail(); obj instanceof JunLispCons; obj = ((JunLispCons) obj).tail()) {
			Object[] anArray = ((JunLispCons) ((JunLispCons) obj).head()).asArray();
			Vertex[] patch = new Vertex[anArray.length];
			for (int i = 0; i < patch.length; i++) {
				int pointIndex = ((Number) anArray[i]).intValue() - 1;
				patch[i] = new Vertex(pointIndex);
			}
			patches.add(patch);
		}
		this.flushComponents();
	}

	/**
	 * Get my colors from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void colorsFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("colors")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		colors = new ArrayList();
		Object[] anArray = ((JunLispCons) list.tail()).asArray();
		for (int i = 0; i < anArray.length; i++) {
			JunLispCons rgbList = (JunLispCons) anArray[i];
			float red = ((Number) rgbList.nth_(1)).floatValue();
			float green = ((Number) rgbList.nth_(2)).floatValue();
			float blue = ((Number) rgbList.nth_(3)).floatValue();
			float alpha = ((Number) rgbList.nth_(4)).floatValue();
			colors.add(new Color(red, green, blue, alpha));
		}
		this.flushComponents();
	}

	/**
	 * Get my color indexes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void colorIndexesFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("colorIndexes")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		if (patches == null || patches.isEmpty()) {
			return;
		}

		ListIterator iterator = patches.listIterator();
		for (Object obj = list.tail(); obj instanceof JunLispCons; obj = ((JunLispCons) obj).tail()) {
			Object[] anArray = ((JunLispCons) ((JunLispCons) obj).head()).asArray();
			Vertex[] patch = (Vertex[]) iterator.next();
			for (int i = 0; i < anArray.length; i++) {
				patch[i].colorIndex = ((Number) anArray[i]).intValue() - 1;
			}
		}
		this.flushComponents();
	}

	/**
	 * Get my normal vectors from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void normalVectorsFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("normalVectors")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		normalVectors = new ArrayList();
		for (Object obj = list.tail(); obj instanceof JunLispCons; obj = ((JunLispCons) obj).tail()) {
			Jun3dPoint aVector = (Jun3dPoint) ((JunLispCons) obj).head();
			normalVectors.add(aVector);
		}
		this.flushComponents();
	}

	/**
	 * Get my normal vector indexes from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void normalVectorIndexesFromLispList(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object anObject) {
				return new Boolean(anObject instanceof JunLispCons && (((JunLispCons) anObject).head() == $("normalVectorIndexes")));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}
		if (patches == null || patches.isEmpty()) {
			return;
		}

		ListIterator iterator = patches.listIterator();
		for (Object obj = list.tail(); obj instanceof JunLispCons; obj = ((JunLispCons) obj).tail()) {
			Object[] anArray = ((JunLispCons) ((JunLispCons) obj).head()).asArray();
			Vertex[] patch = (Vertex[]) iterator.next();
			for (int i = 0; i < anArray.length; i++) {
				patch[i].vectorIndex = ((Number) anArray[i]).intValue() - 1;
			}
		}
		this.flushComponents();
	}

	/**
	 * Write the VRML1.0 string of the receiver on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#vrml10On_(java.io.Writer)
	 * @category vrml support
	 */
	public void vrml10On_(Writer aWriter) {
		// Not supported.
		super.vrml10On_(aWriter);
	}

	/**
	 * Write the VRML2.0 string of the receiver on the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#vrml20On_(java.io.PrintWriter, java.lang.String)
	 * @category vrml support
	 */
	public void vrml20On_(PrintWriter pw, String leader) {
		pw.println(leader + "Transform {");

		pw.println(leader + INDENT + "children [");
		this.vrml20ShapeOn_(pw, leader + INDENT + INDENT);
		pw.println(leader + INDENT + "] # children");

		Jun3dBoundingBox box = this.boundingBox();
		Jun3dPoint center = (Jun3dPoint) box.center();
		pw.print(leader + INDENT + "bboxCenter ");
		pw.print((float) center.x());
		pw.print(' ');
		pw.print((float) center.y());
		pw.print(' ');
		pw.print((float) center.z());
		pw.println();
		pw.print(leader + INDENT + "bboxSize ");
		pw.print((float) box.width());
		pw.print(' ');
		pw.print((float) box.height());
		pw.print(' ');
		pw.print((float) box.depth());
		pw.println();

		pw.println(leader + "} #Transform");
		pw.flush();
	}

	/**
	 * Write my geoemtry as VRML2.0 to the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @see jp.co.sra.jun.opengl.objects.JunOpenGL3dObject#vrml20GeometryOn_(java.io.PrintWriter, java.lang.String)
	 * @category vrml support
	 */
	protected void vrml20GeometryOn_(PrintWriter pw, String leader) {
		pw.println(leader + "geometry " + this.vrml20IndexedSetName() + " {");

		this.vrml20CoordOn_(pw, leader + INDENT);
		this.vrml20CoordIndexOn_(pw, leader + INDENT);
		if (this.hasColors()) {
			this.vrml20ColorOn_(pw, leader + INDENT);
			this.vrml20ColorIndexOn_(pw, leader + INDENT);
		}
		if (this.hasNormalVectors()) {
			this.vrml20NormalOn_(pw, leader + INDENT);
			this.vrml20NormalIndexOn_(pw, leader + INDENT);
		}
		if (this.hasTexture() && this.texture().hasCoordinates()) {
			this.vrml20TexCoordOn_(pw, leader + INDENT);
			this.vrml20TexCoordIndexOn_(pw, leader + INDENT);
		}

		pw.println(leader + "} #" + this.vrml20IndexedSetName());
	}

	/**
	 * Write my coordinate as VRML2.0 on the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @category vrml support
	 */
	protected void vrml20CoordOn_(PrintWriter pw, String leader) {
		pw.println(leader + "coord Coordinate {");
		pw.println(leader + INDENT + "point [");

		for (int i = 0; i < points.size(); i++) {
			Jun3dPoint aPoint = (Jun3dPoint) this.pointAt(i);
			pw.print(leader + INDENT + INDENT);
			pw.print((float) aPoint.x());
			pw.print(' ');
			pw.print((float) aPoint.y());
			pw.print(' ');
			pw.print((float) aPoint.z());
			if (i < points.size() - 1) {
				pw.print(',');
			}
			pw.println(" #" + i);
		}

		pw.println(leader + INDENT + "] #point");
		pw.println(leader + "} #Coordinate");
	}

	/**
	 * Write my coordinate index as VRML2.0 on the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @category vrml support
	 */
	protected void vrml20CoordIndexOn_(PrintWriter pw, String leader) {
		pw.println(leader + "coordIndex [");

		for (int i = 0; i < patches.size(); i++) {
			pw.print(leader + INDENT);
			Vertex[] patch = (Vertex[]) patches.get(i);
			for (int j = 0; j < patch.length; j++) {
				pw.print(patch[j].pointIndex);
				pw.print(", ");
			}
			pw.println(-1);
		}

		pw.println(leader + "] #coordIndex");
	}

	/**
	 * Write my colors as VRML2.0 on the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @category vrml support
	 */
	protected void vrml20ColorOn_(PrintWriter pw, String leader) {
		pw.println(leader + "color Color {");
		pw.println(leader + INDENT + "color [");

		for (int i = 0; i < colors.size(); i++) {
			Color aColor = (Color) this.colorAt(i);
			pw.print(leader + INDENT + INDENT);
			pw.print((float) aColor.getRed() / 255f);
			pw.print(' ');
			pw.print((float) aColor.getGreen() / 255f);
			pw.print(' ');
			pw.print((float) aColor.getBlue() / 255f);
			if (i < colors.size() - 1) {
				pw.print(',');
			}
			pw.println(" #" + i);
		}

		pw.println(leader + INDENT + "] #color");
		pw.println(leader + "} #Color");
	}

	/**
	 * Write my color index as VRML2.0 on the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @category vrml support
	 */
	protected void vrml20ColorIndexOn_(PrintWriter pw, String leader) {
		pw.println(leader + "colorIndex [");

		for (int i = 0; i < patches.size(); i++) {
			pw.print(leader + INDENT);
			Vertex[] patch = (Vertex[]) patches.get(i);
			int index = -1;
			for (int j = 0; j < patch.length; j++) {
				if (patch[j].colorIndex >= 0) {
					index = patch[j].colorIndex;
				}
				pw.print(index);
				pw.print(", ");
			}
			pw.println(-1);
		}

		pw.println(leader + "] #colorIndex");
	}

	/**
	 * Write my normal vectors as VRML2.0 on the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @category vrml support
	 */
	protected void vrml20NormalOn_(PrintWriter pw, String leader) {
		pw.println(leader + "normal Normal {");
		pw.println(leader + INDENT + "vector [");

		for (int i = 0; i < normalVectors.size(); i++) {
			Jun3dPoint aVector = (Jun3dPoint) this.normalVectorAt(i);
			pw.print(leader + INDENT + INDENT);
			pw.print((float) aVector.x());
			pw.print(' ');
			pw.print((float) aVector.y());
			pw.print(' ');
			pw.print((float) aVector.z());
			if (i < normalVectors.size() - 1) {
				pw.print(',');
			}
			pw.println(" #" + i);
		}

		pw.println(leader + INDENT + "] #vector");
		pw.println(leader + "} #Normal");
	}

	/**
	 * Write my normal index as VRML2.0 on the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @category vrml support
	 */
	protected void vrml20NormalIndexOn_(PrintWriter pw, String leader) {
		pw.println(leader + "normalIndex [");

		for (int i = 0; i < patches.size(); i++) {
			pw.print(leader + INDENT);
			Vertex[] patch = (Vertex[]) patches.get(i);
			int index = -1;
			for (int j = 0; j < patch.length; j++) {
				if (patch[j].vectorIndex >= 0) {
					index = patch[j].vectorIndex;
				}
				pw.print(index);
				pw.print(", ");
			}
			pw.println(-1);
		}

		pw.println(leader + "] #normalIndex");
	}

	/**
	 * Write my texture coordinate as VRML2.0 on the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @category vrml support
	 */
	protected void vrml20TexCoordOn_(PrintWriter pw, String leader) {
		pw.println(leader + "texCoord TextureCoordinate {");
		pw.println(leader + INDENT + "point [");

		Jun2dPoint[] coordinates = this.texture().coordinates();
		for (int i = 0; i < coordinates.length; i++) {
			pw.print(leader + INDENT + INDENT);
			pw.print((float) coordinates[i].x());
			pw.print(' ');
			pw.print((float) coordinates[i].y());
			if (i < coordinates.length - 1) {
				pw.print(',');
			}
			pw.println(" #" + i);
		}

		pw.println(leader + INDENT + "] #point");
		pw.println(leader + "} #TextureCoordinate");
	}

	/**
	 * Write my texture coordinate index as VRML2.0 on the writer.
	 * 
	 * @param pw java.io.PrintWriter
	 * @param leader java.lang.String
	 * @category vrml support
	 */
	protected void vrml20TexCoordIndexOn_(PrintWriter pw, String leader) {
		pw.println(leader + "texCoordIndex [");

		int max = this.texture().coordinates().length;
		int index = 0;
		for (int i = 0; i < patches.size(); i++) {
			pw.print(leader + INDENT);
			Vertex[] patch = (Vertex[]) patches.get(i);
			for (int j = 0; j < patch.length; j++) {
				pw.print(index++);
				if (index >= max) {
					index = 0;
				}
				pw.print(", ");
			}
			pw.println(-1);
		}

		pw.println(leader + "] #texCoordIndex");
	}

}
