package jp.co.sra.jun.opengl.flux;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.WindowEvent;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.DateFormat;
import java.util.Date;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StImage;
import jp.co.sra.smalltalk.StInputState;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StMenu;
import jp.co.sra.smalltalk.menu.StMenuBar;
import jp.co.sra.smalltalk.menu.StMenuItem;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.geometry.surfaces.JunPlane;
import jp.co.sra.jun.goodies.animation.JunCartoonMovie;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.files.JunFileModel;
import jp.co.sra.jun.goodies.image.streams.JunImageStream;
import jp.co.sra.jun.goodies.image.streams.JunJpegImageStream;
import jp.co.sra.jun.goodies.movie.framework.JunMoviePlayer;
import jp.co.sra.jun.goodies.movie.support.JunImagesToMovie;
import jp.co.sra.jun.goodies.track.JunTrackerModel;
import jp.co.sra.jun.graphics.navigator.JunFileRequesterDialog;
import jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.projection.JunOpenGLProjection;
import jp.co.sra.jun.system.framework.JunDialog;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunOpenGLFluxModel class
 * 
 *  @author    MATSUDA Ryouichi
 *  @created   1998/11/25 (by MATSUDA Ryouichi)
 *  @updated   1999/06/25 (by nisinaka)
 *  @updated   2003/03/24 (by nisinaka)
 *  @updated   2005/03/02 (by nisinaka)
 *  @updated   2006/04/11 (by m-asada)
 *  @updated   2006/10/13 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun678 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGLFluxModel.java,v 8.17 2008/02/20 06:32:34 nisinaka Exp $
 */
public class JunOpenGLFluxModel extends JunOpenGLDisplayModel {

	protected JunOpenGLFluxObject fluxObject;
	protected JunTrackerModel trackerModel;
	protected int framesPerSecond;

	/**
	 * Create a new instance of JunOpenGLFluxModel and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunOpenGLFluxModel() {
		super();
	}

	/**
	 * Create a new instance of JunOpenGLFluxModel and initialize it with a flux object.
	 * 
	 * @param aFluxObject jp.co.sra.jun.opengl.flux.JunOpenGLFluxObject
	 * @category Instance creation
	 */
	public JunOpenGLFluxModel(JunOpenGLFluxObject aFluxObject) {
		this();
		this.fluxObject_(aFluxObject);
	}

	/**
	 * Initialize the JunOpenGLFluxModel.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		fluxObject = null;
		trackerModel = null;
		framesPerSecond = this.defaultFramesPerSecond();
	}

	/**
	 * Release the JunOpenGLFluxModel.
	 * 
	 * @see jp.co.sra.smalltalk.StObject#release()
	 * @category initialize-release
	 */
	public void release() {
		this.trackerModel().end();
	}

	/**
	 * Answer my current flux object.
	 * 
	 * @return jp.co.sra.jun.opengl.flux.JunOpenGLFluxObject
	 * @category accessing
	 */
	public JunOpenGLFluxObject fluxObject() {
		return fluxObject;
	}

	/**
	 * Set my new flux object.
	 * 
	 * @param aFluxObject jp.co.sra.jun.opengl.flux.JunOpenGLFluxObject
	 * @category accessing
	 */
	public void fluxObject_(JunOpenGLFluxObject aFluxObject) {
		fluxObject = aFluxObject;

		if (fluxObject.size() - 1 <= 0) {
			this.trackerModel().step_(0.0d);
		} else {
			this.trackerModel().step_(1.0d / (fluxObject.size() - 1));
		}
		this.trackerModel().first();
	}

	/**
	 * Answer my current number of frames per second.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int framesPerSecond() {
		return framesPerSecond;
	}

	/**
	 * Set my new number of frames per second.
	 * 
	 * @param aNumber int
	 * @category accessing
	 */
	public void framesPerSecond_(int aNumber) {
		framesPerSecond = aNumber;
	}

	/**
	 * Answer my current flux object as JunOpenGL3dCompoundObject.
	 * 
	 * @return jp.co.sra.jun.opengl.objects.JunOpenGL3dCompoundObject
	 * @category accessing
	 */
	public JunOpenGL3dCompoundObject compoundObject() {
		return this.fluxObject().compoundObject();
	}

	/**
	 * Answer the time for ticking.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int tickTime() {
		return Math.round(1000.0f / this.framesPerSecond());
	}

	/**
	 * Answer my tracker model.
	 * 
	 * @return jp.co.sra.jun.goodies.track.JunTrackerModel
	 * @category accessing
	 */
	public JunTrackerModel trackerModel() {
		if (trackerModel == null) {
			trackerModel = new JunTrackerModel();
			trackerModel.compute_(new StBlockClosure() {
				public Object value_(Object anObject) {
					final double n = ((Number) anObject).doubleValue();
					if (JunOpenGLFluxModel.this.fluxObject() != null) {
						JunOpenGLFluxModel.this.do_framesPerSecond_(new StBlockClosure() {
							public Object value() {
								JunOpenGLFluxModel.this.displayObject_(fluxObject.at_(n));
								JunOpenGLFluxModel.this.changed_($("object"));
								return null;
							}
						}, JunOpenGLFluxModel.this.framesPerSecond());
						Thread.yield();
					}
					return null;
				}
			});
		}

		return trackerModel;
	}

	/**
	 * Start the tracker model.
	 * 
	 * @category accessing
	 */
	public void start() {
		this.trackerModel().start();
	}

	/**
	 * End the tracking process.
	 * 
	 * @category accessing
	 */
	public void end() {
		this.trackerModel().end();
	}

	/**
	 * Answer the bounding box of the receiver.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dBoundingBox
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#boundingBox()
	 * @category bounds accessing
	 */
	public Jun3dBoundingBox boundingBox() {
		if (this.fluxObject() == null) {
			return Jun3dBoundingBox.Origin_corner_(new Jun3dPoint(0, 0, 0), new Jun3dPoint(0, 0, 0));
		}

		return this.fluxObject().boundingBox();
	}

	/**
	 * Enumerate the points and evaluate the Block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#pointsDo_(jp.co.sra.smalltalk.StBlockClosure)
	 * @category enumerating
	 */
	public void pointsDo_(final StBlockClosure aBlock) {
		if (this.fluxObject() == null) {
			super.pointsDo_(aBlock);
		} else {
			this.fluxObject().pointsDo_(aBlock);
		}
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Flux");
	}

	/**
	 * Invoked when a window is in the process of being closed.
	 * 
	 * @param e java.awt.event.WindowEvent
	 * @see jp.co.sra.smalltalk.StApplicationModel#noticeOfWindowClose(java.awt.event.WindowEvent)
	 * @category interface closing
	 */
	public void noticeOfWindowClose(WindowEvent e) {
		super.noticeOfWindowClose(e);
		this.release();
	}

	/**
	 * Update the indication of the file menu.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#updateFileMenuIndication()
	 * @category menu accessing
	 */
	public void updateFileMenuIndication() {
		super.updateFileMenuIndication();

		StMenu fileMenu = (StMenu) this._menuBar().atNameKey_($("fileMenu"));
		if (fileMenu == null) {
			return;
		}

		StMenuItem menuItem;
		boolean displayObjectIsNotEmpty = !this.isEmpty();

		// Save...
		menuItem = fileMenu.atNameKey_($("saveMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty);
		}

		// Save as images...
		menuItem = fileMenu.atNameKey_($("saveAsImagesMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty);
		}

		// Save as movie...
		menuItem = fileMenu.atNameKey_($("saveAsMovieMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty);
		}
	}

	/**
	 * Update the indication of the misc menu.
	 * 
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#updateMiscMenuIndication()
	 * @category menu accessing
	 */
	public void updateMiscMenuIndication() {
		super.updateMiscMenuIndication();

		StMenu miscMenu = (StMenu) this._menuBar().atNameKey_($("miscMenu"));
		if (miscMenu == null) {
			return;
		}

		StMenuItem menuItem;
		boolean displayObjectIsNotEmpty = !this.isEmpty();

		// Convert to animcation...
		menuItem = miscMenu.atNameKey_($("convertToAnimationMenu"));
		if (menuItem != null) {
			menuItem.beEnabled(displayObjectIsNotEmpty);
		}
	}

	/**
	 * Called by menu of 'Open...'
	 * 
	 * @category menu accessing
	 */
	public void openLFT() {
		this.openLFT10();
	}

	/**
	 * Open an LFT1.0 file and get a JunOpenGLFluxObject.
	 * 
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category menu messages
	 */
	public void openLFT10() {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("<1p> files", null, "LFT"), new String[] { "*.lft", "*.LFT" }) };
		File aFile = JunFileRequesterDialog.Request($String("Select an <1p> file.", null, "LFT"), fileTypes, fileTypes[0]);
		if (aFile == null) {
			return;
		}

		JunOpenGLFluxObject anObject;
		try {
			anObject = this.readFromLFT10_(aFile);
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}
		if (anObject == null) {
			return;
		}

		this.fluxObject_(anObject);
		this.resetView();

		if (this.showModel() != null) {
			this.showModel().resetView();
		}
	}

	/**
	 * Called by menu of 'Save...'
	 * 
	 * @category menu messages
	 */
	public void saveLFT() {
		this.saveLFT10();
	}

	/**
	 * Save the flux object to a file.
	 * 
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category menu messages
	 */
	public void saveLFT10() {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("<1p> files", null, "LFT"), new String[] { "*.lft", "*.LFT" }) };
		File file = JunFileRequesterDialog.RequestNewFile($String("Input an <1p> file.", null, "LFT"), new File(this.fluxObject().name() + ".lft"), fileTypes, fileTypes[0]);
		if (file == null) {
			return;
		}

		try {
			this.writeToLFT10_object_(file, this.fluxObject());
		} catch (IOException e) {
			throw new SmalltalkException(e);
		}
	}

	/**
	 * Save the receiver as a series of images.
	 * 
	 * @category menu messages
	 */
	public void saveAsImages() {
		if (this.fluxObject() == null) {
			return;
		}

		this.end();

		File directory = JunFileRequesterDialog.RequestNewDirectory($String("Input a directory name."), new File(this.displayObject().name()));
		if (directory == null) {
			return;
		}

		this.saveAsImagesTo_(directory);
	}

	/**
	 * Save the receiver as a series of images.
	 * 
	 * @param directory java.io.File
	 * @category menu messages
	 */
	protected void saveAsImagesTo_(File directory) {
		if (directory.exists() == false) {
			directory.mkdir();
		}
		Thread.yield();

		StSymbol loopCondition = this.trackerModel().loopCondition();
		double currentValue = this.trackerModel().doubleValue();
		this.trackerModel().loopCondition_($("oneWay"));
		JunCursors cursor = new JunCursors(JunCursors.ExecuteCursor());
		try {
			cursor._show();

			this.trackerModel().playButtonVisual_(true);
			this.trackerModel().first();
			for (int index = 1; this.trackerModel().playButton().value(); index++) {
				String aString = "0000" + index;
				aString = aString.substring(aString.length() - 4);
				File aFile = new File(directory, aString + ".jpg");
				StImage anImage = this.asImage();

				JunCursors writeCursor = new JunCursors(JunCursors.WriteCursor());
				JunImageStream aStream = null;
				try {
					writeCursor._show();
					aStream = JunJpegImageStream.On_(new FileOutputStream(aFile));
					aStream.nextPutImage_(anImage);
				} catch (IOException e) {
					throw new SmalltalkException(e);
				} finally {
					if (aStream != null) {
						try {
							aStream.flush();
							aStream.close();
						} catch (IOException e) {
						} finally {
							aStream = null;
						}
					}
					writeCursor._restore();
				}

				this.trackerModel().next();
				Thread.yield();
			}

			this.trackerModel().playButtonVisual_(false);
		} finally {
			this.trackerModel().loopCondition_(loopCondition);
			this.trackerModel().value_(currentValue);

			cursor._restore();
		}
	}

	/**
	 * Save the receiver as a movie.
	 * 
	 * @category menu messages
	 */
	public void saveAsMovie() {
		if (this.fluxObject() == null) {
			return;
		}

		this.end();

		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("<1p> files", null, $String("Movie")), new String[] { "*.mov", "*.MOV" }) };
		File file = JunFileRequesterDialog.RequestNewFile($String("Input a <1p> file.", null, $String("Movie")), new File(this.displayObject().name() + ".mov"), fileTypes, fileTypes[0]);
		if (file == null) {
			return;
		}

		if (this.saveAsMovieTo_(file)) {
			JunMoviePlayer player = new JunMoviePlayer(file);
			if (player != null) {
				player.open();
			}
		}
	}

	/**
	 * Save the receiver as a movie.
	 * 
	 * @param file java.io.File
	 * @category  menu messages
	 */
	protected boolean saveAsMovieTo_(File file) {
		if (file == null) {
			return false;
		}

		Thread.yield();

		StSymbol loopCondition = this.trackerModel().loopCondition();
		double currentValue = this.trackerModel().doubleValue();
		this.trackerModel().loopCondition_($("oneWay"));
		JunCursors cursor = new JunCursors(JunCursors.ExecuteCursor());
		try {
			cursor._show();

			this.trackerModel().playButtonVisual_(true);
			this.trackerModel().first();
			Dimension extent = this.defaultImageExtent();
			JunImagesToMovie.File_extent_do_(file, extent, new StBlockClosure() {
				public Object value_(Object imagesToMovie) {
					while (JunOpenGLFluxModel.this.trackerModel().playButton().value()) {
						StImage anImage = JunOpenGLFluxModel.this.asImage();
						((JunImagesToMovie) imagesToMovie).add_milliseconds_(anImage, tickTime());
						JunOpenGLFluxModel.this.trackerModel().next();
						Thread.yield();
					}
					return null;
				}
			});
			this.trackerModel().playButtonVisual_(false);
		} finally {
			this.trackerModel().loopCondition_(loopCondition);
			this.trackerModel().value_(currentValue);

			cursor._restore();
		}

		return true;
	}

	/**
	 * Called by menu of 'spawn'
	 * 
	 * @category menu messages
	 */
	public void spawnObject() {
		JunOpenGLDisplayModel displayModel = new JunOpenGLDisplayModel(this.spawningObject());
		displayModel.displayProjection_((JunOpenGLProjection) this.displayProjection().copy());

		StView view = this.getView();
		if (view == null) {
			displayModel.open();
		} else {
			StRectangle box = new StRectangle(view.topComponent().getBounds());
			StRectangle area = new StRectangle(0, 0, box.width(), box.height());
			area = area.align_with_(area.topLeft(), new Point(box.right() + 5, box.top()));
			displayModel.openIn_(area.toRectangle());
		}

		displayModel.changed_($("object"));
	}

	/**
	 * Convert the receiver to the animation.
	 * 
	 * @category menu messages
	 */
	public void convertToAnimation() {
		String aString = JunDialog.Request_($String("Input the interval milliseconds."), String.valueOf(100));
		if (aString == null || aString.length() == 0) {
			return;
		}

		try {
			int intervalMilliseconds = Integer.parseInt(aString);
			this.convertToAnimation_(intervalMilliseconds);
		} catch (NumberFormatException e) {
			JunDialog.Warn_(aString + $String(" is invalid value."));
			return;
		}
	}

	/**
	 * Convert the receiver to the animation.
	 * 
	 * @param intervalMilliseconds int
	 * @category menu messages
	 */
	protected void convertToAnimation_(int intervalMilliseconds) {
		JunCartoonMovie cartoonMovie = new JunCartoonMovie();
		int millisecondValue = Math.max(intervalMilliseconds, cartoonMovie.tickTime());
		File directory = new File(this.defaultBaseName());
		this.saveAsImagesTo_(directory);

		File[] files = directory.listFiles();
		try {
			for (int i = 0; i < files.length; i++) {
				JunImageStream aStream = null;
				try {
					aStream = JunJpegImageStream.On_(new FileInputStream(files[i]));
					StImage anImage = aStream.nextImage();
					cartoonMovie.addImage_keepTime_(anImage, millisecondValue);
				} finally {
					if (aStream != null) {
						aStream.close();
						aStream = null;
					}
				}
			}
		} catch (IOException e) {
			throw new SmalltalkException(e);
		} finally {
			for (int i = 0; i < files.length; i++) {
				files[i].delete();
			}
			directory.delete();
		}

		cartoonMovie.open();
	}

	/**
	 * Read a LFT1.0 file and create a flux object.
	 * 
	 * @param aFile java.io.File
	 * @return jp.co.sra.jun.opengl.flux.JunOpenGLFluxObject
	 * @exception IOException
	 * @category reading
	 */
	public JunOpenGLFluxObject readFromLFT10_(File aFile) throws IOException {
		FileReader aFileReader;
		try {
			aFileReader = new FileReader(aFile);
		} catch (FileNotFoundException e) {
			return null;
		}

		BufferedReader aReader = new BufferedReader(aFileReader);
		JunOpenGLFluxObject anObject = null;
		try {
			anObject = this.loadFromLFT10_(aReader);
		} finally {
			aReader.close();
		}

		return anObject;
	}

	/**
	 * Load a JunOpenGLFluxObject from the reader.
	 * 
	 * @param aReader java.io.BufferedReader
	 * @return jp.co.sra.jun.opengl.flux.JunOpenGLFluxObject
	 * @exception IOException
	 * @category reading
	 */
	public JunOpenGLFluxObject loadFromLFT10_(BufferedReader aReader) throws IOException {
		StringWriter aWriter = new StringWriter();
		JunOpenGLFluxObject anObject = null;
		try {
			int ch;
			while ((ch = aReader.read()) > 0) {
				aWriter.write(ch);
			}
			aWriter.flush();
			anObject = JunOpenGLFluxObject.LoadFrom_(aWriter.toString());
		} finally {
			aWriter.close();
		}

		return anObject;
	}

	/**
	 * Write the JunOpenGLFluxObject to the file.
	 * 
	 * @param aFile java.io.File
	 * @param anObject jp.co.sra.jun.opengl.flux.JunOpenGLFluxObject
	 * @exception java.io.IOException
	 * @category writing
	 */
	public void writeToLFT10_object_(File aFile, JunOpenGLFluxObject anObject) throws IOException {
		if (aFile == null) {
			return;
		}

		BufferedWriter aWriter = new BufferedWriter(new FileWriter(aFile));
		try {
			this.saveToLFT10_object_(aWriter, anObject);
		} finally {
			aWriter.flush();
			aWriter.close();
		}
	}

	/**
	 * Write the JunOpenGLFluxObject to the writer.
	 * 
	 * @param aWriter java.io.BufferedWriter
	 * @param anObject jp.co.sra.jun.opengl.flux.JunOpenGLFluxObject
	 * @exception java.io.IOException
	 * @category writing
	 */
	public void saveToLFT10_object_(BufferedWriter aWriter, JunOpenGLFluxObject anObject) throws IOException {
		aWriter.write(this.defaultStampForLFT10());
		anObject.saveOn_(aWriter);
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category defaults
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunOpenGLFluxViewAwt(this);
		} else {
			return new JunOpenGLFluxViewSwing(this);
		}
	}

	/**
	 * Answer the stamp string for the LFT 1.0 file format.
	 * 
	 * @return java.lang.String
	 * @category defaults
	 */
	public String defaultStampForLFT10() {
		StringWriter sw = new StringWriter();
		PrintWriter pw = new PrintWriter(sw);
		pw.println("% LFT V1.0 List Flux Transmission (Lisp S Expression)");
		pw.println("% This file was created by " + JunSystem.System() + JunSystem.Version());
		pw.println("% " + DateFormat.getInstance().format(new Date()));
		pw.println();
		pw.flush();
		return sw.toString();
	}

	/**
	 * Answer the default number of frames per second.
	 * 
	 * @return int
	 * @category defaults
	 */
	protected int defaultFramesPerSecond() {
		return 15;
	}

	/**
	 * Answer my menu bar.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenuBar
	 * @see jp.co.sra.smalltalk.StApplicationModel#_menuBar()
	 * @category resources
	 */
	public StMenuBar _menuBar() {
		if (_menuBar == null) {
			_menuBar = new StMenuBar();
			_menuBar.add(this._createFileMenu());
			_menuBar.add(this._createEditMenu());
			_menuBar.add(this._createViewMenu());
			_menuBar.add(this._createLightMenu());
			_menuBar.add(this._createMiscMenu());
		}
		return _menuBar;
	}

	/**
	 * Create a "File" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#_createFileMenu()
	 * @category resources
	 */
	protected StMenu _createFileMenu() {
		StMenu fileMenu = new StMenu($String("File"), $("fileMenu"));
		fileMenu.add(new StMenuItem($String("New"), new MenuPerformer(this, "newModel")));
		fileMenu.add(new StMenuItem($String("Open") + "...", new MenuPerformer(this, "openLFT")));
		fileMenu.addSeparator();
		fileMenu.add(new StMenuItem($String("Save") + "...", new MenuPerformer(this, "saveLFT")));
		fileMenu.add(new StMenuItem($String("Save as image..."), $("saveAsImageMenu"), new MenuPerformer(this, "saveAsImage")));
		fileMenu.add(new StMenuItem($String("Save as images..."), $("saveAsImagesMenu"), new MenuPerformer(this, "saveAsImages")));
		fileMenu.add(new StMenuItem($String("Save as movie..."), $("saveAsMovieMenu"), new MenuPerformer(this, "saveAsMovie")));
		fileMenu.addSeparator();
		fileMenu.add(new StMenuItem($String("Quit"), new MenuPerformer(this, "quitDoing")));
		return fileMenu;
	}

	/**
	 * Create a "Misc" menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenu
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#_createMiscMenu()
	 * @category resources
	 */
	protected StMenu _createMiscMenu() {
		StMenu miscMenu = new StMenu($String("Misc"), $("miscMenu"));
		miscMenu.add(new StMenuItem($String("Spawn"), $("spawnMenu"), new MenuPerformer(this, "spawnObject")));
		miscMenu.add(new StMenuItem($String("Viewport"), $("viewportMenu"), new MenuPerformer(this, "spawnViewport")));
		miscMenu.add(new StMenuItem($String("Bounds"), $("boundsMenu"), new MenuPerformer(this, "showBounds")));
		miscMenu.add(new StMenuItem($String("Convert to animation..."), $("convertToAnimationMenu"), new MenuPerformer(this, "convertToAnimation")));
		return miscMenu;
	}

	/**
	 * Compute the current sight point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#computeSightPoint()
	 * @category private
	 */
	protected Jun3dPoint computeSightPoint() {
		if (this.selectedObjects().isEmpty()) {
			if (StInputState.Default().altDown()) {
				return this.displayObject().boundingBox().center();
			} else {
				return this.fluxObject().boundingBox().center();
			}
		} else {
			Jun3dPoint center = new Jun3dPoint(0, 0, 0);
			JunOpenGL3dObject[] selectedObjects = (JunOpenGL3dObject[]) this.selectedObjects().toArray(new JunOpenGL3dObject[this.selectedObjects().size()]);
			for (int i = 0; i < selectedObjects.length; i++) {
				center = center.plus_(selectedObjects[i].boundingBox().center());
			}
			return center.dividedBy_(selectedObjects.length);
		}
	}

	/**
	 * Compute the current zoom height.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#computeZoomHeight()
	 * @category private
	 */
	protected double computeZoomHeight() {
		final double[] maxOffset = new double[2];
		double depth = this.displayProjection().distance();
		Jun3dPoint up = this.displayProjection().translateTo3dPointFromPoint_depth_(new Jun2dPoint(0, -1), depth);
		Jun3dPoint down = this.displayProjection().translateTo3dPointFromPoint_depth_(new Jun2dPoint(0, 1), depth);
		Jun3dPoint right = this.displayProjection().translateTo3dPointFromPoint_depth_(new Jun2dPoint(1, 0), depth);
		Jun3dPoint left = this.displayProjection().translateTo3dPointFromPoint_depth_(new Jun2dPoint(-1, 0), depth);
		final JunPlane horizontal = new JunPlane(this.displayProjection().eyePoint(), left, right);
		final JunPlane vertical = new JunPlane(this.displayProjection().eyePoint(), up, down);

		this.displayObject().pointsDo_(new StBlockClosure() {
			public Object value_(Object value) {
				Jun3dPoint p = (Jun3dPoint) value;
				maxOffset[0] = Math.max(maxOffset[0], vertical.distanceFromPoint_(p));
				maxOffset[1] = Math.max(maxOffset[1], horizontal.distanceFromPoint_(p));
				return null;
			}
		});

		double maxHeight = Math.max(maxOffset[0], maxOffset[1]);

		StView aView = this.getView();
		if (aView != null && maxOffset[0] != 0 && maxOffset[1] != 0) {
			Rectangle aBox = aView.toComponent().getBounds();
			if (aBox.height != 0) {
				if (maxOffset[1] * aBox.width / aBox.height < maxOffset[0]) {
					maxHeight = maxOffset[0] * aBox.height / aBox.width;
				} else {
					maxHeight = maxOffset[1];
				}
			}
		}

		maxHeight *= this.defaultZoomHeightFactor();
		return maxHeight;
	}

}
