package jp.co.sra.jun.opengl.display;

import java.awt.Color;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StBlockValue;
import jp.co.sra.smalltalk.StBlockValued;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.goodies.colors.JunColorChoiceHSB;
import jp.co.sra.jun.goodies.colors.JunColorChoiceModel;
import jp.co.sra.jun.opengl.lights.JunOpenGLAmbientLight;
import jp.co.sra.jun.opengl.lights.JunOpenGLLight;
import jp.co.sra.jun.opengl.lights.JunOpenGLParallelLight;
import jp.co.sra.jun.opengl.lights.JunOpenGLPointLight;
import jp.co.sra.jun.opengl.lights.JunOpenGLSpotLight;
import jp.co.sra.jun.system.framework.JunApplicationModel;
import jp.co.sra.jun.system.support.JunSmallCompiler;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunOpenGLDisplayLight class
 * 
 *  @author    He Weijie
 *  @created   1998/11/09 (by He Weijie)
 *  @updated   1999/05/12 (by nisinaka)
 *  @updated   2000/01/25 (by MATSUDA Ryouichi)
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGLDisplayLight.java,v 8.11 2008/02/20 06:32:19 nisinaka Exp $
 */
public class JunOpenGLDisplayLight extends JunApplicationModel implements StBlockValued {
	protected String lightName;
	protected StValueHolder lightState;
	protected JunColorChoiceModel lightColor;
	protected StValueHolder lightKind;
	protected StValueHolder lightPosition;
	protected StValueHolder lightDirection;
	protected StValueHolder lightCutoffAngle;
	protected StValueHolder coordinateKind;
	protected StValueHolder textValue;
	protected StValueHolder directionTextValue;
	protected StValueHolder cutoffAngleTextValue;

	/**
	 * Create a new instance of <code>JunOpenGLDisplayLight</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunOpenGLDisplayLight() {
		super();
	}

	/**
	 * Create a new instance of <code>JunOpenGLDisplayLight</code> as an ambient light.
	 * 
	 * @param booleanOnOff boolean
	 * @param colorValue java.awt.Color
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight
	 * @category Instance creation
	 */
	public static JunOpenGLDisplayLight AmbientLight_color_(boolean booleanOnOff, Color colorValue) {
		JunOpenGLDisplayLight displayLight = new JunOpenGLDisplayLight();
		displayLight.lightKind().value_($("ambient"));
		displayLight.lightState().value_(Boolean.valueOf(booleanOnOff));
		displayLight.lightColor().color_(colorValue);
		return displayLight;
	}

	/**
	 * Create a new instance of <code>JunOpenGLDisplayLight</code> as a parallel light.
	 * 
	 * @param booleanOnOff boolean
	 * @param colorValue java.awt.Color
	 * @param a3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight
	 * @category Instance creation
	 */
	public static JunOpenGLDisplayLight ParallelLight_color_direction_(boolean booleanOnOff, Color colorValue, Jun3dPoint a3dPoint) {
		JunOpenGLDisplayLight displayLight = new JunOpenGLDisplayLight();
		displayLight.lightKind().value_($("parallel"));
		displayLight.lightState().value_(Boolean.valueOf(booleanOnOff));
		displayLight.lightColor().color_(colorValue);
		displayLight.lightDirection().value_(a3dPoint);

		return displayLight;
	}

	/**
	 * Create a new instance of <code>JunOpenGLDisplayLight</code> as a parallel light.
	 * 
	 * @param booleanOnOff boolean
	 * @param colorValue java.awt.Color
	 * @param a3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight
	 * @category Instance creation
	 */
	public static JunOpenGLDisplayLight ParallelLight_color_position_(boolean booleanOnOff, Color colorValue, Jun3dPoint a3dPoint) {
		JunOpenGLDisplayLight displayLight = new JunOpenGLDisplayLight();
		displayLight.lightKind().value_($("parallel"));
		displayLight.lightState().value_(Boolean.valueOf(booleanOnOff));
		displayLight.lightColor().color_(colorValue);
		displayLight.lightPosition().value_(a3dPoint);

		return displayLight;
	}

	/**
	 * Create a new instance of <code>JunOpenGLDisplayLight</code> as a point light.
	 * 
	 * @param booleanOnOff boolean
	 * @param colorValue java.awt.Color
	 * @param a3dPoint jp.co.sra.jun.opengl.objects.Jun3dPoint
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight
	 * @category Instance creation
	 */
	public static JunOpenGLDisplayLight PointLight_color_position_(boolean booleanOnOff, Color colorValue, Jun3dPoint a3dPoint) {
		JunOpenGLDisplayLight displayLight = new JunOpenGLDisplayLight();
		displayLight.lightKind().value_($("point"));
		displayLight.lightState().value_(Boolean.valueOf(booleanOnOff));
		displayLight.lightColor().color_(colorValue);
		displayLight.lightPosition().value_(a3dPoint);

		return displayLight;
	}

	/**
	 * Create a new instance of <code>JunOpenGLDisplayLight</code> as a spot light.
	 * 
	 * @param booleanOnOff boolean
	 * @param colorValue java.awt.Color
	 * @param a3dPoint jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight
	 * @category Instance creation
	 */
	public static JunOpenGLDisplayLight SpotLight_color_position_(boolean booleanOnOff, Color colorValue, Jun3dPoint a3dPoint) {
		JunOpenGLDisplayLight displayLight = new JunOpenGLDisplayLight();
		displayLight.lightKind().value_($("spot"));
		displayLight.lightState().value_(Boolean.valueOf(booleanOnOff));
		displayLight.lightColor().color_(colorValue);
		displayLight.lightPosition().value_(a3dPoint);

		return displayLight;
	}

	/**
	 * Create a new instance of <code>JunOpenGLDisplayLight</code> as a spot light.
	 * 
	 * @param booleanOnOff boolean
	 * @param colorValue java.awt.Color
	 * @param a3dPoint1 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param a3dPoint2 jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @param aJunAngle jp.co.sra.jun.geometry.basic.JunAngle
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight
	 * @category Instance creation
	 */
	public static JunOpenGLDisplayLight SpotLight_color_position_direction_cutoffAngle_(boolean booleanOnOff, Color colorValue, Jun3dPoint a3dPoint1, Jun3dPoint a3dPoint2, JunAngle aJunAngle) {
		JunOpenGLDisplayLight displayLight = new JunOpenGLDisplayLight();
		displayLight.lightKind().value_($("spot"));
		displayLight.lightState().value_(Boolean.valueOf(booleanOnOff));
		displayLight.lightColor().color_(colorValue);
		displayLight.lightPosition().value_(a3dPoint1);
		displayLight.lightDirection().value_(a3dPoint2);
		displayLight.lightCutoffAngle().value_(aJunAngle);

		return displayLight;
	}

	/**
	 * Answer the kind of coordinate.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category accessing
	 */
	public StValueHolder coordinateKind() {
		if (coordinateKind == null) {
			coordinateKind = new StValueHolder(this.defaultCoordinateKind());

			final JunOpenGLDisplayLight self = this;
			coordinateKind.compute_(new StBlockClosure() {
				public Object value() {
					self.coordinateKindChanged();

					return null;
				}
			});
		}

		return coordinateKind;
	}

	/**
	 * Answer the current light.
	 * 
	 * @return jp.co.sra.jun.opengl.lights.JunOpenGLLight
	 * @category accessing
	 */
	public JunOpenGLLight light() {
		if (this.isOn()) {
			if (this.lightKind().value() == $("ambient")) {
				return JunOpenGLAmbientLight.Color_(this.lightColor().color());
			}

			if (this.lightKind().value() == $("parallel")) {
				return JunOpenGLParallelLight.Direction_color_((Jun3dPoint) this.lightDirection().value(), this.lightColor().color());
			}

			if (this.lightKind().value() == $("point")) {
				return JunOpenGLPointLight.At3dPoint_color_((Jun3dPoint) this.lightPosition().value(), this.lightColor().color());
			}

			if (this.lightKind().value() == $("spot")) {
				return JunOpenGLSpotLight.At3dPoint_direction_cutoffAngle_color_((Jun3dPoint) this.lightPosition().value(), (Jun3dPoint) this.lightDirection().value(), (JunAngle) this.lightCutoffAngle().value(), this.lightColor().color());
			}
		}

		return null;
	}

	/**
	 * Answer my color choice model of the light color.
	 * 
	 * @return jp.co.sra.jun.goodies.colors.JunColorChoiceModel
	 * @category accessing
	 */
	public JunColorChoiceModel lightColor() {
		if (lightColor == null) {
			lightColor = new JunColorChoiceHSB();
			lightColor.compute_(new StBlockClosure() {
				public Object value() {
					lightColorChanged();
					return null;
				}
			});
		}

		return lightColor;
	}

	/**
	 * Answer my value holdler of the light cutoff angle.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category accessing
	 */
	public StValueHolder lightCutoffAngle() {
		if (lightCutoffAngle == null) {
			lightCutoffAngle = new StValueHolder(this.defaultLightCutoffAngle());
			lightCutoffAngle.compute_(new StBlockClosure() {
				public Object value() {
					lightCutoffAngleChanged();
					return null;
				}
			});
		}

		return lightCutoffAngle;
	}

	/**
	 * Answer my value holder of the light direction.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category accessing
	 */
	public StValueHolder lightDirection() {
		if (lightDirection == null) {
			lightDirection = new StValueHolder(this.defaultLightDirection());
			lightDirection.compute_(new StBlockClosure() {
				public Object value() {
					lightDirectionChanged();
					return null;
				}
			});
		}

		return lightDirection;
	}

	/**
	 * Answer my value holder of the light kind.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category accessing
	 */
	public StValueHolder lightKind() {
		if (lightKind == null) {
			lightKind = new StValueHolder(this.defaultLightKind());
			lightKind.compute_(new StBlockClosure() {
				public Object value() {
					lightKindChanged();
					return null;
				}
			});
		}

		return lightKind;
	}

	/**
	 * Answer the name of the light.
	 * 
	 * @return java.lang.String
	 * @category accessing
	 */
	public String lightName() {
		if (lightName == null) {
			lightName = "Light";
		}

		return lightName;
	}

	/**
	 * Set the name of the light.
	 * 
	 * @param aString java.lang.String
	 * @category accessing
	 */
	public void lightName_(String aString) {
		lightName = aString;
	}

	/**
	 * Answer my value holder of the light position.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category accessing
	 */
	public StValueHolder lightPosition() {
		if (lightPosition == null) {
			lightPosition = new StValueHolder(this.defaultLightPosition());
			lightPosition.compute_(new StBlockClosure() {
				public Object value() {
					lightPositionChanged();
					return null;
				}
			});
		}

		return lightPosition;
	}

	/**
	 * Answer my value holder of the light state.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category accessing
	 */
	public StValueHolder lightState() {
		if (lightState == null) {
			lightState = new StValueHolder(this.defaultLightState());
			lightState.compute_(new StBlockClosure() {
				public Object value() {
					lightStateChanged();
					return null;
				}
			});
		}

		return lightState;
	}

	/**
	 * Answer my value.
	 *
	 * @return java.lang.Object
	 * @see jp.co.sra.smalltalk.StValued#value()
	 * @category accessing
	 */
	public Object value() {
		return this;
	}

	/**
	 * Answer the value holder for the cutoff angle text area.
	 *
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	public StValueHolder cutoffAngleTextValue() {
		if (cutoffAngleTextValue == null) {
			cutoffAngleTextValue = new StValueHolder("(" + (float) ((JunAngle) this.lightCutoffAngle().value()).deg() + ")");
		}
		return cutoffAngleTextValue;
	}

	/**
	 * Answer the value holder for the position text area.
	 *
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	public StValueHolder directionTextValue() {
		if (directionTextValue == null) {
			directionTextValue = new StValueHolder(((Jun3dPoint) this.lightDirection().value()).printString());
		}
		return directionTextValue;
	}

	/**
	 * Answer the value holder for the position text area.
	 *
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category aspects
	 */
	public StValueHolder textValue() {
		if (textValue == null) {
			textValue = new StValueHolder(((Jun3dPoint) this.lightPosition().value()).printString());
		}
		return textValue;
	}

	/**
	 * Called when the coordinate kind is changed.
	 * 
	 * @category changing
	 */
	protected void coordinateKindChanged() {
		this.shouldNotImplement();
	}

	/**
	 * Called when a text value for the direction is changed.
	 *
	 * @param aString java.lang.String
	 * @category changing
	 */
	protected void cutoffAngleTextValueChanged_(String aString) {
		try {
			Number number = (Number) JunSmallCompiler.Evaluate_(aString);
			this.lightCutoffAngle().value_(JunAngle.FromDeg_(number.doubleValue()));
		} catch (Exception e) {
		}
	}

	/**
	 * Called when a text value for the direction is changed.
	 *
	 * @param aString java.lang.String
	 * @category changing
	 */
	protected void directionTextValueChanged_(String aString) {
		try {
			Jun3dPoint point = (Jun3dPoint) JunSmallCompiler.Evaluate_(aString);
			this.lightDirection().value_(point);
		} catch (Exception e) {
		}
	}

	/**
	 * Called when the light color is changed.
	 * 
	 * @category changing
	 */
	protected void lightColorChanged() {
		if (this.isOn()) {
			this.changed_($("light"));
		}
	}

	/**
	 * Called when the cutoff angle is changed.
	 * 
	 * @category changing
	 */
	protected void lightCutoffAngleChanged() {
		if (this.isOn()) {
			this.changed_($("light"));
		}
	}

	/**
	 * Called when the light direction is changed.
	 * 
	 * @category changing
	 */
	protected void lightDirectionChanged() {
		if (this.isOn()) {
			this.changed_($("light"));
		}
	}

	/**
	 * Called when the kind of light is changed.
	 * 
	 * @category changing
	 */
	protected void lightKindChanged() {
		if (this.isOn()) {
			this.changed_($("light"));
		}
	}

	/**
	 * Called when the light position is changed.
	 * 
	 * @category changing
	 */
	protected void lightPositionChanged() {
		if (this.isOn()) {
			this.changed_($("light"));
		}
	}

	/**
	 * Called when the light state is changed.
	 * 
	 * @category changing
	 */
	protected void lightStateChanged() {
		this.changed_($("light"));
	}

	/**
	 * Called when a text value for the position is changed.
	 *
	 * @param aString java.lang.String
	 * @category changing
	 */
	protected void textValueChanged_(String aString) {
		try {
			Jun3dPoint point = (Jun3dPoint) JunSmallCompiler.Evaluate_(aString);
			this.lightPosition().value_(point);
		} catch (Exception e) {
		}
	}

	/**
	 * Answer a BlockValue that computes aBlock with the receiver's value  as
	 * the argument. aBlock will become a dependent of the receiver, and will
	 * be sent the message value: when the receiver is sent the message
	 * value:.
	 * 
	 * @return jp.co.sra.smalltalk.StBlockValue
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category constructing
	 */
	public StBlockValue compute_(StBlockClosure aBlock) {
		return new StBlockValue(aBlock, this);
	}

	/**
	 * Answer the default kind of coordinate.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category defaults
	 */
	public StSymbol defaultCoordinateKind() {
		return $("parallel");
	}

	/**
	 * Answer the default cutoff angle.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.JunAngle
	 * @category defaults
	 */
	public JunAngle defaultLightCutoffAngle() {
		return JunAngle.FromDeg_(30);
	}

	/**
	 * Answer the default light direction.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category defaults
	 */
	public Jun3dPoint defaultLightDirection() {
		return (Jun3dPoint) ((Jun3dPoint) this.lightPosition().value()).negated();
	}

	/**
	 * Answer the default kind of light.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category defaults
	 */
	public StSymbol defaultLightKind() {
		return $("parallel");
	}

	/**
	 * Answer the default light position.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category defaults
	 */
	public Jun3dPoint defaultLightPosition() {
		return new Jun3dPoint(0, 0, -100);
	}

	/**
	 * Answer the default light state.
	 * 
	 * @return java.lang.Boolean
	 * @category defaults
	 */
	public Boolean defaultLightState() {
		return Boolean.FALSE;
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunOpenGLDisplayLightViewAwt(this);
		} else {
			return new JunOpenGLDisplayLightViewSwing(this);
		}
	}

	/**
	 * Answer my default view for an ambient light.
	 *
	 * @return jp.co.sra.smalltalk.StView
	 * @category interface opening
	 */
	public StView defaultViewForAmbientLight() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return JunOpenGLDisplayLightViewAwt.ForAmbientLight(this);
		} else {
			return JunOpenGLDisplayLightViewSwing.ForAmbientLight(this);
		}
	}

	/**
	 * Answer my default view for a position light.
	 *
	 * @return jp.co.sra.smalltalk.StView
	 * @category interface opening
	 */
	public StView defaultViewForPositionLight() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return JunOpenGLDisplayLightViewAwt.ForPositionLight(this);
		} else {
			return JunOpenGLDisplayLightViewSwing.ForPositionLight(this);
		}
	}

	/**
	 * Answer the title string.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return JunSystem.$String("Light");
	}

	/**
	 * Open an ambient light view.
	 * 
	 * @category menu messages
	 */
	public void openAmbientLight() {
		this.openView_(this.defaultViewForAmbientLight());
	}

	/**
	 * Open parallel light view.
	 * 
	 * @category menu messages
	 */
	public void openParallelLight() {
		JunOpenGLDisplayLightView view = (JunOpenGLDisplayLightView) this.defaultView();
		//view.setLightView(JunOpenGLDisplayLightView.LIGHT_PARALLEL);
		this.openView_(view);
	}

	/**
	 * Open position light view.
	 * 
	 * @category menu messages
	 */
	public void openPositionLight() {
		this.openView_(this.defaultViewForPositionLight());
	}

	/**
	 * Make the light off.
	 * 
	 * @category switching
	 */
	public void beOff() {
		this.lightState().value_(Boolean.FALSE);
	}

	/**
	 * Make the light on.
	 * 
	 * @category switching
	 */
	public void beOn() {
		this.lightState().value_(Boolean.TRUE);
	}

	/**
	 * Answer true if the light is off, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isOff() {
		return this.isOn() == false;
	}

	/**
	 * Answer true if the light is on, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isOn() {
		return ((Boolean) this.lightState().value()).booleanValue();
	}

	/**
	 * Answer a window resizable.
	 * 
	 * @return boolean
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowResizable()
	 * @category testing
	 */
	protected boolean windowResizable() {
		return false;
	}
}
