package jp.co.sra.jun.opengl.display;

import java.awt.Color;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.event.MouseWheelEvent;
import java.awt.event.MouseWheelListener;
import java.util.Timer;
import java.util.TimerTask;

import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StView;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.boundaries.Jun2dBoundingBox;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.opengl.support.JunOpenGLDrawable;
import jp.co.sra.jun.system.framework.JunAbstractController;

/**
 * JunOpenGLDisplayController class
 * 
 *  @author    He Weijie
 *  @created   1998/10/08 (by r-matuda)
 *  @updated   1998/10/30 (by He Weijie)
 *  @updated   1999/08/11 (by nisinaka)
 *  @updated   2004/09/21 (by nisinaka)
 *  @updated   2007/04/23 (by Mitsuhiro Asada)
 *  @updated   2007/08/23 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun696 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGLDisplayController.java,v 8.19 2008/02/20 06:32:19 nisinaka Exp $
 */
public class JunOpenGLDisplayController extends JunAbstractController implements MouseListener, MouseMotionListener, MouseWheelListener {

	protected static final long _InertiaTime = 100;

	protected Jun2dPoint movementVector;
	protected Point _startPoint;
	protected Point _previousPoint;
	protected Timer _rotationTimer;
	protected long _pressTime;
	protected Rectangle _boundingBox;
	protected Cursor _savedCursor;

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StController#initialize()
	 * @category initialize-release
	 */
	public void initialize() {
		super.initialize();
		movementVector = null;
		_startPoint = null;
		_previousPoint = null;
		_rotationTimer = null;
		_pressTime = -1;
		_boundingBox = null;
		_savedCursor = null;
	}

	/**
	 * Remove references to objects that may refer to the receiver.
	 * Remove the listeners of the view here as well.
	 * 
	 * @see jp.co.sra.smalltalk.StController#release()
	 * @category initialize-release
	 */
	public void release() {
		Component drawable = this.getOpenGLDrawable().toComponent();
		drawable.removeMouseListener(this);
		drawable.removeMouseMotionListener(this);
		drawable.removeMouseWheelListener(this);

		this.flushMovementVector();

		super.release();
	}

	/**
	 * Add myself as a listener of the view.
	 * 
	 * @param aView jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StController#buildListener(jp.co.sra.smalltalk.StView)
	 * @category initialize-release
	 */
	protected void buildListener(StView newView) {
		Component drawable = this.getOpenGLDrawable().toComponent();
		drawable.addMouseListener(this);
		drawable.addMouseMotionListener(this);
		drawable.addMouseWheelListener(this);
	}

	/**
	 * Make sure to terminate the process when finalizing.
	 * 
	 * @throws java.lang.Throwable
	 * @see java.lang.Object#finalize()
	 * @category initialize-release
	 */
	protected void finalize() throws Throwable {
		super.finalize();
		view = null;
	}

	/**
	 * Answer my model as JunOpenGLDisplayModel.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel
	 * @category model accessing
	 */
	protected JunOpenGLDisplayModel getOpenGLDisplayModel() {
		return (JunOpenGLDisplayModel) this.model();
	}

	/**
	 * Answer my view as JunOpenGLDisplayView.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayView
	 * @category view accessing
	 */
	protected JunOpenGLDisplayView getOpenGLDisplayView() {
		return (JunOpenGLDisplayView) this.view();
	}

	/**
	 * Answer the JunOpenGLDrawable on my view.
	 * 
	 * @return jp.co.sra.jun.opengl.support.JunOpenGLDrawable
	 * @category view accessing
	 */
	protected JunOpenGLDrawable getOpenGLDrawable() {
		return this.getOpenGLDisplayView().getOpenGLDrawable();
	}

	/**
	 * Answer the JunOpenGLDrawable instead of JunOpenGLView.
	 * This method is only defined to suppress the error.
	 * 
	 * @return jp.co.sra.jun.opengl.support.JunOpenGLDrawable
	 * @deprecated since Jun683, the former JunOpenGL3dView is replaced with JunOpenGLDrawable.
	 * @category view accessing
	 */
	protected JunOpenGL3dView getOpenGLView() {
		return this.getOpenGLDisplayView().getOpenGLView();
	}

	/**
	 * The mouse button has been pressed.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @see jp.co.sra.jun.system.framework.JunAbstractController#mousePressed(java.awt.event.MouseEvent)
	 * @category mouse events
	 */
	public void mousePressed(MouseEvent e) {
		if (this.getOpenGLDisplayModel().noControl()) {
			return;
		}

		super.mousePressed(e);
	}

	/**
	 * The mouse button has been released.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @see jp.co.sra.jun.system.framework.JunAbstractController#mouseReleased(java.awt.event.MouseEvent)
	 * @category mouse events
	 */
	public void mouseReleased(MouseEvent e) {
		if (this.getOpenGLDisplayModel().noControl()) {
			return;
		}

		super.mouseReleased(e);
	}

	/**
	 * Invoked when a mouse is entered on a component.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @see jp.co.sra.smalltalk.StController#mouseEntered(java.awt.event.MouseEvent)
	 * @category mouse events
	 */
	public void mouseEntered(MouseEvent e) {
		StSymbol state = this.getOpenGLDisplayModel().buttonState();
		if (state == null) {
			return;
		}

		Component aComponent = this.getOpenGLDrawable().toComponent();
		_savedCursor = aComponent.getCursor();

		if (state == $("pick")) {
			aComponent.setCursor(JunCursors.CrossCursor());
		} else if (state == $("grab")) {
			aComponent.setCursor(JunCursors.HandCursor());
		} else if (state == $("drag")) {
			aComponent.setCursor(JunCursors.QuartersCursor());
		} else if (state == $("focus")) {
			aComponent.setCursor(JunCursors.GlassCursor());
		}
	}

	/**
	 * Invoked when a mouse is exited on a component.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @see java.awt.event.MouseListener#mouseExited(java.awt.event.MouseEvent)
	 * @category mouse events
	 */
	public void mouseExited(MouseEvent e) {
		this.getOpenGLDrawable().toComponent().setCursor(_savedCursor);
	}

	/**
	 * The General dragging action.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @see java.awt.event.MouseMotionListener#mouseDragged(java.awt.event.MouseEvent)
	 * @category mouse motion events
	 */
	public void mouseDragged(MouseEvent e) {
		if (_startPoint != null) {
			Point currentPoint = e.getPoint();
			if (currentPoint.equals(_previousPoint) == false) {
				this.mouseDraggedFrom_to_with_(_previousPoint, currentPoint, e);
				_previousPoint = currentPoint;
				_pressTime = System.currentTimeMillis();
			}
		}
		Thread.yield();
	}

	/**
	 * The mouse wheel is moved.
	 * 
	 * @see java.awt.event.MouseWheelListener#mouseWheelMoved(java.awt.event.MouseWheelEvent)
	 * @category mouse wheel events
	 */
	public void mouseWheelMoved(MouseWheelEvent e) {
		this.getOpenGLDisplayModel().wheel_at_in_(e.getUnitsToScroll(), e.getPoint(), this.getOpenGLDrawable().toComponent().getBounds());
	}

	/**
	 * Answer the current movement vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category movements
	 */
	public synchronized Jun2dPoint movementVector() {
		return movementVector;
	}

	/**
	 * Set the current movement vector.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category movements
	 */
	public synchronized void movementVector_(Jun2dPoint aPoint) {
		double x = Math.max(-1, Math.min(aPoint.x(), 1));
		double y = Math.max(-1, Math.min(aPoint.y(), 1));
		movementVector = new Jun2dPoint(x, y);
	}

	/**
	 * Flush the cached value of the movement vector.
	 * 
	 * @category movements
	 */
	public synchronized void flushMovementVector() {
		movementVector = null;
		Thread.yield();
	}

	/**
	 * Answer true if the receiver uses the movement process, otherwise false.
	 * 
	 * @return boolean
	 * @category movements
	 */
	public boolean useMovementProcess() {
		return this.getOpenGLDisplayModel().useMovementProcess();
	}

	/**
	 * Set whether the receiver uses the movement process or not.
	 * 
	 * @param aBoolean boolean
	 * @category movements
	 */
	public void useMovementProcess(boolean aBoolean) {
		this.getOpenGLDisplayModel().useMovementProcess_(aBoolean);
	}

	/**
	 * Initialize the rotation thread.
	 * 
	 * @category movements
	 */
	public void createMovementProcess() {
		if (this.useMovementProcess() == false) {
			return;
		}

		if (_rotationTimer != null) {
			_rotationTimer.cancel();
		}

		_rotationTimer = new Timer();
		_rotationTimer.scheduleAtFixedRate(new TimerTask() {
			public void run() {
				Jun2dPoint aVector = movementVector();
				if (aVector == null || aVector.isZero()) {
					_rotationTimer.cancel();
					_rotationTimer = null;
				} else {
					getOpenGLDisplayModel().grab_(aVector);
				}
			}
		}, 0, 83);
	}

	/**
	 * Activity for the red button pressed.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @see jp.co.sra.jun.system.framework.JunAbstractController#redButtonPressedActivity(java.awt.event.MouseEvent)
	 * @category control defaults
	 */
	protected void redButtonPressedActivity(MouseEvent e) {
		this.flushMovementVector();
		_startPoint = e.getPoint();
		_previousPoint = _startPoint;
		_pressTime = System.currentTimeMillis();

		// Handle picking here.
		// Others, for dragging, are handled at mouseDragged().
		if (this.getOpenGLDisplayModel().buttonState() == $("pick")) {
			this.pickActivityOn_with_(_startPoint, e);
		}
	}

	/**
	 * Activity for the red button released.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @see jp.co.sra.jun.system.framework.JunAbstractController#redButtonReleasedActivity(java.awt.event.MouseEvent)
	 * @category control defaults
	 */
	protected void redButtonReleasedActivity(MouseEvent e) {
		if (_startPoint != null) {
			StSymbol state = this.getOpenGLDisplayModel().buttonState();
			if (state == $("grab")) {
				if (System.currentTimeMillis() - _pressTime <= _InertiaTime) {
					if (this.movementVector() != null) {
						this.createMovementProcess();
					}
				}
			} else if (state == $("focus")) {
				Jun2dBoundingBox area1 = Jun2dBoundingBox.Origin_corner_(new Jun2dPoint(-1, -1), new Jun2dPoint(1, 1));
				Jun2dBoundingBox area2 = this.mouse2dRectangle();
				this._hideBoundingBox();

				if (e.isShiftDown()) {
					this.getOpenGLDisplayModel().focus_to_(area2, area1);
				} else {
					this.getOpenGLDisplayModel().focus_to_(area1, area2);
				}
			}

			_startPoint = null;
		}
	}

	/**
	 * Activity for the yellow button.
	 * 
	 * @param e java.awt.event.MouseEvent
	 * @category control defaults
	 */
	protected void yellowButtonActivity(MouseEvent e) {
		// self yellowButtonShiftActivity ifTrue: [^nil].
		this.flushMovementVector();
		if (this.getOpenGLDisplayModel().useYellowButtonMenu()) {
			this.view()._showPopupMenu(e.getX(), e.getY());
		}
	}

	/**
	 * Action for the mouse drag.
	 * 
	 * @param fromPoint java.awt.Point
	 * @param toPoint java.awt.Point
	 * @param event java.awt.event.MouseEvent
	 * @category control defaults
	 */
	protected void mouseDraggedFrom_to_with_(Point fromPoint, Point toPoint, MouseEvent event) {
		StSymbol state = this.getOpenGLDisplayModel().buttonState();
		if (state == $("grab")) {
			this.grabActivityFrom_to_(fromPoint, toPoint);
		} else if (state == $("drag")) {
			this.dragActivityFrom_to_with_(fromPoint, toPoint, event);
		} else if (state == $("focus")) {
			this.focusActivityFrom_to_(fromPoint, toPoint);
		} else if (this.getOpenGLDisplayModel().usePressActivity()) {
			this.pressActivityOn_with_(toPoint, event);
		}
	}

	/**
	 * Activity for the grab.
	 * 
	 * @param fromPoint java.awt.Point
	 * @param toPoint java.awt.Point
	 * @category control defaults
	 */
	protected void grabActivityFrom_to_(Point fromPoint, Point toPoint) {
		this.flushMovementVector();
		Jun2dPoint regularizedFromPoint = this.mouse2dPoint_(fromPoint);
		Jun2dPoint regularizedToPoint = this.mouse2dPoint_(toPoint);
		this.getOpenGLDisplayModel().grab_xy_(regularizedFromPoint, regularizedToPoint);
		this.movementVector_((Jun2dPoint) regularizedToPoint.minus_(regularizedFromPoint));
	}

	/**
	 * Activity for the drag.
	 * 
	 * @param fromPoint java.awt.Point
	 * @param toPoint java.awt.Point
	 * @param event java.awt.event.MouseEvent
	 * @category control defaults
	 */
	protected void dragActivityFrom_to_with_(Point fromPoint, Point toPoint, MouseEvent event) {
		Jun2dPoint regularizeFromPoint = this.mouse2dPoint_(fromPoint);
		Jun2dPoint regularizeToPoint = this.mouse2dPoint_(toPoint);

		if (event.isShiftDown()) {
			this.getOpenGLDisplayModel().look_xy_(regularizeFromPoint, regularizeToPoint);
		} else {
			this.getOpenGLDisplayModel().slide_xy_(regularizeFromPoint, regularizeToPoint);
		}
	}

	/**
	 * Activity for the focus.
	 * 
	 * @param fromPoint java.awt.Point
	 * @param toPoint java.awt.Point
	 * @category control defaults
	 */
	protected void focusActivityFrom_to_(Point fromPoint, Point toPoint) {
		int originX = Math.min(_startPoint.x, toPoint.x);
		int originY = Math.min(_startPoint.y, toPoint.y);
		int cornerX = Math.max(_startPoint.x, toPoint.x);
		int cornerY = Math.max(_startPoint.y, toPoint.y);
		Rectangle boundingBox = new Rectangle(originX, originY, cornerX - originX, cornerY - originY);
		this._showBoundingBox(boundingBox);
	}

	/**
	 * Activity for the pick.
	 * 
	 * @param aPoint java.awt.Point
	 * @param event java.awt.event.MouseEvent
	 * @category control defaults
	 */
	protected void pickActivityOn_with_(Point aPoint, MouseEvent event) {
		this.flushMovementVector();
		this.getOpenGLDisplayModel().pick_with_(this.mouse2dPoint_(aPoint), event);
	}

	/**
	 * Activity for the press.
	 * 
	 * @param aPoint java.awt.Point
	 * @param event java.awt.event.MouseEvent
	 * @category control defaults
	 */
	protected void pressActivityOn_with_(Point aPoint, MouseEvent event) {
		this.getOpenGLDisplayModel().press_with_(this.mouse2dPoint_(aPoint), event);
	}

	/**
	 * Answer the current bounding box for focusing.
	 * 
	 * @return java.awt.Rectangle
	 * @category bounds accessing
	 */
	protected Rectangle _getBoundingBox() {
		return _boundingBox;
	}

	/**
	 * Show bounding box.
	 * 
	 * @param aRectangle java.awt.Rectangle
	 * @category bounds accessing
	 */
	protected void _showBoundingBox(Rectangle aRectangle) {
		if (_boundingBox != null) {
			this._hideBoundingBox();
		}

		_boundingBox = aRectangle;

		Graphics aGraphics = null;
		try {
			aGraphics = this.getOpenGLDrawable().toComponent().getGraphics();
			aGraphics.setColor(Color.white);
			aGraphics.setXORMode(Color.black);
			aGraphics.drawRect(_boundingBox.x, _boundingBox.y, _boundingBox.width, _boundingBox.height);
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
			}
		}
	}

	/**
	 * Hide the bounding box if exist.
	 * 
	 * @category bounds accessing
	 */
	protected void _hideBoundingBox() {
		if (_boundingBox == null) {
			return;
		}

		Graphics aGraphics = null;
		try {
			aGraphics = this.getOpenGLDrawable().toComponent().getGraphics();
			aGraphics.setColor(Color.white);
			aGraphics.setXORMode(Color.black);
			aGraphics.drawRect(_boundingBox.x, _boundingBox.y, _boundingBox.width, _boundingBox.height);
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
			}
		}

		_boundingBox = null;
	}

	/**
	 * Answer the new focus point.
	 * 
	 * @param aPoint java.awt.Point
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category private
	 */
	protected Jun2dPoint mouse2dPoint_(Point aPoint) {
		return Jun2dPoint.Coerce_(this.regularizePoint_(aPoint));
	}

	/**
	 * Answer the new focus area.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dBoundingBox
	 * @category private
	 */
	protected Jun2dBoundingBox mouse2dRectangle() {
		Rectangle boundingBox = this._getBoundingBox();
		if (boundingBox == null) {
			boundingBox = new Rectangle(_startPoint.x, _startPoint.y, 0, 0);
		} else {
			boundingBox = new Rectangle(boundingBox);
		}

		if (boundingBox.height <= 2) {
			int expandSize = Math.max(this.getOpenGLDrawable().toComponent().getBounds().height / 4, 8);
			boundingBox.grow(expandSize, expandSize);
		}

		Jun2dPoint originPoint = this.regularizePoint_(new Point(boundingBox.x, boundingBox.y));
		Jun2dPoint cornerPoint = this.regularizePoint_(new Point(boundingBox.x + boundingBox.width, boundingBox.y + boundingBox.height));
		return Jun2dBoundingBox.Origin_corner_(originPoint, cornerPoint);
	}

	/**
	 * Regularize the point on the current view.
	 * 
	 * @param aPoint java.awt.Point
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category private
	 */
	protected Jun2dPoint regularizePoint_(Point aPoint) {
		Dimension bounds = this.getOpenGLDrawable().toComponent().getSize();
		double x = (aPoint.x - bounds.width / 2.0) / (bounds.height / 2.0);
		double y = -(aPoint.y - bounds.height / 2.0) / (bounds.height / 2.0);
		return new Jun2dPoint(x, y);
	}

}
