package jp.co.sra.jun.opengl.cosmos;

import java.awt.Color;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StView;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.Jun3dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.boundaries.Jun3dBoundingBox;
import jp.co.sra.jun.geometry.curves.Jun3dLine;
import jp.co.sra.jun.geometry.surfaces.JunSphereSurface;
import jp.co.sra.jun.geometry.transformations.Jun3dTransformation;
import jp.co.sra.jun.goodies.button.JunButtonModel;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight;
import jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel;
import jp.co.sra.jun.opengl.display.JunOpenGLShowModel;
import jp.co.sra.jun.opengl.objects.JunOpenGL3dObject;
import jp.co.sra.jun.opengl.projection.JunOpenGLProjection;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunOpenGLMicrocosmModel class
 * 
 *  @author    nisinaka
 *  @created   1999/08/17 (by nisinaka)
 *  @updated   2005/03/04 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun465 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunOpenGLMicrocosmModel.java,v 8.13 2008/02/20 06:32:18 nisinaka Exp $
 */
public class JunOpenGLMicrocosmModel extends JunOpenGLDisplayModel {

	/**
	 * Answer my show model.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLShowModel
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#showModel()
	 * @category accessing
	 */
	public JunOpenGLShowModel showModel() {
		JunOpenGLShowModel showModel = super.showModel();
		if (showModel == null) {
			showModel = new JunOpenGLShowModel(this) {
				public Jun3dPoint defaultEyePoint() {
					if (this.defaultProjectionTable().get($("eyePoint")) == null) {
						this.defaultEyePoint_(JunOpenGLProjection.DefaultEyePoint);
					}
					return (Jun3dPoint) this.defaultProjectionTable().get($("eyePoint"));
				}

				public void defaultEyePoint_(Jun3dPoint eyePoint) {
					this.defaultProjectionTable().put($("eyePoint"), eyePoint);
				}
			};
		}
		return showModel;
	}

	/**
	 * Answer the default light color.
	 * 
	 * @return java.awt.Color
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#defaultLightColor()
	 * @category defaults
	 */
	public Color defaultLightColor() {
		return Color.getHSBColor(0f, 0f, 0.75f);
	}

	/**
	 * Answer the default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#defaultView()
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunOpenGLMicrocosmViewAwt(this);
		} else {
			return new JunOpenGLMicrocosmViewSwing(this);
		}
	}

	/**
	 * Answer the array of all display lights.
	 * 
	 * @return jp.co.sra.jun.opengl.display.JunOpenGLDisplayLight[]
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#displayLights()
	 * @category lighting
	 */
	public JunOpenGLDisplayLight[] displayLights() {
		if (displayLights == null) {
			displayLights = new JunOpenGLDisplayLight[5];
			displayLights[0] = JunOpenGLDisplayLight.ParallelLight_color_position_(true, this.defaultLightColor(), this.defaultLightPoint());
			displayLights[1] = JunOpenGLDisplayLight.PointLight_color_position_(true, Color.white, new Jun3dPoint(0, 0, 0));
			displayLights[2] = new JunOpenGLDisplayLight();
			displayLights[3] = new JunOpenGLDisplayLight();
			displayLights[4] = JunOpenGLDisplayLight.AmbientLight_color_(true, this.defaultLightColor());
			for (int i = 0; i < displayLights.length; i++) {
				displayLights[i].compute_(new StBlockClosure() {
					public Object value() {
						updateLightMenuIndication();
						changed_($("light"));
						return null;
					}
				});
			}
		}
		return displayLights;
	}

	/**
	 * Set my new display object.
	 * 
	 * @param a3dObject jp.co.sra.jun.opengl.objects.JunOpenGL3dObject
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#displayObject_(jp.co.sra.jun.opengl.objects.JunOpenGL3dObject)
	 * @category accessing
	 */
	public void displayObject_(JunOpenGL3dObject a3dObject) {
		super.displayObject_(a3dObject);

		if (this.defaultProjectionTable().containsKey($("eyePoint")) == false) {
			this.eyePoint_(this.computeEyePoint());
		}
		if (this.defaultProjectionTable().containsKey($("sightPoint")) == false) {
			this.sightPoint_(this.computeSightPoint());
		}
		if (this.defaultProjectionTable().containsKey($("upVector")) == false) {
			this.upVector_(this.computeUpVector());
		}
		if (this.defaultProjectionTable().containsKey($("viewFactor")) == false) {
			this.viewFactor_(this.computeViewFactor());
		}
		if (this.defaultProjectionTable().containsKey($("zoomHeight")) == false) {
			this.zoomHeight_(this.computeZoomHeight());
		}
	}

	/**
	 * Grab and rotate the 3d object.
	 * 
	 * @param from2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param to2dPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#grab_xy_(jp.co.sra.jun.geometry.basic.Jun2dPoint, jp.co.sra.jun.geometry.basic.Jun2dPoint)
	 * @category manipulating
	 */
	public void grab_xy_(Jun2dPoint from2dPoint, Jun2dPoint to2dPoint) {
		if (this.displayObject() == null) {
			return;
		}

		JunOpenGLProjection aProjection = this.displayProjection();
		JunSphereSurface grabSphere = this.grabSphere();
		grabSphere.center_(aProjection.eyePoint());
		Jun3dPoint fromPoint = this.grab3dPoint_(new Jun2dPoint(from2dPoint.x(), from2dPoint.y()));
		Jun3dPoint toPoint = this.grab3dPoint_(new Jun2dPoint(to2dPoint.x(), to2dPoint.y()));
		JunAngle rotationAngle = (new Jun3dLine(grabSphere.center(), toPoint)).angleWithLine_(new Jun3dLine(grabSphere.center(), fromPoint));
		if (Math.abs(rotationAngle.rad()) > ThetaAccuracy) {
			Jun3dLine rotationAxis = new Jun3dLine(grabSphere.center(), (Jun3dPoint) grabSphere.center().minus_(((Jun3dPoint) fromPoint.minus_(grabSphere.center())).product_((Jun3dPoint) toPoint.minus_(grabSphere.center()))));
			Jun3dTransformation transformation = Jun3dTransformation.Rotate_around_(rotationAngle, rotationAxis);
			Jun3dTransformation transformationInv = Jun3dTransformation.Rotate_around_(rotationAngle.mul_(-1), rotationAxis);
			Jun3dPoint upPoint = (Jun3dPoint) aProjection.eyePoint().plus_(aProjection.unitUpVector());
			aProjection.sightPoint_(transformationInv.applyTo_(aProjection.sightPoint()));
			aProjection.upVector_((Jun3dPoint) transformation.applyTo_(upPoint).minus_(aProjection.sightPoint()));
			this.displayProjection_(aProjection);
			this.changed_($("projection"));
		}
	}

	/**
	 * Answer my grab button model.
	 * 
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#grabButton()
	 * @category buttons
	 */
	public JunButtonModel grabButton() {
		if (this.pushButtons().containsKey($("grab")) == false) {
			JunButtonModel button = new JunButtonModel();
			button.value_(false);
			button.visual_(JunCursors.ReadCursorImage());
			button.action_(new StBlockClosure() {
				public Object value_(Object o) {
					JunButtonModel model = (JunButtonModel) o;
					JunOpenGLMicrocosmModel.this.setButtonState($("grab"), !model.value());
					return model;
				}
			});
			this.pushButtons().put($("grab"), button);
		}

		return (JunButtonModel) this.pushButtons().get($("grab"));
	}

	/**
	 * Compute the current eye point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	protected Jun3dPoint computeEyePoint() {
		return (Jun3dPoint) this.boundingBox().center();
	}

	/**
	 * Compute the current sight point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @see jp.co.sra.jun.opengl.display.JunOpenGLDisplayModel#computeSightPoint()
	 * @category private
	 */
	protected Jun3dPoint computeSightPoint() {
		Jun3dBoundingBox box = this.boundingBox();
		double z = Math.min(box.width(), Math.min(box.height(), box.depth())) * 2;
		return (Jun3dPoint) box.center().minus_(new Jun3dPoint(0, 0, z));
	}

	/**
	 * Compute the current up vector.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun3dPoint
	 * @category private
	 */
	protected Jun3dPoint computeUpVector() {
		return new Jun3dPoint(0, 1, 0);
	}

	/**
	 * Compute the current view factor.
	 * 
	 * @return double
	 * @category private
	 */
	protected double computeViewFactor() {
		return 25;
	}

	/**
	 * Compute the current zoom height.
	 * 
	 * @return double
	 * @see jp.co.sra.jun.opengl.display.JunOpenGL3dModel#computeZoomHeight()
	 * @category private
	 */
	protected double computeZoomHeight() {
		Jun3dBoundingBox box = this.boundingBox();
		return Math.min(box.width(), Math.min(box.height(), box.depth())) * 2.5;
	}

	/**
	 * Answer the window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return JunSystem.$String("Microcosm");
	}
}
