package jp.co.sra.jun.graphics.list;

import java.awt.Cursor;
import java.awt.Font;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.io.File;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.MalformedURLException;
import java.net.URL;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StBlockValue;
import jp.co.sra.smalltalk.StComposedText;
import jp.co.sra.smalltalk.StOpaqueImage;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StValueHolder;
import jp.co.sra.smalltalk.SystemResourceSupport;
import jp.co.sra.smalltalk.menu.MenuDispatcher;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StMenuItem;
import jp.co.sra.smalltalk.menu.StPopupMenu;

import jp.co.sra.jun.goodies.calendar.JunCalendarModel;
import jp.co.sra.jun.goodies.cursors.JunCursors;
import jp.co.sra.jun.goodies.icon.JunOpaqueImageIcons;
import jp.co.sra.jun.goodies.tips.JunURL;
import jp.co.sra.jun.goodies.utilities.JunFileUtility;
import jp.co.sra.jun.goodies.utilities.JunStringUtility;
import jp.co.sra.jun.graphics.abstracts.JunAbstractItem;
import jp.co.sra.jun.graphics.abstracts.JunAbstractItem.LookPreferences;
import jp.co.sra.jun.graphics.framework.JunGraphicView;
import jp.co.sra.jun.graphics.item.JunCompositeItem;
import jp.co.sra.jun.graphics.item.JunPrimitiveItem;
import jp.co.sra.jun.graphics.map.JunSequenceMap;
import jp.co.sra.jun.graphics.navigator.JunFileListDispenser;
import jp.co.sra.jun.graphics.navigator.JunFileNavigator;

/**
 * JunFileList class
 * 
 *  @author    nisinaka
 *  @created   2004/01/09 (by nisinaka)
 *  @updated   2006/11/22 (by m-asada)
 *  @updated   2007/03/14 (by m-asada)
 *  @version   699 (with StPL8.9) based on Jun690 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunFileList.java,v 8.21 2008/02/20 06:32:15 nisinaka Exp $
 */
public class JunFileList extends JunSequenceList {
	protected File currentFile;
	protected long lastModified;
	protected String[] filePatterns;
	protected boolean showContents;
	protected File[] currentContents;
	protected StSymbol sortKey;
	protected boolean distinctFolder;
	protected StValueHolder selectedIndexHolder;
	protected StBlockClosure nullItemBlock;
	protected StBlockClosure fileItemBlock;
	protected StBlockClosure directoryItemBlock;
	protected LookPreferences lookPreferences;
	protected StPopupMenu yellowButtonMenuForFile;
	protected StBlockClosure selectBlock;
	protected KeyListener _keyListener;

	public static final File _RootFile = new File(File.separator);
	protected static File[] _RootFiles = null;
	protected static HashMap _ComparatorTable;

	/**
	 * Create a new instance of JunFileList and initialize it.
	 * 
	 * @param aFile java.io.File
	 * @category Instance creation
	 */
	public JunFileList(File aFile) {
		super();
		this.currentFile_(aFile);
	}

	/**
	 * Create a new instance of JunFileList of the root file system.
	 * 
	 * @return jp.co.sra.jun.graphics.list.JunFileList
	 * @category Instance creation
	 */
	public static JunFileList Root() {
		return new JunFileList(_RootFile);
	}

	/**
	 * Create a new instance of JunFileList of the current directory.
	 * 
	 * @return jp.co.sra.jun.graphics.list.JunFileList
	 * @category Instance creation
	 */
	public static JunFileList Current() {
		return new JunFileList(new File("."));
	}

	/**
	 * Create a new instance of JunFileList of the null file list.
	 * 
	 * @return jp.co.sra.jun.graphics.list.JunFileList
	 * @category Instance creation
	 */
	public static JunFileList Null() {
		return new JunFileList(null);
	}

	/**
	 * Answer the files in the root.
	 * 
	 * @return java.io.File[]
	 * @category Utilities
	 */
	public static File[] _RootFiles() {
		if (_RootFiles == null) {
			File[] files = File.listRoots();
			if (files.length == 1 && files[0].isDirectory()) {
				files = files[0].listFiles();
			}

			ArrayList rootFiles = new ArrayList(files.length);
			for (int i = 0; i < files.length; i++) {
				if (files[i].exists()) {
					rootFiles.add(files[i]);
				}
			}
			_RootFiles = (File[]) rootFiles.toArray(new File[rootFiles.size()]);
		}
		return _RootFiles;
	}

	/**
	 * Answer the table of the sorting comparators.
	 * 
	 * @return java.util.Map
	 * @category Private
	 */
	protected static Map _ComparatorTable() {
		if (_ComparatorTable == null) {
			HashMap aHashMap = new HashMap(8);
			aHashMap.put($("nameUp"), new Comparator() {
				public int compare(Object o1, Object o2) {
					return ((File) o1).getName().compareTo(((File) o2).getName());
				}
			});
			aHashMap.put($("nameDown"), new Comparator() {
				public int compare(Object o1, Object o2) {
					return ((File) o2).getName().compareTo(((File) o1).getName());
				}
			});
			aHashMap.put($("dateUp"), new Comparator() {
				public int compare(Object o1, Object o2) {
					return (int) (((File) o1).lastModified() - ((File) o2).lastModified());
				}
			});
			aHashMap.put($("dateDown"), new Comparator() {
				public int compare(Object o1, Object o2) {
					return (int) (((File) o2).lastModified() - ((File) o1).lastModified());
				}
			});
			aHashMap.put($("sizeUp"), new Comparator() {
				public int compare(Object o1, Object o2) {
					return (int) (((File) o1).length() - ((File) o2).length());
				}
			});
			aHashMap.put($("sizeDown"), new Comparator() {
				public int compare(Object o1, Object o2) {
					return (int) (((File) o2).length() - ((File) o1).length());
				}
			});
			aHashMap.put($("extensionUp"), new Comparator() {
				public int compare(Object o1, Object o2) {
					String s1 = JunFileUtility.ExtensionStringOf_((File) o1);
					if (s1 == null) {
						s1 = "";
					}
					String s2 = JunFileUtility.ExtensionStringOf_((File) o2);
					if (s2 == null) {
						s2 = "";
					}
					return s1.compareTo(s2);
				}
			});
			aHashMap.put($("extensionDown"), new Comparator() {
				public int compare(Object o1, Object o2) {
					String s1 = JunFileUtility.ExtensionStringOf_((File) o1);
					if (s1 == null) {
						s1 = "";
					}
					String s2 = JunFileUtility.ExtensionStringOf_((File) o2);
					if (s2 == null) {
						s2 = "";
					}
					return s2.compareTo(s1);
				}
			});

			_ComparatorTable = aHashMap;
		}
		return _ComparatorTable;
	}

	/**
	 * Initialize the JunFileList when created.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		currentFile = null;
		lastModified = 0;
		filePatterns = null;
		showContents = true;
		currentContents = null;
		sortKey = null;
		distinctFolder = true;
		selectedIndexHolder = null;
		nullItemBlock = null;
		fileItemBlock = null;
		directoryItemBlock = null;
		lookPreferences = null;
		yellowButtonMenuForFile = null;
		selectBlock = null;
		_keyListener = null;
	}

	/**
	 * Answer my current file.
	 * 
	 * @return java.io.File
	 * @category accessing
	 */
	public File currentFile() {
		return currentFile;
	}

	/**
	 * Set my new current file.
	 * 
	 * @param aFile java.io.File
	 * @category accessing
	 */
	public void currentFile_(File aFile) {
		if (aFile == null) {
			currentFile = null;
		} else {
			if (aFile.exists()) {
				currentFile = aFile;
				lastModified = currentFile.lastModified();
			}
		}

		JunFileList cloneFileList = JunFileListDispenser.Default().at_(this);
		if (cloneFileList == null) {
			sequenceMap = null;
			currentContents = null;
		} else {
			sequenceMap = cloneFileList.sequenceMap();
			currentContents = cloneFileList.currentContents();
			if (sequenceMap != null) {
				sequenceMap.allItemsDo_(new StBlockClosure() {
					public Object value_(Object anObject) {
						JunAbstractItem each = (JunAbstractItem) anObject;
						each.emphasisState_(false);
						each.lookPreferences_(JunFileList.this.lookPreferences());
						return null;
					}
				});
			}
		}
	}

	/**
	 * Answer the current contents.
	 *  
	 * @return java.io.File[]
	 * @category accessing
	 */
	public File[] currentContents() {
		if (currentContents == null) {
			currentContents = this.privateCurrentContents();
			JunFileListDispenser.Default().add_(this);
		}
		return currentContents;
	}

	/**
	 * Answer the file at the specified index.
	 * 
	 * @param anIndex int
	 * @return java.io.File
	 * @category accessing
	 */
	public File fileAt_(int anIndex) {
		if (anIndex < 0 || this.currentContents().length <= anIndex) {
			return null;
		}
		return this.currentContents()[anIndex];
	}

	/**
	 * Answer my current file patterns.
	 * 
	 * @return java.lang.String[]
	 * @category accessing
	 */
	public String[] filePatterns() {
		if (filePatterns == null) {
			filePatterns = new String[] { "*" };
		}
		return filePatterns;
	}

	/**
	 * Set my new file patterns.
	 * 
	 * @param stringPatternArray java.lang.String[]
	 * @category accessing
	 */
	public void filePatterns_(String[] stringPatternArray) {
		if (Arrays.equals(this.filePatterns(), stringPatternArray)) {
			return;
		}
		filePatterns = stringPatternArray;
		this.resetAll();
	}

	/**
	 * Answer the index of the file.
	 * 
	 * @param aFile java.io.File
	 * @return int
	 * @category accessing
	 */
	public int indexOfFile_(File aFile) {
		if (aFile == null) {
			return 0;
		}
		if (aFile.exists() == false) {
			return 0;
		}

		File[] files = this.currentContents();
		for (int i = 0; i < files.length; i++) {
			if (files[i].equals(aFile)) {
				return i;
			}
		}
		return -1;
	}

	/**
	 * Answer true if the receiver shows the contents of the file, otherwise false.
	 * 
	 * @return boolean
	 * @category accessing
	 */
	public boolean showContents() {
		return showContents;
	}

	/**
	 * Set whether the receiver shows the contents of the file or not.
	 * 
	 * @param aBoolean boolean
	 * @category accessing
	 */
	public void showContents_(boolean aBoolean) {
		if (this.showContents() != aBoolean) {
			showContents = aBoolean;
			this.resetAll();
		}
	}

	/**
	 * Answer my current sort key.
	 * 
	 * @return jp.co.sra.smalltalk.StSymbol
	 * @category accessing
	 */
	public StSymbol sortKey() {
		if (sortKey == null) {
			sortKey = $("nameUp");
		}
		return sortKey;
	}

	/**
	 * Set my new sort key.
	 * 
	 * @param aSymbol jp.co.sra.smalltalk.StSymbol
	 * @category accessing
	 */
	public void sortKey_(StSymbol aSymbol) {
		if (Arrays.asList(new StSymbol[] { $("nameUp"), $("nameDown"), $("dateUp"), $("dateDown"), $("sizeUp"), $("sizeDown"), $("extensionUp"), $("extensionDown") }).contains(aSymbol) == false) {
			return;
		}

		if (this.sortKey() != aSymbol) {
			sortKey = aSymbol;
			this.resetAll();
		}
	}

	/**
	 * Answer whether to handle folders distinctively or not.
	 * 
	 * @return boolean
	 * @category accessing
	 */
	public boolean distinctFolder() {
		return distinctFolder;
	}

	/**
	 * Set whether to handle folders distinctively or not.
	 * 
	 * @param aBoolean boolean
	 * @category accessing
	 */
	public void distinctFolder_(boolean aBoolean) {
		if (this.distinctFolder() != aBoolean) {
			distinctFolder = aBoolean;
			this.resetAll();
		}
	}

	/**
	 * Set the information and update the view.
	 * 
	 * @param aFile java.io.File
	 * @param selectedFile java.io.File
	 * @param aPoint java.awt.Point
	 * @category accessing
	 */
	public void currentFile_selectedFile_scrollOffset_(File aFile, File selectedFile, Point aPoint) {
		this.currentFile_(aFile);
		this.selectedFile_(selectedFile);
		this.changed_($("redisplay"));
		this.scrollOffset_(aPoint);
	}

	/**
	 * Answer true if the current file is null, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isNull() {
		return (this.currentFile() == null);
	}

	/**
	 * Answer true if the current file is the root, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isRoot() {
		if (this.isNull()) {
			return false;
		}
		return _RootFile.equals(this.currentFile());
	}

	/**
	 * Answer true if the current file is a directory, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isDirectory() {
		if (this.isNull()) {
			return false;
		}
		if (this.isRoot()) {
			return true;
		}
		if (this.currentFile().exists() == false) {
			return false;
		}
		return this.currentFile().isDirectory();
	}

	/**
	 * Answer true if the current file is a file, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isFile() {
		if (this.isNull()) {
			return false;
		}
		if (this.isRoot()) {
			return false;
		}
		if (this.currentFile().exists() == false) {
			return false;
		}
		return this.currentFile().isFile();
	}

	/**
	 * Answer true if the file matches the current patterns.
	 * 
	 * @param aFile java.io.File
	 * @return boolean
	 * @category testing
	 */
	public boolean filePatternsMatch_(File aFile) {
		String[] patterns = this.filePatterns();
		if (patterns.length == 0) {
			return false;
		}
		for (int i = 0; i < patterns.length; i++) {
			if ("*".equals(patterns[i])) {
				return true;
			}
		}
		String string = aFile.getPath();
		for (int i = 0; i < patterns.length; i++) {
			if (JunStringUtility.StringMatch_and_(string, patterns[i])) {
				return true;
			}
		}
		return false;

	}

	/**
	 * Make a sequence map.
	 * 
	 * @return jp.co.sra.jun.graphics.map.JunSequenceMap
	 * @see jp.co.sra.jun.graphics.list.JunSequenceList#makeSequenceMap()
	 * @category making
	 */
	protected JunSequenceMap makeSequenceMap() {
		JunSequenceMap aMap = new JunSequenceMap();
		if (this.isNull() == false) {
			if (this.isDirectory() && this.showContents()) {
				JunCursors cursor = new JunCursors(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
				try {
					cursor._show();

					File[] files = this.currentContents();
					for (int i = 0; i < files.length; i++) {
						if (files[i].exists()) {
							JunCompositeItem aCompositeItem = this.makeFileItemFromFile_(files[i]);
							aMap.add_(aCompositeItem);
						}
					}
				} finally {
					cursor._restore();
				}
			} else {
				JunCompositeItem aCompositeItem = this.makeInfoItemFromFile_(this.currentFile());
				aMap.add_(aCompositeItem);
			}
			aMap.arrange();
		}
		aMap.allItemsDo_(new StBlockClosure() {
			public Object value_(Object item) {
				((JunAbstractItem) item).lookPreferences_(JunFileList.this.lookPreferences());
				return null;
			}
		});
		return aMap;
	}

	/**
	 * Create a file item from the file.
	 * 
	 * @param aFile java.io.File
	 * @return jp.co.sra.jun.graphics.item.JunCompositeItem
	 * @category making
	 */
	protected JunCompositeItem makeFileItemFromFile_(File aFile) {
		JunCompositeItem aCompositeItem = new JunCompositeItem(new Point(0, 0), new Point(200, this.lineGrid()));
		aCompositeItem.borderWidth_(1);
		aCompositeItem.borderColor_(aCompositeItem.BackgroundColor());
		aCompositeItem.emphasisState_(false);
		JunPrimitiveItem fileIconItem = this.makeFileIconItemFromFile_(aFile);
		JunPrimitiveItem fileNameItem = this.makeFileNameItemFromFile_(aFile);
		JunPrimitiveItem fileImageItem = this.makeFileImageItemFromFile_(aFile);
		aCompositeItem.add_alignment_(fileIconItem, $("leftCenter"));
		aCompositeItem.add_alignment_with_alignment_offset_(fileNameItem, $("leftCenter"), fileIconItem, $("rightCenter"), new Point(2, 0));
		aCompositeItem.add_alignment_(fileImageItem, $("rightCenter"));
		return aCompositeItem;
	}

	/**
	 * Create a file icon item from the file.
	 * 
	 * @param aFile java.io.File
	 * @return jp.co.sra.jun.graphics.item.JunPrimitiveItem
	 * @category making
	 */
	protected JunPrimitiveItem makeFileIconItemFromFile_(File aFile) {
		StOpaqueImage anImage = JunFileNavigator.ImageFileIconFromFile_(aFile);
		JunPrimitiveItem aPrimitiveItem = new JunPrimitiveItem(new Point(0, 0), new Point(anImage.figure().width() + 4, anImage.figure().height()), anImage);
		aPrimitiveItem.visualAlignment_($("center"));
		aPrimitiveItem.emphasisState_(false);
		return aPrimitiveItem;
	}

	/**
	 * Create a file name item from the file.
	 * 
	 * @param aFile java.io.File
	 * @return jp.co.sra.jun.graphics.item.JunPrimitiveItem
	 * @category making
	 */
	protected JunPrimitiveItem makeFileNameItemFromFile_(File aFile) {
		String aString = aFile.getName();
		if (aString.length() == 0) {
			if (aFile.getParentFile() == null) {
				// This could be a drive on MS Windows.
				aString = aFile.getPath();
			} else {
				// This could be a directory which name ends with a file separator.
				aString = aFile.getParentFile().getName();
			}
		}

		StComposedText aComposedText = new StComposedText(aString, this.defaultListFont());
		JunPrimitiveItem aPrimitiveItem = new JunPrimitiveItem(new Point(0, 0), aComposedText.extent(), aComposedText);
		aPrimitiveItem.visualAlignment_($("leftCenter"));
		aPrimitiveItem.emphasisState_(false);
		return aPrimitiveItem;
	}

	/**
	 * Create a file image item from the file.
	 * 
	 * @param aFile java.io.File
	 * @return jp.co.sra.jun.graphics.item.JunPrimitiveItem
	 * @category making
	 */
	protected JunPrimitiveItem makeFileImageItemFromFile_(File aFile) {
		StOpaqueImage anImage = null;
		if (aFile.isDirectory()) {
			anImage = JunOpaqueImageIcons.Triangle16x16();
		} else {
			anImage = JunOpaqueImageIcons.Empty16x16();
		}

		JunPrimitiveItem aPrimitiveItem = new JunPrimitiveItem(new Point(0, 0), anImage.extent(), anImage);
		aPrimitiveItem.visualAlignment_($("leftCenter"));
		aPrimitiveItem.emphasisState_(false);
		return aPrimitiveItem;
	}

	/**
	 * Create an info item from the file.
	 * 
	 * @param aFile java.io.File
	 * @return jp.co.sra.jun.graphics.item.JunCompositeItem
	 * @category making
	 */
	protected JunCompositeItem makeInfoItemFromFile_(File aFile) {
		JunCompositeItem aCompositeItem = new JunCompositeItem(new Point(0, 0), new Point(0, 0));
		aCompositeItem.borderWidth_(0);
		aCompositeItem.borderColor_(aCompositeItem.BackgroundColor());
		aCompositeItem.emphasisState_(false);
		if (aFile.exists() == false) {
			return aCompositeItem;
		}

		JunPrimitiveItem infoIconItem = this.makeInfoIconItemFromFile_(aFile);
		JunPrimitiveItem infoNameItem = this.makeInfoNameItemFromFile_(aFile);
		JunPrimitiveItem infoCreatedItem = this.makeInfoCreatedItemFromFile_(aFile);
		JunPrimitiveItem infoModifiedItem = this.makeInfoModifiedItemFromFile_(aFile);
		JunPrimitiveItem infoSizeItem = this.makeInfoSizeItemFromFile_(aFile);
		aCompositeItem.add_alignment_(infoIconItem, $("topLeft"));
		aCompositeItem.add_alignment_with_alignment_offset_(infoNameItem, $("topLeft"), infoIconItem, $("bottomLeft"), new Point(0, 12));
		aCompositeItem.add_alignment_with_alignment_offset_(infoCreatedItem, $("topLeft"), infoNameItem, $("bottomLeft"), new Point(0, 2));
		aCompositeItem.add_alignment_with_alignment_offset_(infoModifiedItem, $("topLeft"), infoCreatedItem, $("bottomLeft"), new Point(0, 2));
		aCompositeItem.add_alignment_with_alignment_offset_(infoSizeItem, $("topLeft"), infoModifiedItem, $("bottomLeft"), new Point(0, 2));
		Rectangle aRectangle;
		int xSize = (aRectangle = infoIconItem.bounds()).x + aRectangle.width;
		xSize = Math.max(xSize, (aRectangle = infoNameItem.bounds()).x + aRectangle.width);
		xSize = Math.max(xSize, (aRectangle = infoCreatedItem.bounds()).x + aRectangle.width);
		xSize = Math.max(xSize, (aRectangle = infoModifiedItem.bounds()).x + aRectangle.width);
		xSize = Math.max(xSize, (aRectangle = infoSizeItem.bounds()).x + aRectangle.width);
		int ySize = (aRectangle = infoSizeItem.bounds()).y + aRectangle.height;
		aCompositeItem.extentPoint_(new Point(xSize, ySize));
		return aCompositeItem;
	}

	/**
	 * Create an info icon item from the file.
	 * 
	 * @param aFile java.io.File
	 * @return jp.co.sra.jun.graphics.item.JunPrimitiveItem
	 * @category making
	 */
	protected JunPrimitiveItem makeInfoIconItemFromFile_(File aFile) {
		StOpaqueImage anImage = JunFileNavigator.ImageInfoIconFromFile_(aFile);
		JunPrimitiveItem aPrimitiveItem = new JunPrimitiveItem(new Point(0, 0), anImage.extent(), anImage);
		aPrimitiveItem.visualAlignment_($("center"));
		aPrimitiveItem.emphasisState_(false);
		return aPrimitiveItem;
	}

	/**
	 * Create an info name item from the file.
	 * 
	 * @param aFile java.io.File
	 * @return jp.co.sra.jun.graphics.item.JunPrimitiveItem
	 * @category making
	 */
	protected JunPrimitiveItem makeInfoNameItemFromFile_(File aFile) {
		StringWriter sw = new StringWriter();
		PrintWriter pw = null;

		try {
			pw = new PrintWriter(sw);
			pw.println("Name:");
			pw.print(aFile.getName());
		} finally {
			if (pw != null) {
				pw.flush();
				pw.close();
			}
		}

		String aString = sw.toString();
		StComposedText aComposedText = new StComposedText(aString);
		JunPrimitiveItem aPrimitiveItem = new JunPrimitiveItem(new Point(0, 0), aComposedText.extent(), aComposedText);
		aPrimitiveItem.visualAlignment_($("leftCenter"));
		aPrimitiveItem.emphasisState_(false);
		return aPrimitiveItem;
	}

	/**
	 * Create an info created item from the file.
	 * 
	 * @param aFile java.io.File
	 * @return jp.co.sra.jun.graphics.item.JunPrimitiveItem
	 * @category making
	 */
	protected JunPrimitiveItem makeInfoCreatedItemFromFile_(File aFile) {
		StringWriter sw = new StringWriter();
		PrintWriter pw = null;

		try {
			pw = new PrintWriter(sw);
			pw.println("Created:");
			pw.print("unknown");
		} finally {
			if (pw != null) {
				pw.flush();
				pw.close();
			}
		}

		String aString = sw.toString();
		StComposedText aComposedText = new StComposedText(aString);
		JunPrimitiveItem aPrimitiveItem = new JunPrimitiveItem(new Point(0, 0), aComposedText.extent(), aComposedText);
		aPrimitiveItem.visualAlignment_($("leftCenter"));
		aPrimitiveItem.emphasisState_(false);
		return aPrimitiveItem;
	}

	/**
	 * Create an info modified item from the file.
	 * 
	 * @param aFile java.io.File
	 * @return jp.co.sra.jun.graphics.item.JunPrimitiveItem
	 * @category making
	 */
	protected JunPrimitiveItem makeInfoModifiedItemFromFile_(File aFile) {
		StringWriter sw = new StringWriter();
		PrintWriter pw = null;

		try {
			pw = new PrintWriter(sw);
			pw.println("Modified:");
			pw.print(JunCalendarModel.StringFromDateAndTime_(new Date(aFile.lastModified())));
		} finally {
			if (pw != null) {
				pw.flush();
				pw.close();
			}
		}

		String aString = sw.toString();
		StComposedText aComposedText = new StComposedText(aString);
		JunPrimitiveItem aPrimitiveItem = new JunPrimitiveItem(new Point(0, 0), aComposedText.extent(), aComposedText);
		aPrimitiveItem.visualAlignment_($("leftCenter"));
		aPrimitiveItem.emphasisState_(false);
		return aPrimitiveItem;
	}

	/**
	 * Create an info size item from the file.
	 * 
	 * @param aFile java.io.File
	 * @return jp.co.sra.jun.graphics.item.JunPrimitiveItem
	 * @category making
	 */
	protected JunPrimitiveItem makeInfoSizeItemFromFile_(File aFile) {
		StringWriter sw = new StringWriter();
		PrintWriter pw = null;

		try {
			pw = new PrintWriter(sw);
			if (aFile.isDirectory()) {
				pw.println("Contents:");
				pw.print(aFile.list().length);
				pw.print(" items");
			} else {
				long fileSize = aFile.length();
				pw.println("Size:");
				pw.print(NumberFormat.getInstance().format(fileSize));
				pw.println(" bytes");
				pw.print('(');
				long kiloBytes = 1024;
				long megaBytes = kiloBytes * kiloBytes;
				long gigaBytes = megaBytes * kiloBytes;
				if (fileSize / gigaBytes > 0) {
					pw.print(fileSize * 1000 / gigaBytes / 1000.0f);
					pw.print(" GB");
				} else if (fileSize / megaBytes > 0) {
					pw.print(fileSize * 1000 / megaBytes / 1000.0f);
					pw.print(" MB");
				} else {
					pw.print(fileSize * 1000 / kiloBytes / 1000.0f);
					pw.print(" KB");
				}
				pw.print(')');
			}
		} finally {
			if (pw != null) {
				pw.flush();
				pw.close();
			}
		}

		String aString = sw.toString();
		StComposedText aComposedText = new StComposedText(aString);
		JunPrimitiveItem aPrimitiveItem = new JunPrimitiveItem(new Point(0, 0), aComposedText.extent(), aComposedText);
		aPrimitiveItem.visualAlignment_($("leftCenter"));
		aPrimitiveItem.emphasisState_(false);
		return aPrimitiveItem;
	}

	/**
	 * Answer the sorted current contents.
	 * 
	 * @param directoryContents java.io.File[]
	 * @param sortSymbol jp.co.sra.smalltalk.StSymbol
	 * @return java.io.File[]
	 * @category sorting
	 */
	protected File[] sortedCurrentContents_sortSymbol_(File[] directoryContents, StSymbol sortSymbol) {
		if (directoryContents == null) {
			return new File[0];
		}

		File[] files = (File[]) directoryContents.clone();

		JunCursors cursor = new JunCursors(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		try {
			cursor._show();

			Arrays.sort(files, (Comparator) _ComparatorTable().get(sortSymbol));
		} finally {
			cursor._restore();
		}

		ArrayList list = new ArrayList(files.length);

		cursor = new JunCursors(JunCursors.ExecuteCursor());
		try {
			cursor._show();

			if (this.distinctFolder()) {
				for (int i = 0; i < files.length; i++) {
					if (files[i].isDirectory()) {
						list.add(files[i]);
					}
				}
				for (int i = 0; i < files.length; i++) {
					if (!files[i].isDirectory() && this.filePatternsMatch_(files[i])) {
						list.add(files[i]);
					}
				}
			} else {
				for (int i = 0; i < files.length; i++) {
					if (files[i].isDirectory() || this.filePatternsMatch_(files[i])) {
						list.add(files[i]);
					}
				}
			}
		} finally {
			cursor._restore();
		}

		return (File[]) list.toArray(new File[list.size()]);
	}

	/**
	 * Create a current contents as an array of File.
	 * 
	 * @return java.io.File[]
	 * @category private
	 */
	protected File[] privateCurrentContents() {
		File[] files = null;
		if (this.isNull() || this.isFile()) {
			files = new File[0];
		} else {
			JunCursors cursor = new JunCursors(JunCursors.ReadCursor());
			try {
				cursor._show();
				files = this.isRoot() ? _RootFiles() : this.currentFile().listFiles();
			} finally {
				cursor._restore();
			}

			ArrayList aList = new ArrayList();
			if (files != null) {
				for (int i = 0; i < files.length; i++) {
					if (((Boolean) this.selectBlock().value_(files[i])).booleanValue()) {
						aList.add(files[i]);
					}
				}
			}
			files = (File[]) aList.toArray(new File[aList.size()]);
			files = this.sortedCurrentContents_sortSymbol_(files, this.sortKey());
		}

		return files;
	}

	/**
	 * Answer my popup menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @see jp.co.sra.smalltalk.StApplicationModel#_popupMenu()
	 * @category resources
	 */
	public StPopupMenu _popupMenu() {
		if (this.isNull()) {
			return null;
		}
		if (this.isFile()) {
			return this._popupMenuForFile();
		}
		return this._popupMenuForDirectory();
	}

	/**
	 * Answer the receiver's popup menu for file.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @category resources
	 */
	public StPopupMenu _popupMenuForDirectory() {
		if (yellowButtonMenu == null) {
			StPopupMenu popupMenu = new StPopupMenu();
			popupMenu.add(new StMenuItem($String("Update contents"), $("updateContents"), new MenuDispatcher(this, "updateContents")));
			yellowButtonMenu = popupMenu;
		}
		return yellowButtonMenu;
	}

	/**
	 * Answer the receiver's popup menu for file.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @category resources
	 */
	public StPopupMenu _popupMenuForFile() {
		if (yellowButtonMenuForFile == null) {
			StPopupMenu popupMenu = new StPopupMenu();
			popupMenu.add(new StMenuItem($String("Open"), $("openFile"), new MenuPerformer(this, "openFile")));
			popupMenu.add(new StMenuItem($String("Update contents"), $("updateContents"), new MenuDispatcher(this, "updateContents")));
			yellowButtonMenuForFile = popupMenu;
		}
		return yellowButtonMenuForFile;
	}

	/**
	 * Answer the currently selected file.
	 * 
	 * @return java.io.File
	 * @category selecting
	 */
	public File selectedFile() {
		if (this.showContents()) {
			return this.fileAt_(this.selectedIndex());
		}
		return this.currentFile();
	}

	/**
	 * Set the selection of the file.
	 * 
	 * @param aFile java.io.File
	 * @category selecting
	 */
	public void selectedFile_(File aFile) {
		if (this.isNull()) {
			this.selectedIndex_(-1);
			return;
		}
		if (this.isFile()) {
			this.selectedIndex_(-1);
			return;
		}
		this.selectedIndex_(this.indexOfFile_(aFile));
	}

	/**
	 * Answer the current index of the selection.
	 * 
	 * @return int
	 * @category selecting
	 */
	public int selectedIndex() {
		return this.selectedIndexHolder()._intValue();
	}

	/**
	 * Set the selection with the index.
	 * 
	 * @param anIndex int
	 * @category selecting
	 */
	public void selectedIndex_(int anIndex) {
		if (this.isNull()) {
			this.selectedIndexHolder().value_(-1);
			return;
		}
		if (this.isFile()) {
			this.selectedIndexHolder().value_(-1);
			return;
		}

		JunAbstractItem anItem = this.itemAt_(anIndex);
		if (anItem == null) {
			this.selectedIndexHolder().value_(-1);
			return;
		}

		anItem.withAllItemsDo_(new StBlockClosure() {
			public Object value_(Object each) {
				((JunAbstractItem) each).emphasisState_(true);
				return null;
			}
		});
		Rectangle aRectangle = anItem.bounds();
		this.selectedIndexHolder().value_(anIndex);
		this.changed_with_($("selection"), aRectangle);
	}

	/**
	 * Answer my current selected index holder.
	 * 
	 * @return jp.co.sra.smalltalk.StValueHolder
	 * @category selecting
	 */
	protected StValueHolder selectedIndexHolder() {
		if (selectedIndexHolder == null) {
			selectedIndexHolder = new StValueHolder(-1);
		}
		return selectedIndexHolder;
	}

	/**
	 * Answer the currently selected item.
	 * 
	 * @return jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @category selecting
	 */
	public JunAbstractItem selectedItem() {
		return this.itemAt_(this.selectedIndex());
	}

	/**
	 * Set the selection of the item.
	 * 
	 * @param anItem jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @category selecting
	 */
	public void selectedItem_(JunAbstractItem anItem) {
		if (this.isNull()) {
			this.selectedIndex_(-1);
			return;
		}
		if (this.isFile()) {
			this.selectedIndex_(-1);
			return;
		}
		this.selectedIndex_(this.indexOfItem_(anItem));
	}

	/**
	 * Select the first item.
	 * 
	 * @category selecting
	 */
	public void selectFirstItem() {
		if (this.showContents() == false) {
			return;
		}

		if (this.isDirectory() && !this.isEmpty() && this.selectedItem() == null) {
			this.selectedItem_(this.itemAt_(0));
		}
	}

	/**
	 * Select an item at the specified point.
	 * 
	 * @param aPoint java.awt.Point
	 * @return jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @category selecting
	 */
	public JunAbstractItem selectItemAt_(Point aPoint) {
		if (this.isNull()) {
			return null;
		}
		if (this.isFile()) {
			return null;
		}

		JunAbstractItem newSelectedItem = this.whichItemAt_(aPoint);
		JunAbstractItem oldSelectedItem = this.selectedItem();
		if (oldSelectedItem == null) {
			if (newSelectedItem == null) {
				return null;
			}
		} else {
			if (oldSelectedItem.equals(newSelectedItem)) {
				return null;
			}
		}

		this.deselectItem_(oldSelectedItem);
		this.selectedItem_(newSelectedItem);
		return newSelectedItem;
	}

	/**
	 * Unselect the currently selected item.
	 * 
	 * @return jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @category selecting
	 */
	public JunAbstractItem deselectItem() {
		JunAbstractItem anItem = this.selectedItem();
		if (anItem == null) {
			return null;
		}
		return this.deselectItem_(anItem);
	}

	/**
	 * Unselect the specified item.
	 * 
	 * @param anItem jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @return jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @category selecting
	 */
	public JunAbstractItem deselectItem_(JunAbstractItem anItem) {
		if (anItem == null) {
			return null;
		}

		anItem.withAllItemsDo_(new StBlockClosure() {
			public Object value_(Object each) {
				((JunAbstractItem) each).emphasisState_(false);
				return null;
			}
		});
		Rectangle aRectangle = anItem.bounds();
		this.selectedIndex_(-1);
		this.changed_with_($("selection"), aRectangle);
		return anItem;
	}

	/**
	 * Answer the item at the specified point.
	 * 
	 * @param aPoint java.awt.Point
	 * @return jp.co.sra.jun.graphics.abstracts.JunAbstractItem
	 * @category selecting
	 */
	public JunAbstractItem whichItemAt_(final Point aPoint) {
		return (JunAbstractItem) this.itemsDo_(new StBlockClosure() {
			public Object value_(Object anItem) {
				return ((JunAbstractItem) anItem).containsPoint_(aPoint) ? anItem : null;
			}
		});
	}

	/**
	 * Fix the scrolling offset from the view.
	 * 
	 * @param aView jp.co.sra.jun.graphics.framework.JunGraphicView
	 * @param aBoolean boolean
	 * @category scrolling
	 */
	public void fixScrollingOffsetFrom_keepPosition_(JunGraphicView aView, boolean aBoolean) {
		if (aView == null) {
			return;
		}

		if (aBoolean) {
			super.fixScrollingOffsetFrom_(aView);
		} else {
			JunAbstractItem selectedItem = this.selectedItem();
			if (selectedItem == null) {
				aView.scrollTo_(new Point(0, 0));
			} else {
				aView.makeVisible(selectedItem.bounds());
			}
		}
	}

	/**
	 * Answer the receiver's line grid size.
	 * 
	 * @return int
	 * @category scrolling
	 */
	public int lineGrid() {
		int fontHeight;
		if (this.getView() == null) {
			fontHeight = SystemResourceSupport.getFontMetrics(this.defaultListFont()).getHeight();
		} else {
			fontHeight = this.getView().toComponent().getFontMetrics(this.defaultListFont()).getHeight();
		}
		return Math.max(fontHeight, 18);
	}

	/**
	 * Action for the red button pressed event.
	 * 
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.jun.graphics.framework.JunGraphicModel#redButtonPressedAt_(java.awt.Point)
	 * @category actions
	 */
	public void redButtonPressedAt_(Point aPoint) {
		Point thePoint = aPoint;
		File aFile = this.currentFile();
		if (aFile != null) {
			while (aFile.exists() == false) {
				String aString = aFile.getParent();
				if (aString == null || aString.length() == 0) {
					return;
				}
				aFile = new File(aString);
			}
			if (this.currentFile() != aFile) {
				currentFile = aFile;
				this.updateContents();
				thePoint = new Point(-1, -1);
			}
		}

		if (this.isNull()) {
			this.nullItemAt_(thePoint);
		} else if (this.isFile()) {
			this.fileItemAt_(thePoint);
		} else if (this.isDirectory()) {
			this.directoryItemAt_(thePoint);
		}
	}

	/**
	 * Action for the red button pressed event for the null item.
	 * 
	 * @param aPoint java.awt.Point
	 * @category actions
	 */
	protected void nullItemAt_(Point aPoint) {
		StBlockClosure aBlock = this.nullItemBlock();
		switch (aBlock.numArgs()) {
			case 0:
				aBlock.value();
				return;
			case 1:
				aBlock.value_(null);
				return;
		}
	}

	/**
	 * Answer the block closure for a null item.
	 * 
	 * @return jp.co.sra.smalltalk.StBlockClosure
	 * @category actions
	 */
	protected StBlockClosure nullItemBlock() {
		if (nullItemBlock == null) {
			nullItemBlock = new StBlockClosure();
		}
		return nullItemBlock;
	}

	/**
	 * Set the block closure for a null item.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category actions
	 */
	public void nullItemBlock_(StBlockClosure aBlock) {
		nullItemBlock = aBlock;
	}

	/**
	 * Action for the red button pressed event for the file item.
	 * 
	 * @param aPoint java.awt.Point
	 * @category actions
	 */
	protected void fileItemAt_(Point aPoint) {
		StBlockClosure aBlock = this.fileItemBlock();
		switch (aBlock.numArgs()) {
			case 0:
				aBlock.value();
				return;
			case 1:
				aBlock.value_(this.currentFile());
				return;
		}
	}

	/**
	 * Answer the block closure for a file item.
	 * 
	 * @return jp.co.sra.smalltalk.StBlockClosure
	 * @category actions
	 */
	protected StBlockClosure fileItemBlock() {
		if (fileItemBlock == null) {
			fileItemBlock = new StBlockClosure();
		}
		return fileItemBlock;
	}

	/**
	 * Set the block closure for a file item.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category actions
	 */
	public void fileItemBlock_(StBlockClosure aBlock) {
		fileItemBlock = aBlock;
	}

	/**
	 * Action for the red button pressed event for the directory item.
	 * 
	 * @param aPoint java.awt.Point
	 * @category actions
	 */
	protected void directoryItemAt_(Point aPoint) {
		File selectedFile;
		if (aPoint == new Point(-1, -1)) {
			selectedFile = null;
		} else {
			if (this.showContents()) {
				if (this.selectItemAt_(aPoint) == null) {
					return;
				}
			}
			selectedFile = this.selectedFile();
		}

		StBlockClosure aBlock = this.directoryItemBlock();
		switch (aBlock.numArgs()) {
			case 0:
				aBlock.value();
				return;
			case 1:
				aBlock.value_(selectedFile);
				return;
		}
	}

	/**
	 * Answer my current block closure for directory items.
	 * 
	 * @return jp.co.sra.smalltalk.StBlockClosure
	 * @category actions
	 */
	public StBlockClosure directoryItemBlock() {
		if (directoryItemBlock == null) {
			directoryItemBlock = new StBlockClosure();
		}
		return directoryItemBlock;
	}

	/**
	 * Set my new block closure for directory items.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category actions
	 */
	public void directoryItemBlock_(StBlockClosure aBlock) {
		directoryItemBlock = aBlock;
	}

	/**
	 * Answer a <code>StBlockValue</code> that computes aBlock with the receiver's value as the argument.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return jp.co.sra.smalltalk.StBlockValue
	 * @category constructing
	 */
	public StBlockValue compute_(StBlockClosure aBlock) {
		return this.selectedIndexHolder().compute_(aBlock);
	}

	/**
	 * Answer the receiver's default font.
	 * 
	 * @return java.awt.Font
	 * @category defaults
	 */
	public Font defaultListFont() {
		return StComposedText.DefaultFont();
	}

	/**
	 * Open the file with browser.
	 * 
	 * @category menu messages
	 */
	public synchronized void openFile() {
		File file = this.currentFile();
		JunCursors cursor = new JunCursors(new Cursor(Cursor.WAIT_CURSOR));
		try {
			cursor._show();
			URL url = new URL("file", null, file.getAbsolutePath());
			JunURL.Browse_(url);
		} catch (MalformedURLException e) {
			System.err.print(e.getMessage());
			e.printStackTrace();
		} finally {
			cursor._restore();
		}
	}

	/**
	 * Update the contents.
	 * 
	 * @category menu messages
	 */
	public synchronized void updateContents() {
		JunFileListDispenser.Default().remove_(this);
		File aFile = this.selectedFile();
		this.selectedIndex_(-1);
		this.currentFile_(this.currentFile());
		int anIndex = this.indexOfFile_(aFile);
		if (anIndex >= 0) {
			this.selectedIndex_(anIndex);
		} else {
			this.selectFirstItem();
		}
		this.changed_($("redisplay"));
		this.fixScrollingOffsetFrom_keepPosition_((JunGraphicView) this.getView(), false);
	}

	/**
	 * Answer my current look preferences.
	 * 
	 * @return jp.co.sra.jun.graphics.abstracts.JunAbstractItem.LookPrerences
	 * @category visual properties
	 */
	public LookPreferences lookPreferences() {
		return lookPreferences;
	}

	/**
	 * Answer my KeyListener which handles my keyboard events.
	 *
	 * @return java.awt.event.KeyListener
	 * @see jp.co.sra.jun.graphics.framework.JunGraphicModel#_keyListener()
	 * @category keyboard
	 */
	public KeyListener _keyListener() {
		if (_keyListener == null) {
			_keyListener = new KeyAdapter() {
				public void keyPressed(KeyEvent e) {
					if (_keyboardEvent(e)) {
						e.consume();
					}
				}
			};
		}
		return _keyListener;
	}

	/**
	 * Process the key event.
	 *
	 * @param ev java.awt.event.KeyEvent
	 * @return boolean
	 * @category keyboard
	 */
	protected boolean _keyboardEvent(KeyEvent ev) {
		if (ev.isConsumed()) {
			return false;
		}

		int code = ev.getKeyCode();
		switch (code) {
			case KeyEvent.VK_ENTER:
				this.keyStrokePressENTER();
				return true;
			case KeyEvent.VK_SPACE:
				this.keyStrokePressSPACE();
				return true;
			case KeyEvent.VK_LEFT:
				this.keyStrokePressLEFT();
				return true;
			case KeyEvent.VK_RIGHT:
				this.keyStrokePressRIGHT();
				return true;
			case KeyEvent.VK_UP:
				this.keyStrokePressUP();
				return true;
			case KeyEvent.VK_DOWN:
				this.keyStrokePressDOWN();
				return true;
		}

		return false;
	}

	/**
	 * Handle the key stroke ENTER.
	 * 
	 * @return boolean
	 * @category keyboard
	 */
	protected boolean keyStrokePressENTER() {
		return true;
	}

	/**
	 * Handle the key stroke SPACE.
	 * 
	 * @return boolean
	 * @category keyboard
	 */
	protected boolean keyStrokePressSPACE() {
		return true;
	}

	/**
	 * Handle the key stroke LEFT.
	 * 
	 * @return boolean
	 * @category keyboard
	 */
	protected boolean keyStrokePressLEFT() {
		return true;
	}

	/**
	 * Handle the key stroke RIGHT.
	 * 
	 * @return boolean
	 * @category keyboard
	 */
	protected boolean keyStrokePressRIGHT() {
		return true;
	}

	/**
	 * Handle the key stroke UP.
	 * 
	 * @return boolean
	 * @category keyboard
	 */
	protected boolean keyStrokePressUP() {
		int anIndex = this.selectedIndex();
		if (anIndex == 0) {
			return false;
		}
		anIndex = anIndex - 1;
		anIndex = Math.max(0, anIndex);
		JunAbstractItem newSelectedItem = this.itemAt_(anIndex);
		if (newSelectedItem != null) {
			JunAbstractItem oldSelectedItem = this.selectedItem();
			if (oldSelectedItem == null) {
				if (newSelectedItem == null) {
					return false;
				}
			} else {
				if (oldSelectedItem.equals(newSelectedItem)) {
					return false;
				}
			}

			this.deselectItem_(oldSelectedItem);
			this.selectedItem_(newSelectedItem);
			this.fixScrollingOffsetFrom_keepPosition_((JunGraphicView) this.getView(), false);
		}
		return true;
	}

	/**
	 * Handle the key stroke DOWN.
	 * 
	 * @return boolean
	 * @category keyboard
	 */
	protected boolean keyStrokePressDOWN() {
		int anIndex = this.selectedIndex();
		if (anIndex == this.sequenceMap().componentItems().size() - 1) {
			return false;
		}
		anIndex = anIndex + 1;
		anIndex = Math.min(anIndex, this.sequenceMap().componentItems().size() - 1);
		JunAbstractItem newSelectedItem = this.itemAt_(anIndex);
		if (newSelectedItem != null) {
			JunAbstractItem oldSelectedItem = this.selectedItem();
			if (oldSelectedItem == null) {
				if (newSelectedItem == null) {
					return false;
				}
			} else {
				if (oldSelectedItem.equals(newSelectedItem)) {
					return false;
				}
			}

			this.deselectItem_(oldSelectedItem);
			this.selectedItem_(newSelectedItem);
			this.fixScrollingOffsetFrom_keepPosition_((JunGraphicView) this.getView(), false);
		}
		return true;
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("File List");
	}

	/**
	 * Reset everything.
	 * 
	 * @category private
	 */
	protected void resetAll() {
		JunFileListDispenser.Default().remove_(this);
		File aFile = this.selectedFile();
		this.selectedIndex_(-1);
		sequenceMap = null;
		currentContents = null;
		if (this.isFile()) {
			if (this.filePatternsMatch_(this.currentFile())) {
				// self yourself
			} else {
				this.currentFile_(null);
			}
		} else {
			int index = this.indexOfFile_(aFile);
			if (index > 0) {
				this.selectedIndex_(index);
			} else {
				this.selectFirstItem();
			}
		}
		// this.changed_($("redisplay"));
		this.updateContents();
		// this.fixScrollingOffsetFrom_keepPosition_(this.getView(), false);
	}

	/**
	 * Recompose the receiver in the specified bounds.
	 * 
	 * @param aRectangle java.awt.Rectangle
	 * @see jp.co.sra.jun.graphics.framework.JunGraphicModel#recomposeIn_(java.awt.Rectangle)
	 * @category private
	 */
	public void recomposeIn_(Rectangle aRectangle) {
		if (this.isNull()) {
			return;
		}

		if (this.isDirectory() && this.showContents()) {
			JunSequenceMap aMap = this.sequenceMap();
			JunCompositeItem[] items = new JunCompositeItem[aMap.componentItems().size()];
			aMap.componentItems().toArray(items);
			for (int i = 0; i < items.length; i++) {
				File file = this.fileAt_(i);
				if (file.exists() == false) {
					this.updateContents();
					return;
				}
				JunPrimitiveItem fileIconItem = (JunPrimitiveItem) items[i].itemAt_(0);
				JunPrimitiveItem fileNameItem = (JunPrimitiveItem) items[i].itemAt_(1);
				JunPrimitiveItem fileImageItem = (JunPrimitiveItem) items[i].itemAt_(2);
				/*
				 File file = this.fileAt_(i);
				 StRectangle bounds = new StRectangle(fileImageItem.visualComponent().bounds());
				 if (!file.isFile()) {
				 fileImageItem.extentPoint_(new Point(bounds.width(), bounds.height()));
				 } else {
				 fileImageItem.extentPoint_(new Point(0, bounds.height()));
				 }
				 */
				items[i].extentPoint_(new Point(aRectangle.width, items[i].bounds().height));
				items[i].arrange_alignment_(fileImageItem, $("rightCenter"));
				StRectangle bounds = StRectangle.Origin_corner_(new Point(fileIconItem.originPoint().x + fileIconItem.extentPoint().x, 0), new Point(fileImageItem.originPoint().x, items[i].extentPoint().y - 1));
				items[i].arrange_in_(fileNameItem, bounds.toRectangle());
			}
			aMap.arrange();
		} else {
			JunCompositeItem aCompositeItem = (JunCompositeItem) this.itemAt_(0);
			StRectangle aBox = new StRectangle(aCompositeItem.bounds());
			if (aRectangle.width >= aBox.width() && aRectangle.height >= aBox.height()) {
				aBox = aBox.align_with_(aBox.center(), (new StRectangle(aRectangle)).center());
			} else if (aRectangle.width >= aBox.width()) {
				aBox = aBox.align_with_(aBox.topCenter(), (new StRectangle(aRectangle)).topCenter());
			} else if (aRectangle.height >= aBox.height()) {
				aBox = aBox.align_with_(aBox.leftCenter(), (new StRectangle(aRectangle)).leftCenter());
			} else {
				aBox = aBox.align_with_(aBox.topLeft(), (new StRectangle(aRectangle)).topLeft());
			}
			aCompositeItem.originPoint_(new Point(10, 10));
			aCompositeItem.extentPoint_(new Point(aBox.width(), aBox.height()));
			this.sequenceMap().flushBounds();
		}
	}

	/**
	 * Answer my current select block.
	 * 
	 * @return jp.co.sra.smalltalk.StBlockClosure
	 * @category private
	 */
	protected StBlockClosure selectBlock() {
		if (selectBlock == null) {
			selectBlock = new StBlockClosure() {
				public Object value_(Object aFile) {
					return Boolean.TRUE;
				}
			};
		}
		return selectBlock;
	}

	/**
	 * Set my new select block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category private
	 */
	protected void selectBlock_(final StBlockClosure aBlock) {
		selectBlock = new StBlockClosure() {
			public Object value_(Object aFile) {
				Boolean aBoolean;
				try {
					Object result = aBlock.value_(aFile);
					aBoolean = Boolean.valueOf(result instanceof Boolean && ((Boolean) result).booleanValue());
				} catch (Exception e) {
					aBoolean = Boolean.FALSE;
				}
				return aBoolean;
			}
		};
	}

}
