package jp.co.sra.jun.graphics.framework;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.ScrollPane;
import java.awt.Window;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.KeyListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;

import javax.swing.SwingUtilities;

import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.StController;
import jp.co.sra.smalltalk.StDisplayable;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StViewCanvas;

import jp.co.sra.jun.system.framework.JunAbstractViewPanel;

/**
 * JunGraphicViewAwt class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2006/11/22 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun640 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunGraphicViewAwt.java,v 8.14 2008/02/20 06:32:15 nisinaka Exp $
 */
public class JunGraphicViewAwt extends JunAbstractViewPanel implements JunGraphicView {
	protected StViewCanvas canvas;
	protected ScrollPane scrollPane;

	/**
	 * Create a new instance of <code>JunGraphicViewAwt</code> and initialize it.
	 * 
	 * @param aModel jp.co.sra.jun.graphics.framework.JunGraphicModel
	 * @category Instance creation
	 */
	public JunGraphicViewAwt(JunGraphicModel aModel) {
		super(aModel);
	}

	/**
	 * Initialize the receiver. 
	 * 
	 * @see jp.co.sra.smalltalk.StViewJPanel#initialize()
	 * @category initialize-release 
	 */
	protected void initialize() {
		super.initialize();
		canvas = null;
		scrollPane = null;
	}

	/**
	 * Build the component.
	 * 
	 * @see jp.co.sra.smalltalk.StViewJPanel#buildComponent()
	 * @category initialize-release
	 */
	protected void buildComponent() {
		this.setLayout(new BorderLayout());
		this.add(this.scrollPane(), BorderLayout.CENTER);
		this.scrollPane().setSize(200, 150);
	}

	/**
	 * Answer the receiver's model as <code>JunGraphicModel</code>.
	 * 
	 * @return jp.co.sra.jun.graphics.framework.JunGraphModel
	 * @see jp.co.sra.jun.graphics.framework.JunGraphicView#getGraphicModel()
	 * @category accessing
	 */
	public JunGraphicModel getGraphicModel() {
		return (JunGraphicModel) this.model();
	}

	/**
	 * Display the canvas part of the receiver on the graphics.
	 * The subclasses may override this.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category displaying
	 */
	protected void displayCanvasOn_(Graphics aGraphics) {
		if (this.canvas().isShowing() == false) {
			return;
		}

		Rectangle aRectangle = this.canvas().getBounds();
		aGraphics.setColor(this.canvas().getBackground());
		aGraphics.fillRect(aRectangle.x, aRectangle.y, aRectangle.width, aRectangle.height);

		StDisplayable displayObject = this.getGraphicModel().displayObject();
		if (displayObject != null) {
			displayObject.displayOn_(aGraphics);
		}
	}

	/**
	 * Set up the keyboard for the view on the window.
	 * 
	 * @param aWindow java.awt.Window
	 * @see jp.co.sra.smalltalk.StView#_setupKeyboard(java.awt.Window)
	 * @category keyboard
	 */
	public void _setupKeyboard(final Window aWindow) {
		KeyListener keyListener = this.getGraphicModel()._keyListener();
		if (keyListener == null) {
			return;
		}

		this.addKeyListener(keyListener);

		aWindow.addWindowListener(new WindowAdapter() {
			public void windowActivated(WindowEvent e) {
				SwingUtilities.invokeLater(new Runnable() {
					public void run() {
						Component focusOwner = aWindow.getFocusOwner();
						if (focusOwner == null || focusOwner != JunGraphicViewAwt.this) {
							JunGraphicViewAwt.this.requestFocus();
						}
					}
				});
			}
		});
	}

	/**
	 * Make the specified rectangle area visible.
	 * 
	 * @param aRectangle java.awt.Rectangle
	 * @see jp.co.sra.jun.graphics.framework.JunGraphicView#makeVisible(java.awt.Rectangle)
	 * @category scrolling
	 */
	public void makeVisible(Rectangle aRectangle) {
		Rectangle clipBounds = new Rectangle(this.scrollPane().getScrollPosition(), this.scrollPane().getViewportSize());
		if (clipBounds.contains(aRectangle)) {
			return;
		}

		Point position = this.scrollPane().getScrollPosition();
		if (aRectangle.x + aRectangle.width >= clipBounds.x + clipBounds.width) {
			position.x = aRectangle.x - (clipBounds.width - aRectangle.width);
		}
		if (aRectangle.x <= clipBounds.x || aRectangle.width >= clipBounds.width) {
			position.x = aRectangle.x;
		}
		if (aRectangle.y + aRectangle.height >= clipBounds.y + clipBounds.height) {
			position.y = aRectangle.y - (clipBounds.height - aRectangle.height);
		}
		if (aRectangle.y <= clipBounds.y || aRectangle.height >= clipBounds.height) {
			position.y = aRectangle.y;
		}

		this.scrollPane().setScrollPosition(position);
	}

	/**
	 * Answer the receiver's current scroll offset.
	 * 
	 * @return java.awt.Point
	 * @see jp.co.sra.jun.graphics.framework.JunGraphicView#scrollOffset()
	 * @category scrolling
	 */
	public Point scrollOffset() {
		return this.scrollPane().getScrollPosition();
	}

	/**
	 * Position the view to a particular place, updating the display.
	 * 
	 * @param aPoint java.awt.Point
	 * @see jp.co.sra.jun.graphics.framework.JunGraphicView#scrollTo_(java.awt.Point)
	 * @category scrolling
	 */
	public void scrollTo_(Point aPoint) {
		if (this.scrollOffset().equals(aPoint)) {
			return;
		}

		this.scrollPane().setScrollPosition(aPoint);
	}

	/**
	 * Update the receiver according to the change notification from the model.
	 * 
	 * @param evt jp.co.sra.smalltalk.DependentEvent
	 * @see jp.co.sra.smalltalk.DependentListener#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent evt) {
		StSymbol aSymbol = evt.getAspect();
		if (aSymbol == $("redisplay")) {
			this.updateCanvasSize();
			return;
		}
		if (aSymbol == $("selection") && evt.getParameter() instanceof Rectangle) {
			this.getGraphicModel().recomposeIn_((Rectangle) evt.getParameter());
			this.canvas().repaint();
			return;
		}

		this.getGraphicModel().recomposeIn_(new Rectangle(new Point(0, 0), this.canvas().getSize()));
		// this.fixScrollingOffset();
		super.update_(evt);
	}

	/**
	 * Update the canvas size.
	 * 
	 * @category updating
	 */
	protected void updateCanvasSize() {
		this.getGraphicModel().recomposeIn_(new Rectangle(new Point(0, 0), this.scrollPane().getViewportSize()));
		if (this.getGraphicModel().displayObject() != null) {
			this.canvas().setSize(this.getGraphicModel().displayObject().bounds().getSize());
		}
		this.scrollPane().doLayout();
	}

	/**
	 * Answer the graphic view.
	 * 
	 * @return jp.co.sra.smalltalk.StViewCanvas
	 * @category user interface
	 */
	public StViewCanvas canvas() {
		if (canvas == null) {
			canvas = new StViewCanvas(this.getGraphicModel()) {
				/**
				 * Answer the receiver's default controller
				 * 
				 * @return jp.co.sra.smalltalk.StController
				 * @see jp.co.sra.smalltalk.StViewJPanel#defaultController()
				 * @category controller accessing
				 */
				protected StController defaultController() {
					return new JunGraphicController();
				}

				/**
				 * Display the receiver on the graphics.
				 * 
				 * @param aGraphics java.awt.Graphics
				 * @see jp.co.sra.smalltalk.StViewJPanel#displayOn_(java.awt.Graphics)
				 * @category displaying
				 */
				public void displayOn_(Graphics aGraphics) {
					JunGraphicViewAwt.this.displayCanvasOn_(aGraphics);
				}
			};
			canvas.setBackground(Color.white);
		}

		return canvas;
	}

	/**
	 * Answer the scroll pane.
	 * 
	 * @return java.awt.ScrollPane
	 * @category user interface
	 */
	protected ScrollPane scrollPane() {
		if (scrollPane == null) {
			scrollPane = new ScrollPane(ScrollPane.SCROLLBARS_ALWAYS);
			scrollPane.addComponentListener(new ComponentAdapter() {
				public void componentResized(ComponentEvent e) {
					JunGraphicViewAwt.this.updateCanvasSize();
				}
			});
			scrollPane.add(this.canvas());
			this.updateCanvasSize();
		}

		return scrollPane;
	}
}
