package jp.co.sra.jun.graphics.abstracts;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.util.ArrayList;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StRectangle;

/**
 * JunAbstractItem class
 * 
 *  @author    Ryouichi Matsuda
 *  @created   2003/12/15 (by Ryouichi Matsuda)
 *  @updated   2006/11/22 (by m-asada)
 *  @version   699 (with StPL8.9) based on Jun637 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunAbstractItem.java,v 8.11 2008/02/20 06:32:15 nisinaka Exp $
 */
public class JunAbstractItem extends JunAbstractVisual {
	protected Point originPoint;
	protected Point extentPoint;
	protected int borderWidth;
	protected Color borderColor;
	protected boolean emphasisState;
	protected boolean hiddenState;
	protected LookPreferences lookPreferences;

	/**
	 * LookPreferences class
	 */
	public static class LookPreferences {
		protected Color foregroundColor;
		protected Color backgroundColor;
		protected Color selectionForegroundColor;
		protected Color selectionBackgroundColor;

		/**
		 * Answer the foreground color
		 * 
		 * @return java.awt.Color
		 * @category visual properties
		 */
		public Color foregroundColor() {
			return foregroundColor;
		}

		/**
		 * Set the foreground color.
		 * 
		 * @param aColor java.awt.Color
		 * @category visual properties
		 */
		public void setForegroundColor(Color aColor) {
			foregroundColor = aColor;
		}

		/**
		 * Answer the background color
		 * 
		 * @return java.awt.Color
		 * @category visual properties
		 */
		public Color backgroundColor() {
			return backgroundColor;
		}

		/**
		 * Set the background color.
		 * 
		 * @param aColor java.awt.Color
		 * @category visual properties
		 */
		public void setBackgroundColor(Color aColor) {
			backgroundColor = aColor;
		}

		/**
		 * Answer the selection foreground color
		 * 
		 * @return java.awt.Color
		 * @category visual properties
		 */
		public Color selectionForegroundColor() {
			return selectionForegroundColor;
		}

		/**
		 * Set the selection foreground color.
		 * 
		 * @param aColor java.awt.Color
		 * @category visual properties
		 */
		public void setSelectionForegroundColor(Color aColor) {
			selectionForegroundColor = aColor;
		}

		/**
		 * Answer the selection background color
		 * 
		 * @return java.awt.Color
		 * @category visual properties
		 */
		public Color selectionBackgroundColor() {
			return selectionBackgroundColor;
		}

		/**
		 * Set the selection background color.
		 * 
		 * @param aColor java.awt.Color
		 * @category visual properties
		 */
		public void setSelectionBackgroundColor(Color aColor) {
			selectionBackgroundColor = aColor;
		}
	}

	/**
	 * Create a new instance of <code>JunAbstractItem</code> and initialize it.
	 * 
	 * @param originPoint java.awt.Point
	 * @param extentPoint java.awt.Point
	 * @category Instance creation
	 */
	public JunAbstractItem(Point originPoint, Point extentPoint) {
		super();
		this.originPoint_(originPoint);
		this.extentPoint_(extentPoint);
	}

	/**
	 * Create a new instance of <code>JunAbstractItem</code> and initialize it.
	 * Only for the subclasses.
	 * 
	 * @category Instance creation
	 */
	protected JunAbstractItem() {
		super();
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.graphics.abstracts.JunAbstractVisual#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		originPoint = null;
		extentPoint = null;
		borderWidth = 0;
		borderColor = null;
		emphasisState = false;
		hiddenState = false;
		lookPreferences = null;
	}

	/**
	 * Answer the receiver's origin point.
	 * 
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point originPoint() {
		if (originPoint == null) {
			originPoint = new Point(0, 0);
		}
		return originPoint;
	}

	/**
	 * Set the receiver's origin point.
	 * 
	 * @param aPoint java.awt.Point
	 * @category accessing
	 */
	public void originPoint_(Point aPoint) {
		originPoint = aPoint;
	}

	/**
	 * Answer the receiver's extent point.
	 * 
	 * @return Point java.awt.Point
	 * @category accessing
	 */
	public Point extentPoint() {
		if (extentPoint == null) {
			extentPoint = new Point(0, 0);
		}
		return extentPoint;
	}

	/**
	 * Set the receiver's extent point.
	 * 
	 * @param aPoint java.awt.Point
	 * @category accessing
	 */
	public void extentPoint_(Point aPoint) {
		extentPoint = aPoint;
	}

	/**
	 * Answer the receiver's emphasis state.
	 * 
	 * @return boolean
	 * @category accessing
	 */
	public boolean emphasisState() {
		return emphasisState;
	}

	/**
	 * Set the receiver's emphasis state.
	 * 
	 * @param aBoolean boolean
	 * @category accessing
	 */
	public void emphasisState_(boolean aBoolean) {
		emphasisState = aBoolean;
	}

	/**
	 * Answer the receiver's items.
	 * 
	 * @return jp.co.sra.jun.graphics.abstracts.JunAbstractItem[]
	 * @category accessing
	 */
	public JunAbstractItem[] items() {
		final ArrayList list = new ArrayList();
		this.itemsDo_(new StBlockClosure() {
			public Object value_(Object item) {
				list.add(item);
				return null;
			}
		});
		return (JunAbstractItem[]) list.toArray(new JunAbstractItem[list.size()]);
	}

	/**
	 * Answer the receiver's all items.
	 * 
	 * @return jp.co.sra.jun.graphics.abstracts.JunAbstractItem[]
	 * @category accessing
	 */
	public JunAbstractItem[] allItems() {
		final ArrayList list = new ArrayList();
		this.allItemsDo_(new StBlockClosure() {
			public Object value_(Object item) {
				list.add(item);
				return null;
			}
		});
		return (JunAbstractItem[]) list.toArray(new JunAbstractItem[list.size()]);
	}

	/**
	 * Answer the receiver with all items.
	 * 
	 * @return jp.co.sra.jun.graphics.abstracts.JunAbstractItem[]
	 * @category accessing
	 */
	public JunAbstractItem[] withAllItems() {
		final ArrayList list = new ArrayList();
		this.withAllItemsDo_(new StBlockClosure() {
			public Object value_(Object item) {
				list.add(item);
				return null;
			}
		});
		return (JunAbstractItem[]) list.toArray(new JunAbstractItem[list.size()]);
	}

	/**
	 * Answer the receiver's bounds inside.
	 * 
	 * @return java.awt.Rectangle
	 * @category bounds accessing
	 */
	public Rectangle insideBounds() {
		return (new StRectangle(this.bounds())).insetBy_(this.borderWidth()).toRectangle();
	}

	/**
	 * Answer the receiver's preferred bounds.
	 * 
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.jun.graphics.Abstract.JunAbstractVisual#preferredBounds()
	 * @category bounds accessing
	 */
	public Rectangle preferredBounds() {
		return StRectangle.Origin_extent_(this.originPoint(), this.extentPoint()).toRectangle();
	}

	/**
	 * Display the receiver on the graphics.
	 *
	 * @see jp.co.sra.smalltalk.StDisplayable#displayOn_(java.awt.Graphics)
	 * @category displaying 
	 */
	public void displayOn_(Graphics aGraphics) {
		// no operation
	}

	/**
	 * Answer true if the bounds of the receiver contains the specified point, otherwise false.
	 * 
	 * @param aPoint java.awt.Point
	 * @return boolean
	 * @category testing
	 */
	public boolean containsPoint_(Point aPoint) {
		return this.bounds().contains(aPoint);
	}

	/**
	 * Answer true if the receiver is set to be emphasized, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isEmphasis() {
		return this.emphasisState();
	}
	
	/**
	 * Answer true if the receiver is item object, otherwise false.
	 * 
	 * @return boolean
	 * @see jp.co.sra.jun.graphics.abstracts.JunAbstractVisual#isItem()
	 * @category testing
	 */
	public boolean isItem() {
		return true;
	}

	/**
	 * Answer the receiver's current border width. 
	 * 
	 * @return int
	 * @category visual properties
	 */
	public int borderWidth() {
		return borderWidth;
	}

	/**
	 * Set the receiver's new border width.
	 * 
	 * @param newBorderWidth int
	 * @category visual properties
	 */
	public void borderWidth_(int newBorderWidth) {
		borderWidth = newBorderWidth;
	}

	/**
	 * Answer the receiver's current border color.
	 * 
	 * @return java.awt.Color
	 * @category visual properties
	 */
	public Color borderColor() {
		if (borderColor == null) {
			borderColor = Color.black;
		}
		return borderColor;
	}

	/**
	 * Set the receiver's new border color.
	 * 
	 * @param newBorderColor java.awt.Color
	 * @category visual properties
	 */
	public void borderColor_(Color newBorderColor) {
		borderColor = newBorderColor;
	}

	/**
	 * Answer the receiver's current look preferences.
	 * 
	 * @return jp.co.sra.jun.graphics.abstracts.JunAbstractItem.LookPreferences
	 * @category visual properties
	 */
	public LookPreferences lookPreferences() {
		return lookPreferences;
	}

	/**
	 * Set the receiver's new look preferences.
	 * 
	 * @param newLookPreferences jp.co.sra.jun.graphics.abstracts.JunAbstractItem.LookPreferences
	 * @category visual properties
	 */
	public void lookPreferences_(LookPreferences newLookPreferences) {
		lookPreferences = newLookPreferences;
	}

	/**
	 * Answer the receiver's current foreground color.
	 * 
	 * @return java.awt.Color
	 * @see jp.co.sra.jun.graphics.abstracts.JunAbstractVisual#foregroundColor()
	 * @category visual properties
	 */
	public Color foregroundColor() {
		if (this.lookPreferences() == null) {
			return super.foregroundColor();
		}
		return this.lookPreferences().foregroundColor();
	}

	/**
	 * Set the receiver's new foreground color.
	 * 
	 * @param aColor java.awt.Color
	 * @category visual properties
	 */
	public void foregroundColor_(Color aColor) {
		if (this.lookPreferences() == null) {
			return;
		}
		this.lookPreferences().setForegroundColor(aColor);
	}

	/**
	 * Answer the receiver's current background color.
	 * 
	 * @return java.awt.Color
	 * @see jp.co.sra.jun.graphics.abstracts.JunAbstractVisual#backgroundColor()
	 * @category visual properties
	 */
	public Color backgroundColor() {
		if (this.lookPreferences() == null) {
			return super.backgroundColor();
		}
		return this.lookPreferences().backgroundColor();
	}

	/**
	 * Set the receiver's new background color.
	 * 
	 * @param aColor java.awt.Color
	 * @category visual properties
	 */
	public void backgroundColor_(Color aColor) {
		if (this.lookPreferences() == null) {
			return;
		}
		this.lookPreferences().setBackgroundColor(aColor);
	}

	/**
	 * Answer the receiver's current selection foreground color.
	 * 
	 * @return java.awt.Color
	 * @see jp.co.sra.jun.graphics.abstracts.JunAbstractVisual#selectionForegroundColor()
	 * @category visual properties
	 */
	public Color selectionForegroundColor() {
		if (this.lookPreferences() == null) {
			return super.selectionForegroundColor();
		}
		return this.lookPreferences().selectionForegroundColor();
	}

	/**
	 * Set the receiver's new selection foreground color.
	 * 
	 * @param aColor java.awt.Color
	 * @category visual properties
	 */
	public void selectionSelectionForegroundColor_(Color aColor) {
		if (this.lookPreferences() == null) {
			return;
		}
		this.lookPreferences().setSelectionForegroundColor(aColor);
	}

	/**
	 * Answer the receiver's current selection background color.
	 * 
	 * @return java.awt.Color
	 * @see jp.co.sra.jun.graphics.abstracts.JunAbstractVisual#selectionBackgroundColor()
	 * @category visual properties
	 */
	public Color selectionBackgroundColor() {
		if (this.lookPreferences() == null) {
			return super.selectionBackgroundColor();
		}
		return this.lookPreferences().selectionBackgroundColor();
	}

	/**
	 * Set the receiver's new selection background color.
	 * 
	 * @param aColor java.awt.Color
	 * @category visual properties
	 */
	public void selectionBackgroundColor_(Color aColor) {
		if (this.lookPreferences() == null) {
			return;
		}
		this.lookPreferences().setSelectionBackgroundColor(aColor);
	}
}
