package jp.co.sra.jun.goodies.wavelet;

import jp.co.sra.jun.collections.sequences.JunDoubleMatrix;

/**
 * JunDiscreteWavelet2dTransformation class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2007/05/08 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun660 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunDiscreteWavelet2dTransformation.java,v 8.6 2008/02/20 06:32:14 nisinaka Exp $
 */
public class JunDiscreteWavelet2dTransformation extends JunDiscreteWaveletTransformation {
	protected JunDoubleMatrix sourceCoefficients;
	protected JunDoubleMatrix scalingCoefficients;
	protected JunDoubleMatrix[] waveletCoefficients;
	protected JunDoubleMatrix recomposedCoefficients;

	/**
	 * Create a new instance of <code>JunDiscreteWavelet2dTransformation</code> and initialize it.
	 * 
	 * @param sourceCollection jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @category Instance creation
	 */
	public JunDiscreteWavelet2dTransformation(JunDoubleMatrix sourceCollection) {
		super();
		this.sourceCoefficients_(sourceCollection);
	}

	/**
	 * Create a new instance of <code>JunDiscreteWavelet2dTransformation</code> and initialize it.
	 * 
	 * @param scalingCollection jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @param waveletCollection jp.co.sra.jun.collections.sequences.JunDoubleMatrix[]
	 * @category Instance creation
	 */
	public JunDiscreteWavelet2dTransformation(JunDoubleMatrix scalingCollection, JunDoubleMatrix[] waveletCollection) {
		super();
		this.scalingCoefficients_(scalingCollection);
		this.waveletCoefficients_(waveletCollection);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.goodies.wavelet.JunWaveletTransformation#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		sourceCoefficients = null;
		scalingCoefficients = null;
		waveletCoefficients = null;
		recomposedCoefficients = null;
	}

	/**
	 * Answer the receiver's recomposed coefficients.
	 * 
	 * @return jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @category accessing
	 */
	public JunDoubleMatrix recomposedCoefficients() {
		if (recomposedCoefficients == null) {
			this.computeRecomposedCoefficients();
		}
		return recomposedCoefficients;
	}

	/**
	 * Answer the receiver's scaling coefficients.
	 * 
	 * @return jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @category accessing
	 */
	public JunDoubleMatrix scalingCoefficients() {
		if (scalingCoefficients == null) {
			this.computeScalingAndWaveletCoefficients();
		}
		return scalingCoefficients;
	}

	/**
	 * Set the receiver's scaling coefficients.
	 * 
	 * @param scalingCollection jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @category accessing
	 */
	public void scalingCoefficients_(JunDoubleMatrix scalingCollection) {
		scalingCoefficients = scalingCollection;
		recomposedCoefficients = null;
	}

	/**
	 * Answer the receiver's source coefficients.
	 * 
	 * @return jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @category accessing
	 */
	public JunDoubleMatrix sourceCoefficients() {
		return sourceCoefficients;
	}

	/**
	 * Set the receiver's source coefficients.
	 * 
	 * @param valueCollection jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @category accessing
	 */
	public void sourceCoefficients_(JunDoubleMatrix valueCollection) {
		sourceCoefficients = valueCollection;
		scalingCoefficients = null;
		recomposedCoefficients = null;
	}

	/**
	 * Answer the receiver's wavelet coefficients.
	 * 
	 * @return jp.co.sra.jun.collections.sequences.JunDoubleMatrix[]
	 * @category accessing
	 */
	public JunDoubleMatrix[] waveletCoefficients() {
		if (waveletCoefficients == null) {
			this.computeScalingAndWaveletCoefficients();
		}
		return waveletCoefficients;
	}

	/**
	 * Set the receiver's wavelet coefficients.
	 * 
	 * @param waveletCollection jp.co.sra.jun.collections.sequences.JunDoubleMatrix[]
	 * @category accessing
	 */
	public void waveletCoefficients_(JunDoubleMatrix[] waveletCollection) {
		waveletCoefficients = waveletCollection;
		recomposedCoefficients = null;
	}

	/**
	 * Answer the receiver's diagonal wavelet coefficients.
	 * 
	 * @return jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @category accessing
	 */
	public JunDoubleMatrix diagonalWaveletCoefficients() {
		return this.waveletCoefficients()[2];
	}

	/**
	 * Answer the receiver's horizontal wavelet coefficients.
	 * 
	 * @return jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @category accessing
	 */
	public JunDoubleMatrix horizontalWaveletCoefficients() {
		return this.waveletCoefficients()[0];
	}

	/**
	 * Answer the receiver's vertical wavelet coefficients.
	 * 
	 * @return jp.co.sra.jun.collections.sequences.JunDoubleMatrix
	 * @category accessing
	 */
	public JunDoubleMatrix verticalWaveletCoefficients() {
		return this.waveletCoefficients()[1];
	}

	/**
	 * Apply the specified object to the receiver.
	 * 
	 * @param anObject java.lang.Object
	 * @return jp.co.sra.jun.goodies.wavelet.JunWaveletTransformation
	 * @throws java.lang.IllegalArgumentException
	 * @see jp.co.sra.jun.goodies.wavelet.JunWaveletTransformation#applyTo_(java.lang.Object)
	 * @category applying transformation
	 */
	public JunWaveletTransformation applyTo_(Object anObject) {
		if (anObject instanceof JunDoubleMatrix == false) {
			throw new IllegalArgumentException("anObject must be a JunDoubleMatrix.");
		}

		this.sourceCoefficients_((JunDoubleMatrix) anObject);
		this.scalingCoefficients();
		this.waveletCoefficients();
		return this;
	}

	/**
	 * Apply a transformation 'waveletTransformation' to the receiver.
	 * 
	 * @param waveletTransformation jp.co.sra.jun.goodies.wavelet.JunWaveletTransformation
	 * @return jp.co.sra.jun.goodies.wavelet.JunWaveletTransformation
	 * @throws java.lang.IllegalArgumentException
	 * @see jp.co.sra.jun.goodies.wavelet.JunWaveletTransformation#transform_(jp.co.sra.jun.goodies.wavelet.JunWaveletTransformation)
	 * @category applying transformation
	 */
	public JunWaveletTransformation transform_(JunWaveletTransformation waveletTransformation) {
		if (waveletTransformation instanceof JunDiscreteWavelet2dTransformation == false) {
			throw new IllegalArgumentException("waveletTransformation must be a JunDiscreteWavelet2dTransformation.");
		}

		JunDiscreteWavelet2dTransformation discreteWaveletTransformation = (JunDiscreteWavelet2dTransformation) waveletTransformation;
		return (JunWaveletTransformation) _New(this.getClass(), (discreteWaveletTransformation.sourceCoefficients() == null ? discreteWaveletTransformation.recomposedCoefficients() : null));
	}

	/**
	 * Compute receiver's recomposed coefficients.
	 * 
	 * @category private
	 */
	protected void computeRecomposedCoefficients() {
		if (scalingCoefficients == null) {
			return;
		}
		if (waveletCoefficients == null) {
			return;
		}

		int halfRowSize = scalingCoefficients.rowSize();
		int rowSize = halfRowSize * 2;
		int halfColumnSize = scalingCoefficients.columnSize();
		int columnSize = halfColumnSize * 2;
		JunDoubleMatrix targetScalingCoefficients = (JunDoubleMatrix) scalingCoefficients.transpose();
		JunDoubleMatrix horizontalWaveletCoefficients = (JunDoubleMatrix) this.horizontalWaveletCoefficients().transpose();
		JunDoubleMatrix verticalWaveletCoefficients = (JunDoubleMatrix) this.verticalWaveletCoefficients().transpose();
		JunDoubleMatrix diagonalWaveletCoefficients = (JunDoubleMatrix) this.diagonalWaveletCoefficients().transpose();
		JunDoubleMatrix xScalingCoefficients = new JunDoubleMatrix(halfColumnSize, rowSize);
		JunDoubleMatrix xWaveletCoefficients = new JunDoubleMatrix(halfColumnSize, rowSize);
		for (int index = 0; index < halfColumnSize; index++) {
			JunDiscreteWavelet1dTransformation waveletTransformation = new JunDiscreteWavelet1dTransformation(targetScalingCoefficients._doubleValuesBasicAt_(index), horizontalWaveletCoefficients._doubleValuesBasicAt_(index));
			xScalingCoefficients.basicAt_put_(index, waveletTransformation.recomposedCoefficients());
			waveletTransformation = new JunDiscreteWavelet1dTransformation(verticalWaveletCoefficients._doubleValuesBasicAt_(index), diagonalWaveletCoefficients._doubleValuesBasicAt_(index));
			xWaveletCoefficients.basicAt_put_(index, waveletTransformation.recomposedCoefficients());
		}
		xScalingCoefficients = (JunDoubleMatrix) xScalingCoefficients.transpose();
		xWaveletCoefficients = (JunDoubleMatrix) xWaveletCoefficients.transpose();
		JunDoubleMatrix targetRecomposedCoefficients = new JunDoubleMatrix(rowSize, columnSize);
		for (int index = 0; index < rowSize; index++) {
			JunDiscreteWavelet1dTransformation waveletTransformation = new JunDiscreteWavelet1dTransformation(xScalingCoefficients._doubleValuesBasicAt_(index), xWaveletCoefficients._doubleValuesBasicAt_(index));
			targetRecomposedCoefficients.basicAt_put_(index, waveletTransformation.recomposedCoefficients());
		}
		recomposedCoefficients = targetRecomposedCoefficients;
	}

	/**
	 * Compute receiver's scaling and wavelet coefficients
	 * 
	 * @category private
	 */
	protected void computeScalingAndWaveletCoefficients() {
		if (sourceCoefficients == null) {
			return;
		}

		int rowSize = sourceCoefficients.rowSize();
		int halfRowSize = rowSize / 2;
		int columnSize = sourceCoefficients.columnSize();
		int halfColumnSize = columnSize / 2;
		JunDoubleMatrix xScalingCoefficients = new JunDoubleMatrix(rowSize, halfColumnSize);
		JunDoubleMatrix xWaveletCoefficients = new JunDoubleMatrix(rowSize, halfColumnSize);
		for (int index = 0; index < rowSize; index++) {
			JunDiscreteWavelet1dTransformation waveletTransformation = new JunDiscreteWavelet1dTransformation(sourceCoefficients._doubleValuesBasicAt_(index));
			xScalingCoefficients.basicAt_put_(index, waveletTransformation.scalingCoefficients());
			xWaveletCoefficients.basicAt_put_(index, waveletTransformation.waveletCoefficients());
		}
		xScalingCoefficients = (JunDoubleMatrix) xScalingCoefficients.transpose();
		xWaveletCoefficients = (JunDoubleMatrix) xWaveletCoefficients.transpose();
		JunDoubleMatrix targetScalingCoefficients = new JunDoubleMatrix(halfColumnSize, halfRowSize);
		JunDoubleMatrix horizontalWaveletCoefficients = new JunDoubleMatrix(halfColumnSize, halfRowSize);
		JunDoubleMatrix verticalWaveletCoefficients = new JunDoubleMatrix(halfColumnSize, halfRowSize);
		JunDoubleMatrix diagonalWaveletCoefficients = new JunDoubleMatrix(halfColumnSize, halfRowSize);
		for (int index = 0; index < halfColumnSize; index++) {
			JunDiscreteWavelet1dTransformation waveletTransformation = new JunDiscreteWavelet1dTransformation(xScalingCoefficients._doubleValuesBasicAt_(index));
			targetScalingCoefficients.basicAt_put_(index, waveletTransformation.scalingCoefficients());
			horizontalWaveletCoefficients.basicAt_put_(index, waveletTransformation.waveletCoefficients());
			waveletTransformation = new JunDiscreteWavelet1dTransformation(xWaveletCoefficients._doubleValuesBasicAt_(index));
			verticalWaveletCoefficients.basicAt_put_(index, waveletTransformation.scalingCoefficients());
			diagonalWaveletCoefficients.basicAt_put_(index, waveletTransformation.waveletCoefficients());
		}
		targetScalingCoefficients = (JunDoubleMatrix) targetScalingCoefficients.transpose();
		horizontalWaveletCoefficients = (JunDoubleMatrix) horizontalWaveletCoefficients.transpose();
		verticalWaveletCoefficients = (JunDoubleMatrix) verticalWaveletCoefficients.transpose();
		diagonalWaveletCoefficients = (JunDoubleMatrix) diagonalWaveletCoefficients.transpose();
		scalingCoefficients = targetScalingCoefficients;
		waveletCoefficients = new JunDoubleMatrix[] { horizontalWaveletCoefficients, verticalWaveletCoefficients, diagonalWaveletCoefficients };
	}
}
