package jp.co.sra.jun.goodies.wavelet;

/**
 * JunDiscreteWavelet1dTransformation class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2007/05/07 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun660 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunDiscreteWavelet1dTransformation.java,v 8.6 2008/02/20 06:32:14 nisinaka Exp $
 */
public class JunDiscreteWavelet1dTransformation extends JunDiscreteWaveletTransformation {
	protected double[] sourceCoefficients;
	protected double[] scalingCoefficients;
	protected double[] waveletCoefficients;
	protected double[] recomposedCoefficients;

	/**
	 * Create a new instance of <code>JunDiscreteWavelet1dTransformation</code> and initialize it.
	 * 
	 * @param sourceCollection double[]
	 * @category Instance creation
	 */
	public JunDiscreteWavelet1dTransformation(double[] sourceCollection) {
		super();
		this.sourceCoefficients_(sourceCollection);
	}

	/**
	 * Create a new instance of <code>JunDiscreteWavelet1dTransformation</code> and initialize it.
	 * 
	 * @param scalingCollection double[]
	 * @param waveletCollection double[]
	 * @category Instance creation
	 */
	public JunDiscreteWavelet1dTransformation(double[] scalingCollection, double[] waveletCollection) {
		super();
		this.scalingCoefficients_(scalingCollection);
		this.waveletCoefficients_(waveletCollection);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.goodies.wavelet.JunWaveletTransformation#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		sourceCoefficients = null;
		scalingCoefficients = null;
		waveletCoefficients = null;
		recomposedCoefficients = null;
	}

	/**
	 * Answer the receiver's recomposed coefficients.
	 * 
	 * @return double[]
	 * @category accessing
	 */
	public double[] recomposedCoefficients() {
		if (recomposedCoefficients == null) {
			this.computeRecomposedCoefficients();
		}
		return recomposedCoefficients;
	}

	/**
	 * Answer the receiver's scaling coefficients.
	 * 
	 * @return double[]
	 * @category accessing
	 */
	public double[] scalingCoefficients() {
		if (scalingCoefficients == null) {
			this.computeScalingAndWaveletCoefficients();
		}
		return scalingCoefficients;
	}

	/**
	 * Set the receiver's scaling coefficients.
	 * 
	 * @param scalingCollection double[]
	 * @category accessing
	 */
	public void scalingCoefficients_(double[] scalingCollection) {
		scalingCoefficients = scalingCollection;
		recomposedCoefficients = null;
	}

	/**
	 * Answer the receiver's source coefficients.
	 * 
	 * @return double[]
	 * @category accessing
	 */
	public double[] sourceCoefficients() {
		return sourceCoefficients;
	}

	/**
	 * Set the receiver's source coefficients.
	 * 
	 * @param valueCollection double[]
	 * @category accessing
	 */
	public void sourceCoefficients_(double[] valueCollection) {
		sourceCoefficients = valueCollection;
		scalingCoefficients = null;
		recomposedCoefficients = null;
	}

	/**
	 * Answer the receiver's wavelet coefficients.
	 * 
	 * @return double[]
	 * @category accessing
	 */
	public double[] waveletCoefficients() {
		if (waveletCoefficients == null) {
			this.computeScalingAndWaveletCoefficients();
		}
		return waveletCoefficients;
	}

	/**
	 * Set the receiver's wavelet coefficients.
	 * 
	 * @param waveletCollection double[]
	 * @category accessing
	 */
	public void waveletCoefficients_(double[] waveletCollection) {
		waveletCoefficients = waveletCollection;
		recomposedCoefficients = null;
	}

	/**
	 * Apply the specified object to the receiver.
	 * 
	 * @param anObject java.lang.Object
	 * @return jp.co.sra.jun.goodies.wavelet.JunWaveletTransformation
	 * @throws java.lang.IllegalArgumentException
	 * @see jp.co.sra.jun.goodies.wavelet.JunWaveletTransformation#applyTo_(java.lang.Object)
	 * @category applying transformation
	 */
	public JunWaveletTransformation applyTo_(Object anObject) {
		if (anObject instanceof double[] == false) {
			throw new IllegalArgumentException("anObject must be a double[].");
		}

		this.sourceCoefficients_((double[]) anObject);
		this.scalingCoefficients();
		this.waveletCoefficients();
		return this;
	}

	/**
	 * Apply a transformation 'waveletTransformation' to the receiver.
	 * 
	 * @param waveletTransformation jp.co.sra.jun.goodies.wavelet.JunWaveletTransformation
	 * @return jp.co.sra.jun.goodies.wavelet.JunWaveletTransformation
	 * @throws java.lang.IllegalArgumentException
	 * @see jp.co.sra.jun.goodies.wavelet.JunWaveletTransformation#transform_(jp.co.sra.jun.goodies.wavelet.JunWaveletTransformation)
	 * @category applying transformation
	 */
	public JunWaveletTransformation transform_(JunWaveletTransformation waveletTransformation) {
		if (waveletTransformation instanceof JunDiscreteWavelet1dTransformation == false) {
			throw new IllegalArgumentException("anObject must be a JunDiscreteWavelet1dTransformation.");
		}

		JunDiscreteWavelet1dTransformation discreteWaveletTransformation = (JunDiscreteWavelet1dTransformation) waveletTransformation;
		return (JunWaveletTransformation) _New(this.getClass(), (discreteWaveletTransformation.sourceCoefficients() == null ? discreteWaveletTransformation.recomposedCoefficients() : null));
	}

	/**
	 * Compute receiver's recomposed coefficients.
	 * 
	 * @category private
	 */
	protected void computeRecomposedCoefficients() {
		if (scalingCoefficients == null) {
			return;
		}
		if (waveletCoefficients == null) {
			return;
		}

		int size = scalingCoefficients.length;
		recomposedCoefficients = new double[size * 2];
		int offset = Math.max(1024, size);
		for (int index = 0; index < size; index++) {
			int n = index;
			int nn = n * 2;
			recomposedCoefficients[nn] = 0.0;
			recomposedCoefficients[nn + 1] = 0.0;
			for (int i = 0; i < daubechieScalingSequence.length / 2; i++) {
				int k = i;
				int kk = k * 2;
				int j = (n - k + offset) % size;
				double s = scalingCoefficients[j];
				double w = waveletCoefficients[j];
				recomposedCoefficients[nn] = recomposedCoefficients[nn] + daubechieScalingSequence[kk] * s + daubechieWaveletSequence[kk] * w;
				recomposedCoefficients[nn + 1] = recomposedCoefficients[nn + 1] + daubechieScalingSequence[kk + 1] * s + daubechieWaveletSequence[kk + 1] * w;
			}
		}
	}

	/**
	 * Compute receiver's scaling and wavelet coefficients
	 * 
	 * @category private
	 */
	protected void computeScalingAndWaveletCoefficients() {
		if (sourceCoefficients == null) {
			return;
		}

		int length = sourceCoefficients.length;
		int size = length / 2;
		scalingCoefficients = new double[size];
		waveletCoefficients = new double[size];
		for (int index = 0; index < size; index++) {
			scalingCoefficients[index] = 0.0;
			waveletCoefficients[index] = 0.0;
			for (int i = 0; i < daubechieScalingSequence.length; i++) {
				int j = (i + (2 * index)) % length;
				double v = sourceCoefficients[j];
				scalingCoefficients[index] = scalingCoefficients[index] + daubechieScalingSequence[i] * v;
				waveletCoefficients[index] = waveletCoefficients[index] + daubechieWaveletSequence[i] * v;
			}
		}
	}
}
