package jp.co.sra.jun.goodies.utilities;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.StringReader;
import java.io.Writer;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StInterval;
import jp.co.sra.smalltalk.StSymbol;

import jp.co.sra.jun.system.framework.JunAbstractObject;
import jp.co.sra.jun.system.framework.JunDialog;

/**
 * JunStringUtility class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2003/05/27 (by Mitsuhiro Asada)
 *  @updated   2006/04/17 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on Jun697 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunStringUtility.java,v 8.18 2008/02/20 06:32:14 nisinaka Exp $
 */
public class JunStringUtility extends JunAbstractObject {
	/**
	 * Answer the canonical string using the specified table.
	 * 
	 * @param aString java.lang.String
	 * @param aMap java.util.Map
	 * @return java.lang.String
	 * @category Canonicals
	 */
	public static String CanonicalString_usingTable_(String aString, Map aMap) {
		StringBuffer aStringBuffer = new StringBuffer();

		for (int i = 0; i < aString.length(); i++) {
			Character aCharacter = new Character(aString.charAt(i));
			if (aMap.containsKey(aCharacter)) {
				aStringBuffer.append(aMap.get(aCharacter));
			} else {
				aStringBuffer.append(aCharacter);
			}
		}

		return aStringBuffer.toString();
	}

	/**
	 * Answer the canonical string as HTML.
	 * 
	 * @param aString java.lang.String
	 * @return java.lang.String
	 * @category Canonicals
	 */
	public static String HtmlCanonicalString_(String aString) {
		return HtmlCanonicalString_usingTable_(aString, HtmlCanonicalCharacterTable());
	}

	/**
	 * Answer the canonical string as HTML using the specified table.
	 * 
	 * @param aString java.lang.String
	 * @param aMap java.util.Map
	 * @return java.lang.String
	 * @category Canonicals
	 */
	public static String HtmlCanonicalString_usingTable_(String aString, Map aMap) {
		return CanonicalString_usingTable_(aString, aMap);
	}

	/**
	 * Answer the canonical character table as HTML.
	 * 
	 * @return java.util.Map
	 * @category Canonicals
	 */
	public static Map HtmlCanonicalCharacterTable() {
		return XmlCanonicalCharacterTable();
	}

	/**
	 * Write html tag and attribute string on a writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @param aString java.lang.String
	 * @param attributeString java.lang.String
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @throws java.io.IOException
	 * @category Canonicals
	 */
	public static void On_tag_attributes_do_(Writer aWriter, String aString, String attributeString, StBlockClosure aBlock) throws IOException {
		try {
			aWriter.write("<");
			aWriter.write(aString);
			if (attributeString != null && attributeString.length() > 0) {
				aWriter.write(" ");
				aWriter.write(attributeString);
			}
			aWriter.write(">");

			aBlock.value();
		} finally {
			aWriter.write("</");
			aWriter.write(aString);
			aWriter.write(">");
		}
	}

	/**
	 * Write html tag string on a writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @param aString java.lang.String
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @throws java.io.IOException
	 * @category Canonicals
	 */
	public static void On_tag_do_(Writer aWriter, String aString, StBlockClosure aBlock) throws IOException {
		On_tag_attributes_do_(aWriter, aString, null, aBlock);
	}

	/**
	 * Answer the canonical string as XML.
	 * 
	 * @param aString java.lang.String
	 * @return java.lang.String
	 * @category Canonicals
	 */
	public static String XmlCanonicalString_(String aString) {
		return XmlCanonicalString_usingTable_(aString, XmlCanonicalCharacterTable());
	}

	/**
	 * Answer the canonical string as XML using the specified table.
	 * 
	 * @param aString java.lang.String
	 * @param aMap java.util.Map
	 * @return java.lang.String
	 * @category Canonicals
	 */
	public static String XmlCanonicalString_usingTable_(String aString, Map aMap) {
		return CanonicalString_usingTable_(aString, aMap);
	}

	/**
	 * Answer the canonical character table as XML.
	 * 
	 * @return java.util.Map
	 * @category Canonicals
	 */
	public static Map XmlCanonicalCharacterTable() {
		Map aMap = new HashMap(5);
		aMap.put(new Character('&'), "&amp;");
		aMap.put(new Character('<'), "&lt;");
		aMap.put(new Character('>'), "&gt;");
		aMap.put(new Character('\''), "&apos;");
		aMap.put(new Character('"'), "&quot;");
		return aMap;
	}

	/**
	 * Answer true if the string matches the specified pattern, otherwise false.
	 * 
	 * @param string java.lang.String
	 * @param pattern java.lang.String
	 * @return boolean
	 * @category Comparing
	 */
	public static boolean StringMatch_and_(String string, String pattern) {
		char[] stringCharacters = new char[string.length() + 1];
		string.getChars(0, string.length(), stringCharacters, 0);
		stringCharacters[string.length()] = 0;
		char[] patternCharacters = new char[pattern.length() + 1];
		pattern.getChars(0, pattern.length(), patternCharacters, 0);
		patternCharacters[pattern.length()] = 0;
		return StringMatch_sIndex_and_pIndex_(stringCharacters, 0, patternCharacters, 0);
	}

	/**
	 * Answer true if the string matches the pattern, otherwise false.
	 * 
	 * @param string java.lang.String
	 * @param sindex int
	 * @param pattern java.lang.String
	 * @param pindex int
	 * @return boolean
	 * @category Comparing
	 */
	public static boolean StringMatch_sIndex_and_pIndex_(char[] string, int sindex, char[] pattern, int pindex) {
		int si = sindex;
		int pi = pindex;
		char scc = string[si];
		si++;
		char c = pattern[pi];
		pi++;
		if (c == '[') {
			boolean ok = false;
			char lc = 255;

			while ((c = pattern[pi++]) != 0) {
				if (c == ']') {
					if (ok) {
						return StringMatch_sIndex_and_pIndex_(string, si, pattern, pi);
					} else {
						return false;
					}
				} else {
					if (c == '-') {
						if (lc <= scc && scc <= pattern[pi]) {
							ok = true;
						}
						pi++;
					} else {
						lc = c;
						if (scc == c) {
							ok = true;
						}
					}
				}
			}
		}
		if (c == '?') {
			if (scc != 0) {
				return StringMatch_sIndex_and_pIndex_(string, si, pattern, pi);
			} else {
				return false;
			}
		}
		if (c == '*') {
			if (pattern[pi] == 0) {
				return true;
			}
			si--;
			while (string[si] != 0) {
				if (StringMatch_sIndex_and_pIndex_(string, si, pattern, pi)) {
					return true;
				}
				si++;
			}
			return false;
		}
		if (c == 0) {
			return scc == 0;
		}
		if (c != scc) {
			return false;
		}
		if (scc != 0) {
			return StringMatch_sIndex_and_pIndex_(string, si, pattern, pi);
		} else {
			return false;
		}
	}

	/**
	 * Shorten by ellipsis if too long.
	 *
	 * @return java.lang.String
	 * @param aString java.lang.String
	 * @param charCount int
	 * @category Converting
	 */
	public static String _ContractString_to_(String aString, int charCount) {
		String shortString = (aString == null) ? new String("") : new String(aString);
		if (aString.length() > charCount) {
			int half = charCount / 2;
			shortString = aString.substring(0, half - 1) + "..." + aString.substring(aString.length() - (charCount - half) + 2);
		}
		return shortString;
	}

	/**
	 * Answer the format string.
	 * 
	 * @param argumentString java.lang.String
	 * @param aValue java.lang.String
	 * @return java.lang.String
	 * @category Formatting
	 */
	public static String A_value_(String argumentString, String aValue) {
		String string = aValue.toString();
		int length;
		if (argumentString == null || argumentString.length() == 0) {
			length = string.length();
		} else {
			length = Integer.parseInt(Separate_(argumentString)[0]);
		}
		StringBuffer buffer = new StringBuffer();
		buffer.append(string.substring(0, Math.min(length, string.length())));
		for (int i = 0; i < length - string.length(); i++) {
			buffer.append(" ");
		}
		string = buffer.toString();
		return string;
	}

	/**
	 * Answer the format string.
	 * 
	 * @param argumentString java.lang.String
	 * @param aValue java.lang.String
	 * @return java.lang.String
	 * @category Formatting
	 */
	public static String E_value_(String argumentString, String aValue) {
		return aValue.toString();
	}

	/**
	 * Answer the format string.
	 * 
	 * @param argumentString java.lang.String
	 * @param aValue float
	 * @return java.lang.String
	 * @category Formatting
	 */
	public static String F_value_(String argumentString, float aValue) {
		char[] separators = Separators();
		char[] newSeparators = new char[separators.length + 1];
		for (int i = 0; i < separators.length; i++) {
			newSeparators[i] = separators[i];
		}
		newSeparators[newSeparators.length - 1] = '.';
		String[] collection = Separate_dividers_(argumentString, newSeparators);
		int length;
		int decial;
		if (collection == null || collection.length == 0) {
			length = 10;
			decial = 4;
		} else {
			length = Integer.parseInt(collection[0]);
			decial = Integer.parseInt(collection[collection.length - 1]);
		}
		StringBuffer pattern = new StringBuffer();
		if (aValue < 0.0) {
			pattern.append("-");
		}
		pattern.append("################.");
		for (int i = 0; i < decial; i++) {
			pattern.append("0");
		}
		String string = new DecimalFormat(pattern.toString()).format(aValue);
		StringBuffer buffer = new StringBuffer();
		for (int i = 0; i < length - string.length(); i++) {
			buffer.append(" ");
		}
		buffer.append(string.substring(0, Math.min(length, string.length())));
		string = buffer.toString();
		return string;
	}

	/**
	 * Answer the format string.
	 * 
	 * @param argumentString java.lang.String
	 * @param aValue int
	 * @return java.lang.String
	 * @category Formatting
	 */
	public static String I_value_(String argumentString, int aValue) {
		String string = Integer.toString(aValue);
		String[] collection = Separate_(argumentString);
		int length;
		if (collection == null || collection.length == 0) {
			length = string.length();
		} else {
			length = Integer.parseInt(collection[0]);
		}
		StringBuffer buffer = new StringBuffer();
		for (int i = 0; i < length - string.length(); i++) {
			buffer.append(" ");
		}
		buffer.append(string.substring(0, Math.min(length, string.length())));
		string = buffer.toString();
		return string;
	}

	/**
	 * Answer the spacing message selector.
	 * 
	 * @param messageSelector jp.co.sra.smalltlak.StSymbol
	 * @return java.lang.String
	 * @category Formatting
	 */
	public static String SpacingMessageSelector_(StSymbol messageSelector) {
		return SpacingMessageSelector_with_(messageSelector, " ");
	}

	/**
	 * Answer the spacing message selector with the specified spacing string.
	 * 
	 * @param messageSelector jp.co.sra.smalltlak.StSymbol
	 * @param spacingString java.lang.String
	 * @return java.lang.String
	 * @category Formatting
	 * @param messageSelector
	 */
	public static String SpacingMessageSelector_with_(StSymbol messageSelector, String spacingString) {
		String aString = messageSelector.toString();
		StringBuffer aBuffer = new StringBuffer();
		try {
			for (int i = 0; i < aString.length(); i++) {
				char aCharacter = aString.charAt(i);
				if (Character.isUpperCase(aCharacter)) {
					aBuffer.append(spacingString);
					aBuffer.append(Character.toLowerCase(aCharacter));
				} else {
					aBuffer.append(aCharacter);
				}
			}
		} finally {
			aString = aBuffer.toString();
		}
		return aString;
	}

	/**
	 * Answer the format string.
	 * 
	 * @param argumentString java.lang.String
	 * @return java.lang.String
	 * @category Formatting
	 */
	public static String X_(String argumentString) {
		int length;
		if (argumentString == null || argumentString.length() == 0) {
			length = 1;
		} else {
			length = Integer.parseInt(Separate_(argumentString)[0]);
		}
		StringBuffer buffer = new StringBuffer();
		for (int i = 0; i < length; i++) {
			buffer.append(" ");
		}
		String string = buffer.toString();
		return string;
	}

	/**
	 * Answer the format string.
	 * 
	 * @param aString java.lang.String
	 * @param aValue java.lang.Object
	 * @return java.lang.String
	 * @throws java.lang.IllegalArgumentException
	 * @category Formatting
	 */
	public static String Format_value_(String aString, Object aValue) {
		try {
			String formatString = SkipSeparators_(aString);
			char formatKind = formatString.toLowerCase().charAt(0);
			String formatArgument = formatString.substring(1);
			switch (formatKind) {
				case 'a':
					return A_value_(formatArgument, (String) aValue);
				case 'e':
					return E_value_(formatArgument, (String) aValue);
				case 'f':
					return F_value_(formatArgument, ((Number) aValue).floatValue());
				case 'i':
					return I_value_(formatArgument, ((Number) aValue).intValue());
				case 'x':
					return X_(formatArgument);
			}
		} catch (Exception e) {
			JunDialog.Warn_("unexpected error.");
			throw new IllegalArgumentException(aValue.toString());
		}
		return aValue.toString();
	}

	/**
	 * Answer the string with commas.
	 *  
	 * @param aString java.lang.String
	 * @return java.lang.String
	 * @category Formatting
	 */
	public static String StringWithCommas_(String aString) {
		char[] readStream = aString.toCharArray();
		StringBuffer writeStream = new StringBuffer();
		int index = 1;
		for (int i = readStream.length - 1; i >= 0; i--) {
			writeStream.append(readStream[i]);
			if (i != 0 && index % 3 == 0) {
				writeStream.append(',');
			}
			index++;
		}

		char[] contents = writeStream.toString().toCharArray();
		StringBuffer reverseStream = new StringBuffer();
		for (int i = contents.length - 1; i >= 0; i--) {
			reverseStream.append(contents[i]);
		}
		return reverseStream.toString();
	}

	/**
	 * Get lines from the specified reader.
	 * 
	 * @param aReader java.io.BufferedReader
	 * @return java.lang.String[]
	 * @throws java.io.IOException
	 * @category Reading
	 */
	public static String[] GetLines_(BufferedReader aReader) throws IOException {
		Collection aCollection = new ArrayList();
		String oneLine;
		while ((oneLine = aReader.readLine()) != null) {
			aCollection.add(oneLine);
		}
		return (String[]) aCollection.toArray(new String[aCollection.size()]);
	}

	/**
	 * Get lines from the specified string.
	 * 
	 * @param aString java.lang.String
	 * @return java.lang.String[]
	 * @throws java.io.IOException
	 * @category Reading
	 */
	public static String[] GetLinesFromString_(String aString) throws IOException {
		String[] anArray = null;
		BufferedReader aReader = new BufferedReader(new StringReader(aString));
		try {
			anArray = GetLines_(aReader);
		} finally {
			if (aReader != null) {
				aReader.close();
			}
		}
		return anArray;
	}

	/**
	 * Get the tokens from the specified reader.
	 * 
	 * @param aReader java.io.BufferedReader
	 * @return java.lang.String[][]
	 * @throws IOException
	 * @category Reading
	 */
	public static String[][] GetTokens_(BufferedReader aReader) throws IOException {
		String[] anArray = GetLines_(aReader);
		String[][] aCollection = new String[anArray.length][];
		for (int i = 0; i < anArray.length; i++) {
			aCollection[i] = JunStringUtility.Separate_(anArray[i]);
		}
		return aCollection;
	}

	/**
	 * Get the tokens from the specified string.
	 * 
	 * @param aString java.lang.String
	 * @return java.lang.String[][]
	 * @throws java.io.IOException
	 * @category Reading
	 */
	public static String[][] GetTokensFromString_(String aString) throws IOException {
		String[][] anArray = null;
		BufferedReader aReader = new BufferedReader(new StringReader(aString));
		try {
			anArray = GetTokens_(aReader);
		} finally {
			if (aReader != null) {
				aReader.close();
			}
		}
		return anArray;
	}

	/**
	 * Answer the string with commas.
	 *  
	 * @param aNumber double
	 * @return java.lang.String
	 * @throws java.lang.NumberFormatException
	 * @category Formatting
	 */
	public static String StringWithCommasFor_(double aNumber) {
		NumberFormat format = new DecimalFormat(
				"###,###.#####################################################################################################################################################################################################################################################################################################################################");
		return format.format(aNumber);
	}

	/**
	 * Answer separated strings.
	 *
	 * @param separateCollection java.lang.String
	 * @return java.lang.String[]
	 * @category Separating
	 */
	public static String[] Separate_(String separateCollection) {
		return Separate_dividers_(separateCollection, Separators());
	}

	/**
	 * Answer separated strings.
	 *
	 * @return java.lang.String[]
	 * @param separateCollection java.lang.String
	 * @param dividerCollection char[]
	 * @category Separating
	 */
	public static String[] Separate_dividers_(String separateCollection, char[] dividerCollection) {
		List indexCollection = new ArrayList();
		indexCollection.add(new Integer(-1));
		for (int index = 0; index < separateCollection.length(); index++) {
			char each = separateCollection.charAt(index);
			for (int j = 0; j < dividerCollection.length; j++) {
				if (dividerCollection[j] == each) {
					indexCollection.add(new Integer(index));
				}
			}
		}
		indexCollection.add(new Integer(separateCollection.length()));

		Collection intervalCollection = new ArrayList();
		for (int i = 0; i < indexCollection.size() - 1; i++) {
			int s = ((Integer) indexCollection.get(i)).intValue();
			int e = ((Integer) indexCollection.get(i + 1)).intValue();
			intervalCollection.add(new StInterval(s + 1, e - 1));
		}
		StInterval[] intervales = (StInterval[]) intervalCollection.toArray(new StInterval[intervalCollection.size()]);

		Collection result = new ArrayList();
		for (int i = 0; i < intervales.length; i++) {
			if (intervales[i].size() > 0) {
				String part = separateCollection.substring((int) intervales[i].first(), (int) intervales[i].last() + 1);
				result.add(part);
			}
		}

		return (String[]) result.toArray(new String[result.size()]);
	}

	/**
	 * Answer the separator characters.
	 *
	 * @return char[]
	 * @category Separating
	 */
	public static char[] Separators() {
		char[] separators = new char[5];
		separators[0] = ' ';
		separators[1] = '\t';
		separators[2] = '\r';
		separators[3] = '\f';
		separators[4] = '\n';
		return separators;
	}

	/**
	 * Answer a copy of the string, with trailing an array of character omitted.
	 * 
	 * @param theString java.lang.String
	 * @return java.lang.String
	 * @category Stripping
	 */
	public static String StripBackCharcters_from_(char[] charcterArray, String theString) {
		String aString = theString.toString();
		int startIndex = 0;
		int endIndex = aString.length() - 1;
		for (int i = aString.length() - 1; i >= 0; i--) {
			char aChar = aString.charAt(i);
			boolean includes = false;
			for (int j = 0; j < charcterArray.length; j++) {
				if (aChar == charcterArray[j]) {
					includes = true;
					break;
				}
			}
			if (includes == false) {
				endIndex = i;
				break;
			}
		}
		if (endIndex < startIndex) {
			return "";
		}
		return aString.substring(startIndex, endIndex + 1);
	}

	/**
	 * Answer a copy of the string, with trailing whitespace omitted.
	 * 
	 * @param theString java.lang.String
	 * @return java.lang.String
	 * @category Stripping
	 */
	public static String StripBackSeparators_(String theString) {
		String aString = theString.toString();
		int startIndex = 0;
		int endIndex = aString.length() - 1;
		while (Character.isWhitespace(aString.charAt(endIndex))) {
			endIndex--;
		}
		if (endIndex < startIndex) {
			return "";
		}
		return aString.substring(startIndex, endIndex + 1);
	}

	/**
	 * Answer a copy of the string, with leading and trailing an array of character omitted.
	 * 
	 * @param theString java.lang.String
	 * @return java.lang.String
	 * @category Stripping
	 */
	public static String StripFrontBackCharcters_from_(char[] charcterArray, String theString) {
		return StripFrontCharcters_from_(charcterArray, StripBackCharcters_from_(charcterArray, theString));
	}

	/**
	 * Answer a copy of the string, with leading and trailing whitespace omitted.
	 * 
	 * @param theString java.lang.String
	 * @return java.lang.String
	 * @category Stripping
	 */
	public static String StripFrontBackSeparators_(String theString) {
		return theString.trim();
		// return StripFrontSeparators_(StripBackSeparators_(theString));
	}

	/**
	 * Answer a copy of the string, with leading an array of character omitted.
	 * 
	 * @param theString java.lang.String
	 * @return java.lang.String
	 * @category Stripping
	 */
	public static String StripFrontCharcters_from_(char[] charcterArray, String theString) {
		String aString = theString.toString();
		int startIndex = 0;
		for (int i = 0; i < aString.length(); i++) {
			char aChar = aString.charAt(i);
			boolean includes = false;
			for (int j = 0; j < charcterArray.length; j++) {
				if (aChar == charcterArray[j]) {
					includes = true;
					break;
				}
			}
			if (includes == false) {
				startIndex = i;
				break;
			}
		}
		int endIndex = aString.length();
		if (endIndex < startIndex) {
			return "";
		}
		return aString.substring(startIndex, endIndex);
	}

	/**
	 * Answer a copy of the string, with leading whitespace omitted.
	 * 
	 * @param theString java.lang.String
	 * @return java.lang.String
	 * @category Stripping
	 */
	public static String StripFrontSeparators_(String theString) {
		String aString = theString.toString();
		int startIndex = 0;
		while (Character.isWhitespace(aString.charAt(startIndex))) {
			startIndex++;
		}
		int endIndex = aString.length();
		if (endIndex < startIndex) {
			return "";
		}
		return aString.substring(startIndex, endIndex);
	}

	/**
	 * Answer the string without separators.
	 * 
	 * @param aString java.lang.String
	 * @return java.lang.String
	 * @category Private
	 */
	protected static String SkipSeparators_(String aString) {
		String subString = aString.substring(0);
		while (subString.length() != 0 && Character.isWhitespace(subString.charAt(0))) {
			subString = subString.substring(1);
		}
		return subString;
	}
}
