package jp.co.sra.jun.goodies.spirodesign;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;

import jp.co.sra.jun.geometry.basic.Jun2dPoint;
import jp.co.sra.jun.geometry.basic.JunAngle;
import jp.co.sra.jun.geometry.transformations.Jun2dTransformation;

/**
 * JunSpiroCircle class
 * 
 *  @author    m-asada
 *  @created   2006/03/28 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun676 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunSpiroCircle.java,v 8.11 2008/02/20 06:32:03 nisinaka Exp $
 */
public class JunSpiroCircle extends JunSpiroObject {
	protected Jun2dPoint circleCenter;
	protected double circleRadius;
	protected double circleAngle;

	/**
	 * Create a new instance of <code>JunSpiroCircle</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunSpiroCircle() {
		super();
	}

	/**
	 * Create a new instance of <code>JunSpiroCircle</code> and initialize it.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aRadius double
	 * @category Instance creation
	 */
	public JunSpiroCircle(Jun2dPoint aPoint, double aRadius) {
		super();
		this.center_(aPoint);
		this.radius_(aRadius);
	}

	/**
	 * Create a new instance of <code>JunSpiroCircle</code> and initialize it.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @param aRadius double
	 * @param aColor java.awt.Color
	 * @param aWidth int
	 * @category Instance creation
	 */
	public JunSpiroCircle(Jun2dPoint aPoint, double aRadius, Color aColor, int aWidth) {
		super();
		this.center_(aPoint);
		this.radius_(aRadius);
		this.color_(aColor);
		this.width_(aWidth);
	}

	/**
	 * Initialize the receiver when created.
	 * 
	 * @see jp.co.sra.smalltalk.StApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		circleCenter = null;
		circleRadius = Double.NaN;
		circleAngle = Double.NaN;
	}

	/**
	 * Answer the receiver's circle angle.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double angle() {
		if (Double.isNaN(circleAngle)) {
			circleAngle = this.defaultAngle();
		}
		return circleAngle;
	}

	/**
	 * Set the receiver's circle angle.
	 * 
	 * @param aNumber double
	 * @category accessing
	 */
	public void angle_(double aNumber) {
		circleAngle = aNumber;
	}

	/**
	 * Answer the receiver's circle center point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public Jun2dPoint center() {
		if (circleCenter == null) {
			circleCenter = this.defaultCenter();
		}
		return circleCenter;
	}

	/**
	 * Set the receiver's circle center point.
	 * 
	 * @param aPoint jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category accessing
	 */
	public void center_(Jun2dPoint aPoint) {
		circleCenter = aPoint;
	}

	/**
	 * Answer the receiver's circumference.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double circumference() {
		return 2 * Math.PI * this.radius();
	}

	/**
	 * Answer the receiver's mark.
	 * 
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.jun.goodies.spirodesign.JunSpiroObject#mark()
	 * @category accessing
	 */
	public Rectangle mark() {
		Rectangle mark = new Rectangle(0, 0, Math.max(this.defaultMarkExtent().width, this.width()), Math.max(this.defaultMarkExtent().height, this.width()));
		mark.translate((int) Math.round(this.center().x() - mark.getCenterX()), (int) Math.round(this.center().y() - mark.getCenterY()));
		return mark;
	}

	/**
	 * Answer the receiver's marks.
	 * 
	 * @return java.awt.Rectangle[]
	 * @see jp.co.sra.jun.goodies.spirodesign.JunSpiroObject#marks()
	 * @category accessing
	 */
	public Rectangle[] marks() {
		Rectangle mark = new Rectangle(0, 0, Math.max(this.defaultMarkExtent().width, this.width()), Math.max(this.defaultMarkExtent().height, this.width()));
		Jun2dPoint[] points = this.points();
		Rectangle[] marks = new Rectangle[5];
		for (int i = 0; i < marks.length; i++) {
			marks[i] = new Rectangle(mark);
			marks[i].translate((int) Math.round(points[i].x() - mark.getCenterX()), (int) Math.round(points[i].y() - mark.getCenterY()));
		}
		return marks;
	}

	/**
	 * Answer the receiver's pen points.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint[]
	 * @see jp.co.sra.jun.goodies.spirodesign.JunSpiroObject#points()
	 * @category accessing
	 */
	public Jun2dPoint[] points() {
		Rectangle box = this.boundingBox();
		JunAngle angle = JunAngle.FromDeg_(this.angle());
		Jun2dPoint[] points = new Jun2dPoint[5];
		points[0] = new Jun2dPoint(box.getCenterX(), box.getCenterY());
		points[1] = new Jun2dPoint(box.x + box.width, box.getCenterY());
		points[2] = new Jun2dPoint(box.getCenterX(), box.y + box.height);
		points[3] = new Jun2dPoint(box.x, box.getCenterY());
		points[4] = new Jun2dPoint(box.getCenterX(), box.y);
		for (int i = 0; i < points.length; i++) {
			Jun2dPoint p = new Jun2dPoint(points[i]);
			p = p.transform_(Jun2dTransformation.Rotate_around_(angle, this.center()));
			points[i] = p;
		}
		return points;
	}

	/**
	 * Answer the receiver's circleRadius.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double radius() {
		if (Double.isNaN(circleRadius)) {
			circleRadius = this.defaultRadius();
		}
		return circleRadius;
	}

	/**
	 * Set the receiver's circleRadius.
	 * 
	 * @param aNumber double
	 * @category accessing
	 */
	public void radius_(double aNumber) {
		circleRadius = Math.max(aNumber, 0.0d);
	}

	/**
	 * Answer the receiver's bounding box.
	 * 
	 * @return java.awt.Rectangle
	 * @see jp.co.sra.jun.goodies.spirodesign.JunSpiroObject#boundingBox()
	 * @category bounds accessing
	 */
	public Rectangle boundingBox() {
		Rectangle boundingBox = new Rectangle(this.center()._toPoint(), new Dimension(1, 1));
		boundingBox.grow((int) Math.ceil(this.radius()), (int) Math.ceil(this.radius()));
		return boundingBox;
	}

	/**
	 * Answer the default angle.
	 * 
	 * @return double
	 * @category defaults
	 */
	public double defaultAngle() {
		return 0;
	}

	/**
	 * Answer the default center point.
	 * 
	 * @return jp.co.sra.jun.geometry.basic.Jun2dPoint
	 * @category defaults
	 */
	public Jun2dPoint defaultCenter() {
		return new Jun2dPoint(0, 0);
	}

	/**
	 * Answer the default raduis.
	 * 
	 * @return int
	 * @category defaults
	 */
	public int defaultRadius() {
		return 100;
	}

	/**
	 * Display the receiver with marks on the graphics at the specified point.
	 *
	 * @param graphicsContext java.awt.Graphics
	 * @param displayPoint java.awt.Point
	 * @param aBoolean boolean
	 * @see jp.co.sra.jun.goodies.spirodesign.JunSpiroObject#displayOn_at_with_Marks_(java.awt.Graphics, java.awt.Point, boolean)
	 * @category displaying
	 */
	public void displayOn_at_with_Marks_(Graphics graphicsContext, Point displayPoint, boolean aBoolean) {
		Graphics2D aGraphics = (Graphics2D) graphicsContext.create();
		try {
			Rectangle aRectangle = this.boundingBox();
			this.applyAttributesToGraphicsContext_(aGraphics);
			aGraphics.translate(displayPoint.x, displayPoint.y);
			aGraphics.drawArc(aRectangle.x, aRectangle.y, aRectangle.width, aRectangle.height, 0, 360);
			if (aBoolean) {
				Rectangle[] marks = this.marks();
				for (int i = 0; i < marks.length; i++) {
					Rectangle box = marks[i];
					aGraphics.setColor(this.color());
					aGraphics.fillRect(box.x, box.y, box.width, box.height);
					aGraphics.setColor(Color.white);
					Rectangle insetBox = new Rectangle(box);
					insetBox.grow(-1, -1);
					aGraphics.fillRect(insetBox.x, insetBox.y, insetBox.width, insetBox.height);
				}
			}
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
				aGraphics = null;
			}
		}
	}
}
