package jp.co.sra.jun.goodies.revision;

import java.util.ArrayList;
import java.util.List;

import jp.co.sra.smalltalk.SmalltalkException;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StObject;
import jp.co.sra.smalltalk.StValued;

import jp.co.sra.jun.topology.abstracts.JunAbstractOperator;

/**
 * JunRevisionManager class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2007/08/14 (by m-asada)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun500 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunRevisionManager.java,v 8.5 2008/02/20 06:32:02 nisinaka Exp $
 */
public class JunRevisionManager extends StObject {
	protected boolean inversable;
	protected JunHistoryNode rootNode;
	protected List revisionTipNodes;
	protected int currentRevision;
	protected int currentStep;
	protected JunHistoryNode currentNode;
	protected StBlockClosure stepBlock;

	/**
	 * Create a new instance of <code>JunRevisionManager</code> and initialize it.
	 * 
	 * @throws jp.co.sra.smalltalk.SmalltalkException
	 * @category Instance creation
	 */
	public JunRevisionManager() {
		throw SmalltalkException.ShouldNotImplement();
	}

	/**
	 * Create a new instance of <code>JunRevisionManager</code> and initialize it.
	 * 
	 * @param evaluatableObject jp.co.sra.smalltalk.StValued
	 * @category Instance creation
	 */
	public JunRevisionManager(StValued evaluatableObject) {
		this(evaluatableObject, false);
	}

	/**
	 * Create a new instance of <code>JunRevisionManager</code> and initialize it.
	 * 
	 * @param evaluatableObject jp.co.sra.smalltalk.StValued
	 * @param aBoolean boolean
	 * @category Instance creation
	 */
	public JunRevisionManager(StValued evaluatableObject, boolean aBoolean) {
		super();
		this.inversable_(aBoolean);
		this.setRootNode_(new JunHistoryNode(evaluatableObject));
	}

	/**
	 * Answer the receiver's comment.
	 * 
	 * @return comment
	 * @category accessing
	 */
	public String comment() {
		return this.currentNode().comment();
	}

	/**
	 * Set the receiver's comment.
	 * 
	 * @param aString java.lang.String
	 * @category accessing
	 */
	public void comment_(String aString) {
		this.currentNode().comment_(aString);
	}

	/**
	 * Answer the receiver's current revision.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int currentRevision() {
		return currentRevision;
	}

	/**
	 * Answer the receiver's current step.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int currentStep() {
		return currentStep;
	}

	/**
	 * Answer the receiver's inversable.
	 * 
	 * @return boolean
	 * @category accessing
	 */
	public boolean inversable() {
		return inversable;
	}

	/**
	 * Set the receiver's inversable.
	 * 
	 * @param aBoolean boolean
	 * @category accessing
	 */
	public void inversable_(boolean aBoolean) {
		inversable = aBoolean;
	}

	/**
	 * Answer the receiver's last operation.
	 * 
	 * @return jp.co.sra.smalltalk.StValued
	 * @category accessing
	 */
	public StValued lastOperation() {
		return this.currentNode().operation();
	}

	/**
	 * Answer the receiver's number of revisions.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int numberOfRevisions() {
		return revisionTipNodes.size();
	}

	/**
	 * Answer the number of steps in the specified revision.
	 * 
	 * @param anInteger int
	 * @return int
	 * @category accessing
	 */
	public int numberOfStepsInRevision_(int anInteger) {
		if (anInteger < 1 || revisionTipNodes.size() < anInteger) {
			return 0;
		}
		JunHistoryNode node = (JunHistoryNode) revisionTipNodes.get(anInteger - 1);
		return this.stepIndexOfNode_(node);
	}

	/**
	 * Answer the operation at the specified revision and step.
	 * 
	 * @param anInteger1 int
	 * @param anInteger2 int
	 * @return jp.co.sra.smalltalk.StValued
	 * @category accessing
	 */
	public StValued operationAtRevision_setp_(int anInteger1, int anInteger2) {
		return this.nodeAtRevision_step_(anInteger1, anInteger2).operation();
	}

	/**
	 * Set the receiver's step block.
	 * 
	 * @param aBlockClosure jp.co.sra.smalltalk.StBlockClosure
	 * @category accessing
	 */
	public void stepBlock_(StBlockClosure aBlockClosure) {
		stepBlock = aBlockClosure;
	}

	/**
	 * Set the new operation.
	 * 
	 * @param evaluatableObject jp.co.sra.smalltalk.StValued
	 * @category versioning
	 */
	public void newOperation_(StValued evaluatableObject) {
		JunHistoryNode node = this.currentNode();
		JunHistoryNode tipNode = (JunHistoryNode) revisionTipNodes.get(currentRevision - 1);
		JunHistoryNode newNode = new JunHistoryNode(evaluatableObject);
		newNode.parent_(node);
		if (node.equals(tipNode)) {
			revisionTipNodes.set(currentRevision - 1, newNode);
		} else {
			revisionTipNodes.add(newNode);
			currentRevision = revisionTipNodes.size();
		}
		currentStep = currentStep + 1;
		currentNode = newNode;
		Object result = evaluatableObject.value();
		if (stepBlock != null) {
			stepBlock.value_(result);
		}
	}

	/**
	 * Create the new revision.
	 * 
	 * @category versioning
	 */
	public void newRevision() {
		revisionTipNodes.add(this.currentNode());
		currentRevision = revisionTipNodes.size();
	}

	/**
	 * Redo the operation.
	 * 
	 * @return jp.co.sra.jun.goodies.revision.JunRevisionManager
	 * @category versioning
	 */
	public JunRevisionManager redo() {
		return this.revision_step_(currentRevision, currentStep + 1);
	}

	/**
	 * Remove other revisions.
	 * 
	 * @category versioning
	 */
	public void removeOtherRevisions() {
		JunHistoryNode node = this.currentNode();
		revisionTipNodes = new ArrayList();
		revisionTipNodes.add(node);
		currentRevision = 1;
	}

	/**
	 * Answer the receiver's revision with the specified step.
	 * 
	 * @param anInteger1 int
	 * @param anInteger2 int
	 * @return jp.co.sra.jun.goodies.revision.JunRevisionManager
	 * @category versioning
	 */
	public JunRevisionManager revision_step_(int anInteger1, int anInteger2) {
		int revision = Math.max(Math.min(anInteger1, revisionTipNodes.size()), 1);
		int step = Math.max(Math.min(anInteger2, this.numberOfStepsInRevision_(revision)), 1);
		StValued[] operationArray = this.operationsFromRevision_step_toRevision_step_(currentRevision, currentStep, revision, step);
		currentRevision = revision;
		currentStep = step;
		currentNode = this.nodeAtRevision_step_(currentRevision, currentStep);
		for (int i = 0; i < operationArray.length; i++) {
			StValued evaluatableObject = operationArray[i];
			Object result = evaluatableObject.value();
			if (stepBlock != null) {
				stepBlock.value_(result);
			}
		}
		return this;
	}

	/**
	 * Undo the operation.
	 * 
	 * @return jp.co.sra.jun.goodies.revision.JunRevisionManager
	 * @category versioning
	 */
	public JunRevisionManager undo() {
		return this.revision_step_(currentRevision, currentStep - 1);
	}

	/**
	 * Vanilla the operation.
	 * 
	 * @return jp.co.sra.jun.goodies.revision.JunRevisionManager
	 * @category versioning
	 */
	public JunRevisionManager vanilla() {
		return this.revision_step_(1, 2);
	}

	/**
	 * Answer the receiver's current node.
	 * 
	 * @return jp.co.sra.jun.goodies.revision.JunHistoryNode
	 * @category private
	 */
	protected JunHistoryNode currentNode() {
		return currentNode;
	}

	/**
	 * Answer receiver's node at the specified revision and step.
	 * 
	 * @param anInteger1 int
	 * @param anInteger2 int
	 * @return jp.co.sra.jun.goodies.revision.JunHistoryNode
	 * @category private
	 */
	protected JunHistoryNode nodeAtRevision_step_(int anInteger1, int anInteger2) {
		JunHistoryNode node = (JunHistoryNode) revisionTipNodes.get(anInteger1 - 1);
		int times = this.numberOfStepsInRevision_(anInteger1) - anInteger2;
		for (int i = 0; i < times; i++) {
			node = node.parent();
		}
		return node;
	}

	/**
	 * Answer the operation from revision. 
	 * 
	 * @param anInteger1 int
	 * @param anInteger2 int
	 * @param anInteger3 int
	 * @param anInteger4 int
	 * @return jp.co.sra.smalltalk.StValued[]
	 * @category private
	 */
	protected StValued[] operationsFromRevision_step_toRevision_step_(int anInteger1, int anInteger2, int anInteger3, int anInteger4) {
		if (anInteger1 == anInteger3 && anInteger2 == anInteger4) {
			return new StValued[0];
		}

		JunHistoryNode startNode = this.nodeAtRevision_step_(anInteger1, anInteger2);
		JunHistoryNode endNode = this.nodeAtRevision_step_(anInteger3, anInteger4);
		JunHistoryNode node = null;
		if (anInteger1 == anInteger3 || this.revision_includesNode_(anInteger1, endNode)) {
			if (anInteger2 < anInteger4) {
				StValued[] operations = new StValued[anInteger4 - anInteger2];
				node = endNode;
				for (int i = anInteger4 - anInteger2; i >= 1; i--) {
					operations[i - 1] = node.operation();
					node = node.parent();
				}
				return operations;
			} else {
				if (inversable) {
					StValued[] operations = new StValued[anInteger2 - anInteger4];
					node = startNode;
					for (int i = 0; i < anInteger2 - anInteger4; i++) {
						if ((node.operation() instanceof JunAbstractOperator && ((JunAbstractOperator) node.operation()).isInversable()) == false) {
							return this.operationsFromRevision_step_toRevision_step_(anInteger3, 0, anInteger3, anInteger4);
						}
						operations[i] = ((JunAbstractOperator) node.operation()).inverse();
						node = node.parent();
					}
					return operations;
				} else {
					return this.operationsFromRevision_step_toRevision_step_(anInteger3, 0, anInteger3, anInteger4);
				}
			}
		} else {
			if (inversable) {
				JunHistoryNode junctionNode = startNode;
				int junctionStepIndex = anInteger2;
				while (this.revision_includesNode_(anInteger3, junctionNode) == false) {
					junctionNode = junctionNode.parent();
					junctionStepIndex = junctionStepIndex - 1;
				}
				StValued[] operations1 = this.operationsFromRevision_step_toRevision_step_(anInteger1, anInteger2, anInteger1, junctionStepIndex);
				StValued[] operations2 = this.operationsFromRevision_step_toRevision_step_(anInteger3, junctionStepIndex, anInteger3, anInteger4);
				StValued[] operations = new StValued[operations1.length + operations2.length];
				for (int i = 0; i < operations1.length; i++) {
					operations[i] = operations1[i];
				}
				for (int i = 0; i < operations2.length; i++) {
					operations[operations1.length + i] = operations2[i];
				}
				return operations;
			} else {
				return this.operationsFromRevision_step_toRevision_step_(anInteger3, 0, anInteger3, anInteger4);
			}
		}

		// throw SmalltalkException.ShouldNotImplement();
	}

	/**
	 * Answer <code>true</code> the specified revision includes the specified node, otherwise <code>false</code>.
	 * 
	 * @param anInteger int
	 * @param aJunHistoryNode jp.co.sra.jun.goodies.revision.JunHistoryNode
	 * @return boolean
	 * @category private
	 */
	protected boolean revision_includesNode_(int anInteger, JunHistoryNode aJunHistoryNode) {
		if (anInteger < 1 || revisionTipNodes.size() < anInteger) {
			return false;
		}
		JunHistoryNode node = (JunHistoryNode) revisionTipNodes.get(anInteger - 1);
		while (node != null) {
			if (node == aJunHistoryNode) {
				return true;
			}
			node = node.parent();
		}
		return false;
	}

	/**
	 * Set the receiver's root node.
	 * 
	 * @param aJunHistoryNode jp.co.sra.jun.goodies.revision.JunHistoryNode
	 * @category private
	 */
	protected void setRootNode_(JunHistoryNode aJunHistoryNode) {
		rootNode = aJunHistoryNode;
		revisionTipNodes = new ArrayList();
		revisionTipNodes.add(aJunHistoryNode);
		currentRevision = 1;
		currentStep = 1;
		currentNode = aJunHistoryNode;
		aJunHistoryNode.operation().value();
	}

	/**
	 * Answer the step index of the specified node.
	 * 
	 * @param aJunHistoryNode jp.co.sra.jun.goodies.revision.JunHistoryNode
	 * @return int
	 * @category private
	 */
	protected int stepIndexOfNode_(JunHistoryNode aJunHistoryNode) {
		int index = 0;
		JunHistoryNode node = aJunHistoryNode;
		while (node != null) {
			node = node.parent();
			index = index + 1;
		}
		return index;
	}
}
