package jp.co.sra.jun.goodies.prolog;

import java.io.*;
import java.util.Hashtable;
import jp.co.sra.smalltalk.*;

/**
 * JunPrologSymbol class
 * 
 *  @author    kondo
 *  @created   1999/09/09 (by kondo)
 *  @updated   2003/04/28 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun301 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunPrologSymbol.java,v 8.10 2008/02/20 06:32:02 nisinaka Exp $
 */
public class JunPrologSymbol extends JunPrologString {

	protected static Hashtable JunPrologSymbolTable;
	protected static JunPrologSymbol JunPrologCut;
	protected static JunPrologSymbol JunPrologTrue;
	protected static JunPrologSymbol JunPrologFail;
	protected static JunPrologSymbol JunPrologSend;
	protected static JunPrologSymbol JunPrologVar;
	protected static JunPrologSymbol JunPrologSelf;

	static {
		Initialize();
	}

	/**
	 * Create a new instance of JunPrologSymbol and initialize it.
	 *
	 * @param aString java.lang.String
	 */
	protected JunPrologSymbol(String aString) {
		super(aString);
	}

	/**
	 * Answer the cut symbol.
	 *
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologSymbol
	 */
	public static JunPrologSymbol Cut() {
		return JunPrologCut;
	}

	/**
	 * Install the string as a JunPrologSymbol.
	 *
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologSymbol
	 * @param aString java.lang.String
	 */
	public static JunPrologSymbol Install_(String aString) {
		JunPrologSymbol symbol = (JunPrologSymbol) JunPrologSymbolTable.get(aString);
		if (symbol == null) {
			symbol = new JunPrologSymbol(aString);
			JunPrologSymbolTable.put(aString, symbol);
		}
		return symbol;
	}

	/**
	 * Read a JunPrologSymbol from the stream.
	 *
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @param aStream jp.co.sra.smalltalk.StReadStream
	 */
	public static JunPrologEntity ReadFrom_(StReadStream aStream) {
		boolean[] map = JunPrologCharMap.SymbolMap();
		StringBuffer buffer = new StringBuffer(20);
		char ch = aStream.peek();
		if (ch == '\'') {
			aStream.next();
			while ((ch = aStream.peek()) != 0) {
				if (ch == '\'') {
					aStream.next();
					ch = aStream.peek();
					if (ch != '\'') {
						return Install_(buffer.toString());
					}
				}
				buffer.append(aStream.next());
			}
			throw SmalltalkException.Error("Syntax error near $' unmatched $'\n-- end of file --");
		}

		if (Expect_on_("=..", aStream)) {
			// Non standard (or ad hoc) parsing patterns.
			return Install_("=..");
		}

		while (ch != 0 && map[ch]) {
			buffer.append(aStream.next());
			ch = aStream.peek();
		}
		return Install_(buffer.toString());
	}

	/**
	 * If aStream contains aString at the current position, 
	 * then return true and set the position of aStream to the next char of the string.
	 * else return false and rewind the position.
	 *
	 * @return boolean
	 * @param aString java.lang.String
	 * @param aStream jp.co.sra.smalltalk.StReadStream
	 */
	protected static boolean Expect_on_(String aString, StReadStream aStream) {
		int pos = aStream.position();
		if (aString.equals(aStream.nextAvailable_(aString.length()))) {
			return true;
		} else {
			aStream.position_(pos);
			return false;
		}
	}

	/**
	 * Initialize the class variables.
	 */
	protected static void Initialize() {
		JunPrologSymbolTable = new Hashtable();
		JunPrologCut = Install_("!");
		JunPrologTrue = Install_("true");
		JunPrologFail = Install_("fail");
		JunPrologSend = Install_("send");
		JunPrologVar = Install_("var");
		JunPrologSelf = Install_("self");
	}

	/**
	 * Answer true if the receiver is a built in predicate, otherwise false.
	 *
	 * @return boolean
	 */
	public boolean isBuiltInPredicate() {
		if (this.isJunPrologCut()) {
			return true;
		}
		if (this.isJunPrologTrue()) {
			return true;
		}
		if (this.isJunPrologFail()) {
			return true;
		}
		if (this.isJunPrologSend()) {
			return true;
		}
		if (this.isJunPrologVar()) {
			return true;
		}
		return false;
	}

	/**
	 * Answer true if the receiver is a built in predicate, Cut, otherwise false.
	 *
	 * @return boolean
	 */
	public boolean isJunPrologCut() {
		return this.equals(JunPrologCut);
	}

	/**
	 * Answer true if the receiver is a built in predicate, Fail, otherwise false.
	 *
	 * @return boolean
	 */
	public boolean isJunPrologFail() {
		return this.equals(JunPrologFail);
	}

	/**
	 * Answer true if the receiver is a built in predicate, Self, otherwise false.
	 *
	 * @return boolean
	 */
	public boolean isJunPrologSelf() {
		return this.equals(JunPrologSelf);
	}

	/**
	 * Answer true if the receiver is a built in predicate, Send, otherwise false.
	 *
	 * @return boolean
	 */
	public boolean isJunPrologSend() {
		return this.equals(JunPrologSend);
	}

	/**
	 * Answer true if the receiver is a built in predicate, True, otherwise false.
	 *
	 * @return boolean
	 */
	public boolean isJunPrologTrue() {
		return this.equals(JunPrologTrue);
	}

	/**
	 * Answer true if the receiver is a built in predicate, Var, otherwise false.
	 *
	 * @return boolean
	 */
	public boolean isJunPrologVar() {
		return this.equals(JunPrologVar);
	}

	/**
	 * Print my string representation on the writer.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws IOException if failed.
	 */
	public void printJunPrologOn_(Writer aWriter) throws IOException {
		boolean[] map = JunPrologCharMap.SymbolMap();
		boolean flag = false;
		for (int i = 0; i < string.length(); i++) {
			if (map[string.charAt(i)] == false) {
				flag = true;
				break;
			}
		}

		if (string.length() == 0) {
			aWriter.write("''");
		} else {
			char firstChar = string.charAt(0);
			if (('A' <= firstChar && firstChar <= 'Z') || flag) {
				aWriter.write('\'');
				int length = string.length();
				for (int i = 0; i < length; i++) {
					char x = string.charAt(i);
					aWriter.write(x);
					if (x == '\'') {
						aWriter.write(x);
					}
				}
				aWriter.write('\'');
			} else {
				int length = string.length();
				for (int i = 0; i < length; i++) {
					char x = string.charAt(i);
					aWriter.write(x);
					if (x == '\'') {
						aWriter.write(x);
					}
				}
			}
		}
	}
}
