package jp.co.sra.jun.goodies.prolog;

import jp.co.sra.smalltalk.*;

/**
 * JunPrologParser class
 * 
 *  @author    kondo
 *  @created   1999/09/09 (by kondo)
 *  @updated   2003/04/25 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun301 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunPrologParser.java,v 8.10 2008/02/20 06:32:02 nisinaka Exp $
 */
public class JunPrologParser extends JunPrologScanner {

	protected int prevMark;
	protected Object prevToken;
	protected StSymbol prevTokenType;

	/**
	 * Parse the source stream.
	 * 
	 * @return java.lang.Object
	 * @param sourceStream jp.co.sra.smalltalk.StReadStream
	 */
	public Object parse_(StReadStream sourceStream) {
		return this.parse_ifFail_(sourceStream, new StBlockClosure() {
			public Object value_(Object anObject) {
				String errorMessage = (String) anObject;
				String label = errorMessage + " near " + ContractTo(token.toString(), 10);
				String string = source.upToEnd();
				if (string.length() == 0) {
					string = "--> end of file";
				} else {
					string = "--> " + ContractTo(string, 30);
				}
				throw SmalltalkException.Error(label + "\n" + string);
			}
		});
	}

	/**
	 * Parse the source stream.
	 * 
	 * @return Object
	 * @param sourceStream
	 * @param aBlock DOCUMENT ME!
	 */
	public Object parse_ifFail_(StReadStream sourceStream, StBlockClosure aBlock) {
		this.init_ifFail_(sourceStream, aBlock);
		return this.scan();
	}

	/**
	 * Initialize the prolog parser.
	 * 
	 * @param sourceStream jp.co.sra.smalltalk.StReadStream
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 */
	protected void init_ifFail_(StReadStream sourceStream, StBlockClosure aBlock) {
		super.on_(sourceStream);
		failBlock = aBlock;
		JunPrologVariable.SetZero();
	}

	/**
	 * Get a next token.
	 * 
	 * @return java.lang.Object
	 */
	protected Object nextToken() {
		prevMark = mark;
		prevToken = token;
		prevTokenType = tokenType;

		return super.nextToken();
	}

	/**
	 * Scan the source stream.
	 * 
	 * @return java.lang.Object
	 */
	protected Object scan() {
		if (source.atEnd()) {
			return $("eof");
		}

		this.nextToken();
		if (tokenType == $("eof")) {
			return $("eof");
		}

		if (tokenType == $("neck")) {
			return new JunPrologList(JunPrologBoolean.TRUE, this.scanClause()); // resolve clause
		}

		this.unNextToken();
		return new JunPrologList(JunPrologBoolean.FALSE, this.scanClause()); // assert clause
	}

	/**
	 * Scan a body.
	 *
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologBody
	 */
	protected JunPrologBody scanBody() {
		JunPrologList list = null;
		while (tokenType != $("eof")) {
			list = this.scanStructure().cons_(list);
			this.nextToken();
			if (tokenType == $("rightParenthesis")) {
				return JunPrologBody.FromReverseList_(list);
			} else if (tokenType == $("period")) {
				return JunPrologBody.FromReverseList_(list);
			} else if (tokenType != $("comma")) {
				failBlock.value_("Syntax error");
				return null;
			}
		}

		failBlock.value_("Unexpected eof");
		return null;
	}

	/**
	 * Scan a clause.
	 *
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologClause
	 */
	protected JunPrologClause scanClause() {
		JunPrologStructure head = this.scanStructure();
		this.nextToken();
		if (tokenType == $("neck")) {
			return new JunPrologClause(head, this.scanBody());
		} else if (tokenType == $("comma")) {
			return new JunPrologClause(head, this.scanBody());
		} else if (tokenType == $("period")) {
			return new JunPrologClause(head, null);
		} else if (tokenType == $("rightParenthesis")) {
			return new JunPrologClause(head, null);
		}

		this.unNextToken();
		failBlock.value_("Syntax error");
		return null;
	}

	/**
	 * Scan an expression.
	 *
	 * @return java.lang.Object
	 */
	protected JunPrologEntity scanExpression() {
		this.nextToken();
		if (tokenType == $("eof") || tokenType == $("period")) {
			return null;
		} else if (tokenType == $("number")) {
			return new JunPrologNumber((Number) token);
		} else if (tokenType == $("string")) {
			throw new SmalltalkException("token type is STRING");
			// return token;
		} else if (tokenType == $("object")) {
			throw new SmalltalkException("token type is OBJECT");
			// return new JunPrologObject(token, this.compileAndEvaluate_(token));
		} else if (tokenType == $("leftBracket")) {
			return this.scanList();
		} else if (tokenType == $("leftParenthesis")) {
			return this.scanClause();
		} else if (tokenType == $("symbol") || tokenType == $("variable")) {
			JunPrologEntity symbol = (JunPrologEntity) token;
			this.nextToken();
			if (tokenType == $("leftParenthesis")) {
				this.unNextToken();
				this.unNextToken();
				return this.scanStructure();
			} else {
				this.unNextToken();
				return symbol;
			}
		}

		this.unNextToken();
		failBlock.value_("Syntax error");
		return null;
	}

	/**
	 * Scan a list.
	 *
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 */
	protected JunPrologEntity scanList() {
		this.nextToken();
		if (tokenType == $("comma")) {
			this.nextToken();
		}
		if (tokenType == $("nexk")) {
			this.nextToken();
		}
		if (tokenType == $("rightBracket")) {
			return null;
		} else if (tokenType == $("leftBracket")) {
			return new JunPrologList(this.scanList(), this.scanList());
		} else if (tokenType == $("verticalBar")) {
			JunPrologEntity expression = this.scanExpression();
			this.nextToken();
			if (tokenType == $("rightBracket")) {
				return expression;
			} else {
				failBlock.value_("Syntax error");
				return null;
			}
		}

		this.unNextToken();
		JunPrologEntity expression = this.scanExpression();
		return new JunPrologList(expression, this.scanList());
	}

	/**
	 * Scan a structure.
	 *
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologStructure
	 */
	protected JunPrologStructure scanStructure() {
		Object functor = this.nextToken();
		if (tokenType == $("symbol") || tokenType == $("variable")) {
			JunPrologEntity terms = this.scanTerms();
			return new JunPrologStructure((JunPrologString) functor, (JunPrologTerms) terms);
		}

		failBlock.value_("Syntax error");
		return null;
	}

	/**
	 * Scan terms.
	 *
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 */
	protected JunPrologEntity scanTerms() {
		this.nextToken();
		if (tokenType == $("leftParenthesis")) {
			this.nextToken();
			if (tokenType == $("verticalBar") || tokenType == $("rightParenthesis")) {
				if (tokenType == $("rightParenthesis")) {
					return null;
				}
				JunPrologEntity list = this.scanExpression();
				this.nextToken();
				if (tokenType == $("rightParenthesis")) {
					return list;
				} else {
					failBlock.value_("Syntax error");
					return null;
				}
			} else {
				this.unNextToken();
			}

			JunPrologList list = null;
			while (tokenType != $("eof")) {
				list = new JunPrologList(this.scanExpression(), list);
				this.nextToken();
				if (token == $("rightParenthesis")) { // why not tokenType???
					return JunPrologTerms.FromReverseList_(list);
				}
				if (tokenType == $("verticalBar")) {
					list = list.reverse();
					list = list.nconc_(this.scanExpression());
					this.nextToken();
					if (tokenType == $("rightParenthesis")) {
						return JunPrologTerms.FromList_(list);
					} else {
						failBlock.value_("Syntax error");
						return null;
					}
				}
				if (tokenType != $("comma")) {
					failBlock.value_("Syntax error");
					return null;
				}
			}

			failBlock.value_("Unexpected eof");
			return null;
		} else {
			this.unNextToken();
			return null;
		}
	}

	/**
	 * Push back a token to the source stream.
	 */
	protected void unNextToken() {
		super.unNextToken();

		mark = prevMark;
		token = prevToken;
		tokenType = prevTokenType;
	}
}
