package jp.co.sra.jun.goodies.prolog;

import java.io.*;

/**
 * JunPrologList class
 * 
 *  @author    kondo
 *  @created   1999/09/09 (by kondo)
 *  @updated   2003/04/28 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun301 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunPrologList.java,v 8.11 2008/02/20 06:32:02 nisinaka Exp $
 */
public class JunPrologList extends JunPrologEntity {

	protected static int JunPrologDotPairPrintHorizontalLevel = 100;
	protected static int JunPrologDotPairPrintVerticalLevel = 10;

	protected JunPrologEntity carPart;
	protected JunPrologEntity cdrPart;

	/**
	 * Create a new instance of <code>JunPrologList</code> and initialize it with the car part and the cdr part.
	 *
	 * @param carObject jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @param cdrObject jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @category Instance creation
	 */
	public JunPrologList(JunPrologEntity carObject, JunPrologEntity cdrObject) {
		this.car_cdr_(carObject, cdrObject);
	}

	/**
	 * Create a new instance of <code>JunPrologList</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	protected JunPrologList() {
		super();
	}

	/**
	 * Append the object to the list.
	 *
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologList
	 * @param aJunPrologDotPair jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @category functions
	 */
	public JunPrologList append_(JunPrologEntity aJunPrologDotPair) {
		if (cdrPart.consp() == false) {
			return new JunPrologList(carPart, aJunPrologDotPair);
		}
		return new JunPrologList(carPart, ((JunPrologList) cdrPart).append_(aJunPrologDotPair));
	}

	/**
	 * Answer the arity.
	 *
	 * @return int
	 * @category functions
	 */
	public int arity() {
		return 2;
	}

	/**
	 * Answer the assoc of the object.
	 *
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @param anObject jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @category functions
	 */
	public JunPrologEntity assoc_(JunPrologEntity anObject) {
		JunPrologEntity list = this;
		while (list != null && list.consp()) {
			JunPrologEntity assoc = list.car();
			if (anObject.equals(assoc.car())) {
				return assoc;
			}
			list = list.cdr();
		}

		return null;
	}

	/**
	 * Answer my car part.
	 *
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @see jp.co.sra.jun.goodies.prolog.JunPrologEntity#car()
	 * @category accessing
	 */
	public JunPrologEntity car() {
		return carPart;
	}

	/**
	 * Set my new car part.
	 *
	 * @param newCarPart jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @see jp.co.sra.jun.goodies.prolog.JunPrologEntity#car_(jp.co.sra.jun.goodies.prolog.JunPrologEntity)
	 * @category accessing
	 */
	public void car_(JunPrologEntity newCarPart) {
		carPart = newCarPart;
	}

	/**
	 * Answer my cdr part.
	 *
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @see jp.co.sra.jun.goodies.prolog.JunPrologEntity#cdr()
	 * @category accessing
	 */
	public JunPrologEntity cdr() {
		return cdrPart;
	}

	/**
	 * Set my new cdr part.
	 *
	 * @param newCdrPart jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @see jp.co.sra.jun.goodies.prolog.JunPrologEntity#cdr_(jp.co.sra.jun.goodies.prolog.JunPrologEntity)
	 * @category accessing
	 */
	public void cdr_(JunPrologEntity newCdrPart) {
		cdrPart = newCdrPart;
	}

	/**
	 * Answer the cons with the object.
	 *
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologList
	 * @param anObject jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @see jp.co.sra.jun.goodies.prolog.JunPrologEntity#cons_(jp.co.sra.jun.goodies.prolog.JunPrologEntity)
	 * @category functions
	 */
	public JunPrologList cons_(JunPrologEntity anObject) {
		return new JunPrologList(this, anObject);
	}

	/**
	 * Answer true if the receiver is a kind of JunPrologList, otherwise false.
	 *
	 * @return boolean
	 * @see jp.co.sra.jun.goodies.prolog.JunPrologEntity#consp()
	 * @category testing
	 */
	public boolean consp() {
		return true;
	}

	/**
	 * Answer true if the receiver equals to the object, otherwise false.
	 *
	 * @return boolean
	 * @param anObject java.lang.Object
	 * @see java.lang.Object#equals(java.lang.Object)
	 * @category comparing
	 */
	public boolean equals(Object anObject) {
		if ((anObject instanceof JunPrologList) == false) {
			return false;
		}

		JunPrologList list = (JunPrologList) anObject;
		if (this.car() == null) {
			return (list.car() == null);
		} else if (this.car().equals(list.car())) {
			if (this.cdr() == null) {
				return (list.cdr() == null);
			} else {
				return this.cdr().equals(list.cdr());
			}
		}

		return false;
	}

	/**
	 * Answer the functor.
	 *
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologString
	 * @category functions
	 */
	public JunPrologString functor() {
		return new JunPrologSymbol(".");
	}

	/**
	 * Answer the length of the list.
	 *
	 * @return int
	 * @category functions
	 */
	public int length() {
		JunPrologEntity list = this;
		int count = 0;
		while (list != null && list.consp()) {
			count++;
			list = list.cdr();
		}
		return count;
	}

	/**
	 * Answer the member of the object.
	 *
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @param anObject jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @category functions
	 */
	public JunPrologEntity member_(JunPrologEntity anObject) {
		JunPrologEntity obj = this;
		while (obj.consp()) {
			JunPrologList list = (JunPrologList) obj;
			if (anObject.equals(list.car())) {
				return list;
			}
			obj = list.cdr();
		}
		return null;
	}

	/**
	 * Answer the nconc with the dot pair.
	 *
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologList
	 * @param aJunPrologDotPair jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @category functions
	 */
	public JunPrologList nconc_(JunPrologEntity aJunPrologDotPair) {
		JunPrologEntity obj = this;
		while (obj.consp()) {
			JunPrologList list = (JunPrologList) obj;
			if (list.cdr() != null && list.cdr().consp()) {
				obj = list.cdr();
			} else {
				list.cdr_(aJunPrologDotPair);
				return this;
			}
		}
		return (JunPrologList) aJunPrologDotPair;
	}

	/**
	 * Answer the nth object of the list.
	 *
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @param anInteger int
	 * @category accessing
	 */
	public JunPrologEntity nth_(int anInteger) {
		if (anInteger <= 0) {
			return this;
		}

		int count = 1;
		JunPrologEntity list = this;
		while (list.consp()) {
			if (count >= anInteger) {
				return ((JunPrologList) list).car();
			}
			count++;
			list = ((JunPrologList) list).cdr();
		}

		return null;
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws IOException if failed.
	 * @see jp.co.sra.jun.goodies.prolog.JunPrologEntity#printJunPrologOn_(java.io.Writer)
	 * @category printing
	 */
	public void printJunPrologOn_(Writer aWriter) throws IOException {
		this.printJunPrologOn_level_(aWriter, 1);
	}

	/**
	 * Print my string representation of the cdr part on the writer.
	 *
	 * @param aWriter java.io.Writer
	 * @param tail jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @param level int
	 * @exception java.io.IOException The exception description.
	 * @category printing
	 */
	public void printJunPrologOn_cdr_level_(Writer aWriter, JunPrologEntity tail, int level) throws java.io.IOException {
		JunPrologEntity d = tail;
		int count = 1;
		while (d != null && d.consp()) {
			if (count >= JunPrologDotPairPrintHorizontalLevel) {
				aWriter.write(" ... ]");
				return;
			}
			aWriter.write(',');
			JunPrologList l = (JunPrologList) d;
			if (l.car().consp()) {
				l.car().printJunPrologOn_level_(aWriter, level + 1);
			} else {
				l.car().printJunPrologOn_(aWriter);
			}
			// l.car().printJunPrologOn_level_(aWriter, level + 1);
			count++;
			d = l.cdr();
		}

		if (d == null) {
			aWriter.write(']');
		} else {
			aWriter.write('|');
			d.printJunPrologOn_(aWriter);
			aWriter.write(']');
		}
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @param level int
	 * @throws IOException if failed.
	 * @see jp.co.sra.jun.goodies.prolog.JunPrologEntity#printJunPrologOn_level_(java.io.Writer, int)
	 * @category printing
	 */
	public void printJunPrologOn_level_(Writer aWriter, int level) throws IOException {
		if (level > JunPrologDotPairPrintVerticalLevel) {
			aWriter.write(" ... ");
			return;
		}

		aWriter.write('[');
		if (carPart.consp()) {
			carPart.printJunPrologOn_level_(aWriter, level + 1);
		} else {
			carPart.printJunPrologOn_(aWriter);
		}
		this.printJunPrologOn_cdr_level_(aWriter, cdrPart, level);
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws IOException if failed.
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write(this._className().toString());
		aWriter.write("(");
		this.printJunPrologOn_(aWriter);
		aWriter.write(")");
	}

	/**
	 * Answer my reverse ordered list.
	 *
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologList
	 * @category functions
	 */
	public JunPrologList reverse() {
		JunPrologEntity list = this;
		JunPrologList revlist = null;
		JunPrologList mark = null;
		while (list != null && list.consp()) {
			if (revlist == null) {
				revlist = mark = new JunPrologList(list.car(), revlist);
			} else {
				revlist = new JunPrologList(list.car(), revlist);
			}
			list = list.cdr();
		}
		mark.cdr_(list);
		return revlist;
	}

	/**
	 * Answer the structure list.
	 * Disassemble prolog structure of myself into a list.
	 *
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologList
	 * @category functions
	 */
	public JunPrologList structureList() {
		return new JunPrologList(this.functor(), new JunPrologList(this.car(), new JunPrologList(this.cdr(), null)));
	}

	/**
	 * Create a new instance with the specified arguments.
	 * This method have to be overrided to create a subclass specified instance.
	 *
	 * @return jp.co.sra.jun.goodies.prolog.JunPrologList
	 * @param carEntity jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @param cdrEntity jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @category Private
	 */
	protected JunPrologList _newInstanceWith(JunPrologEntity carEntity, JunPrologEntity cdrEntity) {
		return new JunPrologList(carEntity, cdrEntity);
	}

	/**
	 * Set the car part and the cdr part.
	 *
	 * @param carObject jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @param cdrObject jp.co.sra.jun.goodies.prolog.JunPrologEntity
	 * @category private
	 */
	protected void car_cdr_(JunPrologEntity carObject, JunPrologEntity cdrObject) {
		carPart = carObject;
		cdrPart = cdrObject;
	}
}
