package jp.co.sra.jun.goodies.progress;

import java.awt.Dialog;
import java.awt.Window;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.SystemResourceSupport;

import jp.co.sra.jun.goodies.button.JunButtonModel;
import jp.co.sra.jun.system.framework.JunApplicationModel;

/**
 * JunProgress class
 * 
 *  @author    nisinaka
 *  @created   1998/11/06 (by nisinaka)
 *  @updated   1999/08/05 (by nisinaka)
 *  @updated   2003/01/07 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun433 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunProgress.java,v 8.12 2008/02/20 06:32:01 nisinaka Exp $
 */
public class JunProgress extends JunApplicationModel {

	/** The progress value. */
	protected float progressValue;

	/** The progress message. */
	protected String progressMessage;

	/** The stop condition. */
	protected boolean stopCondition;

	/** The raising flag. */
	protected boolean raisingFlag;

	protected Dialog _dialog;
	protected JunButtonModel _stopButton;

	/**
	 * Initialize this object.
	 * 
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		progressValue = 0;
		progressMessage = new String();
		raisingFlag = false;
		_dialog = null;
		_stopButton = null;
	}

	/**
	 * Answer the current progress value.
	 * 
	 * @return float
	 * @category accessing
	 */
	public float value() {
		return progressValue;
	}

	/**
	 * Set the new progress value.
	 * 
	 * @param normalizedNumber float
	 * @category accessing
	 */
	public void value_(float normalizedNumber) {
		if (0 <= normalizedNumber && normalizedNumber <= 1) {
			float oldValue = progressValue;
			progressValue = normalizedNumber;
			if (oldValue != progressValue) {
				this.changed_($("value"));
			}
		}

		Thread.yield();

		if (this.stopButtonPressed()) {
			throw new RuntimeException("stop");
		}
	}

	/**
	 * Answer the current progress message.
	 * 
	 * @return java.lang.String
	 * @category accessing
	 */
	public String message() {
		return progressMessage;
	}

	/**
	 * Set the new progress message.
	 * 
	 * @param aString java.lang.String
	 * @category accessing
	 */
	public void message_(String aString) {
		String oldString = progressMessage;
		progressMessage = aString;
		if (oldString.equals(progressMessage) == false) {
			this.changed_($("message"));
		}

		Thread.yield();
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @category view accessing
	 */
	public StView defaultView() {
		return this.defaultViewWithNoButton();
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @category view accessing
	 */
	public StView defaultViewWithNoButton() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return JunProgressViewAwt.WithNoButton_(this);
		} else {
			return JunProgressViewSwing.WithNoButton_(this);
		}
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @category view accessing
	 */
	public StView defaultViewWithStopButton() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return JunProgressViewAwt.WithStopButton_(this);
		} else {
			return JunProgressViewSwing.WithStopButton_(this);
		}
	}

	/**
	 * Answer true if one of the progress view is open, otherwise false.
	 *
	 * @return boolean
	 * @category testing
	 */
	public boolean isOpen() {
		if (this.builder() == null) {
			return false;
		}

		Window[] windows = this.builder().windows();
		if (windows.length == 0) {
			return false;
		}

		for (int i = 0; i < windows.length; i++) {
			if (windows[i].isShowing()) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Answer true if the receiver has a stop button, otherwise false.
	 *
	 * @return boolean
	 * @category testing
	 */
	public boolean hasStopButton() {
		return (_stopButton != null);
	}

	/**
	 * Answer true if the stop button is pressed, otherwise false.
	 *
	 * @return boolean
	 * @category testing
	 */
	public boolean stopButtonPressed() {
		return stopCondition;
	}

	/**
	 * Evaluate the BlockClosure.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object do_(final StBlockClosure aBlock) {
		this.openWithStopButton_(false);
		raisingFlag = true;

		Object result = null;
		try {
			result = this.run_(aBlock);
		} finally {
			this.close();
			raisingFlag = false;
		}
		return result;
	}

	/**
	 * Evaluate the BlockClosure.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	public Object doWithStopButton_(final StBlockClosure aBlock) {
		this.openWithStopButton_(true);
		raisingFlag = true;

		Object result = null;
		try {
			result = this.run_(aBlock);
		} finally {
			this.close();
			raisingFlag = false;
		}
		return result;
	}

	/**
	 * Evaluate the block while in progress.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @return java.lang.Object
	 * @category enumerating
	 */
	protected Object run_(StBlockClosure aBlock) {
		Object result = null;
		try {
			if (aBlock.numArgs() == 0) {
				result = aBlock.value();
			} else {
				result = aBlock.value_(this);
			}
		} catch (RuntimeException e) {
			e.printStackTrace();
			if (e.getMessage().equals("stop") == false) {
				throw e;
			}
		}
		return result;
	}

	/**
	 * Open a view with a stop button if a boolean is true, otherwise a view with no button.
	 * Use modal dialog to avoid blocking EventDispatchThread.
	 *
	 * @param aBoolean boolean
	 * @category interface opening
	 */
	public synchronized void openWithStopButton_(boolean aBoolean) {
		stopCondition = false;
		StView aView = aBoolean ? this.defaultViewWithStopButton() : this.defaultViewWithNoButton();

		_dialog = new Dialog(SystemResourceSupport.getFrame(), this.windowTitle(), false);
		_dialog.add(aView.toComponent());
		_dialog.pack();
		_ShowAtMousePoint(_dialog);
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Progress");
	}

	/**
	 * Close the progress.
	 * 
	 * @category interface closing
	 */
	protected synchronized void close() {
		if (_dialog != null) {
			_dialog.setVisible(false);
			_dialog = null;
		}

		stopCondition = true;
	}

	/**
	 * Answer one of the progress views.
	 * 
	 * @return jp.co.sra.jun.goodies.progress.JunProgressView
	 * @category private
	 */
	protected JunProgressView getProgressView() {
		Object[] dependents = this.dependents();
		for (int i = 0; i < dependents.length; i++) {
			if (dependents[i] instanceof JunProgressView) {
				return (JunProgressView) dependents[i];
			}
		}

		return null;
	}

	/**
	 * Answer the current raising flag.
	 * 
	 * @return boolean
	 * @category private
	 */
	protected boolean raisingFlag() {
		return raisingFlag;
	}

	/**
	 * Answer my stop button.
	 * If there is no stop button, return null.
	 *
	 * @return jp.co.sra.jun.goodies.button.JunButtonModel
	 * @category private
	 */
	protected JunButtonModel stopButton() {
		if (_stopButton == null) {
			_stopButton = new JunButtonModel(false, $String("Stop"), new StBlockClosure() {
				public Object value() {
					close();
					return null;
				}
			});
		}
		return _stopButton;
	}

}
