package jp.co.sra.jun.goodies.pen;

import java.awt.Color;
import java.util.Vector;

import jp.co.sra.smalltalk.StAssociation;
import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StValueHolder;

import jp.co.sra.jun.system.framework.JunAbstractObject;

/**
 * JunPen class
 * 
 *  @author    Hirotsugu Kondo
 *  @created   1999/01/05 (by Hirotsugu Kondo)
 *  @updated   2006/03/16 (by m-asada)
 *  @version   699 (with StPL8.9) based on Jun629 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunPen.java,v 8.11 2008/02/20 06:32:01 nisinaka Exp $
 */
public class JunPen extends JunAbstractObject {
	protected JunPenLocation location;
	protected JunPenDirection direction;
	protected boolean down;
	protected int nib;
	protected Color color;
	protected Vector sequence;
	protected StValueHolder drawing;

	/**
	 * Create a new instance of <code>JunPen</code> and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunPen() {
		super();
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunAbstractObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		location = JunPenLocation.Zero();
		direction = JunPenDirection.Longitude_latitude_(0.0d, 0.0d);
		down = true;
		nib = 1;
		color = Color.black;
		drawing = new StValueHolder(new Vector());
	}

	/**
	 * Answer the receiver's color.
	 * 
	 * @return java.awt.Color
	 * @category accessing
	 */
	public Color color() {
		return color;
	}

	/**
	 * Set the receiver's color.
	 * 
	 * @param colorValue java.awt.Color
	 * @category accessing
	 */
	public void color_(Color colorValue) {
		color = colorValue;
	}

	/**
	 * Answer the receiver's direction.
	 * 
	 * @return jp.co.sra.jun.goodies.pen.JunPenDirection
	 * @category accessing
	 */
	public JunPenDirection direction() {
		return direction;
	}

	/**
	 * Set the receiver's direction.
	 * 
	 * @param aDirection double[]
	 * @category accessing
	 */
	public void direction_(double[] aDirection) {
		direction = JunPenDirection.Coerce_(aDirection);
	}

	/**
	 * Set the receiver's direction.
	 * 
	 * @param aDirection double
	 * @category accessing
	 */
	public void direction_(double aDirection) {
		direction = JunPenDirection.Coerce_(aDirection);
	}

	/**
	 * Set the receiver's direction.
	 * 
	 * @param aDirection may be JunPenDirection, Double and Vector of Double (see JunPenDirection.Coerce_(Object))
	 * @category accessing
	 */
	public void direction_(Object aDirection) {
		direction = JunPenDirection.Coerce_(aDirection);
	}

	/**
	 * Answer the receiver's location.
	 * 
	 * @return jp.co.sra.jun.goodies.pen.JunPenLocation
	 * @category accessing
	 */
	public JunPenLocation location() {
		return location;
	}

	/**
	 * Set the receiver's location.
	 * 
	 * @param aLocation double[]
	 * @category accessing
	 */
	public void location_(double[] aLocation) {
		location = JunPenLocation.Coerce_(aLocation);
	}

	/**
	 * Set the receiver's location.
	 * 
	 * @param aLocation double
	 * @category accessing
	 */
	public void location_(double aLocation) {
		location = JunPenLocation.Coerce_(aLocation);
	}

	/**
	 * Set the receiver's location.
	 * 
	 * @param aLocation may be JunPenLocation,JunPoint,Double and Vector (see JunPenLocation.Coerce_(Object))
	 * @category accessing
	 */
	public void location_(Object aLocation) {
		location = JunPenLocation.Coerce_(aLocation);
	}

	/**
	 * Answer the receiver's nib.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int nib() {
		return nib;
	}

	/**
	 * Set the receiver's nib.
	 * 
	 * @param anInteger int
	 * @category accessing
	 */
	public void nib_(int anInteger) {
		nib = Math.max(anInteger, 1);
	}

	/**
	 * Compute the specified block.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category constructing
	 */
	public void compute_(StBlockClosure aBlock) {
		drawing.compute_(aBlock);
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param orderNumbers int
	 * @param distance int
	 * @category designs
	 */
	public void dragon_distance_(int orderNumbers, int distance) {
		this.dragon_distance_interim_(orderNumbers, distance, null);
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param orderNumbers int
	 * @param distance int
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category designs
	 */
	public void dragon_distance_interim_(int orderNumbers, int distance, StBlockClosure aBlock) {
		if (orderNumbers == 0) {
			this.go_(distance);
		} else {
			if (orderNumbers > 0) {
				this.dragon_distance_interim_(orderNumbers - 1, distance, aBlock);
				this.turn_(90);
				this.dragon_distance_interim_(1 - orderNumbers, distance, aBlock);
			} else {
				this.dragon_distance_interim_(-1 - orderNumbers, distance, aBlock);
				this.turn_(-90);
				this.dragon_distance_interim_(1 + orderNumbers, distance, aBlock);
			}
		}
		if (aBlock != null) {
			aBlock.value();
		}
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param orderNumbers int
	 * @param distance int
	 * @param millisecondValue long
	 * @category designs
	 */
	public void dragon_distance_tick_(int orderNumbers, int distance, final long millisecondValue) {
		this.dragon_distance_interim_(orderNumbers, distance, new StBlockClosure() {
			public Object value() {
				try {
					Thread.sleep(millisecondValue);
				} catch (InterruptedException e) {
				}
				return null;
			}
		});
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param npoints int
	 * @param d int
	 * @category designs
	 */
	public void mandala_diameter_(int npoints, int d) {
		this.mandala_diameter_interim_(npoints, d, null);
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param npoints int
	 * @param d int
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category designs
	 */
	public void mandala_diameter_interim_(int npoints, int d, StBlockClosure aBlock) {
		int l = (int) Math.round((Math.PI * d) / npoints);
		this.home();
		this.up();
		this.turn_(-90);
		this.go_(d / 2);
		this.turn_(90);
		this.go_((0 - l) / 2);

		JunPenLocation[] points = new JunPenLocation[npoints];

		for (int i = 0; i < npoints; i++) {
			points[i] = this.location().rounded();
			this.go_(l);
			this.turn_(360 / npoints);
		}

		this.down();

		for (int i = npoints / 2; i > 0; i--) {
			for (int j = 0; j < npoints; j++) {
				this.location_(points[j]);
				this.goto_(points[(j + i) % npoints]);
				if (aBlock != null) {
					aBlock.value();
				}
			}
		}
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param npoints int
	 * @param d int
	 * @param millisecondValue long
	 * @category designs
	 */
	public void mandala_diameter_tick_(int npoints, int d, final long millisecondValue) {
		this.mandala_diameter_interim_(npoints, d, new StBlockClosure() {
			public Object value() {
				try {
					Thread.sleep(millisecondValue);
				} catch (InterruptedException e) {
				}
				return null;
			}
		});
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param n int
	 * @param a int
	 * @category designs
	 */
	public void spiral_angle_(int n, int a) {
		this.spiral_angle_interim_(n, a, null);
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param n int
	 * @param a int
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category designs
	 */
	public void spiral_angle_interim_(int n, int a, StBlockClosure aBlock) {
		for (int i = 1; i < (n + 1); i++) {
			this.go_(i);
			this.turn_(a);
			if (aBlock != null) {
				aBlock.value();
			}
		}
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param n int
	 * @param a int
	 * @param millisecondValue long
	 * @category designs
	 */
	public void spiral_angle_tick_(int n, int a, final long millisecondValue) {
		this.spiral_angle_interim_(n, a, new StBlockClosure() {
			public Object value() {
				try {
					Thread.sleep(millisecondValue);
				} catch (InterruptedException e) {
				}
				return null;
			}
		});
	}

	/**
	 * Begin drawing.
	 * 
	 * @category drawing
	 */
	public void begin() {
		Vector vector = new Vector();
		vector.addElement(this.location());
		sequence = vector;
	}

	/**
	 * The receiver be down.
	 * 
	 * @category drawing
	 */
	public void down() {
		down = true;
	}

	/**
	 * Fill the receiver.
	 * 
	 * @category drawing
	 */
	public void fill() {
		this.fillPolygon_(sequence);
	}

	/**
	 * Fill the receiver with block closure.
	 * 
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category drawing
	 */
	public void fill_(StBlockClosure aBlock) {
		boolean oldDown = this.isDown();
		final StValueHolder savedState = new StValueHolder(this.state());
		final JunPen this_ = this;
		final StBlockClosure aBlock_ = aBlock;
		StBlockClosure assertBlock = new StBlockClosure() {
			public Object value() {
				savedState.value_(this_.state());

				return null;
			}
		};

		StBlockClosure doBlock = new StBlockClosure() {
			public Object value() {
				this_.down();
				this_.begin();
				aBlock_.value();
				this_.fill();

				return null;
			}
		};

		StBlockClosure ensureBlock = new StBlockClosure() {
			public Object value() {
				this_.state_((StAssociation[]) savedState.value());

				return null;
			}
		};

		this.assert_do_ensure_(assertBlock, doBlock, ensureBlock);

		if (oldDown) {
			this.down();
		} else {
			this.up();
		}
	}

	/**
	 * Go the receiver with the specified distance.
	 * 
	 * @param distance double
	 * @category drawing
	 */
	public void go_(double distance) {
		JunPenLocation unitVector = JunPenLocation.X_y_z_(1.0d, 0.0d, 0.0d);
		JunPenTransformation aTransformation = JunPenTransformation.Unity();
		JunPenAngle latitude = this.direction().latitude();
		JunPenAngle longitude = this.direction().longitude();

		if (longitude.abs().isGreaterThan_(new Double(longitude.accuracy()))) {
			JunPenTransformation aT = JunPenTransformation.RotateZ_(longitude);
			aTransformation = aTransformation.product_(aT);
		}

		if (latitude.abs().isGreaterThan_(new Double(latitude.accuracy()))) {
			JunPenLocation aroundVector = JunPenLocation.X_y_z_(0.0d, -1.0d, 0.0d);
			aroundVector = aroundVector.transform_(aTransformation);

			JunPenLocation[] aLine = new JunPenLocation[2];
			aLine[0] = JunPenLocation.Zero();
			aLine[1] = aroundVector;

			JunPenTransformation aT = JunPenTransformation.Rotate_around_(latitude, aLine);
			aTransformation = aTransformation.product_(aT);
		}

		aTransformation = aTransformation.product_(JunPenTransformation.Scale_(new Double(distance)));
		aTransformation = aTransformation.product_(JunPenTransformation.Translate_(location));

		JunPenLocation newLocation = unitVector.transform_(aTransformation);
		this.goto_(newLocation);
	}

	/**
	 * Goto the receiver with the specified point.
	 * 
	 * @param newPoint double[]
	 * @category drawing
	 */
	public void goto_(double[] newPoint) {
		JunPenLocation oldPoint = this.location();
		this.location_(newPoint);
		this.drawLineFrom_to_(oldPoint, this.location());
	}

	/**
	 * Goto the receiver with the specified point.
	 * 
	 * @param newPoint double
	 * @category drawing
	 */
	public void goto_(double newPoint) {
		JunPenLocation oldPoint = this.location();
		this.location_(newPoint);
		this.drawLineFrom_to_(oldPoint, this.location());
	}

	/**
	 * Goto the receiver with the specified point.
	 * 
	 * @param newPoint java.lang.Object
	 * @category drawing
	 */
	public void goto_(Object newPoint) {
		JunPenLocation oldPoint = this.location();
		this.location_(newPoint);
		this.drawLineFrom_to_(oldPoint, this.location());
	}

	/**
	 * Goto the receiver with home position.
	 * 
	 * @category drawing
	 */
	public void home() {
		this.location_(JunPenLocation.Zero());
		this.direction_(JunPenDirection.Zero());
	}

	/**
	 * Answer the current state information.
	 * 
	 * @return jp.co.sra.smalltalk.StAssociation[]
	 * @category drawing
	 */
	public StAssociation[] state() {
		StAssociation[] state = new StAssociation[6];
		state[0] = new StAssociation($("location"), this.location());
		state[1] = new StAssociation($("direction"), this.direction());
		state[2] = new StAssociation($("down"), new Boolean(this.down));
		state[3] = new StAssociation($("nib"), new Integer(this.nib()));
		state[4] = new StAssociation($("color"), this.color());
		state[5] = new StAssociation($("sequence"), sequence);

		return state;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param anArray jp.co.sra.smalltalk.StAssociation[]
	 * @category drawing
	 */
	public void state_(StAssociation[] anArray) {
		for (int index = 0; index < anArray.length; index++) {
			StAssociation association = anArray[index];
			StSymbol key = (StSymbol) association.key();
			Object value = association.value();

			if (key == $("location")) {
				this.location_((JunPenLocation) value);
			}

			if (key == $("direction")) {
				this.direction_((JunPenDirection) value);
			}

			if (key == $("down")) {
				this.down = ((Boolean) value).booleanValue();
			}

			if (key == $("nib")) {
				this.nib_(((Integer) value).intValue());
			}

			if (key == $("color")) {
				this.color_((Color) value);
			}

			if (key == $("sequence")) {
				sequence = (Vector) value;
			}
		}
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param degrees double
	 * @category drawing
	 */
	public void tilt_(double degrees) {
		double longitude = this.direction().longitude().deg();
		double latitude = (this.direction().latitude().deg() + degrees) % 360.0d;
		this.direction_(JunPenDirection.Longitude_latitude_(longitude, latitude));
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param degrees double
	 * @category drawing
	 */
	public void turn_(double degrees) {
		double longitude = (this.direction().longitude().deg() + degrees) % 360.0d;
		double latitude = this.direction().latitude().deg();
		this.direction_(JunPenDirection.Longitude_latitude_(longitude, latitude));
	}

	/**
	 * The receiver be up.
	 * 
	 * @category drawing
	 */
	public void up() {
		down = false;
	}

	/**
	 * Answer true if the receiver is down, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isDown() {
		return down;
	}

	/**
	 * Answer true if the receiver is up, otherwise false.
	 * 
	 * @return boolean
	 * @category testing
	 */
	public boolean isUp() {
		return !down;
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param assertBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param doBlock jp.co.sra.smalltalk.StBlockClosure
	 * @param ensureBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category pirvate
	 */
	private void assert_do_ensure_(StBlockClosure assertBlock, StBlockClosure doBlock, StBlockClosure ensureBlock) {
		assertBlock.value();
		try {
			doBlock.value();
		} catch (Exception e) {
		} finally {
			ensureBlock.value();
		}
	}

	/**
	 * Draw line from the specified point to other point on the receiver.
	 * 
	 * @param startPoint jp.co.sra.jun.goodies.pen.JunPenLocation
	 * @param endPoint jp.co.sra.jun.goodies.pen.JunPenLocation
	 * @category pirvate
	 */
	protected void drawLineFrom_to_(JunPenLocation startPoint, JunPenLocation endPoint) {
		Vector anArray = new Vector();

		if (this.isDown()) {
			anArray.addElement($("stroke"));
			anArray.addElement(new Integer(this.nib()));
			anArray.addElement(this.color());
			anArray.addElement(startPoint);
			anArray.addElement(endPoint);
			drawing.value_(anArray);
		}

		if (sequence != null) {
			sequence.addElement(endPoint);
		}
	}

	/**
	 * DOCUMENT ME!
	 * 
	 * @param collectionOfLocation java.uitl.Vector
	 * @category pirvate
	 */
	protected void fillPolygon_(Vector collectionOfLocation) {
		if (collectionOfLocation == null) {
			return;
		}

		if (collectionOfLocation.size() < 3) {
			return;
		}

		Vector anArray = new Vector();

		if (this.isDown()) {
			anArray.addElement($("fill"));
			anArray.addElement(this.color());
			anArray.addElement(collectionOfLocation);
			drawing.value_(anArray);
		}

		sequence = null;
	}
}
