package jp.co.sra.jun.goodies.nib;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Window;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.StColorValue;
import jp.co.sra.smalltalk.StComposedText;
import jp.co.sra.smalltalk.StRectangle;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.StViewJPanel;
import jp.co.sra.smalltalk.WindowSpecLayout;
import jp.co.sra.jun.goodies.track.JunSimpleSliderViewSwing;
import jp.co.sra.jun.system.framework.JunAbstractViewJPanel;

/**
 * JunNibChoiceViewSwing class
 * 
 *  @author    Nobuto Matsubara
 *  @created   2004/03/19 (By Nobuto Matsubara)
 *  @updated   2004/09/22 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun519 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunNibChoiceViewSwing.java,v 8.11 2008/02/20 06:31:51 nisinaka Exp $
 */
public class JunNibChoiceViewSwing extends JunAbstractViewJPanel implements JunNibChoiceView {

	protected JunSimpleSliderViewSwing sliderView;

	/**
	 * Answer an StView which contains the JunNibChoiceViewSwing with a simple slider.
	 * 
	 * @param aNibChoice jp.co.sra.jun.goodies.nib.JunNibChoice
	 * @return jp.co.sra.smalltalk.StView
	 * @category Instance creation
	 */
	public static StView WithSlider(JunNibChoice aNibChoice) {
		return new StViewJPanel(aNibChoice) {

			protected JunNibChoiceViewSwing nibChoiceView;

			public JunNibChoiceViewSwing nibChoiceView() {
				if (nibChoiceView == null) {
					nibChoiceView = new JunNibChoiceViewSwing((JunNibChoice) this.model());
				}
				return nibChoiceView;
			}

			protected void buildComponent() {
				this.setLayout(new WindowSpecLayout());
				this.add(this.nibChoiceView(), WindowSpecLayout.Position(0f, 1, 0f, 1, 1f, -1, 1f, -20));
				this.add(this.nibChoiceView().sliderView(), WindowSpecLayout.Position(0f, 1, 1f, -19, 1f, -1, 1f, -1));
				this.setSize(new Dimension(234, 55));
			}

			public void _setupKeyboard(Window aWindow) {
				this.nibChoiceView()._setupKeyboard(aWindow);
			}
		};
	}

	/**
	 * Create a new instance of JunNibChoiceViewSwing and initialize it.
	 * 
	 * @param aModel jp.co.sra.jun.goodies.nib.JunNibChoice
	 * @category Instance creation
	 */
	public JunNibChoiceViewSwing(JunNibChoice aModel) {
		super(aModel);
	}

	/**
	 * Answer my model as JunNibChoice
	 * 
	 * @return jp.co.sra.jun.goodies.nib.JunNibChoice
	 * @category model accessing
	 */
	public JunNibChoice getNibChoice() {
		return (JunNibChoice) this.model();
	}

	/**
	 * Answer my slider view.
	 *
	 * @return JunSimpleSliderViewSwing
	 * @category user interface
	 */
	protected JunSimpleSliderViewSwing sliderView() {
		if (sliderView == null) {
			sliderView = new JunSimpleSliderViewSwing(this.getNibChoice().simpleSlider());
		}
		return sliderView;
	}

	/**
	 * Answer the area of nib shape.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category accessing
	 */
	public StRectangle nibShapeArea() {
		StRectangle box = new StRectangle(this.getBounds());
		box = box.insetBy_(new StRectangle(-1, -1, 2, 2));
		return box;
	}

	/**
	 * Answer the area of nib width.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category accessing
	 */
	public StRectangle nibWidthArea() {
		StRectangle box = null;

		Graphics aGraphics = null;
		try {
			aGraphics = this.getGraphics();
			FontMetrics aFontMetrics = aGraphics.getFontMetrics();
			Dimension extent = new Dimension(aFontMetrics.stringWidth(this.nibWidthString()), aFontMetrics.getHeight());
			box = new StRectangle(extent);
			box = box.expandedBy_(new StRectangle(4, 2, 4, 2));
			box = box.align_with_(box.bottomRight(), this.nibShapeArea().bottomRight());
		} finally {
			if (aGraphics != null) {
				aGraphics.dispose();
				aGraphics = null;
			}
		}

		return box;
	}

	/**
	 * Answer the string of nib width.
	 * 
	 * @return jp.co.sra.smalltalk.StRectangle
	 * @category accessing
	 */
	public String nibWidthString() {
		if (this.getNibChoice().nibWidth() - (int) this.getNibChoice().nibWidth() == 0) {
			return String.valueOf((int) this.getNibChoice().nibWidth());
		}
		return String.valueOf(this.getNibChoice().nibWidth());
	}

	/**
	 * Update the view.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		if (!this.isShowing()) {
			return;
		}
		this.displayNibShapeOn_(aGraphics);
		this.displayNibWidthOn_(aGraphics);
	}

	/**
	 * Display the nib shape.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category displaying
	 */
	public void displayNibShapeOn_(Graphics aGraphics) {
		StRectangle box = this.nibShapeArea();
		aGraphics.setColor(Color.white);
		box = box.insetBy_(new StRectangle(1, 1, 0, 0));
		aGraphics.fillRect(box.x(), box.y(), box.width(), box.height());
		StRectangle area = StRectangle.Origin_extent_(new Point(0, 0), new Point((int) this.getNibChoice().nibWidth(), (int) this.getNibChoice().nibWidth()));
		area = area.align_with_(area.center(), box.center());
		if (StColorValue._DistanceFrom(this.getNibChoice().nibColor(), Color.white) / Math.sqrt(StColorValue._MaxDistanceSquared()) < 0.05) {
			aGraphics.setColor(Color.black);
			if (this.getNibChoice().nibShape() == $("rectangle")) {
				StRectangle anArea = area.expandedBy_(new StRectangle(1, 1, 1, 1));
				aGraphics.fillRect(anArea.x(), anArea.y(), anArea.width(), anArea.height());
			}
			if (this.getNibChoice().nibShape() == $("circle")) {
				if (area.width() < 3) {
					StRectangle anArea = area.expandedBy_(new StRectangle(1, 1, 1, 1));
					aGraphics.fillRect(anArea.x(), anArea.y(), anArea.width(), anArea.height());
				} else {
					StRectangle anArea = area.expandedBy_(new StRectangle(1, 1, 2, 2));
					aGraphics.fillArc(anArea.x(), anArea.y(), anArea.width(), anArea.height(), 0, 360);
				}
			}
		}
		aGraphics.setColor(this.getNibChoice().nibColor());
		if (this.getNibChoice().nibShape() == $("rectangle")) {
			aGraphics.fillRect(area.x(), area.y(), area.width(), area.height());
		}
		if (this.getNibChoice().nibShape() == $("circle")) {
			if (area.width() < 3) {
				aGraphics.fillRect(area.x(), area.y(), area.width(), area.height());
			} else {
				StRectangle anArea = area.expandedBy_(new StRectangle(0, 0, 1, 1));
				aGraphics.fillArc(anArea.x(), anArea.y(), anArea.width(), anArea.height(), 0, 360);
			}
		}
		aGraphics.setColor(Color.black);
		box = box.expandedBy_(new StRectangle(1, 1, 0, 0));
		aGraphics.drawRect(box.x(), box.y(), box.width(), box.height());
	}

	/**
	 * Display the nib width on the graphics.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @category displaying
	 */
	public void displayNibWidthOn_(Graphics aGraphics) {
		StRectangle box = this.nibWidthArea();
		StComposedText text = new StComposedText(this.nibWidthString());
		aGraphics.setColor(Color.white);
		box = box.insetBy_(new StRectangle(1, 1, 0, 0));
		aGraphics.fillRect(box.x(), box.y(), box.width(), box.height());
		aGraphics.setColor(Color.black);
		StRectangle area = new StRectangle(text.bounds());
		area = area.align_with_(area.center(), box.center());
		Graphics2D gc = (Graphics2D) aGraphics.create();
		gc.clipRect(box.x(), box.y(), box.width(), box.height());
		text.displayOn_at_(gc, area.origin());
	}

	/**
	 * Action for the update notification.
	 * 
	 * @param evt jp.co.sra.smalltalk.DependentEvent
	 * @category updating
	 */
	public void update_(DependentEvent evt) {
		if (!this.isShowing()) {
			return;
		}

		StSymbol aSymbol = evt.getAspect();
		if (aSymbol == $("nibWidth")) {
			this.displayOn_(this.getGraphics());
		} else if (aSymbol == $("nibColor")) {
			this.displayOn_(this.getGraphics());
		} else if (aSymbol == $("nibShape")) {
			this.displayOn_(this.getGraphics());
		} else {
			super.update_(evt);
		}
	}

	/**
	 * Set up the keyboard for the view on the window.
	 * 
	 * @param aWindow java.awt.Window
	 * @see jp.co.sra.smalltalk.StView#_setupKeyboard(java.awt.Window)
	 * @category keyboard
	 */
	public void _setupKeyboard(final Window aWindow) {
		this.addKeyListener(this.getNibChoice()._keyListener());

		MouseListener aMouseListener = new MouseAdapter() {
			public void mouseEntered(MouseEvent e) {
				Component focusOwner = aWindow.getFocusOwner();
				if (focusOwner != null && focusOwner != JunNibChoiceViewSwing.this) {
					JunNibChoiceViewSwing.this.requestFocus();
				}
			}
		};
		this.addMouseListener(aMouseListener);
		this.sliderView().addMouseListener(aMouseListener);
	}

}
