package jp.co.sra.jun.goodies.movie.support;

import java.awt.Image;
import java.io.BufferedOutputStream;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.net.MalformedURLException;
import java.net.URI;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.TimeZone;

import jp.co.sra.smalltalk.StImage;

import jp.co.sra.jun.goodies.files.JunFileModel;
import jp.co.sra.jun.goodies.image.streams.JunImageStream;
import jp.co.sra.jun.goodies.image.streams.JunJpegImageStream;
import jp.co.sra.jun.goodies.movie.framework.JunMovieHandle;
import jp.co.sra.jun.goodies.movie.framework.JunMoviePlayer;
import jp.co.sra.jun.goodies.tips.JunURL;
import jp.co.sra.jun.graphics.navigator.JunFileRequesterDialog;
import jp.co.sra.jun.system.framework.JunAbstractObject;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunMovieThumbnails class
 * 
 *  @author    nisinaka
 *  @created   2006/02/13 (by nisinaka)
 *  @updated   N/A
 *  @version   699 (with StPL8.9) based on Jun669 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunMovieThumbnails.java,v 8.13 2008/02/20 06:31:50 nisinaka Exp $
 */
public class JunMovieThumbnails extends JunAbstractObject {

	protected static SimpleDateFormat TimeStampFormat;

	static {
		TimeStampFormat = new SimpleDateFormat("HH:mm:ss.SSS");
		TimeStampFormat.setTimeZone(TimeZone.getTimeZone("GMT"));
	}

	protected File movieFile;
	protected double tickMilliseconds;
	protected int thumbnailSize;
	protected StImage[] capturedImages;
	protected StImage[] thumbnailImages;
	protected long[] capturedStamps;
	protected boolean timeStamp;
	protected int numberOfColumns;

	/**
	 * Create a new instance of JunMovieThumbnails and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunMovieThumbnails() {
		super();
	}

	/**
	 * Create a new instance of JunMovieThumbnails and initialize it.
	 *
	 * @param aFile java.io.File
	 * @category Instance creation
	 */
	public JunMovieThumbnails(File aFile) {
		this();
		this.movieFile_(aFile);
	}

	/**
	 * Create a new instance of JunMovieThumbnails and initialize it.
	 *
	 * @param aFile java.io.File
	 * @category Instance creation
	 */
	public JunMovieThumbnails(File aFile, int tickMilliseconds) {
		this.movieFile_(aFile);
		this.tickMilliseconds_(tickMilliseconds);
	}

	/**
	 * Create a new instance of JunMovieThumbnails and initialize it.
	 *
	 * @param aFile java.io.File
	 * @param tickMilliseconds int
	 * @param thumbnailSize int
	 * @category Instance creation
	 */
	public JunMovieThumbnails(File aFile, int tickMilliseconds, int thumbnailSize) {
		this.movieFile_(aFile);
		this.tickMilliseconds_(tickMilliseconds);
		this.thumbnailSize_(thumbnailSize);
	}

	/**
	 * Reqeust for a movie file and create a new instance of JunMovieThumbnails with it.
	 * 
	 * @return jp.co.sra.jun.goodies.movie.framework.support.JunMovieThumbnails
	 * @category Utilities
	 */
	public static JunMovieThumbnails Request() {
		return Request($String("Select a movie file."));
	}

	/**
	 * Reqeust for a movie file and create a new instance of JunMovieThumbnails with it.
	 * 
	 * @param messageString java.lang.String
	 * @return jp.co.sra.jun.goodies.movie.framework.support.JunMovieThumbnails
	 * @category Utilities
	 */
	public static JunMovieThumbnails Request(String messageString) {
		JunFileModel.FileType[] fileTypes = new JunFileModel.FileType[] { new JunFileModel.FileType($String("Movie files"), JunSystem.DefaultMovieExtensionPatterns()), JunFileModel.FileType.All($String("All files")) };
		File aFile = JunFileRequesterDialog.RequestFile(messageString, fileTypes, fileTypes[0]);
		if (aFile == null) {
			return null;
		}
		return new JunMovieThumbnails(aFile);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunAbstractObject#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();
		movieFile = null;
		tickMilliseconds = this.defaultTickMilliseconds();
		thumbnailSize = this.defaultThumbnailSize();
		this.flushImages();
		timeStamp = false;
		numberOfColumns = 10;
	}

	/**
	 * Answer my current movie file.
	 * 
	 * @return java.io.File
	 * @category accessing
	 */
	public File movieFile() {
		return movieFile;
	}

	/**
	 * Set my new movie file.
	 * 
	 * @param aFile java.io.File
	 * @category accessing
	 */
	public void movieFile_(File aFile) {
		movieFile = aFile;
		this.flushImages();
	}

	/**
	 * Answer my current movie file as URI.
	 * 
	 * @return java.net.URI
	 * @category accessing
	 */
	public URI movieURI() {
		if (this.movieFile() == null) {
			return null;
		}

		return this.movieFile().toURI();
	}

	/**
	 * Answer my current value of tickMilliconds.
	 * 
	 * @return double
	 * @category accessing
	 */
	public double tickMilliseconds() {
		return tickMilliseconds;
	}

	/**
	 * Set my new value of tickMilliseconds.
	 * 
	 * @param newTickMilliseconds int
	 * @category accessing
	 */
	public void tickMilliseconds_(double newTickMilliseconds) {
		tickMilliseconds = Math.max(newTickMilliseconds, 10);
	}

	/**
	 * Answer my current value of thumbnailSize.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int thumbnailSize() {
		return thumbnailSize;
	}

	/**
	 * Set my new value of thunbmailSize.
	 * 
	 * @param newThumbnailSize int
	 * @category accessing
	 */
	public void thumbnailSize_(int newThumbnailSize) {
		thumbnailSize = Math.max(newThumbnailSize, 16);
	}

	/**
	 * Answer the captured images.
	 * 
	 * @return jp.co.sra.smalltalk.StImage[]
	 * @category accessing
	 */
	public StImage[] capturedImages() {
		if (capturedImages == null) {
			if (this.movieFile() == null) {
				return null;
			}

			JunMoviePlayer moviePlayer = new JunMoviePlayer(this.movieFile());
			try {
				long[] capturedStamps = this.capturedStamps();
				StImage[] images = new StImage[capturedStamps.length];
				for (int i = 0; i < images.length; i++) {
					moviePlayer.gotoInMilliseconds_(capturedStamps[i]);
					images[i] = moviePlayer.asImage();
				}

				capturedImages = images;
			} finally {
				moviePlayer.release();
			}
		}

		return capturedImages;
	}

	/**
	 * Answer the thumbnail images.
	 * 
	 * @return jp.co.sra.smalltalk.StImage[]
	 * @category accessing
	 */
	public StImage[] thumbnailImages() {
		if (thumbnailImages == null) {
			StImage[] capturedImages = this.capturedImages();
			if (capturedImages == null) {
				return null;
			}

			StImage[] images = new StImage[capturedImages.length];
			for (int i = 0; i < images.length; i++) {
				StImage anImage = capturedImages[i];
				int size = Math.max(anImage.width(), anImage.height());
				double scale = (double) this.thumbnailSize() / size;
				int width = (int) (anImage.width() * scale);
				int height = (int) (anImage.height() * scale);
				images[i] = new StImage(capturedImages[i].image().getScaledInstance(width, height, Image.SCALE_SMOOTH));
			}

			thumbnailImages = images;
		}

		return thumbnailImages;
	}

	/**
	 * Answer the captured stamps.
	 * 
	 * @return long[]
	 * @category accessing
	 */
	public long[] capturedStamps() {
		if (capturedStamps == null) {
			if (this.movieFile() == null) {
				return null;
			}

			JunMovieHandle movieHandle = JunMovieHandle.Filename_(this.movieFile());
			try {
				double durationInMilliseconds = movieHandle.durationInSeconds() * 1000;
				int size = (int) Math.floor(durationInMilliseconds / this.tickMilliseconds());
				capturedStamps = new long[size];
				for (int i = 0; i < size; i++) {
					capturedStamps[i] = (long) (this.tickMilliseconds() * i);
				}
			} finally {
				movieHandle.release();
			}
		}

		return capturedStamps;
	}

	/**
	 * Answer the number of the thumbnail images.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int numberOfThumbnails() {
		return this.thumbnailImages().length;
	}

	/**
	 * Set the number of the thumbnail images.
	 * 
	 * @param numberOfThumbnails int
	 * @category accessing
	 */
	public void numberOfThumbnails_(int numberOfThumbnails) {
		if (this.movieFile() == null) {
			return;
		}

		JunMovieHandle movieHandle = JunMovieHandle.Filename_(this.movieFile());
		try {
			numberOfThumbnails = Math.max(numberOfThumbnails, 1);
			double millisecondValue = movieHandle.durationInSeconds() * 1000;
			this.tickMilliseconds_(millisecondValue / numberOfThumbnails);
		} finally {
			movieHandle.release();
		}
	}

	/**
	 * Answer my current number of columns.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int numberOfColumns() {
		return numberOfColumns;
	}

	/**
	 * Set my new number of columns.
	 * 
	 * @param newNumberOfColumns int
	 * @category accessing 
	 */
	public void numberOfColumns_(int newNumberOfColumns) {
		numberOfColumns = Math.max(newNumberOfColumns, 1);
	}

	/**
	 * Answer true if the receiver generates HTML with timestamp information.
	 * 
	 * @return boolean
	 * @category accessing
	 */
	public boolean timeStamp() {
		return timeStamp;
	}

	/**
	 * Set whether the receiver generates HTML with timestamp information or not.
	 * 
	 * @param aBoolean boolean
	 * @category accessing 
	 */
	public void timeStamp_(boolean aBoolean) {
		timeStamp = aBoolean;
	}

	/**
	 * Answer the default tick in milliseconds.
	 * 
	 * @return double
	 * @category defaults
	 */
	protected double defaultTickMilliseconds() {
		return 1000;
	}

	/**
	 * Answer the default thumbnail size.
	 * 
	 * @return int
	 * @category defaults
	 */
	protected int defaultThumbnailSize() {
		return 64;
	}

	/**
	 * Flush the images.
	 * 
	 * @category flushing
	 */
	protected void flushImages() {
		capturedImages = null;
		thumbnailImages = null;
		capturedStamps = null;
	}

	/**
	 * Show the movie thumbnails.
	 * 
	 * @category viewing
	 */
	public void show() {
		File aDirectory = this.htmlDirectory();
		File aFile = new File(aDirectory, "index.html");
		PrintWriter aWriter = null;
		try {
			aWriter = new PrintWriter(new BufferedWriter(new OutputStreamWriter(new FileOutputStream(aFile), "Shift_JIS")));

			Map[] thumbnailInfo = this.makeThumbnailsIn_(aDirectory);
			Map[] imageInfo = this.makeImagesIn_(aDirectory);
			this.htmlOn_thumbnailInfo_imageInfo_(aWriter, thumbnailInfo, imageInfo);

		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			if (aWriter != null) {
				aWriter.flush();
				aWriter.close();
			}
		}

		try {
			JunURL.Browse_(aFile.toURL());
		} catch (MalformedURLException e) {
			e.printStackTrace();
		}
	}

	/**
	 * Answer the html directory.
	 * 
	 * @return java.io.File
	 * @category html support
	 */
	protected File htmlDirectory() {
		File aDirectory = null;
		do {
			aDirectory = new File(JunSystem.DefaultBaseName());
		} while (aDirectory.exists());

		aDirectory.mkdir();
		return aDirectory;
	}

	/**
	 * Write HTML on the writer.
	 * 
	 * @param aWriter java.io.PrintWriter
	 * @param thumbnailInfo java.util.Map[]
	 * @param imageInfo java.util.Map[]
	 * @throws java.io.IOException
	 * @category html support
	 */
	protected void htmlOn_thumbnailInfo_imageInfo_(PrintWriter aWriter, Map[] thumbnailInfo, Map[] imageInfo) throws IOException {
		aWriter.println("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">");

		String htmlTag = "HTML";
		try {
			aWriter.println("<" + htmlTag + ">");

			this.htmlHeadOn_(aWriter);
			this.htmlBodyOn_thumbnailInfo_imageInfo_(aWriter, thumbnailInfo, imageInfo);
		} finally {
			aWriter.println("</" + htmlTag + ">");
		}
	}

	/**
	 * Write the head part of the HTML on the writer.
	 * 
	 * @param aWriter java.io.PrintWriter
	 * @throws java.io.IOException 
	 * @category html support
	 */
	protected void htmlHeadOn_(PrintWriter aWriter) throws IOException {
		String headTag = "HEAD";
		try {
			aWriter.println("<" + headTag + ">");
			aWriter.println("<META http-equiv=\"Content-Type\" content=\"text/html; charset=Shit_JIS\">");

			String titleTag = "TITLE";
			try {
				aWriter.print("<" + titleTag + ">");
				aWriter.print(this.htmlTitleString());
			} finally {
				aWriter.println("</" + titleTag + ">");
			}
		} finally {
			aWriter.println("</" + headTag + ">");
		}
	}

	/**
	 * Write the body part of the HTML on the writer.
	 * 
	 * @param aWriter java.io.PrintWriter
	 * @param thumbnailInfo java.util.Map[]
	 * @param imageInfo java.util.Map[]
	 * @throws java.io.IOException 
	 * @category html support
	 */
	protected void htmlBodyOn_thumbnailInfo_imageInfo_(PrintWriter aWriter, Map[] thumbnailInfo, Map[] imageInfo) throws IOException {
		String bodyTag = "BODY";
		try {
			aWriter.println("<" + bodyTag + " bgcolor=\"#ffffff\">");

			this.htmlMovieOn_(aWriter);
			this.htmlTableOn_thumbnailInfo_imageInfo_(aWriter, thumbnailInfo, imageInfo);
		} finally {
			aWriter.println("</" + bodyTag + ">");
		}
	}

	/**
	 * Answer the title string.
	 * 
	 * @return java.lang.String
	 * @category html support
	 */
	protected String htmlTitleString() {
		if (this.movieFile() == null) {
			return "";
		}

		return this.movieFile().getName();
	}

	/**
	 * Write the movie part of the HTML on the writer.
	 * 
	 * @param aWriter java.io.PrintWriter
	 * @throws IOException 
	 * @category html support
	 */
	protected void htmlMovieOn_(PrintWriter aWriter) throws IOException {
		String movieTag = "H2";
		try {
			aWriter.print("<" + movieTag + ">");
			aWriter.print("<A HREF=\"" + this.movieURI() + "\">" + this.htmlTitleString() + "</A>");
		} finally {
			aWriter.println("</" + movieTag + ">");
		}
	}

	/**
	 * Write the table part of the HTML on the writer.
	 * 
	 * @param aWriter java.io.PrintWriter
	 * @param thumbnailInfo java.util.Map[]
	 * @param imageInfo java.util.Map[]
	 * @category html support
	 */
	protected void htmlTableOn_thumbnailInfo_imageInfo_(PrintWriter aWriter, Map[] thumbnailInfo, Map[] imageInfo) {
		String tableTag = "TABLE";
		try {
			aWriter.print("<" + tableTag);
			if (this.timeStamp()) {
				aWriter.print(" border=\"0\" cellpadding=\"0\" cellspacing=\"2\" ");
			} else {
				aWriter.print(" border=\"0\" cellpadding=\"0\" cellspacing=\"0\" ");
			}
			aWriter.println(">");

			String tbodyTag = "TBODY";
			try {
				aWriter.println("  <" + tbodyTag + ">");
				aWriter.println("    <TR>");
				for (int i = 0; i < this.numberOfThumbnails(); i++) {
					aWriter.print("      <TD align=\"center\">");
					this.htmlThumbnailOn_thumbnailMap_imageMap_(aWriter, thumbnailInfo[i], imageInfo[i]);
					aWriter.println("</TD>");

					if ((i + 1) % this.numberOfColumns() == 0) {
						aWriter.println("    </TR>");
						aWriter.println("    <TR>");
					}
				}
				aWriter.println("    </TR>");
			} finally {
				aWriter.println("  </" + tbodyTag + ">");
			}
		} finally {
			aWriter.println("</" + tableTag + ">");
		}
	}

	/**
	 * Write the thumbnail part of the HTML on the writer.
	 * 
	 * @param aWriter java.io.PrintWriter
	 * @param thumbnailMap java.util.Map
	 * @param imageMap java.util.Map
	 * @category html support 
	 */
	protected void htmlThumbnailOn_thumbnailMap_imageMap_(PrintWriter aWriter, Map thumbnailMap, Map imageMap) {
		aWriter.print("<A HREF=\"" + imageMap.get($("src")) + "\">");
		aWriter.print("<IMG");
		aWriter.print(" align=\"middle\"");
		aWriter.print(" hspace=\"0\"");
		aWriter.print(" vspace=\"1\"");
		aWriter.print(" border=\"0\"");
		aWriter.print(" width=\"" + thumbnailMap.get($("width")) + "\"");
		aWriter.print(" height=\"" + thumbnailMap.get($("height")) + "\"");
		aWriter.print(" src=\"" + thumbnailMap.get($("src")) + "\"");
		aWriter.print(" alt=\"" + thumbnailMap.get($("alt")) + "\"");
		aWriter.print(">");
		aWriter.print("</A>");

		if (this.timeStamp()) {
			int time = Integer.parseInt((String) thumbnailMap.get($("alt")));
			aWriter.print("<BR>");
			aWriter.print("<TT>");
			aWriter.print("&nbsp;");
			aWriter.print(TimeStampFormat.format(new Date(time)));
			aWriter.print("&nbsp;");
			aWriter.print("</TT>");
		}
	}

	/**
	 * Make thumbnails in the specified directory.
	 * 
	 * @param aDirectory java.io.File
	 * @return java.util.Map[]
	 * @category private
	 */
	protected Map[] makeThumbnailsIn_(File aDirectory) {
		ArrayList aList = new ArrayList();

		File thumbnailsDirectory = new File(aDirectory, "thumbnails");
		thumbnailsDirectory.mkdir();

		long[] capturedStamps = this.capturedStamps();
		if (capturedStamps != null && capturedStamps.length > 0) {
			int length = String.valueOf(capturedStamps[capturedStamps.length - 1]).length();
			String pattern = "";
			for (int i = 0; i < length; i++) {
				pattern += '0';
			}
			DecimalFormat format = new DecimalFormat(pattern);
			for (int i = 0; i < capturedStamps.length; i++) {
				String stamp = format.format(capturedStamps[i]);
				File aFile = new File(thumbnailsDirectory, stamp + ".jpg");
				StImage anImage = this.thumbnailImages()[i];

				JunImageStream writeStream = null;
				try {
					writeStream = JunJpegImageStream.On_(new BufferedOutputStream(new FileOutputStream(aFile)));
					writeStream.nextPutImage_(anImage);
				} catch (IOException e) {
					e.printStackTrace();
				} finally {
					if (writeStream != null) {
						try {
							writeStream.flush();
							writeStream.close();
						} catch (IOException e) {
						} finally {
							writeStream = null;
						}
					}
				}

				HashMap aMap = new HashMap();
				aMap.put($("alt"), stamp);
				aMap.put($("src"), "./thumbnails/" + stamp + ".jpg");
				aMap.put($("width"), new Integer(anImage.width()));
				aMap.put($("height"), new Integer(anImage.height()));
				aList.add(aMap);
			}
		}

		return (Map[]) aList.toArray(new HashMap[aList.size()]);
	}

	/**
	 * Make images in the specified directory.
	 * 
	 * @param aDirectory java.io.File
	 * @return java.util.Map[]
	 * @category private
	 */
	protected Map[] makeImagesIn_(File aDirectory) {
		ArrayList aList = new ArrayList();

		File thumbnailsDirectory = new File(aDirectory, "images");
		thumbnailsDirectory.mkdir();

		long[] capturedStamps = this.capturedStamps();
		if (capturedStamps != null && capturedStamps.length > 0) {
			int length = String.valueOf(capturedStamps[capturedStamps.length - 1]).length();
			String pattern = "";
			for (int i = 0; i < length; i++) {
				pattern += '0';
			}
			DecimalFormat format = new DecimalFormat(pattern);
			for (int i = 0; i < capturedStamps.length; i++) {
				String stamp = format.format(capturedStamps[i]);
				File aFile = new File(thumbnailsDirectory, stamp + ".jpg");
				StImage anImage = this.capturedImages()[i];

				JunImageStream writeStream = null;
				try {
					writeStream = JunJpegImageStream.On_(new BufferedOutputStream(new FileOutputStream(aFile)));
					writeStream.nextPutImage_(anImage);
				} catch (IOException e) {
					e.printStackTrace();
				} finally {
					if (writeStream != null) {
						try {
							writeStream.flush();
							writeStream.close();
						} catch (IOException e) {
						} finally {
							writeStream = null;
						}
					}
				}

				HashMap aMap = new HashMap();
				aMap.put($("alt"), stamp);
				aMap.put($("src"), "./images/" + stamp + ".jpg");
				aMap.put($("width"), new Integer(anImage.width()));
				aMap.put($("height"), new Integer(anImage.height()));
				aList.add(aMap);
			}
		}

		return (Map[]) aList.toArray(new HashMap[aList.size()]);
	}

}
