package jp.co.sra.jun.goodies.milliseconds;

import java.awt.Toolkit;
import java.awt.datatransfer.StringSelection;
import java.io.IOException;
import java.io.Writer;
import java.util.Calendar;
import java.util.StringTokenizer;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.StView;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StMenuItem;
import jp.co.sra.smalltalk.menu.StPopupMenu;

import jp.co.sra.jun.system.framework.JunApplicationModel;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunMillisecondModel class
 * 
 *  @author    nisinaka
 *  @created   2003/05/08 (by nisinaka)
 *  @updated   2005/03/03 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun473 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunMillisecondModel.java,v 8.11 2008/02/20 06:31:49 nisinaka Exp $
 */
public class JunMillisecondModel extends JunApplicationModel {

	protected long originalValue;
	protected long millisecondValue;
	protected int msecs;
	protected int seconds;
	protected int minutes;
	protected int hours;
	protected StBlockClosure validateBlock;
	protected StPopupMenu _popupMenu;

	/**
	 * Create a new instance of <code>JunMillisecondModel</code> and initialize it.
	 *
	 * @category Instance creation
	 */
	public JunMillisecondModel() {
		this.initialize();
	}

	/**
	 * Create a new instance of <code>JunMillisecondModel</code> with the specified numbers.
	 *
	 * @param hourInteger int
	 * @param minuteInteger int
	 * @param secondInteger int
	 * @param msecInteger int
	 * @category Instance creation
	 */
	public JunMillisecondModel(int hourInteger, int minuteInteger, int secondInteger, int msecInteger) {
		long hours = 3600 * 1000 * Math.max(0, hourInteger);
		long minutes = 60 * 1000 * Math.max(0, Math.min(minuteInteger, 59));
		long seconds = 1000 * Math.max(0, Math.min(secondInteger, 59));
		long msecs = Math.max(0, Math.min(msecInteger, 999));
		this.milliseconds_(hours + minutes + seconds + msecs);
	}

	/**
	 * Create a new instance of <code>JunMillisecondModel</code> and initialize it.
	 *
	 * @param millisecondClockValue long
	 * @category Instance creation
	 */
	public JunMillisecondModel(long millisecondClockValue) {
		this.initialize_(millisecondClockValue);
	}

	/**
	 * Convert the number in milliseconds to a string representation.
	 *
	 * @return java.lang.String
	 * @param milliseconds long
	 * @category Converting
	 */
	public static String ConvertMillisecondsToString_(long milliseconds) {
		long decimal = milliseconds % 1000;
		long value = milliseconds / 1000;
		int seconds = (int) (value % 60);
		int minutes = (int) (value / 60 % 60);
		int hours = (int) (value / 3600);

		StringBuffer buf = new StringBuffer();
		if (hours < 10) {
			buf.append('0');
		}
		buf.append(hours);
		buf.append(':');
		if (minutes < 10) {
			buf.append('0');
		}
		buf.append(minutes);
		buf.append(':');
		if (seconds < 10) {
			buf.append('0');
		}
		buf.append(seconds);
		buf.append('.');
		if (decimal < 100) {
			buf.append('0');
		}
		if (decimal < 10) {
			buf.append('0');
		}
		buf.append(decimal);
		return buf.toString();
	}

	/**
	 * Convert the string representation of milliseconds to a long number.
	 *
	 * @return long
	 * @param aString java.lang.String
	 * @category Converting
	 */
	public static long ConvertStringToMilliseconds_(String aString) {
		final String delimiters = " \t\r\n\f:.";
		StringTokenizer tokenizer = new StringTokenizer(aString, delimiters);
		long hours = Integer.valueOf(tokenizer.nextToken()).intValue() * 3600 * 1000;
		long minutes = Integer.valueOf(tokenizer.nextToken()).intValue() * 60 * 1000;
		long seconds = Integer.valueOf(tokenizer.nextToken()).intValue() * 1000;
		int decimal = 0;
		if (tokenizer.hasMoreElements()) {
			decimal = Integer.valueOf(tokenizer.nextToken()).intValue();
		}
		return hours + minutes + seconds + decimal;
	}

	/**
	 * Menu message: Copy the current value to the clipboard.
	 * 
	 * @category menu messages
	 */
	public void copyValue() {
		Toolkit.getDefaultToolkit().getSystemClipboard().setContents(new StringSelection(this.printString()), null);
	}

	/**
	 * Answer a default view.
	 * 
	 * @return jp.co.sra.smalltalk.StView
	 * @see jp.co.sra.smalltalk.StApplicationModel#defaultView()
	 * @category interface opening
	 */
	public StView defaultView() {
		if (GetDefaultViewMode() == VIEW_AWT) {
			return new JunMillisecondViewAwt(this);
		} else {
			return new JunMillisecondViewSwing(this);
		}
	}

	/**
	 * An action for decrementing the number of hour.
	 * 
	 * @category actions
	 */
	public void hourDown() {
		this.milliseconds_(this.milliseconds() - 3600000);
	}

	/**
	 * Answer my current number of hours.
	 *
	 * @return int
	 * @category accessing
	 */
	public int hours() {
		return hours;
	}

	/**
	 * Set the new number of hours.
	 *
	 * @param hourInteger int
	 * @category accessing
	 */
	public void hours_(int hourInteger) {
		if (hourInteger < 0) {
			return;
		}

		long newMilliseconds = (new JunMillisecondModel(hourInteger, this.minutes(), this.seconds(), this.msecs())).milliseconds();
		this.milliseconds_(newMilliseconds);
	}

	/**
	 * An action for incrementing the number of hour.
	 * 
	 * @category actions
	 */
	public void hourUp() {
		this.milliseconds_(this.milliseconds() + 3600000);
	}

	/**
	 * Answer my current millisecond value.
	 *
	 * @return long
	 * @category accessing
	 */
	public long milliseconds() {
		return millisecondValue;
	}

	/**
	 * Set my new millisecond value.
	 *
	 * @param aValue long
	 * @category accessing
	 */
	public void milliseconds_(long aValue) {
		long oldValue = this.milliseconds();
		Long newValueObject = (Long) this.validateBlock().value_value_(new Long(oldValue), new Long(aValue));
		if (newValueObject == null) {
			return;
		}
		long newValue = newValueObject.longValue();

		millisecondValue = newValue;
		msecs = (int) (millisecondValue % 1000);
		newValue = millisecondValue / 1000;
		seconds = (int) (newValue % 60);
		minutes = (int) (newValue / 60 % 60);
		hours = (int) (newValue / 3600);
		if (oldValue != millisecondValue) {
			this.changed_($("value"));
		}
	}

	/**
	 * An action for decrementing the number of minute.
	 * 
	 * @category actions
	 */
	public void minuteDown() {
		this.milliseconds_(this.milliseconds() - 60000);
	}

	/**
	 * Answer my current number of minutes.
	 *
	 * @return int
	 * @category accessing
	 */
	public int minutes() {
		return minutes;
	}

	/**
	 * Set the new number of minutes.
	 *
	 * @param minutesInteger int
	 * @category accessing
	 */
	public void minutes_(int minutesInteger) {
		if (minutesInteger < 0 || 60 < minutesInteger) {
			return;
		}

		long newMilliseconds = (new JunMillisecondModel(this.hours(), minutesInteger, this.seconds(), this.msecs())).milliseconds();
		this.milliseconds_(newMilliseconds);
	}

	/**
	 * An action for incrementing the number of minute.
	 * 
	 * @category actions
	 */
	public void minuteUp() {
		this.milliseconds_(this.milliseconds() + 60000);
	}

	/**
	 * An action for decrementing the number of msec.
	 * 
	 * @category actions
	 */
	public void msecDown() {
		this.milliseconds_(this.milliseconds() - 1);
	}

	/**
	 * Answer my current number of msecs.
	 *
	 * @return int
	 * @category accessing
	 */
	public int msecs() {
		return msecs;
	}

	/**
	 * Set the new number of msecs.
	 *
	 * @param msecsInteger int
	 * @category accessing
	 */
	public void msecs_(int msecsInteger) {
		if (msecsInteger < 0 || 999 < msecsInteger) {
			return;
		}

		long newMilliseconds = (new JunMillisecondModel(this.hours(), this.minutes(), this.seconds(), msecsInteger)).milliseconds();
		this.milliseconds_(newMilliseconds);
	}

	/**
	 * An action for incrementing the number of msec.
	 * 
	 * @category actions
	 */
	public void msecUp() {
		this.milliseconds_(this.milliseconds() + 1);
	}

	/**
	 * Menu message: Set the current time.
	 * 
	 * @category menu messages
	 */
	public void nowValue() {
		Calendar aCalendar = Calendar.getInstance();
		int hour_of_day = aCalendar.get(Calendar.HOUR_OF_DAY);
		int minute = aCalendar.get(Calendar.MINUTE);
		int second = aCalendar.get(Calendar.SECOND);
		int millisecond = aCalendar.get(Calendar.MILLISECOND);
		long nowInMilliseconds = (new JunMillisecondModel(hour_of_day, minute, second, millisecond)).milliseconds();
		this.milliseconds_(nowInMilliseconds);
	}

	/**
	 * Answer my original millisecond value.
	 *
	 * @return long
	 * @category accessing
	 */
	public long original() {
		return originalValue;
	}

	/**
	 * Menu message: Set the original time.
	 * 
	 * @category menu messages
	 */
	public void originalValue() {
		this.milliseconds_(this.original());
	}

	/**
	 * Print my string representation on aWriter.
	 * 
	 * @param aWriter java.io.Writer
	 * @throws IOException if failed.
	 * @see jp.co.sra.smalltalk.StObject#printOn_(java.io.Writer)
	 * @category printing
	 */
	public void printOn_(Writer aWriter) throws IOException {
		aWriter.write(ConvertMillisecondsToString_(this.milliseconds()));
	}

	/**
	 * An action for decrementing the number of second.
	 * 
	 * @category actions
	 */
	public void secondDown() {
		this.milliseconds_(this.milliseconds() - 1000);
	}

	/**
	 * Answer my current number of seconds.
	 *
	 * @return int
	 * @category accessing
	 */
	public int seconds() {
		return seconds;
	}

	/**
	 * Set the new number of seconds.
	 *
	 * @param secondsInteger int
	 * @category accessing
	 */
	public void seconds_(int secondsInteger) {
		if (secondsInteger < 0 || 60 < secondsInteger) {
			return;
		}

		long newMilliseconds = (new JunMillisecondModel(this.hours(), this.minutes(), secondsInteger, this.msecs())).milliseconds();
		this.milliseconds_(newMilliseconds);
	}

	/**
	 * An action for incrementing the number of second.
	 * 
	 * @category actions
	 */
	public void secondUp() {
		this.milliseconds_(this.milliseconds() + 1000);
	}

	/**
	 * Answer my validation block.
	 *
	 * @return jp.co.sra.smalltalk.StBlockClosure
	 * @category accessing
	 */
	public StBlockClosure validateBlock() {
		if (validateBlock == null) {
			validateBlock = new StBlockClosure() {
				public Object value_value_(Object oldValue, Object newValue) {
					return newValue;
				}
			};
		}
		return validateBlock;
	}

	/**
	 * Set my new validation block.
	 *
	 * @param aBlock jp.co.sra.smalltalk.StBlockClosure
	 * @category accessing
	 */
	public void validateBlock_(StBlockClosure aBlock) {
		validateBlock = aBlock;
	}

	/**
	 * Initialize the receiver.
	 *
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		this.initialize_(0);
	}

	/**
	 * Initialize the receiver with the specified milliseconds value.
	 *
	 * @param milliseconds long
	 * @category initialize-release
	 */
	protected void initialize_(long milliseconds) {
		super.initialize();
		originalValue = milliseconds;
		this.milliseconds_(milliseconds);
		validateBlock = null;
		_popupMenu = null;
	}

	/**
	 * Answer my popup menu.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StPopupMenu
	 * @see jp.co.sra.smalltalk.StApplicationModel#_popupMenu()
	 * @category resources
	 */
	public StPopupMenu _popupMenu() {
		if (_popupMenu == null) {
			_popupMenu = new StPopupMenu();
			_popupMenu.add(new StMenuItem(JunSystem.$String("Copy"), new MenuPerformer(this, "copyValue")));
			_popupMenu.addSeparator();
			_popupMenu.add(new StMenuItem(JunSystem.$String("Now"), new MenuPerformer(this, "nowValue")));
			_popupMenu.add(new StMenuItem(JunSystem.$String("Original msec"), new MenuPerformer(this, "originalValue")));
			_popupMenu.addSeparator();
			_popupMenu.add(new StMenuItem(JunSystem.$String("Hour down"), new MenuPerformer(this, "hourDown")));
			_popupMenu.add(new StMenuItem(JunSystem.$String("Hour up"), new MenuPerformer(this, "hourUp")));
			_popupMenu.addSeparator();
			_popupMenu.add(new StMenuItem(JunSystem.$String("Minute down"), new MenuPerformer(this, "minuteDown")));
			_popupMenu.add(new StMenuItem(JunSystem.$String("Minute up"), new MenuPerformer(this, "minuteUp")));
			_popupMenu.addSeparator();
			_popupMenu.add(new StMenuItem(JunSystem.$String("Second down"), new MenuPerformer(this, "secondDown")));
			_popupMenu.add(new StMenuItem(JunSystem.$String("Second up"), new MenuPerformer(this, "secondDown")));
			_popupMenu.addSeparator();
			_popupMenu.add(new StMenuItem(JunSystem.$String("Msec down"), new MenuPerformer(this, "msecDown")));
			_popupMenu.add(new StMenuItem(JunSystem.$String("Msec up"), new MenuPerformer(this, "msecUp")));
		}
		return _popupMenu;
	}

	/**
	 * Answer a window title.
	 * 
	 * @return java.lang.String
	 * @see jp.co.sra.smalltalk.StApplicationModel#windowTitle()
	 * @category interface opening
	 */
	protected String windowTitle() {
		return JunSystem.$String("Milliseconds");
	}
}