package jp.co.sra.jun.goodies.image.streams;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import javax.imageio.IIOException;
import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.imageio.ImageWriter;
import javax.imageio.stream.ImageInputStream;
import javax.imageio.stream.ImageOutputStream;
import jp.co.sra.smalltalk.StImage;

/**
 * JunPngImageStream class
 * 
 *  @author    Mitsuhiro Asada
 *  @created   2000/11/16 (by Mitsuhiro Asada)
 *  @updated   2005/08/15 (by Mitsuhiro Asada)
 *  @version   699 (with StPL8.9) based on JunXXX for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunPngImageStream.java,v 8.11 2008/02/20 06:31:35 nisinaka Exp $
 */
public class JunPngImageStream extends JunImageStream {
	public static final String DefaultImageExtension = "png";

	/**
	 * Constructor with an input stream.
	 * 
	 * @return jp.co.sra.jun.goodies.image.streams.JunImageStream
	 * @param stream java.io.InputStream
	 * @exception java.io.IOException
	 * @see jp.co.sra.jun.goodies.image.streams.JunImageStream#On_(java.io.InputStream)
	 * @category Instance creation
	 */
	public static JunImageStream On_(InputStream stream) throws IOException {
		return On_(_CreateImageStream(), stream);
	}

	/**
	 * Constructor with an output stream.
	 * 
	 * @return jp.co.sra.jun.goodies.image.streams.JunImageStream
	 * @param stream java.io.OutputStream
	 * @exception java.io.IOException
	 * @see jp.co.sra.jun.goodies.image.streams.JunImageStream#On_(java.io.OutputStream)
	 * @category Instance creation
	 */
	public static JunImageStream On_(OutputStream stream) throws IOException {
		return On_(_CreateImageStream(), stream);
	}

	/**
	 * Answer an instance of png image stream.
	 * 
	 * @return jp.co.sra.jun.goodies.image.streams.JunImageStream
	 * @category Instance creation
	 */
	protected static JunImageStream _CreateImageStream() {
		return new JunPngImageStream();
	}

	/**
	 * Read the image from the input stream.
	 * 
	 * @return jp.co.sra.smalltalk.StImage
	 * @exception java.io.IOException
	 * @see jp.co.sra.jun.goodies.image.streams.JunImageStream#nextImage()
	 * @category accessing
	 */
	public StImage nextImage() throws IOException {
		if (inStream == null) {
			throw new IIOException("the inStream is null.");
		}

		ImageReader imageReader = null;
		ImageInputStream imageStream = null;
		try {
			imageReader = (ImageReader) ImageIO.getImageReadersBySuffix(this.defaultImageExtension()).next();
			imageStream = ImageIO.createImageInputStream(inStream);
			imageReader.setInput(imageStream);
			imageObject = new StImage(imageReader.read(0));
		} finally {
			if (imageStream != null) {
				imageStream.flush();
				imageStream.close();
			}
			if (imageReader != null) {
				imageReader.reset();
				imageReader.dispose();
			}
		}
		return imageObject;
	}

	/**
	 * Write the image on the output stream.
	 *
	 * @param newImage jp.co.sra.smalltalk.StImage
	 * @exception java.io.IOException
	 * @see jp.co.sra.jun.goodies.image.streams.JunImageStream#nextPutImage_(jp.co.sra.smalltalk.StImage)
	 * @category accessing
	 */
	public void nextPutImage_(StImage newImage) throws IOException {
		if (outStream == null) {
			throw new IIOException("the outStream is null.");
		}

		ImageWriter imageWriter = null;
		ImageOutputStream imageStream = null;
		try {
			imageWriter = (ImageWriter) ImageIO.getImageWritersBySuffix(this.defaultImageExtension()).next();
			imageStream = ImageIO.createImageOutputStream(outStream);
			imageWriter.setOutput(imageStream);
			imageWriter.write(newImage.image());
		} finally {
			if (imageStream != null) {
				imageStream.flush();
				imageStream.close();
			}
			if (imageWriter != null) {
				imageWriter.reset();
				imageWriter.dispose();
			}
		}
	}

	/**
	 * Answer the receiver's image file extension.
	 *
	 * @return java.lang.String
	 * @see jp.co.sra.jun.goodies.image.streams.JunImageStream#defaultImageExtension()
	 * @category defaults
	 */
	public String defaultImageExtension() {
		return DefaultImageExtension;
	}
}
