package jp.co.sra.jun.goodies.freehand;

import java.util.Vector;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.menu.MenuPerformer;
import jp.co.sra.smalltalk.menu.StMenu;
import jp.co.sra.smalltalk.menu.StMenuBar;
import jp.co.sra.smalltalk.menu.StMenuItem;

import jp.co.sra.jun.goodies.lisp.JunLispCons;
import jp.co.sra.jun.goodies.lisp.JunLispList;

/**
 * JunFreehandNote class
 * 
 *  @author    nisinaka
 *  @created   2002/07/09 (by nisinaka)
 *  @updated   2005/03/03 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun633 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunFreehandNote.java,v 8.12 2008/02/20 06:31:34 nisinaka Exp $
 */
public class JunFreehandNote extends JunFreehandPage {

	protected int pageIndex;
	protected Vector pageCollection;
	protected StMenuBar _menuBar;

	/**
	 * Create an instance of JunFreehandNote and initialize it.
	 * 
	 * @category Instance creation
	 */
	public JunFreehandNote() {
		super();
	}

	/**
	 * Create an instance of JunFreehandNote and initialize it with the JunLispList.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category Instance creation
	 */
	public JunFreehandNote(JunLispList aList) {
		this();
		this.fromLispList_(aList);
	}

	/**
	 * Initialize the instance variables.
	 * 
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		this.flushNote();
		_menuBar = null;
	}

	/**
	 * Flush the note.
	 * 
	 * @category accessing
	 */
	public void flushNote() {
		pageIndex = -1;
		pageCollection = new Vector();
		this.makePage();
	}

	/**
	 * Answer the current collection of the pages.
	 * 
	 * @return java.util.Vector
	 * @category accessing
	 */
	public Vector pageCollection() {
		return pageCollection;
	}

	/**
	 * Answer the current index of the pages.
	 * 
	 * @return int
	 * @category accessing
	 */
	public int pageIndex() {
		return pageIndex;
	}

	/**
	 * Answer my menu bar.
	 * 
	 * @return jp.co.sra.smalltalk.menu.StMenuBar
	 * @see jp.co.sra.smalltalk.StApplicationModel#_menuBar()
	 * @category resources
	 */
	public StMenuBar _menuBar() {
		if (_menuBar == null) {
			_menuBar = new StMenuBar();

			StMenu fileMenu = new StMenu($String("File"));
			fileMenu.add(new StMenuItem($String("New"), new MenuPerformer(this, "newNote")));
			fileMenu.addSeparator();
			fileMenu.add(new StMenuItem($String("Quit"), new MenuPerformer(this, "closeRequest")));
			_menuBar.add(fileMenu);

			StMenu editMenu = new StMenu($String("Edit"));
			editMenu.add(new StMenuItem($String("Back"), new MenuPerformer(this, "goBackward")));
			editMenu.add(new StMenuItem($String("Forward"), new MenuPerformer(this, "goForward")));
			editMenu.addSeparator();
			editMenu.add(new StMenuItem($String("Nib"), new MenuPerformer(this, "changeNibWidthAndColor")));
			editMenu.add(new StMenuItem($String("Clear"), new MenuPerformer(this, "clearPage")));
			_menuBar.add(editMenu);

			StMenu pageMenu = new StMenu($String("Page"));
			pageMenu.add(new StMenuItem($String("Next (New Page)"), new MenuPerformer(this, "goToNextPage")));
			pageMenu.add(new StMenuItem($String("Previous"), new MenuPerformer(this, "goToPreviousPage")));
			pageMenu.addSeparator();
			pageMenu.add(new StMenuItem($String("First"), new MenuPerformer(this, "goToFirstPage")));
			pageMenu.add(new StMenuItem($String("Last"), new MenuPerformer(this, "goToLastPage")));
			_menuBar.add(pageMenu);

			StMenu miscMenu = new StMenu($String("Misc"));
			miscMenu.add(new StMenuItem($String("Viewfinder"), new MenuPerformer(this, "openViewfinder")));
			_menuBar.add(miscMenu);
		}
		return _menuBar;
	}

	/**
	 * Menu message: go to the first page.
	 * 
	 * @category menu messages
	 */
	public void goToFirstPage() {
		this.savePage();
		pageIndex = 0;
		this.loadPage();
		this.changed();
	}

	/**
	 * Menu message: go to the last page.
	 * 
	 * @category menu messages
	 */
	public void goToLastPage() {
		this.savePage();
		pageIndex = this.pageCollection().size() - 1;
		this.loadPage();
		this.changed();
	}

	/**
	 * Menu message: go to the next page.
	 * 
	 * @category menu messages
	 */
	public void goToNextPage() {
		this.savePage();

		if (this.pageIndex() < (this.pageCollection().size() - 1)) {
			pageIndex++;
		} else if (!this.isEmpty()) {
			this.makePage();
		}

		this.loadPage();
		this.changed();
	}

	/**
	 * Menu message: go to the previous page.
	 * 
	 * @category menu messages
	 */
	public void goToPreviousPage() {
		this.savePage();

		if (this.pageIndex() <= 0) {
			return;
		}

		pageIndex--;
		this.loadPage();
		this.changed();
	}

	/**
	 * Menu message: create a new note.
	 * 
	 * @category menu messages
	 */
	public void newNote() {
		(new JunFreehandNote()).open();
	}

	/**
	 * Answer the title string.
	 * 
	 * @return java.lang.String
	 * @category interface opening
	 */
	protected String windowTitle() {
		return $String("Freehand Note");
	}

	/**
	 * Load the current page.
	 * 
	 * @category private
	 */
	protected void loadPage() {
		JunFreehandPage aPage = (JunFreehandPage) this.pageCollection().get(this.pageIndex());
		this.nibColor_nibWidth_nibCollection_locusCollection_scalarCollection_currentIndex_(aPage.nibColor(), aPage.nibWidth(), aPage.nibCollection(), aPage.locusCollection(), aPage.scalarCollection(), aPage.currentIndex());
	}

	/**
	 * Make a page.
	 * 
	 * @category private
	 */
	protected void makePage() {
		this.pageCollection().add(new JunFreehandPage());
		pageIndex++;
	}

	/**
	 * Save the current page.
	 * 
	 * @category private
	 */
	protected void savePage() {
		JunFreehandPage aPage = new JunFreehandPage();
		aPage.nibColor_nibWidth_nibCollection_locusCollection_scalarCollection_currentIndex_(this.nibColor(), this.nibWidth(), this.nibCollection(), this.locusCollection(), this.scalarCollection(), this.currentIndex());
		this.pageCollection().set(this.pageIndex(), aPage);
	}

	/**
	 * Get my attributes from the lisp list.
	 * 
	 * @param aList aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @see jp.co.sra.jun.goodies.freehand.JunFreehandPage#fromLispList(jp.co.sra.jun.goodies.lisp.JunLispList)
	 * @category lisp support
	 */
	public void fromLispList_(JunLispList aList) {
		super.fromLispList_(aList);
		this.pageIndexFromLispList_(aList);
		this.pageCollectionFromLispList_(aList);
	}

	/**
	 * Get my page index from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void pageIndexFromLispList_(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("pageIndex"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		pageIndex = ((Number) list.tail()).intValue();
	}

	/**
	 * Get my page collection from the lisp list.
	 * 
	 * @param aList jp.co.sra.jun.goodies.lisp.JunLispList
	 * @category lisp support
	 */
	protected void pageCollectionFromLispList_(JunLispList aList) {
		JunLispCons list = (JunLispCons) aList.detect_ifNone_(new StBlockClosure() {
			public Object value_(Object car) {
				return new Boolean(car instanceof JunLispCons && ((JunLispCons) car).head() == $("pageCollection"));
			}
		}, new StBlockClosure());
		if (list == null) {
			return;
		}

		Object[] anArray = ((JunLispCons) list.tail()).asArray();
		Vector aVector = new Vector(anArray.length);
		for (int i = 0; i < anArray.length; i++) {
			aVector.add(new JunFreehandPage((JunLispCons) anArray[i]));
		}

		pageCollection = aVector;
	}

	/**
	 * Convert the receiver to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @see jp.co.sra.jun.system.framework.JunApplicationModel#toLispList()
	 * @see jp.co.sra.jun.goodies.freehand.JunFreehandPage#toLispList()
	 * @category lisp support
	 */
	public JunLispCons toLispList() {
		JunLispCons list = super.toLispList();
		list.add_(this.pageIndexToLispList());
		list.add_(this.pageCollectionToLispList());
		return list;
	}

	/**
	 * Convert the receiver's page index to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons pageIndexToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("pageIndex"));
		list.tail_(new Integer(this.pageIndex()));
		return list;
	}

	/**
	 * Convert the receiver's nib collection to a lisp list.
	 * 
	 * @return jp.co.sra.jun.goodies.lisp.JunLispCons
	 * @category lisp support
	 */
	protected JunLispCons pageCollectionToLispList() {
		JunLispCons list = this.lispCons();
		list.head_($("pageCollection"));
		JunLispList[] pageList = new JunLispList[this.pageCollection().size()];
		for (int i = 0; i < pageList.length; i++) {
			pageList[i] = ((JunFreehandPage) this.pageCollection().get(i)).toLispList();
		}
		list.tail_(JunLispList.List_(pageList));
		return list;
	}

}