package jp.co.sra.jun.goodies.files;

import java.awt.Choice;
import java.awt.Font;
import java.awt.Label;
import java.awt.List;
import java.awt.TextField;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.io.File;

import jp.co.sra.smalltalk.StBlockClosure;
import jp.co.sra.smalltalk.WindowSpecLayout;

import jp.co.sra.jun.system.framework.JunAbstractViewPanel;
import jp.co.sra.jun.system.support.JunSystem;

/**
 * JunFileViewAwt class
 * 
 *  @author    nisinaka
 *  @created   2003/03/05 (by nisinaka)
 *  @updated   2004/09/21 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun433 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunFileViewAwt.java,v 8.10 2008/02/20 06:31:33 nisinaka Exp $
 */
public class JunFileViewAwt extends JunAbstractViewPanel implements JunFileView {

	/**
	 * A BoldLabel is a Label with BOLD font.
	 */
	class BoldLabel extends Label {
		public BoldLabel(String text) {
			super(text);
		}

		public void addNotify() {
			super.addNotify();
			this.setFont(this.getFont().deriveFont(Font.BOLD));
		}
	}

	protected Label fileLabel;
	protected TextField fileName;
	protected List fileList;
	protected Label fileTypeLabel;
	protected Choice fileType;
	protected Label directoryLabel;
	protected Label currentDirectoryLabel;
	protected List directoryList;
	protected Label driveLabel;
	protected Choice currentDrive;

	/**
	 * Create a new instance of JunFileViewAwt and initialize it.
	 *
	 * @param aFileModel jp.co.sra.jun.goodies.files.JunFileModel
	 * @category Instance creation
	 */
	public JunFileViewAwt(JunFileModel aFileModel) {
		super(aFileModel);
	}

	/**
	 * Build this component.
	 * 
	 * @category initialize-release
	 */
	protected void buildComponent() {
		this.setLayout(new WindowSpecLayout());
		this.add(this.fileLabel(), WindowSpecLayout.Position(0f, 8, 0f, 4));
		this.add(this.fileName(), WindowSpecLayout.Position(0f, 8, 0f, 32, 0.5f, -4, 0f, 60));
		this.add(this.fileList(), WindowSpecLayout.Position(0f, 8, 0f, 68, 0.5f, -4, 1f, -72));
		this.add(this.fileTypeLabel(), WindowSpecLayout.Position(0f, 8, 1f, -62));
		this.add(this.fileType(), WindowSpecLayout.Position(0f, 8, 1f, -36, 0.5f, -4, 1f, -8));
		this.add(this.directoryLabel(), WindowSpecLayout.Position(0.5f, 4, 0f, 4));
		this.add(this.currentDirectoryLabel(), WindowSpecLayout.Position(0.5f, 4, 0f, 32, 1f, -8, 0f, 60));
		this.add(this.directoryList(), WindowSpecLayout.Position(0.5f, 4, 0f, 68, 1f, -8, 1f, -72));
		this.add(this.driveLabel(), WindowSpecLayout.Position(0.5f, 4, 1f, -62));
		this.add(this.currentDrive(), WindowSpecLayout.Position(0.5f, 4, 1f, -36, 1f, -8, 1f, -8));
		this.setSize(500, 300);

		// postBuildWith:
		this.getFileModel().currentDirectory();
		this.getFileModel().updateCurrentDirectoryLabel();
		this.getFileModel().updateDirectoryAndFileList();
		this.getFileModel().updateCurrentDrive();
	}

	/**
	 * Answer my model as JunFileModel.
	 *
	 * @return jp.co.sra.jun.goodies.files.JunFileModel
	 * @see jp.co.sra.jun.goodies.files.JunFileView#getFileModel()
	 * @category model accessing
	 */
	public JunFileModel getFileModel() {
		return (JunFileModel) this.model();
	}

	/**
	 * Answer the current directory label.
	 *
	 * @return java.awt.Label
	 * @category user interface
	 */
	protected Label currentDirectoryLabel() {
		if (currentDirectoryLabel == null) {
			currentDirectoryLabel = new Label();

			this.getFileModel().currentDirectoryLabel().compute_(new StBlockClosure() {
				public Object value_(Object newValue) {
					currentDirectoryLabel.setText((String) newValue);
					return null;
				}
			});
		}
		return currentDirectoryLabel;
	}

	/**
	 * Answer the current drive choice.
	 *
	 * @return java.awt.Choice
	 * @category user interface
	 */
	protected Choice currentDrive() {
		if (currentDrive == null) {
			currentDrive = new Choice();
			File[] drives = this.getFileModel().drives();
			for (int i = 0; i < drives.length; i++) {
				currentDrive.addItem(drives[i].getPath().toUpperCase());
			}
			currentDrive.addItem(this.getFileModel().DefaultDirectoryItem);
			currentDrive.select(((File) this.getFileModel().currentDrive().value()).getPath().toUpperCase());

			currentDrive.addItemListener(new ItemListener() {
				public void itemStateChanged(ItemEvent e) {
					if (e.getStateChange() == ItemEvent.SELECTED) {
						getFileModel().currentDriveChanged(currentDrive.getSelectedItem());
					}
				}
			});

			this.getFileModel().currentDrive().compute_(new StBlockClosure() {
				public Object value_(Object newValue) {
					String newItem = ((File) newValue).getPath().toUpperCase();
					if (newItem.equals(currentDrive.getSelectedItem()) == false) {
						currentDrive.select(newItem);
					}
					return null;
				}
			});
		}
		return currentDrive;
	}

	/**
	 * Answer the directory label.
	 *
	 * @return java.awt.Label
	 * @category user interface
	 */
	protected Label directoryLabel() {
		if (directoryLabel == null) {
			directoryLabel = new JunFileViewAwt.BoldLabel(JunSystem.$String("Directory") + ":");
		}
		return directoryLabel;
	}

	/**
	 * Answer the directory list.
	 *
	 * @return java.awt.List
	 * @category user interface
	 */
	protected List directoryList() {
		if (directoryList == null) {
			directoryList = new List();

			directoryList.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					getFileModel().directoryListDoubleClicked(directoryList.getSelectedIndex(), e.getActionCommand());
				}
			});

			this.getFileModel().directoryList().compute_(new StBlockClosure() {
				public Object value_(Object newValue) {
					Object[] directories = (Object[]) newValue;
					int level = getFileModel().currentDirectoryLevel();
					String indent = "";
					directoryList.removeAll();
					for (int i = 0; i < directories.length; i++) {
						directoryList.add(indent + directories[i]);
						if (i < level) {
							indent += "  ";
						}
					}
					return null;
				}
			});
		}
		return directoryList;
	}

	/**
	 * Answer the drive label.
	 *
	 * @return java.awt.Label
	 * @category user interface
	 */
	protected Label driveLabel() {
		if (driveLabel == null) {
			driveLabel = new JunFileViewAwt.BoldLabel(JunSystem.$String("Drive") + ":");
		}
		return driveLabel;
	}

	/**
	 * Answer the file label.
	 *
	 * @return java.awt.Label
	 * @category user interface
	 */
	protected Label fileLabel() {
		if (fileLabel == null) {
			fileLabel = new JunFileViewAwt.BoldLabel(JunSystem.$String("File") + ":");
		}
		return fileLabel;
	}

	/**
	 * Answer the file list.
	 *
	 * @return java.awt.List
	 * @category user interface
	 */
	protected List fileList() {
		if (fileList == null) {
			fileList = new List();

			fileList.addItemListener(new ItemListener() {
				public void itemStateChanged(ItemEvent e) {
					if (fileList.getSelectedIndex() < 0) {
						return;
					}
					getFileModel().fileListChanged(fileList.getSelectedItem());
				}
			});

			this.getFileModel().fileList().compute_(new StBlockClosure() {
				public Object value_(Object newValue) {
					Object[] files = (Object[]) newValue;
					fileList.removeAll();
					for (int i = 0; i < files.length; i++) {
						fileList.add((String) files[i]);
					}
					return null;
				}
			});

			this.getFileModel().fileName().compute_(new StBlockClosure() {
				public Object value_(Object newValue) {
					String[] items = fileList.getItems();
					for (int i = 0; i < items.length; i++) {
						if (items[i].equals(newValue)) {
							fileList.select(i);
							return null;
						}
					}
					fileList.deselect(fileList.getSelectedIndex());
					return null;
				}
			});
		}
		return fileList;
	}

	/**
	 * Ansewr the file name text field.
	 *
	 * @return java.awt.TextField
	 * @category user interface
	 */
	protected TextField fileName() {
		if (fileName == null) {
			fileName = new TextField();
			fileName.setText((String) this.getFileModel().fileName().value());
			fileName.selectAll();

			fileName.addActionListener(new ActionListener() {
				public void actionPerformed(ActionEvent e) {
					if (fileName.getText().equals(getFileModel().fileName()) == false) {
						getFileModel().fileNameChanged(fileName.getText());
					}
				}
			});

			fileName.addFocusListener(new FocusAdapter() {
				public void focusLost(FocusEvent e) {
					if (fileName.getText().equals(getFileModel().fileName()) == false) {
						getFileModel().fileNameChanged(fileName.getText());
					}
				}
			});

			this.getFileModel().fileName().compute_(new StBlockClosure() {
				public Object value_(Object newValue) {
					fileName.setText((String) newValue);
					return null;
				}
			});

		}
		return fileName;
	}

	/**
	 * Answer the file type choice.
	 *
	 * @return java.awt.Choice
	 * @category user interface
	 */
	protected Choice fileType() {
		if (fileType == null) {
			fileType = new Choice();
			JunFileModel.FileType[] fileTypes = this.getFileModel().fileTypes();
			for (int i = 0; i < fileTypes.length; i++) {
				fileType.addItem(fileTypes[i].name());
			}
			fileType.select(((JunFileModel.FileType) this.getFileModel().fileType().value()).name());

			fileType.addItemListener(new ItemListener() {
				public void itemStateChanged(ItemEvent e) {
					if (e.getStateChange() == ItemEvent.SELECTED) {
						getFileModel().fileTypeChanged(fileType.getSelectedIndex());
					}
				}
			});

			this.getFileModel().fileType().compute_(new StBlockClosure() {
				public Object value_(Object newValue) {
					String newItem = ((JunFileModel.FileType) newValue).name();
					if (newItem.equals(fileType.getSelectedItem()) == false) {
						fileType.select(newItem);
					}
					return null;
				}
			});
		}
		return fileType;
	}

	/**
	 * Answer the file type label.
	 *
	 * @return java.awt.Label
	 * @category user interface
	 */
	protected Label fileTypeLabel() {
		if (fileTypeLabel == null) {
			fileTypeLabel = new JunFileViewAwt.BoldLabel(JunSystem.$String("File Type") + ":");
		}
		return fileTypeLabel;
	}

}
