package jp.co.sra.jun.goodies.display;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Insets;
import java.awt.Point;

import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;

import jp.co.sra.smalltalk.DependentEvent;
import jp.co.sra.smalltalk.StController;
import jp.co.sra.smalltalk.StDisplayable;
import jp.co.sra.smalltalk.StSymbol;
import jp.co.sra.smalltalk.StViewJPanel;
import jp.co.sra.smalltalk.WindowSpecLayout;

import jp.co.sra.jun.system.framework.JunAbstractViewJPanel;

/**
 * JunDisplayViewSwing class
 * 
 *  @author    nisinaka
 *  @created   1998/11/16 (by nisinaka)
 *  @updated   1999/11/11 (by MATSUDA Ryouichi)
 *  @updated   2002/01/09 (by nisinaka)
 *  @updated   2002/11/21 (by nisinaka)
 *  @updated   2003/03/19 (by nisinaka)
 *  @updated   2004/09/21 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun519 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunDisplayViewSwing.java,v 8.17 2008/02/20 06:31:22 nisinaka Exp $
 */
public class JunDisplayViewSwing extends JunAbstractViewJPanel implements JunDisplayView {

	protected StViewJPanel canvas;
	protected JScrollPane scrollPane;

	/**
	 * Create a new instance of JunDisplayViewSwing and initialize it.
	 * 
	 * @param aDisplayModel jp.co.sra.jun.goodies.display.JunDisplayM
	 * @category Instance creation
	 */
	public JunDisplayViewSwing(JunDisplayModel aDisplayModel) {
		super(aDisplayModel);
	}

	/**
	 * Create a new instance of JunDisplayViewSwing and initialize it.
	 *
	 * @param aDisplayModel jp.co.sra.jun.goodies.display.JunDisplayModel
	 * @param aDimension java.awt.Dimension
	 * @category Instance creation
	 */
	public JunDisplayViewSwing(JunDisplayModel aDisplayModel, Dimension aDimension) {
		this(aDisplayModel);
		this.setPreferredSize(aDimension);
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.smalltalk.StViewJPanel#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		canvas = null;
		scrollPane = null;
	}

	/**
	 * Build this component.
	 * 
	 * @see jp.co.sra.smalltalk.StViewJPanel#buildComponent()
	 * @category initialize-release
	 */
	protected void buildComponent() {
		this.setLayout(new WindowSpecLayout());
		this.add(this.scrollPane(), WindowSpecLayout.Position(0f, 0, 0f, 0, 1f, 0, 1f, 0));
	}

	/**
	 * Answer my visual object.
	 * 
	 * @return jp.co.sra.smalltalk.StDisplayable
	 * @category accessing
	 */
	public StDisplayable visual() {
		return this.getDisplayModel().visual();
	}

	/**
	 * Gets the preferred size of this component.
	 *
	 * @return A dimension object indicating this component's preferred size.
	 * @see #getMinimumSize
	 * @see java.awt.LayoutManager
	 * @see javax.swing.JComponent#getPreferredSize()
	 * @category bounds accessing
	 */
	public Dimension getPreferredSize() {
		Point location = this.scrollPane().getLocation();
		Dimension size = this.scrollPane().getPreferredSize();
		return new Dimension(size.width + location.x * 2, size.height + location.y * 2);
	}

	/**
	 * Answer my viewport size.
	 * 
	 * @return java.awt.Dimension
	 * @see jp.co.sra.jun.goodies.display.JunDisplayView#_getViewportSize()
	 * @category bounds accessing
	 */
	public Dimension _getViewportSize() {
		Insets insets = this.scrollPane().getInsets();
		return new Dimension(this.scrollPane().getWidth() - insets.top - insets.bottom, this.scrollPane().getHeight() - insets.left - insets.right);
	}

	/**
	 * Answer my model as JunDisplayModel.
	 * 
	 * @return jp.co.sra.jun.goodies.display.JunDisplayModel
	 * @see jp.co.sra.jun.goodies.display.JunDisplayView#getDisplayModel()
	 * @category model accessing
	 */
	public JunDisplayModel getDisplayModel() {
		return (JunDisplayModel) this.model();
	}

	/**
	 * Answer a default controller.
	 * 
	 * @return jp.co.sra.smalltalk.StController
	 * @see jp.co.sra.smalltalk.StViewJPanel#defaultController()
	 * @category controller accessing
	 */
	protected StController defaultController() {
		return new JunDisplayController();
	}

	/**
	 * Answer my canvas.
	 *
	 * @return jp.co.sra.smalltalk.StViewJPanel
	 * @category user interface
	 */
	public StViewJPanel canvas() {
		if (canvas == null) {
			canvas = new StViewJPanel() {
				public Dimension getPreferredSize() {
					if (getDisplayModel()._isScrollable()) {
						return visual().bounds().getSize();
					}
					return super.getPreferredSize();
				}

				public void paint(Graphics g) {
					super.paint(g);
					g.setColor(Color.white);
					Dimension visualSize = visual().bounds().getSize();
					int x = (this.getWidth() - visualSize.width) / 2;
					int y = (this.getHeight() - visualSize.height) / 2;
					visual().displayOn_at_(g, new Point(x, y));
				}
			};
		}

		return canvas;
	}

	/**
	 * Answer my canvas as Component.
	 * 
	 * @return java.awt.Component
	 * @see jp.co.sra.jun.goodies.display.JunDisplayView#canvasComponent()
	 * @category user interface
	 */
	public Component canvasComponent() {
		return this.canvas();
	}

	/**
	 * Answer my scroll pane, which already contains the canvas.
	 *
	 * @return javax.swing.JScrollPane
	 * @category user interface
	 */
	protected JScrollPane scrollPane() {
		if (scrollPane == null) {
			scrollPane = new JScrollPane(this.canvas()) {
				public Dimension getPreferredSize() {
					Dimension size = visual().bounds().getSize();
					Insets insets = this.getInsets();
					return new Dimension(size.width + insets.left + insets.right, size.height + insets.top + insets.bottom);
				}

				public void setBounds(int x, int y, int width, int height) {
					super.setBounds(x, y, width, height);

					if (getDisplayModel()._isScrollable() == false) {
						Insets insets = this.getInsets();
						canvas().setSize(width - insets.left - insets.right, height - insets.top - insets.bottom);
					}
				}
			};
		}

		return scrollPane;
	}

	/**
	 * Do the layout on the scroll pane.
	 * 
	 * @see jp.co.sra.smalltalk.StViewJPanel#addNotify()
	 * @category actions
	 */
	public void addNotify() {
		super.addNotify();

		this.scrollPane().doLayout();
	}

	/**
	 * Show the popup menu at the specified point on the view.
	 * 
	 * @param x int
	 * @param y int
	 * @see jp.co.sra.smalltalk.StView#_showPopupMenu(int, int)
	 * @category popup menu
	 */
	public void _showPopupMenu(int x, int y) {
		JPopupMenu popupMenu = this.popupMenuView();
		if (popupMenu != null) {
			popupMenu.show(this.canvas(), x, y);
		}
	}

	/**
	 * Update the receiver according to the change notification from the model.
	 * 
	 * @param e jp.co.sra.smalltalk.DependentEvent
	 * @see jp.co.sra.smalltalk.StViewJPanel#update_(jp.co.sra.smalltalk.DependentEvent)
	 * @category updating
	 */
	public void update_(DependentEvent e) {
		if (this.topComponent().isShowing() == false) {
			return;
		}

		StSymbol aspect = e.getAspect();
		if (aspect == $("redisplay")) {
			this.canvas().repaint();
		} else if (aspect == $("visual")) {
			//Update client's preferred size because the area taken up
			//by the graphics has gotten larger or smaller (if cleared)
			this.canvas().setPreferredSize(this.canvas().getPreferredSize());
			//This lets the scroll pane know to update itself
			//and its scroll bars.
			this.canvas().revalidate();
			// Finally repaint.
			this.canvas().repaint();
		} else {
			super.update_(e);
		}
	}

}
