package jp.co.sra.jun.goodies.bulletinboard;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Panel;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.image.BufferedImage;
import jp.co.sra.smalltalk.StApplicationModel;
import jp.co.sra.smalltalk.StController;
import jp.co.sra.jun.system.framework.JunAbstractViewCanvas;

/**
 * JunBulletinBoardViewAwt class
 * 
 *  @author    nisinaka
 *  @created   2002/11/27 (by nisinaka)
 *  @updated   2004/09/21 (by nisinaka)
 *  @version   699 (with StPL8.9) based on Jun629 for Smalltalk
 *  @copyright 1999-2008 SRA (Software Research Associates, Inc.)
 *  @copyright 1999-2005 Information-technology Promotion Agency, Japan (IPA)
 *  @copyright 2001-2008 SRA/KTL (SRA Key Technology Laboratory, Inc.)
 * 
 * $Id: JunBulletinBoardViewAwt.java,v 8.13 2008/02/20 06:31:11 nisinaka Exp $
 */
public class JunBulletinBoardViewAwt extends JunAbstractViewCanvas implements JunBulletinBoardView {
	protected int displaySpeed;
	protected Point displayPoint;

	/**
	 * Create a new instance of <code>JunBulletinBoardViewAwt</code> and initialize it.
	 *
	 * @param aBulletinBoard jp.co.sra.jun.goodies.bulletinboard.JunBulletinBoard
	 * @category Instance creation
	 */
	public JunBulletinBoardViewAwt(JunBulletinBoard aBulletinBoard) {
		super(aBulletinBoard);
	}

	/**
	 * Create a new instance of JunBulletinBoardViewAwt on the model and open it.
	 *
	 * @return java.awt.Frame
	 * @param aModel jp.co.sra.jun.goodies.bulletinboard.JunBulletinBoard
	 * @category Instance creation
	 */
	public static Frame OpenOn_(JunBulletinBoard aModel) {
		final JunBulletinBoardViewAwt aView = new JunBulletinBoardViewAwt(aModel) {
			public Dimension getMinimumSize() {
				return DefaultWindowSize;
			}
		};

		Panel aPanel = StApplicationModel._PanelWithBorder();
		aPanel.setLayout(new BorderLayout());
		aPanel.add(aView, BorderLayout.CENTER);

		Frame aFrame = new Frame(DefaultWindowLabel);
		aFrame.setLayout(new BorderLayout());
		aFrame.add(aPanel, BorderLayout.CENTER);
		aFrame.addWindowListener(new WindowAdapter() {
			public void windowOpened(WindowEvent e) {
				((JunBulletinBoardController) aView.controller()).resume();
			}

			public void windowClosing(WindowEvent e) {
				((JunBulletinBoardController) aView.controller()).terminate();
				e.getWindow().dispose();
			}
		});
		aFrame.pack();
		StApplicationModel._ShowAtMousePoint(aFrame);

		return aFrame;
	}

	/**
	 * Display the current content.
	 * 
	 * @see jp.co.sra.jun.goodies.bulletinboard.JunBulletinBoardView#display()
	 * @category displaying
	 */
	public void display() {
		Graphics aGraphics = this.getGraphics();
		if (aGraphics == null) {
			return;
		}

		try {
			this.displayOn_(aGraphics);
		} finally {
			aGraphics.dispose();
		}
	}

	/**
	 * Paints this component.
	 * 
	 * @param aGraphics java.awt.Graphics
	 * @see jp.co.sra.smalltalk.StViewCanvas#displayOn_(java.awt.Graphics)
	 * @category displaying
	 */
	public void displayOn_(Graphics aGraphics) {
		if (this.isShowing() == false) {
			return;
		}

		BufferedImage anImage = this.getBulletinBoard().imageLocation_width_(this.displayPoint().x, this.getBounds().width);
		int y = (this.getBounds().height - anImage.getHeight()) / 2;
		aGraphics.drawImage(anImage, 0, y, null);

		Thread.yield();
	}

	/**
	 * Answer my current display point.
	 *
	 * @return java.awt.Point
	 * @category accessing
	 */
	public Point displayPoint() {
		int modelWidth = this.getBulletinBoard().width();
		int startLocation = this.getBounds().width * -1;
		if (displayPoint == null) {
			displayPoint = new Point(startLocation, 0);
		} else {
			displayPoint.x += this.displaySpeed();
		}
		if (displayPoint.x > modelWidth) {
			displayPoint = new Point(startLocation, 0);
		}
		return displayPoint;
	}

	/**
	 * Answer the current display speed.
	 *
	 * @return int
	 * @category accessing
	 */
	public int displaySpeed() {
		return displaySpeed;
	}

	/**
	 * Set the new display speed.
	 *
	 * @param newDisplaySpeed int
	 * @category accessing
	 */
	public void displaySpeed_(int newDisplaySpeed) {
		displaySpeed = Math.max(1, Math.min(newDisplaySpeed, this.getBulletinBoard().width()));
	}

	/**
	 * Answer my model as a JunBulletinBoard.
	 *
	 * @return jp.co.sra.jun.goodies.bulletinboard.JunBulletinBoard
	 * @category model accessing
	 */
	public JunBulletinBoard getBulletinBoard() {
		return (JunBulletinBoard) this.model();
	}

	/**
	 * Set the bounds of the view.
	 * Also needs to change the font.
	 *
	 * @param x The new <i>x</i>-coordinate of this component.
	 * @param y The new <i>y</i>-coordinate of this component.
	 * @param width The new <code>width</code> of this component.
	 * @param height The new <code>height</code> of this component.
	 * @see java.awt.Component#setBounds(int, int, int, int)
	 * @see jp.co.sra.jun.system.framework.JunAbstractViewCanvas#setBounds(int, int, int, int)
	 * @category bounds accessing
	 */
	public void setBounds(int x, int y, int width, int height) {
		super.setBounds(x, y, width, height);

		JunBulletinBoard aBulletinBoard = this.getBulletinBoard();
		Rectangle oldModelBounds = aBulletinBoard.bounds();
		aBulletinBoard.fontHeight_(Math.round(height * 0.8f));
		Rectangle newModelBounds = aBulletinBoard.bounds();
		if (displayPoint != null) {
			displayPoint.x = displayPoint.x * newModelBounds.width / oldModelBounds.width;
		}

		// this.displaySpeed_(displaySpeed * newModelBounds.width / oldModelBounds.width);
	}

	/**
	 * Build this component.
	 * 
	 * @see jp.co.sra.smalltalk.StViewCanvas#buildComponent()
	 * @category initialize-release
	 */
	protected void buildComponent() {
		this.setSize(DefaultWindowSize.width, JunBulletinBoard.DefaultBulletinHeight);
		this.setBackground(this.getBulletinBoard().backgroundColor());
	}

	/**
	 * Answer my default controller. Subclasses may override this.
	 * 
	 * @return jp.co.sra.smalltalk.StController
	 * @see jp.co.sra.smalltalk.StViewCanvas#defaultController()
	 * @category defaults
	 */
	protected StController defaultController() {
		return new JunBulletinBoardController();
	}

	/**
	 * Initialize the receiver.
	 * 
	 * @see jp.co.sra.jun.system.framework.JunAbstractViewCanvas#initialize()
	 * @category initialize-release
	 */
	protected void initialize() {
		super.initialize();

		displaySpeed = 2;
		displayPoint = null;
	}
}
